<?php

namespace App\Api\Http\Controllers\Api\V1\Lead;

use App\Api\Http\Requests\Lead\UnattendedLeadRequest;
use App\Api\Http\Resources\Leads\LeadCollection;
use App\Http\Controllers\Controller;
use App\Lead;
use App\Location;
use App\Task;
use App\User;
use Illuminate\Support\Facades\DB;
use Spatie\QueryBuilder\QueryBuilder;
use Spatie\QueryBuilder\AllowedFilter;

class UnattendedLeadsController extends Controller
{
    public function index(): LeadCollection
    {
        $userRoles = auth()
            ->user()
            ->userRoles();
        $all_reporting_users_id_array = [];
        $authUser = auth()->user();
        if ($authUser->hasRole('admin') || $authUser->hasRole('super-admin') || $authUser->hasRole('ceo') || $authUser->hasRole('head-sales')) {
            $all_reporting_users_id_array = User::with('roles')->whereHas('roles', function ($q) {
                $q->where('role_id', 8);
            })->where('status', 1)->get()->pluck('id')->toArray();
        } elseif ($userRoles->contains('slug', 'sales-head')) {
            $all_reporting_users_id_array = User::whereHas('roles', function ($q) {
                $q->where('role_id', 8);
            })->where('branch_id', auth()->user()->branch_id)
            ->where('status', 1)
            ->pluck('id')
            ->toArray();
        } elseif ($authUser->hasRole('executive')) {
            $all_reporting_users_id_array = auth()->user()->id;
        } else {
            // $all_reporting_users_id_array = $authUser->getAllReportingUsersIds();
            $all_reporting_users_id_array = array_merge($authUser->getAllReportingUsersIds(), [auth()->user()->id]);
        }

        $leads = QueryBuilder::for(Lead::class)
            ->with([
                'getLeadCategory',
                'getLeadStage',
                'getUnit',
                'getSource',
                'getProjectType',
                'getPreferredProject',
                'getBudget',
                'getLeadCategory',
            ])
            ->where('is_ceo_verified', '1')
            ->where('is_attended', 0)
            ->where('runo', 0)
            ->whereHas('getSource', function ($query){
                $query->where('status', 0);
            })

            ->when(is_array($all_reporting_users_id_array), function ($query) use ($all_reporting_users_id_array) {
                return $query->whereIn('attended_by', $all_reporting_users_id_array);
            }, function ($query) use ($all_reporting_users_id_array) {
                return $query->where('attended_by', $all_reporting_users_id_array);
            })
            ->orderByDesc('allocation_date')
            ->allowedFilters([
                AllowedFilter::exact('id'),
                // AllowedFilter::exact('created_by'),
                AllowedFilter::scope('sales_person'),
                AllowedFilter::exact('source'),
            ])
            ->jsonPaginate()
            ->appends(request()->query());

        return (new LeadCollection($leads))->additional(['message' => 'Unattended Leads list!']);
    }

    public function update(UnattendedLeadRequest $request, Lead $lead): ?\Illuminate\Http\JsonResponse
    {
        DB::beginTransaction();
        if ($request->location_id != 15 && $request->location_id != 16 && $request->location_id != '') {
            $locat = Location::find($request->location_id);
            $location = $locat->name;
        } else {
            $location = $request->location_name;
        }
        if ($request->create_task_id == 2 || $request->create_task_id == 3) {
            $category = 1;
        } elseif ($request->create_task_id == 4) {
            $category = 3;
        } elseif ($request->create_task_id != '') {
            $category = 4;
        } else {
            $category = null;
        }
        try {
            $lead->lead_category = $category;
            $lead->lead_stage_id = $request->lead_stage_id;
            $lead->comment = $request->comment;
            $lead->is_attended = 1;
            $lead->location_id = $request->location_id ?? '';
            $lead->location = $location ?? '';
            $lead->budget = $request->budget;
            $lead->project_type = $request->project_type;
            $lead->preferred_project = $request->preferred_project;
            $lead->profession = $request->profession;
            $lead->investment_purpose = $request->investment_purpose;
            $lead->preferred_investment_location = $location;
            $lead->residential_status = $request->residential_status;
            $lead->job = $request->job;
            $lead->save();

            //New-Task-Creation
            if ($request->create_task_id != 17) {
                $task = new Task();
                $task->lead_id = $lead->id;
                $task->create_task_id = $request->create_task_id;
                $task->followup_by = auth()->user()->id;
                $task->lead_category = $category;
                $task->schedule_date = $request->reminder ? date('Y-m-d', strtotime($request->reminder)) : '';
                $task->reminder = $request->reminder ? date('Y-m-d', strtotime($request->reminder)) : '';
                $task->created_by = auth()->user()->id;
                $task->remarks = $request->comment;
                $task->lead_stage = $request->lead_stage_id;

                $task->save();
            }

            DB::commit();

            return $this->success('Status Updated');
        } catch (\Throwable $th) {
            info($th);
            DB::rollBack();

            return $this->error('Something went wrong! Please try again', 400);
        }
    }
}
