<?php

namespace App\Api\Http\Controllers\Api\V2\Receipts;

use Carbon\Carbon;
use App\Api\Http\Requests\Units\UnitsRequest;
use App\Api\Http\Resources\Bank\BankCollection;
use App\Api\Http\Resources\Units\UnitsCollection;
use App\Bank;
use App\Http\Controllers\Controller;
use App\Http\Requests\Receipt\ReceiptRequest;
use App\Helpers\ActivityLog;
use App\PriceList;
use App\Receipt;
use App\SalesActivity;
use App\User;
use Auth;
use DB;
use Illuminate\Http\Request;

class ReceiptsController extends Controller
{
    public function store(ReceiptRequest $request)
    {
        DB::beginTransaction();

        try {
            $receipts = new Receipt();
            $receipts->client_id = $request->client_id;
            $receipts->project_id = $request->project_id;
            $receipts->unit_id = $request->unit_id ?? 0;
            $receipts->user_id = auth()->user()->id;
            $receipts->sale_id = $request->sale_activity_id;
            $receipts->include_co_applicant = $request->include_co_applicant ?? 0;
            $receipts->currency = 'INR';
            $receipts->created_by = auth()->user()->id;
            $receipts->receipt_date = ($request->receipt_date) ? Carbon::parse($request->receipt_date)->format('Y-m-d') : null;
            $receipts->mode_of_payment = $request->mode_of_payment;
            $receipts->amount = $request->amount;
            if ($request->bank == 'Other') {
                $receipts->bank = $request->other_bank;
                $newBank = new Bank();
                $newBank->title = $request->other_bank;
                $newBank->save();
            } else {
                $receipts->bank = $request->bank;
            }
            $receipts->remarks = $request->remarks ? $request->remarks : '';

            if ('Cheque' == $request->mode_of_payment) {
                $receipts->cheque_no = $request->cheque_no;
                $receipts->receipt_tracking_no = $request->cheque_no;
                $receipts->cheque_date = ($request->cheque_date) ? Carbon::parse($request->cheque_date)->format('Y-m-d') : null;
                $receipts->cheque_submit_date = ($request->cheque_submit_date) ? Carbon::parse($request->cheque_submit_date)->format('Y-m-d') : null;
            }

            if ('EFT' == $request->mode_of_payment) {
                $receipts->fund_transfer_date = ($request->fund_transfer_date) ? Carbon::parse($request->fund_transfer_date)->format('Y-m-d') : null;
                $receipts->transaction_ref_no = $request->transaction_ref_no;
                $receipts->receipt_tracking_no = $request->transaction_ref_no;
            }

            if ('DD' == $request->mode_of_payment) {
                $receipts->dd_no = $request->dd_no;
                $receipts->receipt_tracking_no = $request->dd_no;
                $receipts->dd_date = ($request->dd_date) ? Carbon::parse($request->dd_date)->format('Y-m-d') : null;
            }

            if ('UPI' == $request->mode_of_payment) {
                $receipts->upi_reference_no = $request->transaction_ref_no;
                $receipts->upi_paid_date = ($request->upi_paid_date) ? Carbon::parse($request->upi_paid_date)->format('Y-m-d') : null;
                $receipts->upi_provider = $request->upi_provider;

                $receipts->receipt_tracking_no = $request->transaction_ref_no;
            }

            if ($request->hasfile('payment_file')) {
                $receipts->payment_file = $request->file('payment_file')->store('payment', 'uploads');
            }

            if ($request->sales_owner) {
                $sale = SalesActivity::find($request->sale_activity_id);
                $user = User::find($request->sales_owner);

                if ($sale) {
                    $sale->sales_owner = $request->sales_owner;
                    if ($user->hasRole('executive')) {
                        $team_leader_id = $user->reporting_to;
                    } else {
                        $team_leader_id = $user->id;
                    }
                    $sale->team_leader_id = $team_leader_id;
                    $sale->receipt_owner_change = 1;
                    $sale->receipt_owner_change_by = auth()->user()->id;
                    $sale->save();
                }
            }

            if (Auth::user()->hasRole('crm') || Auth::user()->hasRole('cr-support') || auth()->user()->id == 249) {
                $receipts->collection_type = 'CR';
            } else {
                $receipts->collection_type = 'Sales';
            }

            if ($receipts->save()) {
                $activity_subject = auth()->user()->name . ' created new receipt with id ' . $receipts->id;
                ActivityLog::addToLog($activity_subject, 'crm');
                DB::commit();
                return $this->success(
                    'Receipt has been added successfully.',
                );
            }
        } catch (\Throwable $th) {
            dd($th);
            DB::rollback();

            return $this->success(
                'Faild to add receipt.',
            );
        }
    }

    public function bank()
    {
        $bank = Bank::get();
        
        $other_bank = [
            'id' => '',
            'title' => 'Other Bank'
        ];
    
        $bank[] = $other_bank;
    
        $data = array_map(function ($item) {
            return is_array($item) ? (object)$item : $item;
        }, $bank->toArray());
    
        return [
            'message' => 'Banks',
            'data' => new BankCollection($data),
        ];
    }
    

    public function units(UnitsRequest $request)
    {
        $project_id = $request->project_id;

        $priceList = PriceList::where('project_id', $project_id)
            ->where('booking_status_id', 2)
            ->get();

        if ($priceList->isEmpty()) {
            return [
                'message' => 'Units',
                'data' => [
                    [
                        'id' => '',
                        'name' => ''
                    ]
                ]
            ];
        } else {
            return [
                'message' => 'Units',
                'data' => new UnitsCollection($priceList)
            ];
        }
    }

    public function customers(Request $request)
    {
        $project_id = $request->project_id;
        $unit_id = $request->unit_id;

        $salesActivity = SalesActivity::where('project', $project_id)
            ->where('project_unit', $unit_id)
            ->whereNotIn('booking_status', ['hold', 'cancelled'])
            ->where('is_verified', '!=', 5)
            ->where(function ($query) {
                $query->whereNull('is_cancelled')->orWhere('is_cancelled', 'rejected');
            })
            ->get();

        $data = $salesActivity->map(function ($sales) {
            return [
                'name' => $sales->getLeads ? $sales->getLeads->enquirer_name : '',
                'lead_id' => (string)$sales->lead_id ?? '',
                'sale_id' => (string)$sales->id ?? '',
            ];
        });

        return [
            'message' => 'customers',
            'data' => $data
        ];
    }

    public function saleData(Request $request)
    {
        $project_id = $request->project_id;
        $customer_id = $request->customer_id;

        $salesActivity = SalesActivity::where('project', $project_id)
            ->where('lead_id', $customer_id)
            ->whereNotIn('booking_status', ['hold', 'cancelled'])
            ->where('is_verified', '!=', 5)
            ->get();

        $data = $salesActivity->map(function ($sales){
            return [
                'sale_id' => (string)$sales->id ?? "",
                'proposal_id' => (string)$sales->proposal_id ?? '',
            ];
        });

        return [
            'message' => 'Sale Data',
            'data' => $data
        ];
    }

    public function commercialAvailableUnits(UnitsRequest $request)
    {
        $project_id = $request->project_id;

        $priceList = PriceList::where('project_id', $project_id)
            ->where('booking_status_id', 1)
            ->get();

        if ($priceList->isEmpty()) {
            return [
                'message' => 'Units',
                'data' => [
                    [
                        'id' => '',
                        'name' => ''
                    ]
                ]
            ];
        } else {
            return [
                'message' => 'Units',
                'data' => new UnitsCollection($priceList)
            ];
        }
    }
}
