<?php

namespace App\Api\Http\Controllers\Api\V2\Tasks;

use App\Api\Http\Requests\Task\TaskListRequest;
use App\Api\Http\Resources\Leads\LeadCollection;
use App\Api\Http\Resources\Tasks\TaskCollection;
use App\DailyActivityCalculation;
use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\Builder;
use App\Lead;
use App\Quarter;
use App\Task;
use App\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;

class TasksController extends Controller
{
    public function index(TaskListRequest $request): TaskCollection
    {
        $lists = $request->list;
        $week = Carbon::today()->subDay(15);
        $month = Carbon::now()->subMonth();
        $seven = Carbon::today()->subDays(7);
        $currentDate = date('Y-m-d');
        $quarter = Quarter::whereRaw('? between date_from and date_to', $currentDate)
            ->first();
            
            if($quarter){
                $from_date = $quarter->date_from;
                $to_date = $quarter->date_to;
            }

        $all_reporting_users_id_array = [];
        $userRoles = auth()
            ->user()
            ->userRoles();

        $authUser = auth()->user();
        if (!$userRoles->contains('slug', 'ceo') && !$userRoles->contains('slug', 'super-admin') && !$userRoles->contains('slug', 'admin') && !$userRoles->contains('slug', 'head-sales')) { //Except superadmin and admin
            $all_reporting_users_id_array = $authUser->getAllReportingUsersIds();
            array_push($all_reporting_users_id_array, $authUser->id);
        } elseif ($userRoles->contains('slug', 'sales-head')) {
            $all_reporting_users_id_array = User::whereHas('roles', function ($q) {
                $q->where('role_id', 8);
            })->where('branch_id', auth()->user()->branch_id)
                ->where('status', 1)
                ->get()
                ->pluck('id')
                ->toArray();
        }

        $tasks = QueryBuilder::for(Task::class)
            ->with([
                'getTaskType',
                'getFollowupBy',
                'getProject',
                'getLeadCategory',
                'getLeadStage',
                'getFollowupStatus',
                'getSiteVisitStatus',
                'getBudget',
                'getCreatedBy',
                'getLead'
            ])
            ->when($lists == 1, function ($query) {
                $query->whereDate('schedule_date', today());
            })
            ->when($lists == 2, function ($query) use ($week) {
                $query->whereDate('schedule_date', '>=', $week);
            })
            ->when($lists == 3, function ($query) use ($month) {
                $query->whereDate('schedule_date', '>=', $month);
            })
            ->when($lists == 4, function ($query) use ($request) {
                $from = $request->from_date;
                $to = $request->to_date;
                $query->whereBetween('schedule_date', [$from, $to]);
            })
            ->when($lists == 5, function ($query) {
                $yesterday = Carbon::yesterday();
                $query->whereDate('schedule_date', $yesterday);
            })
            ->when($lists == 6, function ($query) {
                $query->whereBetween('schedule_date', ['2023-11-20', '2023-12-31']);
            })
            ->when($lists == 7, function ($query) use ($seven) {
                $query->whereDate('schedule_date', '>=', $seven);
            })
            ->when($lists == 8, function ($query) use($from_date, $to_date){
                $query->whereBetween('schedule_date', [$from_date, $to_date]);
            })
            ->where(function ($query) use ($all_reporting_users_id_array) {
                $query->whereIn('followup_by', $all_reporting_users_id_array)
                    ->orWhereIn('created_by', $all_reporting_users_id_array);
            })
            ->defaultSort('-created_at')
            ->allowedFilters([
                AllowedFilter::exact('id'),
                AllowedFilter::exact('lead_id'),
                AllowedFilter::scope('today_tasks'),
                AllowedFilter::exact('type', 'create_task_id'),
                AllowedFilter::scope('starts_between'),
                AllowedFilter::scope('task_completed_status'),
                AllowedFilter::scope('task_created'),
                AllowedFilter::scope('search_name'),
            ])
            ->jsonPaginate(15)
            ->appends(request()->query());

        return (new TaskCollection($tasks))->additional(['message' => "Tasks"]);
    }

    public function usersTask(Request $request)
    {
        $userRoles = auth()
            ->user()
            ->userRoles();
        $all_reporting_users_id_array = [];
        $authUser = auth()->user();
        $branch_id = $request->branch_id;
        $team_leader = $request->team_leader;
        if ($authUser->hasRole('admin') || $authUser->hasRole('super-admin') || $authUser->hasRole('ceo') || $authUser->hasRole('head-sales')) {
            $all_reporting_users_id_array = User::with('roles')->whereHas('roles', function ($q) {
                $q->where('role_id', 8);
            })->when($branch_id, function ($query) use ( $branch_id) {
                $query->where('branch_id', $branch_id);
            })
            ->when($team_leader, function ($query) use ($team_leader) {
                $query->where('reporting_to', $team_leader);
            })
            ->where('status', 1)->get()->pluck('id');
        } elseif ($userRoles->contains('slug', 'sales-head')) {
            $all_reporting_users_id_array = User::whereHas('roles', function ($q) {
                $q->where('role_id', 8);
            })->where('branch_id', auth()->user()->branch_id)
            ->when($team_leader, function ($query) use ($team_leader) {
                $query->where('reporting_to', $team_leader);
            })
                ->where('status', 1)
                ->pluck('id')
                ->toArray();
        } elseif ($authUser->hasRole('executive')) {
            $all_reporting_users_id_array = [$authUser->id];
        } elseif ($authUser->hasRole('team-leader')) {
            $all_reporting_users_id_array = $authUser->getAllReportingUsersIds();
            array_push($all_reporting_users_id_array, $authUser->id);
        } else {
            $all_reporting_users_id_array = $authUser->getAllReportingUsersIds();
        }


        $create_task_id = $request->create_task_id;
        $lists = $request->list;
        $week = Carbon::today()->subDays(15);
        $month = Carbon::now()->subMonth();
        $yesterday = Carbon::yesterday()->toDateString();
        $seven = Carbon::today()->subDays(7);
        $currentDate = date('Y-m-d');
        $quarter = Quarter::whereRaw('? between date_from and date_to', $currentDate)
            ->first();
            
            if($quarter){
                $from_date = $quarter->date_from;
                $to_date = $quarter->date_to;
            }

        $tasks = QueryBuilder::for(Task::class)
            ->allowedFilters([
                AllowedFilter::exact('id'),
                AllowedFilter::exact('lead_id'),
                AllowedFilter::scope('today_tasks'),
                AllowedFilter::exact('type', 'create_task_id'),
                AllowedFilter::scope('starts_between'),
                AllowedFilter::scope('task_completed_status'),
                AllowedFilter::scope('task_created'),
                AllowedFilter::scope('search_name'),
            ])
            ->whereIn('created_by', $all_reporting_users_id_array)
            ->where('task_completed_status', 1)
            ->when($lists == 1, function ($query) {
                $query->whereDate('schedule_date', today());
            })
            ->when($lists == 2, function ($query) use ($week) {
                $query->whereDate('schedule_date', '>=', $week);
            })
            ->when($lists == 3, function ($query) use ($month) {
                $query->whereDate('schedule_date', '>=', $month);
            })
            ->when($lists == 4, function ($query) use ($request) {
                $from = $request->from_date;
                $to = $request->to_date;
                $query->whereBetween('schedule_date', [$from, $to]);
            })
            ->when($lists == 5, function ($query) use ($yesterday) {
                $query->whereDate('schedule_date', '=', $yesterday);
            })
       
            ->when($lists == 6, function ($query) {
                $query->whereBetween('schedule_date', ['2023-11-20', '2023-12-31']);
            })
            ->when($lists == 7, function ($query) use ($seven) {
                $query->whereDate('schedule_date', '>=', $seven);
            })
            ->when($lists == 8, function ($query) use($from_date, $to_date){
                $query->whereBetween('schedule_date', [$from_date, $to_date]);
            })
            ->whereIn('create_task_id', [2, 3, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19])
            ->where(function ($query) {
                $query->whereIn('create_task_id', [2, 3, 13, 14, 12, 19])
                    ->where('team_leader_approved', 1);
            })
            ->orWhere(function ($query) use ($all_reporting_users_id_array, $lists, $week, $month, $request, $yesterday,$seven, $from_date, $to_date) {
                $query->whereIn('created_by', $all_reporting_users_id_array)
                    ->whereIn('create_task_id', [10, 11, 15, 16, 17, 18])
                    ->when($lists == 1, function ($query) {
                        $query->whereDate('schedule_date', today());
                    })
                    ->when($lists == 2, function ($query) use ($week) {
                        $query->whereDate('schedule_date', '>=', $week);
                    })
                    ->when($lists == 3, function ($query) use ($month) {
                        $query->whereDate('schedule_date', '>=', $month);
                    })
                    ->when($lists == 4, function ($query) use ($request) {
                        $from = $request->from_date;
                        $to = $request->to_date;
                        $query->whereBetween('schedule_date', [$from, $to]);
                    })
                    ->when($lists == 5, function ($query) use ($yesterday) {
                        $query->whereDate('schedule_date', '=', $yesterday);
                    })
                    ->when($lists == 6, function ($query) {
                        $query->whereBetween('schedule_date', ['2023-11-20', '2023-12-31']);
                    })
                    ->when($lists == 7, function ($query) use ($seven) {
                        $query->whereDate('schedule_date', '>=', $seven);
                    })
                    ->when($lists == 8, function ($query) use($from_date, $to_date){
                        $query->whereBetween('schedule_date', [$from_date, $to_date]);
                    });
            })
            ->select('tasks.*', DB::raw('(SELECT MAX(point) FROM daily_activity_calculations WHERE create_task_id = tasks.create_task_id) as max_point'))
            ->allowedFilters([
                AllowedFilter::exact('id'),
                AllowedFilter::exact('lead_id'),
                AllowedFilter::scope('today_tasks'),
                AllowedFilter::exact('type', 'create_task_id'),
                AllowedFilter::scope('starts_between'),
                AllowedFilter::scope('task_completed_status'),
                AllowedFilter::scope('task_created'),
                AllowedFilter::scope('search_name'),
            ])
            ->get()
            ->groupBy('lead_id');

        $tasks->transform(function ($leadTasks) {
            return $leadTasks->sortByDesc('max_point')->first();
        });

        if($create_task_id == 15){
            $filteredTasks = $tasks->whereIn('create_task_id', [15,17]);
            $filteredTasks->where('create_task_id', 17)
                    ->where('duration', '>', 30);
        }elseif($create_task_id == 16){
            $filteredTasks = $tasks->whereIn('create_task_id', [16,18]);
            $filteredTasks->where('create_task_id', 18)
                    ->where('duration', '>', 30);
        }else{
            $filteredTasks = $tasks->where('create_task_id', $create_task_id);
        }

        
        $filteredTasks = $filteredTasks->sortByDesc('max_point');
        $filteredTasks->transform(function ($leadTasks) {
            return $leadTasks;
        });

        $data = [];
        foreach ($filteredTasks as $task) {
            $data[] = [
                'id' => $task->id ? $task->id : 0,
                'lead_id' => $task->lead_id ? $task->lead_id : 0,
                'name' => (string)$task->getLead  ? $task->getLead->enquirer_name : '',
                'email' => (string)$task->getLead->email ?? '',
                'project_type_id' => (string)$task->getLead->project_type ?? '',
                'task_sale_status' => (string)$task->getLead->task_sale_status ?? '',
                'mobile' => (string)$task->getLead ? $task->getLead->mobile : '',
                'schedule_date' => (string) $task->schedule_date ?? '',
                'schedule_time' => (string)$task->schedule_time ?? '',
                'task_completed_status' => (string) $task->task_completed_status ?? '',
                'task_completion_remark' => (string)$task->task_completion_remarks ?? '',
                'created_at' => (string) \Carbon\Carbon::createFromFormat('Y-m-d H:i:s', $task->created_at)
                ->format('Y-m-d h:i A'),
                'remarks' => (string) $task->remarks ?? '',
                'sales_person' => (string)$task->getCreatedBy->name ?? '',
                'getTaskType' => $task->getTaskType ? [
                    'id' => $task->getTaskType->id ?? 0,
                    'name' => (string)$task->getTaskType->name ?? '',
                    'status' => $task->getTaskType->status ?? 0
                ] : (object)[],
                'project' => $task->getProject ? [
                    'id' => $task->getProject ? $task->getProject->id : '',
                    'project_name' => (string) $task->getProject->project_name ?? '',
                    'project_short' => (string)$task->getProject ? $task->getProject->project_code : '',
                ] : (object)[],
                'resident' => (string) $task->getLead->residential_status ?? '',
                'location' => (string)$task->getLead->location ?? '',
                'lead_status' =>  $task->getLeadCategory ? [
                    'id' => $task->getLeadCategory ? $task->getLeadCategory->id : 0,
                    'name' => (string)$task->getLeadCategory ? $task->getLeadCategory->name : '',
                    'status' => $task->getLeadCategory ? $task->getLeadCategory->status : 0,
                ] : (object)[],
                'lead_source' => (string)$task->getLead->getSource ? $task->getLead->getSource->name : '',
                'place' => (string) $task->getLead->place_address ?? '',
                'duration' => (string)$task->duration ?? '',
                'call_date' => (string) $task->runo_created_at ?? '',
                'latitude' => $task->getLead->latitude ?? '',
                'longitude' => $task->getLead->longitude ?? '',
            ];
        }

        return response()->json([
            'status' => true,
            'message' => 'Success',
            'data' => $data,
        ]);
    }

    public function listNewLead(Request $request)
    {

        $userRoles = auth()
            ->user()
            ->userRoles();
        $all_reporting_users_id_array = [];
        $authUser = auth()->user();
        $branch_id = $request->branch_id;
        $team_leader = $request->team_leader;
        if ($authUser->hasRole('admin') || $authUser->hasRole('super-admin') || $authUser->hasRole('ceo') || $authUser->hasRole('head-sales')) {
            $all_reporting_users_id_array = User::with('roles')->whereHas('roles', function ($q) {
                $q->where('role_id', 8);
            })
            ->when($branch_id, function ($query) use ( $branch_id) {
                $query->where('branch_id', $branch_id);
            })
            ->when($team_leader, function ($query) use ($team_leader) {
                $query->where('reporting_to', $team_leader);
            })
            ->where('status', 1)->get()->pluck('id');
        } elseif ($userRoles->contains('slug', 'sales-head')) {

            $all_reporting_users_id_array = User::whereHas('roles', function ($q) {
                $q->where('role_id', 8);
            })->where('branch_id', auth()->user()->branch_id)
                ->where('status', 1)->get()
                ->pluck('id')
                ->toArray();
        } elseif ($authUser->hasRole('executive')) {
            $all_reporting_users_id_array = [$authUser->id];
        } else {
            $all_reporting_users_id_array = $authUser->getAllReportingUsersIds();
        }

        $lists = $request->list;
        $week = Carbon::today()->subDays(15);
        $month = Carbon::now()->subMonth();
        $yesterday = Carbon::yesterday()->toDateString();
        $seven = Carbon::today()->subDays(7);
        $currentDate = date('Y-m-d');
        $quarter = Quarter::whereRaw('? between date_from and date_to', $currentDate)
            ->first();
            
            if($quarter){
                $from_date = $quarter->date_from;
                $to_date = $quarter->date_to;
            }

        $tasks = Task::whereIn('created_by', $all_reporting_users_id_array)
            ->where('task_completed_status', 1)
            ->when($lists == 1, function ($query) {
                $query->whereDate('schedule_date', today());
            })
            ->when($lists == 2, function ($query) use ($week) {
                $query->whereDate('schedule_date', '>=', $week);
            })
            ->when($lists == 3, function ($query) use ($month) {
                $query->whereDate('schedule_date', '>=', $month);
            })
            ->when($lists == 4, function ($query) use ($request) {
                $from = $request->from_date;
                $to = $request->to_date;
                $query->whereBetween('schedule_date', [$from, $to]);
            })
            ->when($lists == 5, function ($query) use ($yesterday) {
                $query->whereDate('schedule_date', '=', $yesterday);
            })
            ->when($lists == 6, function ($query) {
                $query->whereBetween('schedule_date', ['2023-11-20', '2023-12-31']);
            })
            ->when($lists == 7, function ($query) use ($seven) {
                $query->whereDate('schedule_date', '>=', $seven);
            })
            ->when($lists == 8, function ($query) use($from_date, $to_date){
                $query->whereBetween('schedule_date', [$from_date, $to_date]);
            })
            ->whereIn('create_task_id', [2, 3, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19])
            ->where(function ($query) {
                $query->whereIn('create_task_id', [2, 3, 13, 14, 12, 19])
                    ->where('team_leader_approved', 1);
            })
            ->orWhere(function ($query) use ($all_reporting_users_id_array, $lists, $week, $month, $request, $yesterday,$seven, $from_date, $to_date) {
                $query->whereIn('created_by', $all_reporting_users_id_array)
                    ->whereIn('create_task_id', [10, 11, 15, 16, 17, 18])
                    ->when($lists == 1, function ($query) {
                        $query->whereDate('schedule_date', today());
                    })
                    ->when($lists == 2, function ($query) use ($week) {
                        $query->whereDate('schedule_date', '>=', $week);
                    })
                    ->when($lists == 3, function ($query) use ($month) {
                        $query->whereDate('schedule_date', '>=', $month);
                    })
                    ->when($lists == 4, function ($query) use ($request) {
                        $from = $request->from_date;
                        $to = $request->to_date;
                        $query->whereBetween('schedule_date', [$from, $to]);
                    })
                    ->when($lists == 5, function ($query) use ($yesterday) {
                        $query->whereDate('schedule_date', '=', $yesterday);
                    })
                    ->when($lists == 6, function ($query) {
                        $query->whereBetween('schedule_date', ['2023-11-20', '2023-12-31']);
                    })
                    ->when($lists == 7, function ($query) use ($seven) {
                        $query->whereDate('schedule_date', '>=', $seven);
                    })
                    ->when($lists == 8, function ($query) use($from_date, $to_date){
                        $query->whereBetween('schedule_date', [$from_date, $to_date]);
                    });
            })
            ->select('tasks.*', DB::raw('(SELECT MAX(point) FROM daily_activity_calculations WHERE create_task_id = tasks.create_task_id) as max_point'))
            ->get()
            ->groupBy('lead_id');



        $tasks->transform(function ($leadTasks) {
            return $leadTasks->sortByDesc('max_point')->first();
        });

        $leadsToSkip = $tasks->pluck('lead_id');



        $newLeadCount = QueryBuilder::for(Lead::class)
            ->where(function ($query) use ($all_reporting_users_id_array) {
                $query->whereIn('created_by', $all_reporting_users_id_array)
                    ->orWhereIn('attended_by', $all_reporting_users_id_array);
            })
            ->when($lists == 1, function ($query) {
                $query->whereDate('lead_date', today());
            })

            ->when($lists == 2, function ($query) use ($week) {
                $query->whereDate('lead_date', '>=', $week);
            })
            ->when($lists == 3, function ($query) use ($month) {
                $query->whereDate('lead_date', '>=', $month);
            })
            ->when($lists == 4, function ($query) use ($request) {
                $from = $request->from_date;
                $to = $request->to_date;
                $query->whereBetween('lead_date', [$from, $to]);
            })
            ->when($lists == 5, function ($query) use ($yesterday) {
                $query->whereDate('lead_date', '=', $yesterday);
            })
            ->when($lists == 6, function ($query) {
                $query->whereBetween('lead_date', ['2023-11-20', '2023-12-31']);
            })
            ->when($lists == 7, function ($query) use ($seven) {
                $query->whereDate('lead_date', '>=', $seven);
            })
            ->when($lists == 8, function ($query) use($from_date, $to_date){
                $query->whereBetween('lead_date', [$from_date, $to_date]);
            })
            ->where('duplicate_entry', 'no')
            ->whereNotIn('id', $leadsToSkip)
            ->allowedFilters([
                AllowedFilter::callback('is_today', function (Builder $query, $value) {
                    $query->whereDate('created_at', today());
                }),
                AllowedFilter::callback('project_type', function (Builder $query, $value) {
                    if ($value == 0) {
                        $query->whereNull('project_type');
                    } else {
                        $query->where('project_type', $value);
                    }
                }),
                AllowedFilter::exact('id'),
                AllowedFilter::exact('created_by'),
                AllowedFilter::exact('source', 'getSource.id'),
                AllowedFilter::exact('project', 'getPreferredProject.id'),
                AllowedFilter::scope('starts_between'),
            ])
            ->jsonPaginate(25)
            ->appends(request()->query());
        return new LeadCollection($newLeadCount);
    }

    public function leadTaskDetails(Request $request)
    {
        $task = Task::find($request->lead_id);
        
            $lead_id = $task->lead_id;
        
        $lead = Lead::find($lead_id);

        if (!$lead) {
            return response()->json([
                'status' => false,
                'message' => 'Lead not found',
                'data' => [],
            ]);
        }

        $tasks = Task::where('lead_id', $lead->id)->get();

        $taskDetails = [];

        foreach ($tasks as $task) {
            $taskDetails[] = [
                'id' => $task->id,
                'status' => $task->status,
                'create_task_id' => $task->getTaskType ? $task->getTaskType->name : '',
                'time' => date('h:ia', strtotime($task->created_at)),
                'date' => date('M d Y', strtotime($task->created_at)),
                'remarks' =>$task->remarks ?? '',
                'task_completion_remarks' =>$task->task_completed_status ?? '',
                'team_leader_approved' =>$task->team_leader_approved ?? ''
            ];
        }
        
        $data = [
            'id' => $lead->id,
            'name' => $lead->enquirer_name ?? '',
            'phone_code' => strval($lead->phone_code) ?? '',
            'mobile' => $lead->mobile ?? '',
            'email' => $lead->email ?? '',
            'project' => $lead->getPreferredProject ? $lead->getPreferredProject->project_name : '',
            'source' => $lead->getSource ? $lead->getSource->name : '',
            'attended_by' => $lead->getAttendedBy ? $lead->getAttendedBy->name : '',
            'location' => strval($lead->location) ?? '',
            'remarks' => strval($lead->remarks) ?? '',
            'lead_date'=>date('d-m-y', strtotime($lead->lead_date)) ?? '',
            'whatsapp_no' => strval($lead->whatsapp_no) ?? '',
            'tasks' => $taskDetails,
        ];
        
        return response()->json([
            'status' => true,
            'message' => 'Success',
            'data' => $data ?? [],
        ]);
    }

    public function newLeadTaskDetails(Request $request)
    {
     
         $lead_id = $request->lead_id;
        
        $lead = Lead::find($lead_id);

        if (!$lead) {
            return response()->json([
                'status' => false,
                'message' => 'Lead not found',
                'data' => [],
            ]);
        }

        $tasks = Task::where('lead_id', $lead->id)->get();

        $taskDetails = [];

        foreach ($tasks as $task) {
            $taskDetails[] = [
                'id' => $task->id,
                'status' => $task->status,
                'create_task_id' => $task->getTaskType ? $task->getTaskType->name : '',
                'time' => date('h:ia', strtotime($task->created_at)),
                'date' => date('M d Y', strtotime($task->created_at)),
                'remarks' =>$task->remarks ?? '',
                'task_completion_remarks' =>$task->task_completed_status ?? '',
                'team_leader_approved' =>$task->team_leader_approved ?? ''
            ];
        }
        
        $data = [
            'id' => $lead->id,
            'name' => $lead->enquirer_name ?? '',
            'phone_code' => strval($lead->phone_code) ?? '',
            'mobile' => $lead->mobile ?? '',
            'email' => $lead->email ?? '',
            'project' => $lead->getPreferredProject ? $lead->getPreferredProject->project_name : '',
            'source' => $lead->getSource ? $lead->getSource->name : '',
            'attended_by' => $lead->getAttendedBy ? $lead->getAttendedBy->name : '',
            'location' => strval($lead->location) ?? '',
            'remarks' => strval($lead->remarks) ?? '',
            'lead_date'=>date('d-m-y', strtotime($lead->lead_date)) ?? '',
            'whatsapp_no' => strval($lead->whatsapp_no) ?? '',
            'tasks' => $taskDetails,
        ];
        
        return response()->json([
            'status' => true,
            'message' => 'Success',
            'data' => $data ?? [],
        ]);
    }
}
