<?php

namespace App\Exports;

use App\LeadBank;
use App\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;

class LeadBankExport implements FromQuery, WithHeadings, WithMapping, ShouldAutoSize
{
    use Exportable;

    protected $request;

    public function __construct($request)
    {
        $this->request = $request;
    }

    public function query()
    {


        $userPermissions = auth()
            ->user()
            ->userPermissions();

        $userRoles = auth()
            ->user()
            ->userRoles();



        $authUser = auth()->user(); 
       

        $active_lost_lead = $this->request->active_lost_lead ? $this->request->active_lost_lead : 'Lead Bank';
        $project = $this->request->s_project ?? '';
        $salesPerson = $this->request->s_sales_person ?? '';
        $leadSource = $this->request->s_source ?? '';
        $leadStage = $this->request->s_stage;
        $leadCategory = $this->request->s_category ?? "";
        $fromDate = $this->request->s_from ?? '';
        $toDate = $this->request->s_to ?? '';

        $user_id = $this->request->user_id;
        $project_type_id = $this->request->project_type_id;

        $teamLeader = intval($this->request->s_team_leader);

        $phoneCodeFilter = $this->request->phone_code_filter;



        if ($this->request->s_from) {

            $fromDate = Carbon::parse($this->request->s_from)->startOfDay()->format('Y-m-d H:i:s');
        }

        if ($this->request->s_to) {
            $toDate = Carbon::parse($this->request->s_to)->endOfDay()->format('Y-m-d H:i:s');;
        }

        $rows = LeadBank::query()
            ->with('getSource', 'getLeadStage', 'getLeadCategory', 'getProjectType', 'getBudget', 'getPreferredProject','getProfession')
         
            ->when($project, function ($q, $project) {
                return $q->whereIn('preferred_project', $project);
            })

            ->when($salesPerson, function ($q, $salesPerson) {
                return $q->whereIn('residential_status', $salesPerson);
            })
            ->when($leadSource, function ($q, $leadSource) {
                return $q->whereIn('source', $leadSource);
            })
            ->when($leadStage, function ($q, $leadStage) {
                return $q->whereIn('age', $leadStage);
            })
            ->when($leadCategory, function ($q, $leadCategory) {
                return $q->whereIn('profession', $leadCategory);
            })
            ->when($fromDate, function ($q, $fromDate) {
                return $q->whereDate('created_at', '>=', $fromDate);
            })
            ->when($toDate, function ($q, $toDate) {
                return $q->whereDate('created_at', '<=', $toDate);
            })
            ->when($phoneCodeFilter, function ($q, $phoneCodeFilter) {
                return $q->where('phone_code', $phoneCodeFilter);
            });

        if ('active' == $active_lost_lead) {
            $rows = $rows->where(function ($query) {
                $query->whereNotIn('lead_stage_id', [6, 7, 8])->orWhereNull('lead_stage_id'); // 6 not interested
            });

            // $rows = $rows->where('lead_stage_id', '<>',6)->orWhereNull('lead_stage_id');// 6 not interested
        } elseif ('lost' == $active_lost_lead) {
            $rows = $rows->where(function ($query) {
                $query->where('lead_stage_id', '=', 6)->where('lead_stage_id', '<>', 7)->orWhere('lead_stage_id', '=', 8);
            });
            $rows = $rows->where(function ($query) {
                $query->where('lost_lead_approval', 1); // 6 not interested,8 invalid
            });
        } elseif ('lost-pending' == $active_lost_lead) {
            $rows = $rows->where(function ($query) {
                $query->where('lead_stage_id', '=', 6)->where('lead_stage_id', '<>', 7)->orWhere('lead_stage_id', '=', 8);
            });
            $rows = $rows->where(function ($query) {
                $query->where('lost_lead_approval', 0)->orWhereNull('lost_lead_approval'); // 6 not interested,8 invalid
            });
        } elseif ('duplicate' == $active_lost_lead) {
            $rows = $rows->where('duplicate_entry', 'yes')->where(function ($q) {
                $q->where('sales_head_approval', 0)
                    ->orWhereNull('sales_head_approval');
            })
                ->where(function ($q) {
                    $q->where('lead_stage_id', '<>', 7)
                        ->orWhereNull('lead_stage_id');
                })->whereNotNull('duplicate_request_at');
        } elseif ('customer' == $active_lost_lead) {
            $rows = $rows->where(function ($query) {
                $query->where('lead_stage_id', 7)->where('sale_status', 'yes');
            });
        } elseif ('all' == $active_lost_lead) {
            $rows = $rows->where(function ($query) {
                $query->where('sale_status', 'no');
            });
        }

        if ($user_id) {


            $rows = $rows->where(function ($query) use ($user_id) {
                $query->where('created_by', $user_id)->orWhere('attended_by', $user_id);
            });
        }
        if ($project_type_id) {

            if ($project_type_id == 'other') {
                $rows = $rows->whereNull('project_type');
            } else {
                $rows = $rows->where('project_type', $project_type_id);
            }
        }

        return $rows->orderBy('created_at', 'desc');
    }

    public function headings(): array
    {
        $set1 = [
            'EnquirerName',
            'Email',
            'Mobile',
            'AltMobile',
            'Source',
            'Stage',
            'ProjectType',
            'PreferredProject',
            'Budget',
            'Remarks',
            'AttendedBy',
            'Date',
            'Remark'
        ];
        $set2 = [
            'EnquirerName',
            'Email',
            'Mobile',
            'AltMobile',
            'Source',
            'Stage',
            'Category',
            'ProjectType',
            'PreferredProject',
            'Budget',
            'Profession',
            'Age',
            'Residential Status',
            'Date',
            'Remark',
        ];
        if ($this->request->active_lost_lead &&  $this->request->active_lost_lead == 'lost') {
            return  $set1;
        }
        return  $set2;
    }

    public function map($lead): array
    {
        $set1 = [
            $lead->enquirer_name,
            $lead->email,
            $lead->phone_code . ' ' . $lead->mobile,
            $lead->alt_mobile,
            isset($lead->getSource) ? $lead->getSource->name : '',
            isset($lead->getLeadStage) ? $lead->getLeadStage->name : '',
            isset($lead->getLeadCategory) ? $lead->getLeadCategory->name : '',
            isset($lead->getProjectType) ? $lead->getProjectType->name : '',
            isset($lead->getPreferredProject) ? $lead->getPreferredProject->project_name : '',
            isset($lead->getBudget) ? $lead->getBudget->name : '',
            isset($lead->getLeadStage) ? $lead->getLeadStage->percentage . ' % ' : 0,
            isset($lead->getProfession) ? $lead->getProfession->name : '',
            isset($lead->getAge) ? $lead->getAge->age : '',
            $lead->residential_status,
            date('d-m-Y', strtotime($lead->created_at)),
            $lead->remarks,
        ];
        $set2 = [
            $lead->enquirer_name,
            $lead->email,
            $lead->phone_code . ' ' . $lead->mobile,
            $lead->alt_mobile,
            isset($lead->getSource) ? $lead->getSource->name : '',
            isset($lead->getLeadStage) ? $lead->getLeadStage->name : '',
            isset($lead->getLeadCategory) ? $lead->getLeadCategory->name : '',
            isset($lead->getProjectType) ? $lead->getProjectType->name : '',
            isset($lead->getPreferredProject) ? $lead->getPreferredProject->project_name : '',
            isset($lead->getBudget) ? $lead->getBudget->name : '',
            isset($lead->getProfession) ? $lead->getProfession->name : '',
            isset($lead->getAge) ? $lead->getAge->age : '',
            $lead->residential_status,
            date('d-m-Y', strtotime($lead->created_at)),
            $lead->remarks,
        ];
        if ($this->request->active_lost_lead &&  $this->request->active_lost_lead == 'lost') {
            return  $set1;
        }
        return  $set2;
    }

    public function fields(): array
    {
        $set1 = [
            'enquirer_name',
            'email',
            'mobile',
            'alt_mobile',
            'source',
            'lead_stage',
            'lead_category',
            'project_type',
            'pref_project',
            'budget',
            'remarks',
            'Profession',
            'Age',
            'Residential Status',
            'newDate',
            'premarks',
        ];
        $set2 = [
            'enquirer_name',
            'email',
            'mobile',
            'alt_mobile',
            'source',
            'lead_stage',
            'lead_category',
            'project_type',
            'pref_project',
            'budget',
        
            'Profession',
            'Age',
            'Residential Status',
            'newDate',
            'premarks',
        ];
        if ($this->request->active_lost_lead &&  $this->request->active_lost_lead == 'lost') {
            return  $set1;
        }
        return  $set2;
    }
}
