<?php

namespace App\Exports\Receipt;

use App\Receipt;
use Carbon\Carbon;
use Illuminate\Support\Facades\URL;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithCustomStartCell;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadings;
use PhpOffice\PhpSpreadsheet\Calculation\LookupRef\Hyperlink;

class ExportReceipts implements FromCollection, WithHeadings, WithStrictNullComparison, WithCustomStartCell, WithEvents, ShouldAutoSize
{
    /**
    * @return \Illuminate\Support\Collection
    */

    use Exportable;


    public function __construct($dateFrom, $dateTo, $project, $unit,$customer,$collection_type,$quadra_receipt)
    {
        $this->dateFrom = $dateFrom;
        $this->dateTo = $dateTo;
        $this->project = $project;
        $this->unit = $unit;
        $this->client_id = $customer;

        $this->collection_type = $collection_type;
        $this->quadra_receipt = $quadra_receipt;

        $this->headerDateFrom = Carbon::parse($dateFrom)->format('d-m-Y');
        $this->headerDateTo = Carbon::parse($dateTo)->format('d-m-Y');
    }

    public function collection()
    {
        $data = Receipt::with('getClient', 'getUnit', 'getUnit.getProjects', 'getSalesData.getProject');

        if ($this->project) {
            $project = $this->project;
            $data->whereHas('getProject', function ($query) use ($project) {
                $query->where('project_id', $project);
            });
        }

        if ($this->client_id) {
            $data->where('client_id', $this->client_id);
        }

        if ($this->unit) {
            $data->where('unit_id', $this->unit);
        }

        if ($this->dateFrom) {
            $data->where('receipt_date', '>=', date('Y-m-d', strtotime($this->dateFrom)));
        }

        if ($this->dateTo) {
            $data->where('receipt_date', '<=', date('Y-m-d', strtotime($this->dateTo)));
        }

        if ($this->collection_type) {
            $data->where('collection_type', $this->collection_type);
        }

        if ($this->quadra_receipt) {
            $data->where('quadra_receipt', $this->quadra_receipt);
        }

        $row = $data->get();

        $result = [];

        foreach ($row as $key => $data) {
            $isApproved = '';
            $quadra_receipt = '';
            if($data->is_approved == 1){
                $isApproved = 'Approved';
            }
            elseif($data->is_approved == 2){
                $isApproved = 'Rejected';
            }
            else{
                $isApproved = 'Pending';
            }
            $file = '';
            if($data->payment_file){
                $file = URL::asset('uploads/' . $data->payment_file) ;
            }
            else{
                $file = 'Nil';
            }
            $units = '';

            // Check if $data->getUnit exists and is truthy
            if ($data->getUnit) {
                // If true, assign $data->getUnit->ap_no to $units
                $units = $data->getUnit->ap_no;
            } else {
                // If $data->getUnit is falsy, check if $data->getSalesData exists and is truthy
                if ($data->getSalesData && $data->getSalesData->getProjectUnit) {
                    // If true, assign $data->getSalesData->no_of_unit to $units
                    $units = $data->getSalesData->getProjectUnit->ap_no;
                } else {
                    $units = $data->getSalesData ? $data->getSalesData->no_of_unit : '';
                }
            }


            if($data->quadra_receipt == 1){
                $quadra_receipt = 'Yes';
            }
            else{
                $quadra_receipt = 'No';
            }
            
            $result[] = [
                'Project Name' =>  $data->getSalesData ? $data->getSalesData->getProject->project_name : '',
                'Unit' => $units,
                'Client' => $data->getClient ? $data->getClient->enquirer_name : '',
                'Receipt No' => $data->receipt_no ?? '',
                'ref No' => $data->transaction_ref_no ? $data->transaction_ref_no : '',
                'Receipt Date' => $data->receipt_date ? date('d-m-Y', strtotime($data->receipt_date)) : '',
                'Mode Of Payment' => $data->mode_of_payment ? $data->mode_of_payment : '',
                'Amount' => $data->amount ? $data->amount: '',
                'Bank' => $data->bank ? $data->bank:'',
                'File' => $file,
                'Status' => $isApproved,
                'Created By' => $data->getCreatedBy ? $data->getCreatedBy->name : '',
                'Quadra Satus' => $quadra_receipt,
                'Collection Type' => $data->collection_type ? $data->collection_type: '',
            ];
        }
        $result = collect($result);

        return $result;

    }

      public function startCell(): string
    {
        return 'A2';
    }


    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                /** @var Sheet $sheet */
                $sheet = $event->sheet;

                $sheet->mergeCells('A1:J1');
                

                $sheet->setCellValue('A1', 'All Receipts List');

                $default_font_style = [
                    'font' => ['name' => 'Arial', 'size' => 15],
                ];
                $header_font = [
                    'font' => ['name' => 'Arial', 'size' => 11],
                ];

                $styleArray = [
                    'alignment' => [
                        'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
                    ],
                ];

                $cellRange = 'A1:J1'; // All headers
                $event->sheet->getDelegate()->getStyle($cellRange)->applyFromArray($styleArray);
                $event->sheet->getStyle('A1:J1')->applyFromArray($default_font_style);
                $event->sheet->getStyle('A2:J2')->applyFromArray($header_font);
            },
        ];
    }
    public function headings(): array
    {
        return [
            [
        
                'Project',
                'Unit/No Of Shares',
                'Client',
                'Receipt No',
                'Ref No.',
                'Receipt Date',
                'Mode Of Payment',
                'Amount',
                'Bank',
                'File',
                'Status',
                'Created By',
                'Quadra Satus',
                'Collection Type'

            ],
        ];
    }
}
