<?php

namespace App\Http\Controllers\Dashboard;

use App\Role;
use App\User;
use App\Branch;
use App\Country;
use App\Helpers\ActivityLog;
use Illuminate\Http\Request;
use Redirect, Response, File;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use App\Http\Requests\UserImageRequest;
use Illuminate\Support\Facades\Validator;
use App\Actions\Users\UpdatePasswordAction;
use App\Actions\Users\UpdateUserProfileAction;
use App\Helpers\LeadDistributeHelper;
use App\Http\Requests\User\ProfileUpdateRequest;
use App\Http\Requests\User\PasswordUpdateRequest;

class UserController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function index()
    {
        $users = User::with('getBranch')->where('id', '<>', 0);
        if (!\Auth::user()->hasRole('super-admin') && !\Auth::user()->hasRole('management') && !\Auth::user()->hasRole('sales-head')) {
            $users = $users->where('reporting_to', \Auth::user()->id);
        }
        $users = $users->get();
        $roles = Role::where('id', '<>', 1)->get(); //Exclude super admin
        $activity_subject = auth()->user()->name . ' Viewed users list';
        ActivityLog::addToLog($activity_subject, 'leads');
        return view('dashboard.users.index', compact('users', 'roles'));
    }

    public function create()
    {
        $reporting_to = User::where('status', 1)->get();
        $roles = Role::where('id', '<>', 1)->get(); //Exclude super admin
        $branches = Branch::where('is_active', 1)->get();

        $activity_subject = auth()->user()->name . ' viewed user create form';
        ActivityLog::addToLog($activity_subject, 'leads');
        return view('dashboard.users.create', compact('reporting_to', 'roles', 'branches'));
    }
    public function store(Request $request)
    {
        $name = $request->name;
        $username = $request->username;
        $email = $request->email;
        $password = $request->password;
        $password_confirmation = $request->password_confirmation;
        $mobile = $request->mobile;
        $reporting_to = $request->reporting_to;
        $role = $request->role;
        $file = $request->file;
        $designation = $request->designation;

        $branch_id = $request->branch_id;

        $data = [
            'name' => $name,
            'username' => $username,
            'email' => $email,
            'password' => $password,
            'password_confirmation' => $password_confirmation,
            'mobile' => $mobile,
            'reporting_to' => $reporting_to,
            'role' => $role,
            'file' => $file,
            'designation' => $designation,
            'branch_id' => $branch_id,
        ];
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'username' => 'required|string|max:255|unique:users',
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'mobile' => ['required', 'min:8'],
            'reporting_to' => ['required'],
            'role' => ['required'],
            'designation' => ['required'],
            'file' => ['nullable', 'mimes:jpg,jpeg,png'],
            'branch_id' => ['required'],

        ];
        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput($request->all());
        }
        $obj = new User();
        $obj->name = $name;
        $obj->branch_id = $branch_id;
        $obj->username = $username;
        $obj->email = $email;
        $obj->password = Hash::make($password);
        $obj->mobile = $mobile;
        $obj->reporting_to = $reporting_to;
        $obj->designation = $designation;
        if ($request->hasfile('file')) {
            $usr_file = $request->file('file');

            $file_name = $usr_file->getClientOriginalName();
            $file_name = time() . $file_name;
            $usr_file->move('./uploads/profile/', $file_name);

            $obj->image = $file_name;
        }
        $obj->created_by = \Auth::user()->id;
        $obj->updated_by = \Auth::user()->id;
        if ($obj->save()) {
            $role_obj = Role::find($role);
            $obj->roles()->attach($role_obj);

            $activity_subject = auth()->user()->name . ' added new user with id ' . $obj->id . ' and name ' . $obj->name;
            ActivityLog::addToLog($activity_subject, 'leads');

            return Redirect::to("dashboard/users/create")
                ->withSuccess('User has been added successfully.');
        }
        return Redirect::to("dashboard/users/create")
            ->withError('Failed to add User.');
    }
    public function edit($id)
    {
        $row = User::find($id);
        if ($row->id == 1 && \Auth::user()->id != 1) {
            abort(404);
        }

        $branches = Branch::where('is_active', 1)->get();

        $roles = Role::where('id', '<>', 1)->get();
        $reporting_to = User::where('status', 1)->get();

        $activity_subject = auth()->user()->name . ' viewed user edit form with id  ' . $row->id;
        ActivityLog::addToLog($activity_subject, 'leads');
        return view('dashboard.users.edit', compact('row', 'reporting_to', 'roles', 'branches'));
    }
    public function update(Request $request)
    {
        $id = $request->id;

        $branch_id = $request->branch_id;

        $name = $request->name;
        $username = $request->username ? $request->username : NULL;
        $email = $request->email;
        $mobile = $request->mobile;
        $reporting_to = $request->reporting_to;
        $status = $request->status;
        $file = $request->file;
        $new_password = $request->new_password ? $request->new_password : NULL;
        $designation = $request->designation;

        $role = $request->role;


        $data = [
            'name' => $name,
            'username' => $username,
            'email' => $email,
            'mobile' => $mobile,
            'reporting_to' => $reporting_to,
            'status' => $status,
            'file' => $file,
            'role' => $role,
            'branch_id' => $branch_id

        ];
        $rules = [
            'name' => ['required', 'string', 'max:255'],

            'branch_id' => ['required'],

            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $id],
            'mobile' => ['required', 'min:8'],
            'reporting_to' => ['required'],
            'status' => ['required'],
            'file' => ['nullable', 'mimes:jpg,jpeg,png', 'dimensions:min_width=200,min_height=200,max_width=250,max_height=250'],


            'role' => ['required'],


        ];
        if (\Auth::user()->hasRole('super-admin') || \Auth::user()->hasRole('admin')) {
            if (!empty($new_password)) {
                $data['new_password'] = $new_password;
                $rules['new_password'] = ['string', 'min:8'];
            }

            $data['username'] = $username;
            $rules['username'] = ['required', 'string', 'max:255', 'unique:users,username,' . $id];
            ;
        }

        $validator = Validator::make($data, $rules);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput($request->all());
        }
        $obj = User::find($id);
        $obj->name = $name;

        $obj->branch_id = $branch_id;
        $obj->email = $email;
        $obj->mobile = $mobile;
        $obj->status = $status;
        $obj->designation = $designation;
        $obj->reporting_to = $reporting_to;
        $obj->updated_by = \Auth::user()->id;

        if (\Auth::user()->hasRole('super-admin') || \Auth::user()->hasRole('sales-head')) {
            if (!empty($new_password)) {
                $obj->password = Hash::make($new_password);
            }
            if (!empty($username)) {
                $obj->username = $username;
            }
        }

        if ($request->hasfile('file')) {

            $obj->image = $request->file('file')->store('users', 'uploads');
        }

        //Updateing Role
        $obj->roles()->sync([$role]);
        if ($obj->save()) {
            $activity_subject = auth()->user()->name . ' updated user with id  ' . $obj->id;
            ActivityLog::addToLog($activity_subject, 'leads');
            return Redirect::to("dashboard/users")
                ->withSuccess('User has been updated successfully.');
        }

        return Redirect::to("dashboard/users")
            ->withError('Failed to update User.');
    }
    public function profile()
    {

        $user = auth()->user();
        $countries = Country::orderBy('phonecode', 'asc')->get();


        $activity_subject = auth()->user()->name . ' viewed profile image change form';
        ActivityLog::addToLog($activity_subject, 'leads');
        return view('dashboard.users.profile', compact('user', 'countries'));
    }

    public function updateProfile(ProfileUpdateRequest $request, UpdateUserProfileAction $updateUserProfileAction)
    {
        $response = $updateUserProfileAction->execute(collect($request->validated()));
        if (!$response) {

            return redirect()->back()->withError('Something Went Wrong');
        }
        return redirect()->back()->withSuccess('Profile updated successfully');
    }

    public function changePassword(PasswordUpdateRequest $request, UpdatePasswordAction $updatePasswordAction)
    {
        $response = $updatePasswordAction->execute(collect($request->validated()));
        if (!$response) {
            return Redirect::to("dashboard/profile")
                ->withError('Something Went Wrong')->with('password-change', true);
        }
        return Redirect::to("dashboard/profile")
            ->withSuccess('Password changed successfully')->with('password-change', true);
    }
}
