<?php

namespace App\Http\Controllers\MallSoftware;

use App\Actions\MallSoftware\LeaseRentalAgreement\LeaseRentalAgreementStoreAction;
use App\Actions\MallSoftware\LeaseRentalAgreement\LeaseRentalAgreementUpdateAction;
use App\Floor;
use App\Http\Controllers\Controller;
use App\Http\Requests\MallSoftware\LeaseRentalAgreement\LeaseRentalAgreementStoreRequest;
use App\Http\Requests\MallSoftware\LeaseRentalAgreement\LeaseRentalAgreementUpdateRequest;
use App\LeaseRentalAgreement;
use App\LeaseRentalAgreementDetail;
use App\LesseEntry;
use App\PriceList;
use App\Project;
use App\RateItemDefinition;
use App\UnitDefinition;
use App\UnitType;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class LeaseRentalAgreementController extends Controller
{
    public function index()
    {
        return view('mall_software.lease_rental_agreement.index');
    }

    public function create()
    {
        $floors = Floor::get();
        $types = UnitType::get();
        $projects = Project::where('status', 1)->get();
        $lesseeData = LesseEntry::get();
        return view('mall_software.lease_rental_agreement.create', compact('floors', 'types', 'projects', 'lesseeData'));
    }

    public function getList()
    {
        $data = LeaseRentalAgreement::with('getProjectName', 'getLesseeName', 'getProjectUnit', 'getFloor', 'getType')->get();
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('agreement_no', function ($data) {
                $link = '<a href="' . route('mall-software.lease_rental_agreement.viewAgreement', $data->id) . '">' . $data->agreement_no . '</a>';
                return $link;
            })
            ->addColumn('lessee_name', function ($data) {
                if ($data->getLesseeName->lessee_type == "Individual") {
                    return $data->getLesseeName->individual_name;
                } elseif ($data->getLesseeName->lessee_type == "Company") {
                    return $data->getLesseeName->company_name;
                }
            })
            ->addColumn('project_name', function ($data) {
                return $data->getProjectName->project_name;
            })
            ->addColumn('project_unit', function ($data) {
                return $data->getProjectUnit->unit_name;
            })
            ->addColumn('notice_period', function ($data) {
                return $data->notice_period ?? "--";
            })
            ->addColumn('approval_status', function ($data) {
                if ($data->approval_status == 0) {
                    return '<span class="label label-warning">Pending</span>';
                } elseif ($data->approval_status == 1) {
                    return '<span class="label label-success">Approved</span>';
                }
            })
            ->addColumn('actions', function ($data) {
                $csrfField = csrf_field();
                $deleteRoute = route('mall-software.lease_rental_agreement.destroy', $data->id);
                $editbtn = '<a href="' . route('mall-software.lease_rental_agreement.edit', $data->id) . '" class="btn">Edit</a>';
                $deletebtn = <<<HTML
                <form action="{$deleteRoute}" method="post" style ="display:inline">
                {$csrfField}
                <button type="submit" class="btn"><i class='fas fa-trash'></i></button>
                </form>
                HTML;
                return $editbtn . " " . $deletebtn;
                // return $deletebtn;
            })
            ->rawColumns(['actions', 'project_name', 'project_unit', 'agreement_no', 'lessee_name', 'approval_status', 'notice_period'])
            ->make(true);
    }

    public function viewAgreement(LeaseRentalAgreement $leaseRentalAgreement)
    {
        $lessee = LesseEntry::where('id', $leaseRentalAgreement->lessee_id)->get();
        $rateItem = RateItemDefinition::where('id', $leaseRentalAgreement->rate_item_id)->get();
        return view('mall_software.lease_rental_agreement.view_agreement', compact('leaseRentalAgreement', 'lessee', 'rateItem'));
    }

    public function viewAgreementList(LeaseRentalAgreement $leaseRentalAgreement)
    {
        // $data = LeaseRentalAgreement::where('id', $leaseRentalAgreement->id)->with('getLesseeName', 'getRateItem', 'getProjectUnit', 'getType')->get();
        $data = LeaseRentalAgreementDetail::with('getRateItem')->where('agreement_no_id', $leaseRentalAgreement->id)->get();
        $gstAmt = 0;
        $tdsAmt = 0;
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('rate_type', function ($data) {
                return $data->getRateItem->rate_type ?? "--";
            })
            ->addColumn('rate_item_name', function ($data) {
                return $data->getRateItem->rate_item_name ?? "--";
            })
            ->addColumn('area', function ($data) {
                return $data->area;
            })
            ->addColumn('rate', function ($data) {
                return $data->rate ?? "--";
            })
            ->addColumn('total', function ($data) {
                return $data->total_amount ?? "--";
            })
            ->addColumn('gst_rate', function ($data) {
                if ($data->gst_rate == null) {
                    return "--";
                } else {
                    return $data->gst_rate . "%" ?? "--";
                }
            })
            ->addColumn('gst_amount', function ($data) use (&$gstAmt) {
                return $data->gst_amount ?? "--";
            })
            ->addColumn('tds_rate', function ($data) {
                if ($data->tds_rate == null) {
                    return "--";
                } else {
                    return $data->tds_rate . "%" ?? "--";
                }
            })
            ->addColumn('tds_amount', function ($data) use (&$tdsAmt) {
                return $data->tds_amount ?? "--";
            })
            ->addColumn('net_amount', function ($data) use (&$gstAmt, &$tdsAmt) {
                return $data->net_amount ?? "--";
            })
            ->rawColumns(['rate_item_name', 'agreement_no', 'total', 'gst_amount', 'tds_amount', 'net_amount', 'area', 'tds_rate', 'rate', 'gst_rate', 'rate_type'])
            ->make(true);
    }

    public function getUnitList(Request $request)
    {
        $projectUnits = UnitDefinition::where('project_id', $request->project_id)->get();
        return response()->json(['data' => $projectUnits, 'status' => true]);
    }
    public function getFloorsList(Request $request)
    {
        $unitDefinition = UnitDefinition::where('id', $request->unit_id)->first();

        $type_id = $unitDefinition->type_id;
        $floors = Floor::find($unitDefinition->floor_id);
        $unit_types = UnitType::find($type_id);
        $data = [
            'floor' => $floors,
            'unit_types' => $unit_types,
        ];
        return response()->json(['data' => $data, 'status' => true]);
    }
    public function getTotal(Request $request)
    {
        $area_value = $request->area_value;
        $rate_amount = $request->rate_amount;
        $gst_rate = $request->gst_rate;
        $tds_rate = $request->tds_rate;
        $gst_amount = '';
        $tds_amount = '';
        $net_amount = '';

        $total_amount = $area_value * $rate_amount;
        if ($total_amount) {
            $gst_amount = ($gst_rate / 100) * $total_amount;
            $tds_amount = ($tds_rate / 100) * $total_amount;
        }
        $net_amount = ($total_amount + $gst_amount) - $tds_amount;
        return response()->json([
            'total_amount' => $total_amount,
            'gst_amount' => $gst_amount,
            'tds_amount' => $tds_amount,
            'net_amount' => $net_amount,
        ]);
    }

    public function getSecurityDepositList()
    {
        $data = RateItemDefinition::where('rate_type', 'Security Deposit')->get();
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('checkbox', function ($data) {
                $btn = '<td><label><input data-id="' . $data->id . '" class="security_deposit" type="checkbox" name="security_deposit_select[' . $data->id . ']" /><span></span></label></td>';
                return $btn;
            })
            ->addColumn('rate_item_name', function ($data) {
                $rate_item_name = '<input type="hidden" readonly style="border-bottom: 0" class="rate_item_name' . $data->id . '" value="' . $data->id . '" name="SD_rate_definition_id[' . $data->id . ']">';
                $rate_item_name .= '<input readonly style="border-bottom: 0" class="rate_item_name' . $data->id . '" value="' . $data->rate_item_name . '">';
                return $rate_item_name;
            })
            ->addColumn('calculation_method', function ($data) {
                $calculation_method = '<input readonly style="border-bottom: 0" class="calculation_method' . $data->id . '" value="' . $data->rate_calculation_method . '" name="SD_calculation_method[' . $data->id . ']">';
                return $calculation_method;
            })
            ->addColumn('area_input', function ($data) {
                $area = '<input type="text" class="security_deposit_area_input_' . $data->id . '" name="SD_area_input[' . $data->id . ']">';
                return $area;
            })
            ->addColumn('rate', function ($data) {
                $rate_amount = '<input readonly style="border-bottom: 0" class="rate_amount_' . $data->id . '" value="' . $data->default_rate . '" name="SD_rate[' . $data->id . ']">';
                return $rate_amount;
            })
            ->addColumn('total_amount', function ($data) {
                $total_amount = '<input readonly style="border-bottom: 0" type="text" class="total_amount_' . $data->id . '" name="SD_total_amount[' . $data->id . ']">';
                return $total_amount;
            })
            ->addColumn('GST_rate', function ($data) {
                if ($data->gst_percentage == null) {
                    return "--";
                } else {
                    $gst_rate = '<input readonly style="border-bottom: 0" class="gst_rate_' . $data->id . '" value="' . $data->gst_percentage . '" name="SD_gst_rate[' . $data->id . ']">';
                    return $gst_rate;
                }
            })
            ->addColumn('GST_Amount', function ($data) {
                $gst_amount = '<input readonly style="border-bottom: 0" class="gst_amount_' . $data->id . '" name="SD_gst_amount[' . $data->id . ']">';
                return $gst_amount;
            })
            ->addColumn('TDS_rate', function ($data) {
                if ($data->tds_percentage == null) {
                    return '--';
                } else {
                    $tds_rate = '<input readonly style="border-bottom: 0" class="tds_rate_' . $data->id . '" value="' . $data->tds_percentage . '" name="SD_tds_rate[' . $data->id . ']">';
                    return $tds_rate;
                }
            })
            ->addColumn('TDS_Amount', function ($data) {
                $tds_amount = '<input readonly style="border-bottom: 0" class="tds_amount_' . $data->id . '" name="SD_tds_amount[' . $data->id . ']">';
                return $tds_amount;
            })
            ->addColumn('Other Taxes', function ($data) {
                return ($data->other_taxes) ? $data->other_taxes : "--";
            })
            ->addColumn('Net_Amount', function ($data) {
                $net_amount = '<input readonly style="border-bottom: 0" class="net_amount_' . $data->id . '" name="SD_net_amount[' . $data->id . ']">';
                return $net_amount;
            })
            ->rawColumns(['checkbox', 'rate_item_name', 'calculation_method', 'area_input', 'rate', 'GST_rate', 'GST_Amount', 'TDS_rate', 'TDS_Amount', 'Other Taxes', 'Net_Amount', 'total_amount'])
            ->make(true);
    }

    public function getFixedChargesList()
    {
        $data = RateItemDefinition::where('rate_type', 'Fixed Charges')->get();
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('checkbox', function ($data) {
                $btn = '<td><label><input data-id="' . $data->id . '" class="fixedCharges" type="checkbox" name="fixed_charges_select[' . $data->id . ']" /><span></span></label></td>';
                return $btn;
            })
            ->addColumn('rate_item_name', function ($data) {
                $rate_item_name = '<input type="hidden" readonly style="border-bottom: 0" class="rate_item_name' . $data->id . '" value="' . $data->id . '" name="FC_rate_definition_id[' . $data->id . ']">';
                $rate_item_name .= '<input readonly style="border-bottom: 0" class="rate_item_name' . $data->id . '" value="' . $data->rate_item_name . '">';
                return $rate_item_name;
            })
            ->addColumn('calculation_method', function ($data) {
                $calculation_method = '<input readonly style="border-bottom: 0" class="calculation_method' . $data->id . '" value="' . $data->rate_calculation_method . '" name="FC_calculation_method[' . $data->id . ']">';
                return $calculation_method;
            })
            ->addColumn('area_input', function ($data) {
                $area = '<input type="text" class="fixed_charges_area_input_' . $data->id . '" name="FC_area_input[' . $data->id . ']">';
                return $area;
            })
            ->addColumn('rate', function ($data) {
                $rate_amount = '<input readonly style="border-bottom: 0" class="rate_amount_' . $data->id . '" value="' . $data->default_rate . '" name="FC_rate[' . $data->id . ']">';
                return $rate_amount;
            })
            ->addColumn('total_amount', function ($data) {
                $total_amount = '<input readonly style="border-bottom: 0" type="text" class="total_amount_' . $data->id . '" name="FC_total_amount[' . $data->id . ']">';
                return $total_amount;
            })
            ->addColumn('GST_rate', function ($data) {
                if ($data->gst_percentage == null) {
                    return "--";
                } else {
                    $gst_rate = '<input readonly style="border-bottom: 0" class="gst_rate_' . $data->id . '" value="' . $data->gst_percentage . '" name="FC_gst_rate[' . $data->id . ']">';
                    return $gst_rate;
                }
            })
            ->addColumn('GST_Amount', function ($data) {
                $gst_amount = '<input readonly style="border-bottom: 0" class="gst_amount_' . $data->id . '" name="FC_gst_amount[' . $data->id . ']">';
                return $gst_amount;
            })
            ->addColumn('TDS_rate', function ($data) {
                if ($data->tds_percentage == null) {
                    return '--';
                } else {
                    $tds_rate = '<input readonly style="border-bottom: 0" class="tds_rate_' . $data->id . '" value="' . $data->tds_percentage . '" name="FC_tds_rate[' . $data->id . ']">';
                    return $tds_rate;
                }
            })
            ->addColumn('TDS_Amount', function ($data) {
                $tds_amount = '<input readonly style="border-bottom: 0" class="tds_amount_' . $data->id . '" name="FC_tds_amount[' . $data->id . ']">';
                return $tds_amount;
            })
            ->addColumn('Other Taxes', function ($data) {
                if ($data->other_taxes == null) {
                    return "--";
                } else {
                    return $data->other_taxes;
                }
            })
            ->addColumn('Net_Amount', function ($data) {
                $net_amount = '<input readonly style="border-bottom: 0" class="net_amount_' . $data->id . '" name="FC_net_amount[' . $data->id . ']">';
                return $net_amount;
            })
            ->rawColumns(['checkbox', 'rate_item_name', 'calculation_method', 'GST_Amount', 'TDS_Amount', 'Other Taxes', 'Net_Amount', 'total_amount', 'area_input', 'rate', 'GST_rate', 'TDS_rate'])
            ->make(true);
    }

    public function getVariableChargesList()
    {
        $data = RateItemDefinition::where('rate_type', 'Variable Charges')->get();
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('checkbox', function ($data) {
                $btn = '<td><label><input id="" type="checkbox" name="variable_charges_select[' . $data->id . ']" /><span></span></label></td>';
                return $btn;
            })
            ->addColumn('rate_item_name', function ($data) {
                $rate_item_name = '<input type="hidden" readonly style="border-bottom: 0" class="rate_item_name' . $data->id . '" value="' . $data->id . '" name="VC_rate_definition_id[' . $data->id . ']">';
                $rate_item_name .= '<input readonly style="border-bottom: 0" class="rate_item_name' . $data->id . '" value="' . $data->rate_item_name . '">';
                return $rate_item_name;
            })
            ->addColumn('rate', function ($data) {
                $rate_amount = '<input id="" readonly style="border-bottom: 0" type="text" name="VC_rate[' . $data->id . ']" value="' . $data->default_rate . '"/>';
                return $rate_amount;
            })
            // ->addColumn('amount', function ($data) {
            //     return "-";
            // })
            ->rawColumns(['checkbox', 'rate_item_name', 'rate', 'amount'])
            ->make(true);
    }

    public function store(LeaseRentalAgreementStoreRequest $request, LeaseRentalAgreementStoreAction $action)
    {
        try {
            $response = $action->execute(collect($request->validated()));
            if ($response) {
                return redirect()->route('mall-software.lease_rental_agreement')->with('success', 'Successfully Created');
            } else {
                return redirect()->back()->with(['error' => 'Something went wrong']);
            }
        } catch (\Throwable $th) {
            info($th);
            return redirect()->back()->with(['error' => 'Something went wrong']);
        }
    }


    public function edit(LeaseRentalAgreement $leaseRentalAgreement)
    {
        $projects = Project::where('status', 1)->get();
        $unit_definition = UnitDefinition::where('id', $leaseRentalAgreement->project_units_id)->first();
        $floor = Floor::where('id', $unit_definition->floor_id)->first();
        $type = UnitType::where('id', $unit_definition->type_id)->first();
        $lesseeData = LesseEntry::where('id', $leaseRentalAgreement->lessee_id)->first();
        return view('mall_software.lease_rental_agreement.edit', compact('floor', 'unit_definition', 'projects', 'lesseeData', 'leaseRentalAgreement', 'type'));
    }

    public function update(LeaseRentalAgreementUpdateRequest $request, LeaseRentalAgreementUpdateAction $action, LeaseRentalAgreement $leaseRentalAgreement)
    {
        try {
            $response = $action->execute($leaseRentalAgreement, collect($request->validated()));
            if ($response) {
                return redirect()->route('mall-software.lease_rental_agreement')->with('success', 'Successfully Updated');
            } else {
                return redirect()->back()->with(['error' => 'Something went wrong']);
            }
        } catch (\Throwable $th) {
            info($th);
            return redirect()->back()->with(['error' => 'Something went wrong']);
        }
    }

    public function destroy(LeaseRentalAgreement $leaseRentalAgreement)
    {
        try {
            $leaseRentalAgreement->delete();
            return redirect()->route('mall-software.lease_rental_agreement')->with('success', 'Successfully Deleted');
        } catch (\Throwable $th) {
            info($th);
            return redirect()->back()->with(['error' => 'Something went wrong']);
        }
    }
}
