<?php

namespace App\Providers;

use Illuminate\Routing\RouteRegistrar;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Str;
use Illuminate\View\Compilers\BladeCompiler;
use Symfony\Component\Finder\Finder;

abstract class ModuleServiceProvider extends ServiceProvider
{
    /**
     * This namespace is applied to your controller Routes.
     *
     * In addition, it is set as the URL generator's root namespace.
     *
     * @var string
     */
    protected $namespace = 'App\Http\Controllers';

    /**
     *
     * @var string
     */
    protected $name = null;

    /**
     *
     * @var string
     */
    protected $path =  null;

    /**
     *
     * @var bool
     */
    protected $useRootDomain = false;

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {
        $this->mapRoutes();
        $this->configureView();
        $this->configureLang();
        $this->configureComponents();
    }

    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        //
    }

    protected function getModulePath(): string
    {
        return rtrim($this->path, '/');
    }

    /**
     * Configure the views for the module.
     *
     * @return void
     */
    protected function configureView()
    {
        View::addNamespace($this->name, $this->getModulePath() . '/Resources/views');
    }

    /**
     * Configure the lang for the module.
     *
     * @return void
     */
    protected function configureLang()
    {
        Lang::addNamespace($this->name, $this->getModulePath() . '/Resources/lang');
    }

    /**
     * Define the Routes for the module.
     *
     * @return void
     */
    public function mapRoutes()
    {
        $this->routeRegistrar()->group(function () {
            $this->mapApiRoutes();

            $this->mapWebRoutes();
        });
    }

    /**
     * Get the route registrar for the module.
     *
     * @return RouteRegistrar
     */
    protected function routeRegistrar(): RouteRegistrar
    {
        $routeRegistrar = Route::name($this->name . '.');

        $useSubdomain = config('module.subdomain', false);

        if (!$useSubdomain && $this->useRootDomain) {
            return $routeRegistrar;
        }

        if ($useSubdomain && !$this->useRootDomain) {
            return $routeRegistrar->domain($this->name . '.' . config('module.domain'));
        }

        if ($useSubdomain && $this->useRootDomain) {
            return $routeRegistrar->domain(config('module.domain'));
        }

        return $routeRegistrar->prefix($this->name);
    }

    /**
     * Define the "web" Routes for the application.
     *
     * These Routes all receive session state, CSRF protection, etc.
     *
     * @return void
     */
    protected function mapWebRoutes()
    {
        Route::middleware('web')
//            ->namespace($this->namespace)
            ->group($this->getModulePath() . '/Routes/web.php');
    }

    /**
     * Define the "api" Routes for the application.
     *
     * These Routes are typically stateless.
     *
     * @return void
     */
    protected function mapApiRoutes()
    {
        Route::prefix('api')
            ->name('api.')
            ->middleware('api')
//            ->namespace($this->namespace)
            ->group($this->getModulePath() . '/Routes/api.php');
    }

    /**
     * Configure the Blade components.
     *
     * @return void
     */
    protected function configureComponents()
    {
        $this->callAfterResolving(BladeCompiler::class, function () {
            $this->registerComponents($this->getModulePath() . '/Resources/views/components');
        });
    }

    /**
     * Register the given component.
     *
     * @param  string  $component
     * @return void
     */
    protected function registerComponent(string $component)
    {
        Blade::component($this->name . '::components.'.$component, $this->name . '-'.$component);
    }

    /**
     * Register all of the commands in the given directory.
     *
     * @param  array|string  $paths
     * @return void
     */
    protected function registerComponents($paths)
    {
        $paths = array_unique(Arr::wrap($paths));

        $paths = array_filter($paths, function ($path) {
            return is_dir($path);
        });

        if (empty($paths)) {
            return;
        }

        foreach ((new Finder)->in($paths)->files() as $component) {
            $component = Str::of(
                $component->getFilenameWithoutExtension()
            )->replaceLast(".blade", "");

            $this->registerComponent($component);
        }
    }
}
