/// <reference path="global.d.ts" />
/// <reference types="react" />
import * as React$2 from 'react';
import React__default, { CSSProperties, SVGProps, AriaAttributes, DOMAttributes as DOMAttributes$1, AriaRole, ClipboardEventHandler, CompositionEventHandler, ReactEventHandler, FormEventHandler, HTMLAttributeAnchorTarget, HTMLAttributeReferrerPolicy, ReactNode, ReactElement, KeyboardEvent as KeyboardEvent$2, FocusEvent as FocusEvent$1, SyntheticEvent, JSX as JSX$1, ElementType, JSXElementConstructor, HTMLAttributes, Key as Key$2, Dispatch, SetStateAction, ReactHTML, ForwardedRef } from 'react';

declare const SignatureSaveMode: {
    readonly ALWAYS: "ALWAYS";
    readonly NEVER: "NEVER";
    readonly USING_UI: "USING_UI";
};
type ISignatureSaveMode = (typeof SignatureSaveMode)[keyof typeof SignatureSaveMode];

/**
 * Copyright (c) 2014-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

/**
 * Immutable data encourages pure functions (data-in, data-out) and lends itself
 * to much simpler application development and enabling techniques from
 * functional programming such as lazy evaluation.
 *
 * While designed to bring these powerful functional concepts to JavaScript, it
 * presents an Object-Oriented API familiar to Javascript engineers and closely
 * mirroring that of Array, Map, and Set. It is easy and efficient to convert to
 * and from plain Javascript types.
 *
 * ## How to read these docs
 *
 * In order to better explain what kinds of values the Immutable.js API expects
 * and produces, this documentation is presented in a statically typed dialect of
 * JavaScript (like [Flow][] or [TypeScript][]). You *don't need* to use these
 * type checking tools in order to use Immutable.js, however becoming familiar
 * with their syntax will help you get a deeper understanding of this API.
 *
 * **A few examples and how to read them.**
 *
 * All methods describe the kinds of data they accept and the kinds of data
 * they return. For example a function which accepts two numbers and returns
 * a number would look like this:
 *
 * ```js
 * sum(first: number, second: number): number
 * ```
 *
 * Sometimes, methods can accept different kinds of data or return different
 * kinds of data, and this is described with a *type variable*, which is
 * typically in all-caps. For example, a function which always returns the same
 * kind of data it was provided would look like this:
 *
 * ```js
 * identity<T>(value: T): T
 * ```
 *
 * Type variables are defined with classes and referred to in methods. For
 * example, a class that holds onto a value for you might look like this:
 *
 * ```js
 * class Box<T> {
 *   constructor(value: T)
 *   getValue(): T
 * }
 * ```
 *
 * In order to manipulate Immutable data, methods that we're used to affecting
 * a Collection instead return a new Collection of the same type. The type
 * `this` refers to the same kind of class. For example, a List which returns
 * new Lists when you `push` a value onto it might look like:
 *
 * ```js
 * class List<T> {
 *   push(value: T): this
 * }
 * ```
 *
 * Many methods in Immutable.js accept values which implement the JavaScript
 * [Iterable][] protocol, and might appear like `Iterable<string>` for something
 * which represents sequence of strings. Typically in JavaScript we use plain
 * Arrays (`[]`) when an Iterable is expected, but also all of the Immutable.js
 * collections are iterable themselves!
 *
 * For example, to get a value deep within a structure of data, we might use
 * `getIn` which expects an `Iterable` path:
 *
 * ```
 * getIn(path: Iterable<string | number>): any
 * ```
 *
 * To use this method, we could pass an array: `data.getIn([ "key", 2 ])`.
 *
 *
 * Note: All examples are presented in the modern [ES2015][] version of
 * JavaScript. Use tools like Babel to support older browsers.
 *
 * For example:
 *
 * ```js
 * // ES2015
 * const mappedFoo = foo.map(x => x * x);
 * // ES5
 * var mappedFoo = foo.map(function (x) { return x * x; });
 * ```
 *
 * [ES2015]: https://developer.mozilla.org/en-US/docs/Web/JavaScript/New_in_JavaScript/ECMAScript_6_support_in_Mozilla
 * [TypeScript]: http://www.typescriptlang.org/
 * [Flow]: https://flowtype.org/
 * [Iterable]: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Iteration_protocols
 */



  /**
   * Lists are ordered indexed dense collections, much like a JavaScript
   * Array.
   *
   * Lists are immutable and fully persistent with O(log32 N) gets and sets,
   * and O(1) push and pop.
   *
   * Lists implement Deque, with efficient addition and removal from both the
   * end (`push`, `pop`) and beginning (`unshift`, `shift`).
   *
   * Unlike a JavaScript Array, there is no distinction between an
   * "unset" index and an index set to `undefined`. `List#forEach` visits all
   * indices from 0 to size, regardless of whether they were explicitly defined.
   */
  declare module List {

    /**
     * True if the provided value is a List
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable');
     * List.isList([]); // false
     * List.isList(List()); // true
     * ```
     */
    function isList(maybeList: any): maybeList is List<any>;

    /**
     * Creates a new List containing `values`.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable');
     * List.of(1, 2, 3, 4)
     * // List [ 1, 2, 3, 4 ]
     * ```
     *
     * Note: Values are not altered or converted in any way.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable');
     * List.of({x:1}, 2, [3], 4)
     * // List [ { x: 1 }, 2, [ 3 ], 4 ]
     * ```
     */
    function of<T>(...values: Array<T>): List<T>;
  }

  /**
   * Create a new immutable List containing the values of the provided
   * collection-like.
   *
   * Note: `List` is a factory function and not a class, and does not use the
   * `new` keyword during construction.
   *
   * <!-- runkit:activate -->
   * ```js
   * const { List, Set } = require('immutable')
   *
   * const emptyList = List()
   * // List []
   *
   * const plainArray = [ 1, 2, 3, 4 ]
   * const listFromPlainArray = List(plainArray)
   * // List [ 1, 2, 3, 4 ]
   *
   * const plainSet = Set([ 1, 2, 3, 4 ])
   * const listFromPlainSet = List(plainSet)
   * // List [ 1, 2, 3, 4 ]
   *
   * const arrayIterator = plainArray[Symbol.iterator]()
   * const listFromCollectionArray = List(arrayIterator)
   * // List [ 1, 2, 3, 4 ]
   *
   * listFromPlainArray.equals(listFromCollectionArray) // true
   * listFromPlainSet.equals(listFromCollectionArray) // true
   * listFromPlainSet.equals(listFromPlainArray) // true
   * ```
   */
  declare function List(): List<any>;
  declare function List<T>(): List<T>;
  declare function List<T>(collection: Iterable<T>): List<T>;

  interface List<T> extends Collection$1.Indexed<T> {

    /**
     * The number of items in this List.
     */
    readonly size: number;

    // Persistent changes

    /**
     * Returns a new List which includes `value` at `index`. If `index` already
     * exists in this List, it will be replaced.
     *
     * `index` may be a negative number, which indexes back from the end of the
     * List. `v.set(-1, "value")` sets the last item in the List.
     *
     * If `index` larger than `size`, the returned List's `size` will be large
     * enough to include the `index`.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * const originalList = List([ 0 ]);
     * // List [ 0 ]
     * originalList.set(1, 1);
     * // List [ 0, 1 ]
     * originalList.set(0, 'overwritten');
     * // List [ "overwritten" ]
     * originalList.set(2, 2);
     * // List [ 0, undefined, 2 ]
     *
     * List().set(50000, 'value').size;
     * // 50001
     * ```
     *
     * Note: `set` can be used in `withMutations`.
     */
    set(index: number, value: T): List<T>;

    /**
     * Returns a new List which excludes this `index` and with a size 1 less
     * than this List. Values at indices above `index` are shifted down by 1 to
     * fill the position.
     *
     * This is synonymous with `list.splice(index, 1)`.
     *
     * `index` may be a negative number, which indexes back from the end of the
     * List. `v.delete(-1)` deletes the last item in the List.
     *
     * Note: `delete` cannot be safely used in IE8
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * List([ 0, 1, 2, 3, 4 ]).delete(0);
     * // List [ 1, 2, 3, 4 ]
     * ```
     *
     * Since `delete()` re-indexes values, it produces a complete copy, which
     * has `O(N)` complexity.
     *
     * Note: `delete` *cannot* be used in `withMutations`.
     *
     * @alias remove
     */
    delete(index: number): List<T>;
    remove(index: number): List<T>;

    /**
     * Returns a new List with `value` at `index` with a size 1 more than this
     * List. Values at indices above `index` are shifted over by 1.
     *
     * This is synonymous with `list.splice(index, 0, value)`.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * List([ 0, 1, 2, 3, 4 ]).insert(6, 5)
     * // List [ 0, 1, 2, 3, 4, 5 ]
     * ```
     *
     * Since `insert()` re-indexes values, it produces a complete copy, which
     * has `O(N)` complexity.
     *
     * Note: `insert` *cannot* be used in `withMutations`.
     */
    insert(index: number, value: T): List<T>;

    /**
     * Returns a new List with 0 size and no values in constant time.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * List([ 1, 2, 3, 4 ]).clear()
     * // List []
     * ```
     *
     * Note: `clear` can be used in `withMutations`.
     */
    clear(): List<T>;

    /**
     * Returns a new List with the provided `values` appended, starting at this
     * List's `size`.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * List([ 1, 2, 3, 4 ]).push(5)
     * // List [ 1, 2, 3, 4, 5 ]
     * ```
     *
     * Note: `push` can be used in `withMutations`.
     */
    push(...values: Array<T>): List<T>;

    /**
     * Returns a new List with a size ones less than this List, excluding
     * the last index in this List.
     *
     * Note: this differs from `Array#pop` because it returns a new
     * List rather than the removed value. Use `last()` to get the last value
     * in this List.
     *
     * ```js
     * List([ 1, 2, 3, 4 ]).pop()
     * // List[ 1, 2, 3 ]
     * ```
     *
     * Note: `pop` can be used in `withMutations`.
     */
    pop(): List<T>;

    /**
     * Returns a new List with the provided `values` prepended, shifting other
     * values ahead to higher indices.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * List([ 2, 3, 4]).unshift(1);
     * // List [ 1, 2, 3, 4 ]
     * ```
     *
     * Note: `unshift` can be used in `withMutations`.
     */
    unshift(...values: Array<T>): List<T>;

    /**
     * Returns a new List with a size ones less than this List, excluding
     * the first index in this List, shifting all other values to a lower index.
     *
     * Note: this differs from `Array#shift` because it returns a new
     * List rather than the removed value. Use `first()` to get the first
     * value in this List.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * List([ 0, 1, 2, 3, 4 ]).shift();
     * // List [ 1, 2, 3, 4 ]
     * ```
     *
     * Note: `shift` can be used in `withMutations`.
     */
    shift(): List<T>;

    /**
     * Returns a new List with an updated value at `index` with the return
     * value of calling `updater` with the existing value, or `notSetValue` if
     * `index` was not set. If called with a single argument, `updater` is
     * called with the List itself.
     *
     * `index` may be a negative number, which indexes back from the end of the
     * List. `v.update(-1)` updates the last item in the List.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * const list = List([ 'a', 'b', 'c' ])
     * const result = list.update(2, val => val.toUpperCase())
     * // List [ "a", "b", "C" ]
     * ```
     *
     * This can be very useful as a way to "chain" a normal function into a
     * sequence of methods. RxJS calls this "let" and lodash calls it "thru".
     *
     * For example, to sum a List after mapping and filtering:
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * function sum(collection) {
     *   return collection.reduce((sum, x) => sum + x, 0)
     * }
     *
     * List([ 1, 2, 3 ])
     *   .map(x => x + 1)
     *   .filter(x => x % 2 === 0)
     *   .update(sum)
     * // 6
     * ```
     *
     * Note: `update(index)` can be used in `withMutations`.
     *
     * @see `Map#update`
     */
    update(index: number, notSetValue: T, updater: (value: T) => T): this;
    update(index: number, updater: (value: T) => T): this;
    update<R>(updater: (value: this) => R): R;

    /**
     * Returns a new List with size `size`. If `size` is less than this
     * List's size, the new List will exclude values at the higher indices.
     * If `size` is greater than this List's size, the new List will have
     * undefined values for the newly available indices.
     *
     * When building a new List and the final size is known up front, `setSize`
     * used in conjunction with `withMutations` may result in the more
     * performant construction.
     */
    setSize(size: number): List<T>;


    // Deep persistent changes

    /**
     * Returns a new List having set `value` at this `keyPath`. If any keys in
     * `keyPath` do not exist, a new immutable Map will be created at that key.
     *
     * Index numbers are used as keys to determine the path to follow in
     * the List.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable')
     * const list = List([ 0, 1, 2, List([ 3, 4 ])])
     * list.setIn([3, 0], 999);
     * // List [ 0, 1, 2, List [ 999, 4 ] ]
     * ```
     *
     * Plain JavaScript Object or Arrays may be nested within an Immutable.js
     * Collection, and setIn() can update those values as well, treating them
     * immutably by creating new copies of those values with the changes applied.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable')
     * const list = List([ 0, 1, 2, { plain: 'object' }])
     * list.setIn([3, 'plain'], 'value');
     * // List([ 0, 1, 2, { plain: 'value' }])
     * ```
     *
     * Note: `setIn` can be used in `withMutations`.
     */
    setIn(keyPath: Iterable<any>, value: any): this;

    /**
     * Returns a new List having removed the value at this `keyPath`. If any
     * keys in `keyPath` do not exist, no change will occur.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable')
     * const list = List([ 0, 1, 2, List([ 3, 4 ])])
     * list.deleteIn([3, 0]);
     * // List [ 0, 1, 2, List [ 4 ] ]
     * ```
     *
     * Plain JavaScript Object or Arrays may be nested within an Immutable.js
     * Collection, and removeIn() can update those values as well, treating them
     * immutably by creating new copies of those values with the changes applied.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable')
     * const list = List([ 0, 1, 2, { plain: 'object' }])
     * list.removeIn([3, 'plain']);
     * // List([ 0, 1, 2, {}])
     * ```
     *
     * Note: `deleteIn` *cannot* be safely used in `withMutations`.
     *
     * @alias removeIn
     */
    deleteIn(keyPath: Iterable<any>): this;
    removeIn(keyPath: Iterable<any>): this;

    /**
     * Note: `updateIn` can be used in `withMutations`.
     *
     * @see `Map#updateIn`
     */
    updateIn(keyPath: Iterable<any>, notSetValue: any, updater: (value: any) => any): this;
    updateIn(keyPath: Iterable<any>, updater: (value: any) => any): this;

    /**
     * Note: `mergeIn` can be used in `withMutations`.
     *
     * @see `Map#mergeIn`
     */
    mergeIn(keyPath: Iterable<any>, ...collections: Array<any>): this;

    /**
     * Note: `mergeDeepIn` can be used in `withMutations`.
     *
     * @see `Map#mergeDeepIn`
     */
    mergeDeepIn(keyPath: Iterable<any>, ...collections: Array<any>): this;

    // Transient changes

    /**
     * Note: Not all methods can be safely used on a mutable collection or within
     * `withMutations`! Check the documentation for each method to see if it
     * allows being used in `withMutations`.
     *
     * @see `Map#withMutations`
     */
    withMutations(mutator: (mutable: this) => any): this;

    /**
     * An alternative API for withMutations()
     *
     * Note: Not all methods can be safely used on a mutable collection or within
     * `withMutations`! Check the documentation for each method to see if it
     * allows being used in `withMutations`.
     *
     * @see `Map#asMutable`
     */
    asMutable(): this;

    /**
     * @see `Map#wasAltered`
     */
    wasAltered(): boolean;

    /**
     * @see `Map#asImmutable`
     */
    asImmutable(): this;

    // Sequence algorithms

    /**
     * Returns a new List with other values or collections concatenated to this one.
     *
     * Note: `concat` can be used in `withMutations`.
     *
     * @alias merge
     */
    concat<C>(...valuesOrCollections: Array<Iterable<C> | C>): List<T | C>;
    merge<C>(...collections: Array<Iterable<C>>): List<T | C>;

    /**
     * Returns a new List with values passed through a
     * `mapper` function.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * List([ 1, 2 ]).map(x => 10 * x)
     * // List [ 10, 20 ]
     * ```
     */
    map<M>(
      mapper: (value: T, key: number, iter: this) => M,
      context?: any
    ): List<M>;

    /**
     * Flat-maps the List, returning a new List.
     *
     * Similar to `list.map(...).flatten(true)`.
     */
    flatMap<M>(
      mapper: (value: T, key: number, iter: this) => Iterable<M>,
      context?: any
    ): List<M>;

    /**
     * Returns a new List with only the values for which the `predicate`
     * function returns true.
     *
     * Note: `filter()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filter<F extends T>(
      predicate: (value: T, index: number, iter: this) => value is F,
      context?: any
    ): List<F>;
    filter(
      predicate: (value: T, index: number, iter: this) => any,
      context?: any
    ): this;

    /**
     * Returns a List "zipped" with the provided collection.
     *
     * Like `zipWith`, but using the default `zipper`: creating an `Array`.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * const a = List([ 1, 2, 3 ]);
     * const b = List([ 4, 5, 6 ]);
     * const c = a.zip(b); // List [ [ 1, 4 ], [ 2, 5 ], [ 3, 6 ] ]
     * ```
     */
    zip<U>(other: Collection$1<any, U>): List<[T,U]>;
    zip<U,V>(other: Collection$1<any, U>, other2: Collection$1<any,V>): List<[T,U,V]>;
    zip(...collections: Array<Collection$1<any, any>>): List<any>;

    /**
     * Returns a List "zipped" with the provided collections.
     *
     * Unlike `zip`, `zipAll` continues zipping until the longest collection is
     * exhausted. Missing values from shorter collections are filled with `undefined`.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * const a = List([ 1, 2 ]);
     * const b = List([ 3, 4, 5 ]);
     * const c = a.zipAll(b); // List [ [ 1, 3 ], [ 2, 4 ], [ undefined, 5 ] ]
     * ```
     *
     * Note: Since zipAll will return a collection as large as the largest
     * input, some results may contain undefined values. TypeScript cannot
     * account for these without cases (as of v2.5).
     */
    zipAll<U>(other: Collection$1<any, U>): List<[T,U]>;
    zipAll<U,V>(other: Collection$1<any, U>, other2: Collection$1<any,V>): List<[T,U,V]>;
    zipAll(...collections: Array<Collection$1<any, any>>): List<any>;

    /**
     * Returns a List "zipped" with the provided collections by using a
     * custom `zipper` function.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { List } = require('immutable');" }
     * -->
     * ```js
     * const a = List([ 1, 2, 3 ]);
     * const b = List([ 4, 5, 6 ]);
     * const c = a.zipWith((a, b) => a + b, b);
     * // List [ 5, 7, 9 ]
     * ```
     */
    zipWith<U, Z>(
      zipper: (value: T, otherValue: U) => Z,
      otherCollection: Collection$1<any, U>
    ): List<Z>;
    zipWith<U, V, Z>(
      zipper: (value: T, otherValue: U, thirdValue: V) => Z,
      otherCollection: Collection$1<any, U>,
      thirdCollection: Collection$1<any, V>
    ): List<Z>;
    zipWith<Z>(
      zipper: (...any: Array<any>) => Z,
      ...collections: Array<Collection$1<any, any>>
    ): List<Z>;
  }


  /**
   * Immutable Map is an unordered Collection.Keyed of (key, value) pairs with
   * `O(log32 N)` gets and `O(log32 N)` persistent sets.
   *
   * Iteration order of a Map is undefined, however is stable. Multiple
   * iterations of the same Map will iterate in the same order.
   *
   * Map's keys can be of any type, and use `Immutable.is` to determine key
   * equality. This allows the use of any value (including NaN) as a key.
   *
   * Because `Immutable.is` returns equality based on value semantics, and
   * Immutable collections are treated as values, any Immutable collection may
   * be used as a key.
   *
   * <!-- runkit:activate -->
   * ```js
   * const { Map, List } = require('immutable');
   * Map().set(List([ 1 ]), 'listofone').get(List([ 1 ]));
   * // 'listofone'
   * ```
   *
   * Any JavaScript object may be used as a key, however strict identity is used
   * to evaluate key equality. Two similar looking objects will represent two
   * different keys.
   *
   * Implemented by a hash-array mapped trie.
   */
  declare module Map$1 {

    /**
     * True if the provided value is a Map
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * Map.isMap({}) // false
     * Map.isMap(Map()) // true
     * ```
     */
    function isMap(maybeMap: any): maybeMap is Map$1<any, any>;

    /**
     * Creates a new Map from alternating keys and values
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * Map.of(
     *   'key', 'value',
     *   'numerical value', 3,
     *    0, 'numerical key'
     * )
     * // Map { 0: "numerical key", "key": "value", "numerical value": 3 }
     * ```
     *
     * @deprecated Use Map([ [ 'k', 'v' ] ]) or Map({ k: 'v' })
     */
    function of(...keyValues: Array<any>): Map$1<any, any>;
  }

  /**
   * Creates a new Immutable Map.
   *
   * Created with the same key value pairs as the provided Collection.Keyed or
   * JavaScript Object or expects a Collection of [K, V] tuple entries.
   *
   * Note: `Map` is a factory function and not a class, and does not use the
   * `new` keyword during construction.
   *
   * <!-- runkit:activate -->
   * ```js
   * const { Map } = require('immutable')
   * Map({ key: "value" })
   * Map([ [ "key", "value" ] ])
   * ```
   *
   * Keep in mind, when using JS objects to construct Immutable Maps, that
   * JavaScript Object properties are always strings, even if written in a
   * quote-less shorthand, while Immutable Maps accept keys of any type.
   *
   * <!-- runkit:activate
   *      { "preamble": "const { Map } = require('immutable');" }
   * -->
   * ```js
   * let obj = { 1: "one" }
   * Object.keys(obj) // [ "1" ]
   * assert.equal(obj["1"], obj[1]) // "one" === "one"
   *
   * let map = Map(obj)
   * assert.notEqual(map.get("1"), map.get(1)) // "one" !== undefined
   * ```
   *
   * Property access for JavaScript Objects first converts the key to a string,
   * but since Immutable Map keys can be of any type the argument to `get()` is
   * not altered.
   */
  declare function Map$1<K, V>(collection: Iterable<[K, V]>): Map$1<K, V>;
  declare function Map$1<T>(collection: Iterable<Iterable<T>>): Map$1<T, T>;
  declare function Map$1<V>(obj: {[key: string]: V}): Map$1<string, V>;
  declare function Map$1<K, V>(): Map$1<K, V>;
  declare function Map$1(): Map$1<any, any>;

  interface Map$1<K, V> extends Collection$1.Keyed<K, V> {

    /**
     * The number of entries in this Map.
     */
    readonly size: number;

    // Persistent changes

    /**
     * Returns a new Map also containing the new key, value pair. If an equivalent
     * key already exists in this Map, it will be replaced.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const originalMap = Map()
     * const newerMap = originalMap.set('key', 'value')
     * const newestMap = newerMap.set('key', 'newer value')
     *
     * originalMap
     * // Map {}
     * newerMap
     * // Map { "key": "value" }
     * newestMap
     * // Map { "key": "newer value" }
     * ```
     *
     * Note: `set` can be used in `withMutations`.
     */
    set(key: K, value: V): this;

    /**
     * Returns a new Map which excludes this `key`.
     *
     * Note: `delete` cannot be safely used in IE8, but is provided to mirror
     * the ES6 collection API.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const originalMap = Map({
     *   key: 'value',
     *   otherKey: 'other value'
     * })
     * // Map { "key": "value", "otherKey": "other value" }
     * originalMap.delete('otherKey')
     * // Map { "key": "value" }
     * ```
     *
     * Note: `delete` can be used in `withMutations`.
     *
     * @alias remove
     */
    delete(key: K): this;
    remove(key: K): this;

    /**
     * Returns a new Map which excludes the provided `keys`.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const names = Map({ a: "Aaron", b: "Barry", c: "Connor" })
     * names.deleteAll([ 'a', 'c' ])
     * // Map { "b": "Barry" }
     * ```
     *
     * Note: `deleteAll` can be used in `withMutations`.
     *
     * @alias removeAll
     */
    deleteAll(keys: Iterable<K>): this;
    removeAll(keys: Iterable<K>): this;

    /**
     * Returns a new Map containing no keys or values.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * Map({ key: 'value' }).clear()
     * // Map {}
     * ```
     *
     * Note: `clear` can be used in `withMutations`.
     */
    clear(): this;

    /**
     * Returns a new Map having updated the value at this `key` with the return
     * value of calling `updater` with the existing value.
     *
     * Similar to: `map.set(key, updater(map.get(key)))`.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const aMap = Map({ key: 'value' })
     * const newMap = aMap.update('key', value => value + value)
     * // Map { "key": "valuevalue" }
     * ```
     *
     * This is most commonly used to call methods on collections within a
     * structure of data. For example, in order to `.push()` onto a nested `List`,
     * `update` and `push` can be used together:
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map, List } = require('immutable');" }
     * -->
     * ```js
     * const aMap = Map({ nestedList: List([ 1, 2, 3 ]) })
     * const newMap = aMap.update('nestedList', list => list.push(4))
     * // Map { "nestedList": List [ 1, 2, 3, 4 ] }
     * ```
     *
     * When a `notSetValue` is provided, it is provided to the `updater`
     * function when the value at the key does not exist in the Map.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable');" }
     * -->
     * ```js
     * const aMap = Map({ key: 'value' })
     * const newMap = aMap.update('noKey', 'no value', value => value + value)
     * // Map { "key": "value", "noKey": "no valueno value" }
     * ```
     *
     * However, if the `updater` function returns the same value it was called
     * with, then no change will occur. This is still true if `notSetValue`
     * is provided.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable');" }
     * -->
     * ```js
     * const aMap = Map({ apples: 10 })
     * const newMap = aMap.update('oranges', 0, val => val)
     * // Map { "apples": 10 }
     * assert.strictEqual(newMap, map);
     * ```
     *
     * For code using ES2015 or later, using `notSetValue` is discourged in
     * favor of function parameter default values. This helps to avoid any
     * potential confusion with identify functions as described above.
     *
     * The previous example behaves differently when written with default values:
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable');" }
     * -->
     * ```js
     * const aMap = Map({ apples: 10 })
     * const newMap = aMap.update('oranges', (val = 0) => val)
     * // Map { "apples": 10, "oranges": 0 }
     * ```
     *
     * If no key is provided, then the `updater` function return value is
     * returned as well.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable');" }
     * -->
     * ```js
     * const aMap = Map({ key: 'value' })
     * const result = aMap.update(aMap => aMap.get('key'))
     * // "value"
     * ```
     *
     * This can be very useful as a way to "chain" a normal function into a
     * sequence of methods. RxJS calls this "let" and lodash calls it "thru".
     *
     * For example, to sum the values in a Map
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable');" }
     * -->
     * ```js
     * function sum(collection) {
     *   return collection.reduce((sum, x) => sum + x, 0)
     * }
     *
     * Map({ x: 1, y: 2, z: 3 })
     *   .map(x => x + 1)
     *   .filter(x => x % 2 === 0)
     *   .update(sum)
     * // 6
     * ```
     *
     * Note: `update(key)` can be used in `withMutations`.
     */
    update(key: K, notSetValue: V, updater: (value: V) => V): this;
    update(key: K, updater: (value: V) => V): this;
    update<R>(updater: (value: this) => R): R;

    /**
     * Returns a new Map resulting from merging the provided Collections
     * (or JS objects) into this Map. In other words, this takes each entry of
     * each collection and sets it on this Map.
     *
     * Note: Values provided to `merge` are shallowly converted before being
     * merged. No nested values are altered.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const one = Map({ a: 10, b: 20, c: 30 })
     * const two = Map({ b: 40, a: 50, d: 60 })
     * one.merge(two) // Map { "a": 50, "b": 40, "c": 30, "d": 60 }
     * two.merge(one) // Map { "b": 20, "a": 10, "d": 60, "c": 30 }
     * ```
     *
     * Note: `merge` can be used in `withMutations`.
     *
     * @alias concat
     */
    merge<KC, VC>(...collections: Array<Iterable<[KC, VC]>>): Map$1<K | KC, V | VC>;
    merge<C>(...collections: Array<{[key: string]: C}>): Map$1<K | string, V | C>;
    concat<KC, VC>(...collections: Array<Iterable<[KC, VC]>>): Map$1<K | KC, V | VC>;
    concat<C>(...collections: Array<{[key: string]: C}>): Map$1<K | string, V | C>;

    /**
     * Like `merge()`, `mergeWith()` returns a new Map resulting from merging
     * the provided Collections (or JS objects) into this Map, but uses the
     * `merger` function for dealing with conflicts.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const one = Map({ a: 10, b: 20, c: 30 })
     * const two = Map({ b: 40, a: 50, d: 60 })
     * one.mergeWith((oldVal, newVal) => oldVal / newVal, two)
     * // { "a": 0.2, "b": 0.5, "c": 30, "d": 60 }
     * two.mergeWith((oldVal, newVal) => oldVal / newVal, one)
     * // { "b": 2, "a": 5, "d": 60, "c": 30 }
     * ```
     *
     * Note: `mergeWith` can be used in `withMutations`.
     */
    mergeWith(
      merger: (oldVal: V, newVal: V, key: K) => V,
      ...collections: Array<Iterable<[K, V]> | {[key: string]: V}>
    ): this;

    /**
     * Like `merge()`, but when two Collections conflict, it merges them as well,
     * recursing deeply through the nested data.
     *
     * Note: Values provided to `merge` are shallowly converted before being
     * merged. No nested values are altered unless they will also be merged at
     * a deeper level.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const one = Map({ a: Map({ x: 10, y: 10 }), b: Map({ x: 20, y: 50 }) })
     * const two = Map({ a: Map({ x: 2 }), b: Map({ y: 5 }), c: Map({ z: 3 }) })
     * one.mergeDeep(two)
     * // Map {
     * //   "a": Map { "x": 2, "y": 10 },
     * //   "b": Map { "x": 20, "y": 5 },
     * //   "c": Map { "z": 3 }
     * // }
     * ```
     *
     * Note: `mergeDeep` can be used in `withMutations`.
     */
    mergeDeep(...collections: Array<Iterable<[K, V]> | {[key: string]: V}>): this;

    /**
     * Like `mergeDeep()`, but when two non-Collections conflict, it uses the
     * `merger` function to determine the resulting value.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const one = Map({ a: Map({ x: 10, y: 10 }), b: Map({ x: 20, y: 50 }) })
     * const two = Map({ a: Map({ x: 2 }), b: Map({ y: 5 }), c: Map({ z: 3 }) })
     * one.mergeDeepWith((oldVal, newVal) => oldVal / newVal, two)
     * // Map {
     * //   "a": Map { "x": 5, "y": 10 },
     * //   "b": Map { "x": 20, "y": 10 },
     * //   "c": Map { "z": 3 }
     * // }
     * ```

     * Note: `mergeDeepWith` can be used in `withMutations`.
     */
    mergeDeepWith(
      merger: (oldVal: any, newVal: any, key: any) => any,
      ...collections: Array<Iterable<[K, V]> | {[key: string]: V}>
    ): this;


    // Deep persistent changes

    /**
     * Returns a new Map having set `value` at this `keyPath`. If any keys in
     * `keyPath` do not exist, a new immutable Map will be created at that key.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const originalMap = Map({
     *   subObject: Map({
     *     subKey: 'subvalue',
     *     subSubObject: Map({
     *       subSubKey: 'subSubValue'
     *     })
     *   })
     * })
     *
     * const newMap = originalMap.setIn(['subObject', 'subKey'], 'ha ha!')
     * // Map {
     * //   "subObject": Map {
     * //     "subKey": "ha ha!",
     * //     "subSubObject": Map { "subSubKey": "subSubValue" }
     * //   }
     * // }
     *
     * const newerMap = originalMap.setIn(
     *   ['subObject', 'subSubObject', 'subSubKey'],
     *   'ha ha ha!'
     * )
     * // Map {
     * //   "subObject": Map {
     * //     "subKey": "subvalue",
     * //     "subSubObject": Map { "subSubKey": "ha ha ha!" }
     * //   }
     * // }
     * ```
     *
     * Plain JavaScript Object or Arrays may be nested within an Immutable.js
     * Collection, and setIn() can update those values as well, treating them
     * immutably by creating new copies of those values with the changes applied.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const originalMap = Map({
     *   subObject: {
     *     subKey: 'subvalue',
     *     subSubObject: {
     *       subSubKey: 'subSubValue'
     *     }
     *   }
     * })
     *
     * originalMap.setIn(['subObject', 'subKey'], 'ha ha!')
     * // Map {
     * //   "subObject": {
     * //     subKey: "ha ha!",
     * //     subSubObject: { subSubKey: "subSubValue" }
     * //   }
     * // }
     * ```
     *
     * If any key in the path exists but cannot be updated (such as a primitive
     * like number or a custom Object like Date), an error will be thrown.
     *
     * Note: `setIn` can be used in `withMutations`.
     */
    setIn(keyPath: Iterable<any>, value: any): this;

    /**
     * Returns a new Map having removed the value at this `keyPath`. If any keys
     * in `keyPath` do not exist, no change will occur.
     *
     * Note: `deleteIn` can be used in `withMutations`.
     *
     * @alias removeIn
     */
    deleteIn(keyPath: Iterable<any>): this;
    removeIn(keyPath: Iterable<any>): this;

    /**
     * Returns a new Map having applied the `updater` to the entry found at the
     * keyPath.
     *
     * This is most commonly used to call methods on collections nested within a
     * structure of data. For example, in order to `.push()` onto a nested `List`,
     * `updateIn` and `push` can be used together:
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map, List } = require('immutable')
     * const map = Map({ inMap: Map({ inList: List([ 1, 2, 3 ]) }) })
     * const newMap = map.updateIn(['inMap', 'inList'], list => list.push(4))
     * // Map { "inMap": Map { "inList": List [ 1, 2, 3, 4 ] } }
     * ```
     *
     * If any keys in `keyPath` do not exist, new Immutable `Map`s will
     * be created at those keys. If the `keyPath` does not already contain a
     * value, the `updater` function will be called with `notSetValue`, if
     * provided, otherwise `undefined`.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable')" }
     * -->
     * ```js
     * const map = Map({ a: Map({ b: Map({ c: 10 }) }) })
     * const newMap = map.updateIn(['a', 'b', 'c'], val => val * 2)
     * // Map { "a": Map { "b": Map { "c": 20 } } }
     * ```
     *
     * If the `updater` function returns the same value it was called with, then
     * no change will occur. This is still true if `notSetValue` is provided.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable')" }
     * -->
     * ```js
     * const map = Map({ a: Map({ b: Map({ c: 10 }) }) })
     * const newMap = map.updateIn(['a', 'b', 'x'], 100, val => val)
     * // Map { "a": Map { "b": Map { "c": 10 } } }
     * assert.strictEqual(newMap, aMap)
     * ```
     *
     * For code using ES2015 or later, using `notSetValue` is discourged in
     * favor of function parameter default values. This helps to avoid any
     * potential confusion with identify functions as described above.
     *
     * The previous example behaves differently when written with default values:
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable')" }
     * -->
     * ```js
     * const map = Map({ a: Map({ b: Map({ c: 10 }) }) })
     * const newMap = map.updateIn(['a', 'b', 'x'], (val = 100) => val)
     * // Map { "a": Map { "b": Map { "c": 10, "x": 100 } } }
     * ```
     *
     * Plain JavaScript Object or Arrays may be nested within an Immutable.js
     * Collection, and updateIn() can update those values as well, treating them
     * immutably by creating new copies of those values with the changes applied.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Map } = require('immutable')" }
     * -->
     * ```js
     * const map = Map({ a: { b: { c: 10 } } })
     * const newMap = map.updateIn(['a', 'b', 'c'], val => val * 2)
     * // Map { "a": { b: { c: 20 } } }
     * ```
     *
     * If any key in the path exists but cannot be updated (such as a primitive
     * like number or a custom Object like Date), an error will be thrown.
     *
     * Note: `updateIn` can be used in `withMutations`.
     */
    updateIn(keyPath: Iterable<any>, notSetValue: any, updater: (value: any) => any): this;
    updateIn(keyPath: Iterable<any>, updater: (value: any) => any): this;

    /**
     * A combination of `updateIn` and `merge`, returning a new Map, but
     * performing the merge at a point arrived at by following the keyPath.
     * In other words, these two lines are equivalent:
     *
     * ```js
     * map.updateIn(['a', 'b', 'c'], abc => abc.merge(y))
     * map.mergeIn(['a', 'b', 'c'], y)
     * ```
     *
     * Note: `mergeIn` can be used in `withMutations`.
     */
    mergeIn(keyPath: Iterable<any>, ...collections: Array<any>): this;

    /**
     * A combination of `updateIn` and `mergeDeep`, returning a new Map, but
     * performing the deep merge at a point arrived at by following the keyPath.
     * In other words, these two lines are equivalent:
     *
     * ```js
     * map.updateIn(['a', 'b', 'c'], abc => abc.mergeDeep(y))
     * map.mergeDeepIn(['a', 'b', 'c'], y)
     * ```
     *
     * Note: `mergeDeepIn` can be used in `withMutations`.
     */
    mergeDeepIn(keyPath: Iterable<any>, ...collections: Array<any>): this;

    // Transient changes

    /**
     * Every time you call one of the above functions, a new immutable Map is
     * created. If a pure function calls a number of these to produce a final
     * return value, then a penalty on performance and memory has been paid by
     * creating all of the intermediate immutable Maps.
     *
     * If you need to apply a series of mutations to produce a new immutable
     * Map, `withMutations()` creates a temporary mutable copy of the Map which
     * can apply mutations in a highly performant manner. In fact, this is
     * exactly how complex mutations like `merge` are done.
     *
     * As an example, this results in the creation of 2, not 4, new Maps:
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * const map1 = Map()
     * const map2 = map1.withMutations(map => {
     *   map.set('a', 1).set('b', 2).set('c', 3)
     * })
     * assert.equal(map1.size, 0)
     * assert.equal(map2.size, 3)
     * ```
     *
     * Note: Not all methods can be used on a mutable collection or within
     * `withMutations`! Read the documentation for each method to see if it
     * is safe to use in `withMutations`.
     */
    withMutations(mutator: (mutable: this) => any): this;

    /**
     * Another way to avoid creation of intermediate Immutable maps is to create
     * a mutable copy of this collection. Mutable copies *always* return `this`,
     * and thus shouldn't be used for equality. Your function should never return
     * a mutable copy of a collection, only use it internally to create a new
     * collection.
     *
     * If possible, use `withMutations` to work with temporary mutable copies as
     * it provides an easier to use API and considers many common optimizations.
     *
     * Note: if the collection is already mutable, `asMutable` returns itself.
     *
     * Note: Not all methods can be used on a mutable collection or within
     * `withMutations`! Read the documentation for each method to see if it
     * is safe to use in `withMutations`.
     *
     * @see `Map#asImmutable`
     */
    asMutable(): this;

    /**
     * Returns true if this is a mutable copy (see `asMutable()`) and mutative
     * alterations have been applied.
     *
     * @see `Map#asMutable`
     */
    wasAltered(): boolean;

    /**
     * The yin to `asMutable`'s yang. Because it applies to mutable collections,
     * this operation is *mutable* and may return itself (though may not
     * return itself, i.e. if the result is an empty collection). Once
     * performed, the original mutable copy must no longer be mutated since it
     * may be the immutable result.
     *
     * If possible, use `withMutations` to work with temporary mutable copies as
     * it provides an easier to use API and considers many common optimizations.
     *
     * @see `Map#asMutable`
     */
    asImmutable(): this;

    // Sequence algorithms

    /**
     * Returns a new Map with values passed through a
     * `mapper` function.
     *
     *     Map({ a: 1, b: 2 }).map(x => 10 * x)
     *     // Map { a: 10, b: 20 }
     */
    map<M>(
      mapper: (value: V, key: K, iter: this) => M,
      context?: any
    ): Map$1<K, M>;

    /**
     * @see Collection.Keyed.mapKeys
     */
    mapKeys<M>(
      mapper: (key: K, value: V, iter: this) => M,
      context?: any
    ): Map$1<M, V>;

    /**
     * @see Collection.Keyed.mapEntries
     */
    mapEntries<KM, VM>(
      mapper: (entry: [K, V], index: number, iter: this) => [KM, VM],
      context?: any
    ): Map$1<KM, VM>;

    /**
     * Flat-maps the Map, returning a new Map.
     *
     * Similar to `data.map(...).flatten(true)`.
     */
    flatMap<KM, VM>(
      mapper: (value: V, key: K, iter: this) => Iterable<[KM, VM]>,
      context?: any
    ): Map$1<KM, VM>;

    /**
     * Returns a new Map with only the entries for which the `predicate`
     * function returns true.
     *
     * Note: `filter()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filter<F extends V>(
      predicate: (value: V, key: K, iter: this) => value is F,
      context?: any
    ): Map$1<K, F>;
    filter(
      predicate: (value: V, key: K, iter: this) => any,
      context?: any
    ): this;

    /**
     * @see Collection.Keyed.flip
     */
    flip(): Map$1<V, K>;
  }


  /**
   * A type of Map that has the additional guarantee that the iteration order of
   * entries will be the order in which they were set().
   *
   * The iteration behavior of OrderedMap is the same as native ES6 Map and
   * JavaScript Object.
   *
   * Note that `OrderedMap` are more expensive than non-ordered `Map` and may
   * consume more memory. `OrderedMap#set` is amortized O(log32 N), but not
   * stable.
   */

  declare module OrderedMap {

    /**
     * True if the provided value is an OrderedMap.
     */
    function isOrderedMap(maybeOrderedMap: any): maybeOrderedMap is OrderedMap<any, any>;
  }

  /**
   * Creates a new Immutable OrderedMap.
   *
   * Created with the same key value pairs as the provided Collection.Keyed or
   * JavaScript Object or expects a Collection of [K, V] tuple entries.
   *
   * The iteration order of key-value pairs provided to this constructor will
   * be preserved in the OrderedMap.
   *
   *     let newOrderedMap = OrderedMap({key: "value"})
   *     let newOrderedMap = OrderedMap([["key", "value"]])
   *
   * Note: `OrderedMap` is a factory function and not a class, and does not use
   * the `new` keyword during construction.
   */
  declare function OrderedMap<K, V>(collection: Iterable<[K, V]>): OrderedMap<K, V>;
  declare function OrderedMap<T>(collection: Iterable<Iterable<T>>): OrderedMap<T, T>;
  declare function OrderedMap<V>(obj: {[key: string]: V}): OrderedMap<string, V>;
  declare function OrderedMap<K, V>(): OrderedMap<K, V>;
  declare function OrderedMap(): OrderedMap<any, any>;

  interface OrderedMap<K, V> extends Map$1<K, V> {

    /**
     * The number of entries in this OrderedMap.
     */
    readonly size: number;

    /**
     * Returns a new OrderedMap also containing the new key, value pair. If an
     * equivalent key already exists in this OrderedMap, it will be replaced
     * while maintaining the existing order.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { OrderedMap } = require('immutable')
     * const originalMap = OrderedMap({a:1, b:1, c:1})
     * const updatedMap = originalMap.set('b', 2)
     *
     * originalMap
     * // OrderedMap {a: 1, b: 1, c: 1}
     * updatedMap
     * // OrderedMap {a: 1, b: 2, c: 1}
     * ```
     *
     * Note: `set` can be used in `withMutations`.
     */
    set(key: K, value: V): this;

    /**
     * Returns a new OrderedMap resulting from merging the provided Collections
     * (or JS objects) into this OrderedMap. In other words, this takes each
     * entry of each collection and sets it on this OrderedMap.
     *
     * Note: Values provided to `merge` are shallowly converted before being
     * merged. No nested values are altered.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { OrderedMap } = require('immutable')
     * const one = OrderedMap({ a: 10, b: 20, c: 30 })
     * const two = OrderedMap({ b: 40, a: 50, d: 60 })
     * one.merge(two) // OrderedMap { "a": 50, "b": 40, "c": 30, "d": 60 }
     * two.merge(one) // OrderedMap { "b": 20, "a": 10, "d": 60, "c": 30 }
     * ```
     *
     * Note: `merge` can be used in `withMutations`.
     *
     * @alias concat
     */
    merge<KC, VC>(...collections: Array<Iterable<[KC, VC]>>): OrderedMap<K | KC, V | VC>;
    merge<C>(...collections: Array<{[key: string]: C}>): OrderedMap<K | string, V | C>;
    concat<KC, VC>(...collections: Array<Iterable<[KC, VC]>>): OrderedMap<K | KC, V | VC>;
    concat<C>(...collections: Array<{[key: string]: C}>): OrderedMap<K | string, V | C>;

    // Sequence algorithms

    /**
     * Returns a new OrderedMap with values passed through a
     * `mapper` function.
     *
     *     OrderedMap({ a: 1, b: 2 }).map(x => 10 * x)
     *     // OrderedMap { "a": 10, "b": 20 }
     *
     * Note: `map()` always returns a new instance, even if it produced the same
     * value at every step.
     */
    map<M>(
      mapper: (value: V, key: K, iter: this) => M,
      context?: any
    ): OrderedMap<K, M>;

    /**
     * @see Collection.Keyed.mapKeys
     */
    mapKeys<M>(
      mapper: (key: K, value: V, iter: this) => M,
      context?: any
    ): OrderedMap<M, V>;

    /**
     * @see Collection.Keyed.mapEntries
     */
    mapEntries<KM, VM>(
      mapper: (entry: [K, V], index: number, iter: this) => [KM, VM],
      context?: any
    ): OrderedMap<KM, VM>;

    /**
     * Flat-maps the OrderedMap, returning a new OrderedMap.
     *
     * Similar to `data.map(...).flatten(true)`.
     */
    flatMap<KM, VM>(
      mapper: (value: V, key: K, iter: this) => Iterable<[KM, VM]>,
      context?: any
    ): OrderedMap<KM, VM>;

    /**
     * Returns a new OrderedMap with only the entries for which the `predicate`
     * function returns true.
     *
     * Note: `filter()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filter<F extends V>(
      predicate: (value: V, key: K, iter: this) => value is F,
      context?: any
    ): OrderedMap<K, F>;
    filter(
      predicate: (value: V, key: K, iter: this) => any,
      context?: any
    ): this;

    /**
     * @see Collection.Keyed.flip
     */
    flip(): OrderedMap<V, K>;
  }


  /**
   * A Collection of unique values with `O(log32 N)` adds and has.
   *
   * When iterating a Set, the entries will be (value, value) pairs. Iteration
   * order of a Set is undefined, however is stable. Multiple iterations of the
   * same Set will iterate in the same order.
   *
   * Set values, like Map keys, may be of any type. Equality is determined using
   * `Immutable.is`, enabling Sets to uniquely include other Immutable
   * collections, custom value types, and NaN.
   */
  declare module Set$1 {

    /**
     * True if the provided value is a Set
     */
    function isSet(maybeSet: any): maybeSet is Set$1<any>;

    /**
     * Creates a new Set containing `values`.
     */
    function of<T>(...values: Array<T>): Set$1<T>;

    /**
     * `Set.fromKeys()` creates a new immutable Set containing the keys from
     * this Collection or JavaScript Object.
     */
    function fromKeys<T>(iter: Collection$1<T, any>): Set$1<T>;
    function fromKeys(obj: {[key: string]: any}): Set$1<string>;

    /**
     * `Set.intersect()` creates a new immutable Set that is the intersection of
     * a collection of other sets.
     *
     * ```js
     * const { Set } = require('immutable')
     * const intersected = Set.intersect([
     *   Set([ 'a', 'b', 'c' ])
     *   Set([ 'c', 'a', 't' ])
     * ])
     * // Set [ "a", "c"" ]
     * ```
     */
    function intersect<T>(sets: Iterable<Iterable<T>>): Set$1<T>;

    /**
     * `Set.union()` creates a new immutable Set that is the union of a
     * collection of other sets.
     *
     * ```js
     * const { Set } = require('immutable')
     * const unioned = Set.union([
     *   Set([ 'a', 'b', 'c' ])
     *   Set([ 'c', 'a', 't' ])
     * ])
     * // Set [ "a", "b", "c", "t"" ]
     * ```
     */
    function union<T>(sets: Iterable<Iterable<T>>): Set$1<T>;
  }

  /**
   * Create a new immutable Set containing the values of the provided
   * collection-like.
   *
   * Note: `Set` is a factory function and not a class, and does not use the
   * `new` keyword during construction.
   */
  declare function Set$1(): Set$1<any>;
  declare function Set$1<T>(): Set$1<T>;
  declare function Set$1<T>(collection: Iterable<T>): Set$1<T>;

  interface Set$1<T> extends Collection$1.Set<T> {

    /**
     * The number of items in this Set.
     */
    readonly size: number;

    // Persistent changes

    /**
     * Returns a new Set which also includes this value.
     *
     * Note: `add` can be used in `withMutations`.
     */
    add(value: T): this;

    /**
     * Returns a new Set which excludes this value.
     *
     * Note: `delete` can be used in `withMutations`.
     *
     * Note: `delete` **cannot** be safely used in IE8, use `remove` if
     * supporting old browsers.
     *
     * @alias remove
     */
    delete(value: T): this;
    remove(value: T): this;

    /**
     * Returns a new Set containing no values.
     *
     * Note: `clear` can be used in `withMutations`.
     */
    clear(): this;

    /**
     * Returns a Set including any value from `collections` that does not already
     * exist in this Set.
     *
     * Note: `union` can be used in `withMutations`.
     * @alias merge
     * @alias concat
     */
    union<C>(...collections: Array<Iterable<C>>): Set$1<T | C>;
    merge<C>(...collections: Array<Iterable<C>>): Set$1<T | C>;
    concat<C>(...collections: Array<Iterable<C>>): Set$1<T | C>;

    /**
     * Returns a Set which has removed any values not also contained
     * within `collections`.
     *
     * Note: `intersect` can be used in `withMutations`.
     */
    intersect(...collections: Array<Iterable<T>>): this;

    /**
     * Returns a Set excluding any values contained within `collections`.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { OrderedSet } = require('immutable')
     * OrderedSet([ 1, 2, 3 ]).subtract([1, 3])
     * // OrderedSet [2]
     * ```
     *
     * Note: `subtract` can be used in `withMutations`.
     */
    subtract(...collections: Array<Iterable<T>>): this;


    // Transient changes

    /**
     * Note: Not all methods can be used on a mutable collection or within
     * `withMutations`! Check the documentation for each method to see if it
     * mentions being safe to use in `withMutations`.
     *
     * @see `Map#withMutations`
     */
    withMutations(mutator: (mutable: this) => any): this;

    /**
     * Note: Not all methods can be used on a mutable collection or within
     * `withMutations`! Check the documentation for each method to see if it
     * mentions being safe to use in `withMutations`.
     *
     * @see `Map#asMutable`
     */
    asMutable(): this;

    /**
     * @see `Map#wasAltered`
     */
    wasAltered(): boolean;

    /**
     * @see `Map#asImmutable`
     */
    asImmutable(): this;

    // Sequence algorithms

    /**
     * Returns a new Set with values passed through a
     * `mapper` function.
     *
     *     Set([1,2]).map(x => 10 * x)
     *     // Set [10,20]
     */
    map<M>(
      mapper: (value: T, key: T, iter: this) => M,
      context?: any
    ): Set$1<M>;

    /**
     * Flat-maps the Set, returning a new Set.
     *
     * Similar to `set.map(...).flatten(true)`.
     */
    flatMap<M>(
      mapper: (value: T, key: T, iter: this) => Iterable<M>,
      context?: any
    ): Set$1<M>;

    /**
     * Returns a new Set with only the values for which the `predicate`
     * function returns true.
     *
     * Note: `filter()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filter<F extends T>(
      predicate: (value: T, key: T, iter: this) => value is F,
      context?: any
    ): Set$1<F>;
    filter(
      predicate: (value: T, key: T, iter: this) => any,
      context?: any
    ): this;
  }


  /**
   * A type of Set that has the additional guarantee that the iteration order of
   * values will be the order in which they were `add`ed.
   *
   * The iteration behavior of OrderedSet is the same as native ES6 Set.
   *
   * Note that `OrderedSet` are more expensive than non-ordered `Set` and may
   * consume more memory. `OrderedSet#add` is amortized O(log32 N), but not
   * stable.
   */
  declare module OrderedSet {

    /**
     * True if the provided value is an OrderedSet.
     */
    function isOrderedSet(maybeOrderedSet: any): boolean;

    /**
     * Creates a new OrderedSet containing `values`.
     */
    function of<T>(...values: Array<T>): OrderedSet<T>;

    /**
     * `OrderedSet.fromKeys()` creates a new immutable OrderedSet containing
     * the keys from this Collection or JavaScript Object.
     */
    function fromKeys<T>(iter: Collection$1<T, any>): OrderedSet<T>;
    function fromKeys(obj: {[key: string]: any}): OrderedSet<string>;
  }

  /**
   * Create a new immutable OrderedSet containing the values of the provided
   * collection-like.
   *
   * Note: `OrderedSet` is a factory function and not a class, and does not use
   * the `new` keyword during construction.
   */
  declare function OrderedSet(): OrderedSet<any>;
  declare function OrderedSet<T>(): OrderedSet<T>;
  declare function OrderedSet<T>(collection: Iterable<T>): OrderedSet<T>;

  interface OrderedSet<T> extends Set$1<T> {

    /**
     * The number of items in this OrderedSet.
     */
    readonly size: number;

    /**
     * Returns an OrderedSet including any value from `collections` that does
     * not already exist in this OrderedSet.
     *
     * Note: `union` can be used in `withMutations`.
     * @alias merge
     * @alias concat
     */
    union<C>(...collections: Array<Iterable<C>>): OrderedSet<T | C>;
    merge<C>(...collections: Array<Iterable<C>>): OrderedSet<T | C>;
    concat<C>(...collections: Array<Iterable<C>>): OrderedSet<T | C>;

    // Sequence algorithms

    /**
     * Returns a new Set with values passed through a
     * `mapper` function.
     *
     *     OrderedSet([ 1, 2 ]).map(x => 10 * x)
     *     // OrderedSet [10, 20]
     */
    map<M>(
      mapper: (value: T, key: T, iter: this) => M,
      context?: any
    ): OrderedSet<M>;

    /**
     * Flat-maps the OrderedSet, returning a new OrderedSet.
     *
     * Similar to `set.map(...).flatten(true)`.
     */
    flatMap<M>(
      mapper: (value: T, key: T, iter: this) => Iterable<M>,
      context?: any
    ): OrderedSet<M>;

    /**
     * Returns a new OrderedSet with only the values for which the `predicate`
     * function returns true.
     *
     * Note: `filter()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filter<F extends T>(
      predicate: (value: T, key: T, iter: this) => value is F,
      context?: any
    ): OrderedSet<F>;
    filter(
      predicate: (value: T, key: T, iter: this) => any,
      context?: any
    ): this;

    /**
     * Returns an OrderedSet of the same type "zipped" with the provided
     * collections.
     *
     * Like `zipWith`, but using the default `zipper`: creating an `Array`.
     *
     * ```js
     * const a = OrderedSet([ 1, 2, 3 ])
     * const b = OrderedSet([ 4, 5, 6 ])
     * const c = a.zip(b)
     * // OrderedSet [ [ 1, 4 ], [ 2, 5 ], [ 3, 6 ] ]
     * ```
     */
    zip<U>(other: Collection$1<any, U>): OrderedSet<[T,U]>;
    zip<U,V>(other1: Collection$1<any, U>, other2: Collection$1<any, V>): OrderedSet<[T,U,V]>;
    zip(...collections: Array<Collection$1<any, any>>): OrderedSet<any>;

    /**
     * Returns a OrderedSet of the same type "zipped" with the provided
     * collections.
     *
     * Unlike `zip`, `zipAll` continues zipping until the longest collection is
     * exhausted. Missing values from shorter collections are filled with `undefined`.
     *
     * ```js
     * const a = OrderedSet([ 1, 2 ]);
     * const b = OrderedSet([ 3, 4, 5 ]);
     * const c = a.zipAll(b); // OrderedSet [ [ 1, 3 ], [ 2, 4 ], [ undefined, 5 ] ]
     * ```
     *
     * Note: Since zipAll will return a collection as large as the largest
     * input, some results may contain undefined values. TypeScript cannot
     * account for these without cases (as of v2.5).
     */
    zipAll<U>(other: Collection$1<any, U>): OrderedSet<[T,U]>;
    zipAll<U,V>(other1: Collection$1<any, U>, other2: Collection$1<any, V>): OrderedSet<[T,U,V]>;
    zipAll(...collections: Array<Collection$1<any, any>>): OrderedSet<any>;

    /**
     * Returns an OrderedSet of the same type "zipped" with the provided
     * collections by using a custom `zipper` function.
     *
     * @see Seq.Indexed.zipWith
     */
    zipWith<U, Z>(
      zipper: (value: T, otherValue: U) => Z,
      otherCollection: Collection$1<any, U>
    ): OrderedSet<Z>;
    zipWith<U, V, Z>(
      zipper: (value: T, otherValue: U, thirdValue: V) => Z,
      otherCollection: Collection$1<any, U>,
      thirdCollection: Collection$1<any, V>
    ): OrderedSet<Z>;
    zipWith<Z>(
      zipper: (...any: Array<any>) => Z,
      ...collections: Array<Collection$1<any, any>>
    ): OrderedSet<Z>;

  }


  /**
   * Stacks are indexed collections which support very efficient O(1) addition
   * and removal from the front using `unshift(v)` and `shift()`.
   *
   * For familiarity, Stack also provides `push(v)`, `pop()`, and `peek()`, but
   * be aware that they also operate on the front of the list, unlike List or
   * a JavaScript Array.
   *
   * Note: `reverse()` or any inherent reverse traversal (`reduceRight`,
   * `lastIndexOf`, etc.) is not efficient with a Stack.
   *
   * Stack is implemented with a Single-Linked List.
   */
  declare module Stack {

    /**
     * True if the provided value is a Stack
     */
    function isStack(maybeStack: any): maybeStack is Stack<any>;

    /**
     * Creates a new Stack containing `values`.
     */
    function of<T>(...values: Array<T>): Stack<T>;
  }

  /**
   * Create a new immutable Stack containing the values of the provided
   * collection-like.
   *
   * The iteration order of the provided collection is preserved in the
   * resulting `Stack`.
   *
   * Note: `Stack` is a factory function and not a class, and does not use the
   * `new` keyword during construction.
   */
  declare function Stack(): Stack<any>;
  declare function Stack<T>(): Stack<T>;
  declare function Stack<T>(collection: Iterable<T>): Stack<T>;

  interface Stack<T> extends Collection$1.Indexed<T> {

    /**
     * The number of items in this Stack.
     */
    readonly size: number;

    // Reading values

    /**
     * Alias for `Stack.first()`.
     */
    peek(): T | undefined;


    // Persistent changes

    /**
     * Returns a new Stack with 0 size and no values.
     *
     * Note: `clear` can be used in `withMutations`.
     */
    clear(): Stack<T>;

    /**
     * Returns a new Stack with the provided `values` prepended, shifting other
     * values ahead to higher indices.
     *
     * This is very efficient for Stack.
     *
     * Note: `unshift` can be used in `withMutations`.
     */
    unshift(...values: Array<T>): Stack<T>;

    /**
     * Like `Stack#unshift`, but accepts a collection rather than varargs.
     *
     * Note: `unshiftAll` can be used in `withMutations`.
     */
    unshiftAll(iter: Iterable<T>): Stack<T>;

    /**
     * Returns a new Stack with a size ones less than this Stack, excluding
     * the first item in this Stack, shifting all other values to a lower index.
     *
     * Note: this differs from `Array#shift` because it returns a new
     * Stack rather than the removed value. Use `first()` or `peek()` to get the
     * first value in this Stack.
     *
     * Note: `shift` can be used in `withMutations`.
     */
    shift(): Stack<T>;

    /**
     * Alias for `Stack#unshift` and is not equivalent to `List#push`.
     */
    push(...values: Array<T>): Stack<T>;

    /**
     * Alias for `Stack#unshiftAll`.
     */
    pushAll(iter: Iterable<T>): Stack<T>;

    /**
     * Alias for `Stack#shift` and is not equivalent to `List#pop`.
     */
    pop(): Stack<T>;


    // Transient changes

    /**
     * Note: Not all methods can be used on a mutable collection or within
     * `withMutations`! Check the documentation for each method to see if it
     * mentions being safe to use in `withMutations`.
     *
     * @see `Map#withMutations`
     */
    withMutations(mutator: (mutable: this) => any): this;

    /**
     * Note: Not all methods can be used on a mutable collection or within
     * `withMutations`! Check the documentation for each method to see if it
     * mentions being safe to use in `withMutations`.
     *
     * @see `Map#asMutable`
     */
    asMutable(): this;

    /**
     * @see `Map#wasAltered`
     */
    wasAltered(): boolean;

    /**
     * @see `Map#asImmutable`
     */
    asImmutable(): this;

    // Sequence algorithms

    /**
     * Returns a new Stack with other collections concatenated to this one.
     */
    concat<C>(...valuesOrCollections: Array<Iterable<C> | C>): Stack<T | C>;

    /**
     * Returns a new Stack with values passed through a
     * `mapper` function.
     *
     *     Stack([ 1, 2 ]).map(x => 10 * x)
     *     // Stack [ 10, 20 ]
     *
     * Note: `map()` always returns a new instance, even if it produced the same
     * value at every step.
     */
    map<M>(
      mapper: (value: T, key: number, iter: this) => M,
      context?: any
    ): Stack<M>;

    /**
     * Flat-maps the Stack, returning a new Stack.
     *
     * Similar to `stack.map(...).flatten(true)`.
     */
    flatMap<M>(
      mapper: (value: T, key: number, iter: this) => Iterable<M>,
      context?: any
    ): Stack<M>;

    /**
     * Returns a new Set with only the values for which the `predicate`
     * function returns true.
     *
     * Note: `filter()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filter<F extends T>(
      predicate: (value: T, index: number, iter: this) => value is F,
      context?: any
    ): Set$1<F>;
    filter(
      predicate: (value: T, index: number, iter: this) => any,
      context?: any
    ): this;

    /**
     * Returns a Stack "zipped" with the provided collections.
     *
     * Like `zipWith`, but using the default `zipper`: creating an `Array`.
     *
     * ```js
     * const a = Stack([ 1, 2, 3 ]);
     * const b = Stack([ 4, 5, 6 ]);
     * const c = a.zip(b); // Stack [ [ 1, 4 ], [ 2, 5 ], [ 3, 6 ] ]
     * ```
     */
    zip<U>(other: Collection$1<any, U>): Stack<[T,U]>;
    zip<U,V>(other: Collection$1<any, U>, other2: Collection$1<any,V>): Stack<[T,U,V]>;
    zip(...collections: Array<Collection$1<any, any>>): Stack<any>;

    /**
     * Returns a Stack "zipped" with the provided collections.
     *
     * Unlike `zip`, `zipAll` continues zipping until the longest collection is
     * exhausted. Missing values from shorter collections are filled with `undefined`.
     *
     * ```js
     * const a = Stack([ 1, 2 ]);
     * const b = Stack([ 3, 4, 5 ]);
     * const c = a.zipAll(b); // Stack [ [ 1, 3 ], [ 2, 4 ], [ undefined, 5 ] ]
     * ```
     *
     * Note: Since zipAll will return a collection as large as the largest
     * input, some results may contain undefined values. TypeScript cannot
     * account for these without cases (as of v2.5).
     */
    zipAll<U>(other: Collection$1<any, U>): Stack<[T,U]>;
    zipAll<U,V>(other: Collection$1<any, U>, other2: Collection$1<any,V>): Stack<[T,U,V]>;
    zipAll(...collections: Array<Collection$1<any, any>>): Stack<any>;

    /**
     * Returns a Stack "zipped" with the provided collections by using a
     * custom `zipper` function.
     *
     * ```js
     * const a = Stack([ 1, 2, 3 ]);
     * const b = Stack([ 4, 5, 6 ]);
     * const c = a.zipWith((a, b) => a + b, b);
     * // Stack [ 5, 7, 9 ]
     * ```
     */
    zipWith<U, Z>(
      zipper: (value: T, otherValue: U) => Z,
      otherCollection: Collection$1<any, U>
    ): Stack<Z>;
    zipWith<U, V, Z>(
      zipper: (value: T, otherValue: U, thirdValue: V) => Z,
      otherCollection: Collection$1<any, U>,
      thirdCollection: Collection$1<any, V>
    ): Stack<Z>;
    zipWith<Z>(
      zipper: (...any: Array<any>) => Z,
      ...collections: Array<Collection$1<any, any>>
    ): Stack<Z>;
  }


  /**
   * A record is similar to a JS object, but enforces a specific set of allowed
   * string keys, and has default values.
   *
   * The `Record()` function produces new Record Factories, which when called
   * create Record instances.
   *
   * ```js
   * const { Record } = require('immutable')
   * const ABRecord = Record({ a: 1, b: 2 })
   * const myRecord = ABRecord({ b: 3 })
   * ```
   *
   * Records always have a value for the keys they define. `remove`ing a key
   * from a record simply resets it to the default value for that key.
   *
   * ```js
   * myRecord.size // 2
   * myRecord.get('a') // 1
   * myRecord.get('b') // 3
   * const myRecordWithoutB = myRecord.remove('b')
   * myRecordWithoutB.get('b') // 2
   * myRecordWithoutB.size // 2
   * ```
   *
   * Values provided to the constructor not found in the Record type will
   * be ignored. For example, in this case, ABRecord is provided a key "x" even
   * though only "a" and "b" have been defined. The value for "x" will be
   * ignored for this record.
   *
   * ```js
   * const myRecord = ABRecord({ b: 3, x: 10 })
   * myRecord.get('x') // undefined
   * ```
   *
   * Because Records have a known set of string keys, property get access works
   * as expected, however property sets will throw an Error.
   *
   * Note: IE8 does not support property access. Only use `get()` when
   * supporting IE8.
   *
   * ```js
   * myRecord.b // 3
   * myRecord.b = 5 // throws Error
   * ```
   *
   * Record Types can be extended as well, allowing for custom methods on your
   * Record. This is not a common pattern in functional environments, but is in
   * many JS programs.
   *
   * However Record Types are more restricted than typical JavaScript classes.
   * They do not use a class constructor, which also means they cannot use
   * class properties (since those are technically part of a constructor).
   *
   * While Record Types can be syntactically created with the JavaScript `class`
   * form, the resulting Record function is actually a factory function, not a
   * class constructor. Even though Record Types are not classes, JavaScript
   * currently requires the use of `new` when creating new Record instances if
   * they are defined as a `class`.
   *
   * ```
   * class ABRecord extends Record({ a: 1, b: 2 }) {
   *   getAB() {
   *     return this.a + this.b;
   *   }
   * }
   *
   * var myRecord = new ABRecord({b: 3})
   * myRecord.getAB() // 4
   * ```
   *
   *
   * **Flow Typing Records:**
   *
   * Immutable.js exports two Flow types designed to make it easier to use
   * Records with flow typed code, `RecordOf<TProps>` and `RecordFactory<TProps>`.
   *
   * When defining a new kind of Record factory function, use a flow type that
   * describes the values the record contains along with `RecordFactory<TProps>`.
   * To type instances of the Record (which the factory function returns),
   * use `RecordOf<TProps>`.
   *
   * Typically, new Record definitions will export both the Record factory
   * function as well as the Record instance type for use in other code.
   *
   * ```js
   * import type { RecordFactory, RecordOf } from 'immutable';
   *
   * // Use RecordFactory<TProps> for defining new Record factory functions.
   * type Point3DProps = { x: number, y: number, z: number };
   * const defaultValues: Point3DProps = { x: 0, y: 0, z: 0 };
   * const makePoint3D: RecordFactory<Point3DProps> = Record(defaultValues);
   * export makePoint3D;
   *
   * // Use RecordOf<T> for defining new instances of that Record.
   * export type Point3D = RecordOf<Point3DProps>;
   * const some3DPoint: Point3D = makePoint3D({ x: 10, y: 20, z: 30 });
   * ```
   *
   * **Flow Typing Record Subclasses:**
   *
   * Records can be subclassed as a means to add additional methods to Record
   * instances. This is generally discouraged in favor of a more functional API,
   * since Subclasses have some minor overhead. However the ability to create
   * a rich API on Record types can be quite valuable.
   *
   * When using Flow to type Subclasses, do not use `RecordFactory<TProps>`,
   * instead apply the props type when subclassing:
   *
   * ```js
   * type PersonProps = {name: string, age: number};
   * const defaultValues: PersonProps = {name: 'Aristotle', age: 2400};
   * const PersonRecord = Record(defaultValues);
   * class Person extends PersonRecord<PersonProps> {
   *   getName(): string {
   *     return this.get('name')
   *   }
   *
   *   setName(name: string): this {
   *     return this.set('name', name);
   *   }
   * }
   * ```
   *
   * **Choosing Records vs plain JavaScript objects**
   *
   * Records offer a persistently immutable alternative to plain JavaScript
   * objects, however they're not required to be used within Immutable.js
   * collections. In fact, the deep-access and deep-updating functions
   * like `getIn()` and `setIn()` work with plain JavaScript Objects as well.
   *
   * Deciding to use Records or Objects in your application should be informed
   * by the tradeoffs and relative benefits of each:
   *
   * - *Runtime immutability*: plain JS objects may be carefully treated as
   *   immutable, however Record instances will *throw* if attempted to be
   *   mutated directly. Records provide this additional guarantee, however at
   *   some marginal runtime cost. While JS objects are mutable by nature, the
   *   use of type-checking tools like [Flow](https://medium.com/@gcanti/immutability-with-flow-faa050a1aef4)
   *   can help gain confidence in code written to favor immutability.
   *
   * - *Value equality*: Records use value equality when compared with `is()`
   *   or `record.equals()`. That is, two Records with the same keys and values
   *   are equal. Plain objects use *reference equality*. Two objects with the
   *   same keys and values are not equal since they are different objects.
   *   This is important to consider when using objects as keys in a `Map` or
   *   values in a `Set`, which use equality when retrieving values.
   *
   * - *API methods*: Records have a full featured API, with methods like
   *   `.getIn()`, and `.equals()`. These can make working with these values
   *   easier, but comes at the cost of not allowing keys with those names.
   *
   * - *Default values*: Records provide default values for every key, which
   *   can be useful when constructing Records with often unchanging values.
   *   However default values can make using Flow and TypeScript more laborious.
   *
   * - *Serialization*: Records use a custom internal representation to
   *   efficiently store and update their values. Converting to and from this
   *   form isn't free. If converting Records to plain objects is common,
   *   consider sticking with plain objects to begin with.
   */
  declare module Record$1 {

    /**
     * True if `maybeRecord` is an instance of a Record.
     */
    export function isRecord(maybeRecord: any): maybeRecord is Record$1<any>;

    /**
     * Records allow passing a second parameter to supply a descriptive name
     * that appears when converting a Record to a string or in any error
     * messages. A descriptive name for any record can be accessed by using this
     * method. If one was not provided, the string "Record" is returned.
     *
     * ```js
     * const { Record } = require('immutable')
     * const Person = Record({
     *   name: null
     * }, 'Person')
     *
     * var me = Person({ name: 'My Name' })
     * me.toString() // "Person { "name": "My Name" }"
     * Record.getDescriptiveName(me) // "Person"
     * ```
     */
    export function getDescriptiveName(record: Record$1<any>): string;

    /**
     * A Record.Factory is created by the `Record()` function. Record instances
     * are created by passing it some of the accepted values for that Record
     * type:
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Record } = require('immutable')" }
     * -->
     * ```js
     * // makePerson is a Record Factory function
     * const makePerson = Record({ name: null, favoriteColor: 'unknown' });
     *
     * // alan is a Record instance
     * const alan = makePerson({ name: 'Alan' });
     * ```
     *
     * Note that Record Factories return `Record<TProps> & Readonly<TProps>`,
     * this allows use of both the Record instance API, and direct property
     * access on the resulting instances:
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Record } = require('immutable');const makePerson = Record({ name: null, favoriteColor: 'unknown' });const alan = makePerson({ name: 'Alan' });" }
     * -->
     * ```js
     * // Use the Record API
     * console.log('Record API: ' + alan.get('name'))
     *
     * // Or direct property access (Readonly)
     * console.log('property access: ' + alan.name)
     * ```
     *
     * **Flow Typing Records:**
     *
     * Use the `RecordFactory<TProps>` Flow type to get high quality type checking of
     * Records:
     *
     * ```js
     * import type { RecordFactory, RecordOf } from 'immutable';
     *
     * // Use RecordFactory<TProps> for defining new Record factory functions.
     * type PersonProps = { name: ?string, favoriteColor: string };
     * const makePerson: RecordFactory<PersonProps> = Record({ name: null, favoriteColor: 'unknown' });
     *
     * // Use RecordOf<T> for defining new instances of that Record.
     * type Person = RecordOf<PersonProps>;
     * const alan: Person = makePerson({ name: 'Alan' });
     * ```
     */
    export module Factory {}

    export interface Factory<TProps extends Object> {
      (values?: Partial<TProps> | Iterable<[string, any]>): Record$1<TProps> & Readonly<TProps>;
      new (values?: Partial<TProps> | Iterable<[string, any]>): Record$1<TProps> & Readonly<TProps>;
    }

    export function Factory<TProps extends Object>(values?: Partial<TProps> | Iterable<[string, any]>): Record$1<TProps> & Readonly<TProps>;
  }

  /**
   * Unlike other types in Immutable.js, the `Record()` function creates a new
   * Record Factory, which is a function that creates Record instances.
   *
   * See above for examples of using `Record()`.
   *
   * Note: `Record` is a factory function and not a class, and does not use the
   * `new` keyword during construction.
   */
  declare function Record$1<TProps extends object>(defaultValues: TProps, name?: string): Record$1.Factory<TProps>;

  interface Record$1<TProps extends Object> {

    // Reading values

    has(key: string): key is keyof TProps & string;

    /**
     * Returns the value associated with the provided key, which may be the
     * default value defined when creating the Record factory function.
     *
     * If the requested key is not defined by this Record type, then
     * notSetValue will be returned if provided. Note that this scenario would
     * produce an error when using Flow or TypeScript.
     */
    get<K extends keyof TProps>(key: K, notSetValue?: any): TProps[K];
    get<T>(key: string, notSetValue: T): T;

    // Reading deep values

    hasIn(keyPath: Iterable<any>): boolean;
    getIn(keyPath: Iterable<any>): any;

    // Value equality

    equals(other: any): boolean;
    hashCode(): number;

    // Persistent changes

    set<K extends keyof TProps>(key: K, value: TProps[K]): this;
    update<K extends keyof TProps>(key: K, updater: (value: TProps[K]) => TProps[K]): this;
    merge(...collections: Array<Partial<TProps> | Iterable<[string, any]>>): this;
    mergeDeep(...collections: Array<Partial<TProps> | Iterable<[string, any]>>): this;

    mergeWith(
      merger: (oldVal: any, newVal: any, key: keyof TProps) => any,
      ...collections: Array<Partial<TProps> | Iterable<[string, any]>>
    ): this;
    mergeDeepWith(
      merger: (oldVal: any, newVal: any, key: any) => any,
      ...collections: Array<Partial<TProps> | Iterable<[string, any]>>
    ): this;

    /**
     * Returns a new instance of this Record type with the value for the
     * specific key set to its default value.
     *
     * @alias remove
     */
    delete<K extends keyof TProps>(key: K): this;
    remove<K extends keyof TProps>(key: K): this;

    /**
     * Returns a new instance of this Record type with all values set
     * to their default values.
     */
    clear(): this;

    // Deep persistent changes

    setIn(keyPath: Iterable<any>, value: any): this;
    updateIn(keyPath: Iterable<any>, updater: (value: any) => any): this;
    mergeIn(keyPath: Iterable<any>, ...collections: Array<any>): this;
    mergeDeepIn(keyPath: Iterable<any>, ...collections: Array<any>): this;

    /**
     * @alias removeIn
     */
    deleteIn(keyPath: Iterable<any>): this;
    removeIn(keyPath: Iterable<any>): this;

    // Conversion to JavaScript types

    /**
     * Deeply converts this Record to equivalent native JavaScript Object.
     *
     * Note: This method may not be overridden. Objects with custom
     * serialization to plain JS may override toJSON() instead.
     */
    toJS(): { [K in keyof TProps]: any };

    /**
     * Shallowly converts this Record to equivalent native JavaScript Object.
     */
    toJSON(): TProps;

    /**
     * Shallowly converts this Record to equivalent JavaScript Object.
     */
    toObject(): TProps;

    // Transient changes

    /**
     * Note: Not all methods can be used on a mutable collection or within
     * `withMutations`! Only `set` may be used mutatively.
     *
     * @see `Map#withMutations`
     */
    withMutations(mutator: (mutable: this) => any): this;

    /**
     * @see `Map#asMutable`
     */
    asMutable(): this;

    /**
     * @see `Map#wasAltered`
     */
    wasAltered(): boolean;

    /**
     * @see `Map#asImmutable`
     */
    asImmutable(): this;

    // Sequence algorithms

    toSeq(): Seq.Keyed<keyof TProps, TProps[keyof TProps]>;

    [Symbol.iterator](): IterableIterator<[keyof TProps, TProps[keyof TProps]]>;
  }

  /**
   * `Seq` describes a lazy operation, allowing them to efficiently chain
   * use of all the higher-order collection methods (such as `map` and `filter`)
   * by not creating intermediate collections.
   *
   * **Seq is immutable** — Once a Seq is created, it cannot be
   * changed, appended to, rearranged or otherwise modified. Instead, any
   * mutative method called on a `Seq` will return a new `Seq`.
   *
   * **Seq is lazy** — `Seq` does as little work as necessary to respond to any
   * method call. Values are often created during iteration, including implicit
   * iteration when reducing or converting to a concrete data structure such as
   * a `List` or JavaScript `Array`.
   *
   * For example, the following performs no work, because the resulting
   * `Seq`'s values are never iterated:
   *
   * ```js
   * const { Seq } = require('immutable')
   * const oddSquares = Seq([ 1, 2, 3, 4, 5, 6, 7, 8 ])
   *   .filter(x => x % 2 !== 0)
   *   .map(x => x * x)
   * ```
   *
   * Once the `Seq` is used, it performs only the work necessary. In this
   * example, no intermediate arrays are ever created, filter is called three
   * times, and map is only called once:
   *
   * ```js
   * oddSquares.get(1); // 9
   * ```
   *
   * Any collection can be converted to a lazy Seq with `Seq()`.
   *
   * <!-- runkit:activate -->
   * ```js
   * const { Map } = require('immutable')
   * const map = Map({ a: 1, b: 2, c: 3 }
   * const lazySeq = Seq(map)
   * ```
   *
   * `Seq` allows for the efficient chaining of operations, allowing for the
   * expression of logic that can otherwise be very tedious:
   *
   * ```js
   * lazySeq
   *   .flip()
   *   .map(key => key.toUpperCase())
   *   .flip()
   * // Seq { A: 1, B: 1, C: 1 }
   * ```
   *
   * As well as expressing logic that would otherwise seem memory or time
   * limited, for example `Range` is a special kind of Lazy sequence.
   *
   * <!-- runkit:activate -->
   * ```js
   * const { Range } = require('immutable')
   * Range(1, Infinity)
   *   .skip(1000)
   *   .map(n => -n)
   *   .filter(n => n % 2 === 0)
   *   .take(2)
   *   .reduce((r, n) => r * n, 1)
   * // 1006008
   * ```
   *
   * Seq is often used to provide a rich collection API to JavaScript Object.
   *
   * ```js
   * Seq({ x: 0, y: 1, z: 2 }).map(v => v * 2).toObject();
   * // { x: 0, y: 2, z: 4 }
   * ```
   */

  declare module Seq {
    /**
     * True if `maybeSeq` is a Seq, it is not backed by a concrete
     * structure such as Map, List, or Set.
     */
    function isSeq(maybeSeq: any): maybeSeq is Seq.Indexed<any> | Seq.Keyed<any, any> | Seq.Set<any>;


    /**
     * `Seq` which represents key-value pairs.
     */
    export module Keyed {}

    /**
     * Always returns a Seq.Keyed, if input is not keyed, expects an
     * collection of [K, V] tuples.
     *
     * Note: `Seq.Keyed` is a conversion function and not a class, and does not
     * use the `new` keyword during construction.
     */
    export function Keyed<K, V>(collection: Iterable<[K, V]>): Seq.Keyed<K, V>;
    export function Keyed<V>(obj: {[key: string]: V}): Seq.Keyed<string, V>;
    export function Keyed<K, V>(): Seq.Keyed<K, V>;
    export function Keyed(): Seq.Keyed<any, any>;

    export interface Keyed<K, V> extends Seq<K, V>, Collection$1.Keyed<K, V> {
      /**
       * Deeply converts this Keyed Seq to equivalent native JavaScript Object.
       *
       * Converts keys to Strings.
       */
      toJS(): Object;

      /**
       * Shallowly converts this Keyed Seq to equivalent native JavaScript Object.
       *
       * Converts keys to Strings.
       */
      toJSON(): { [key: string]: V };

      /**
       * Shallowly converts this collection to an Array.
       */
      toArray(): Array<[K, V]>;

      /**
       * Returns itself
       */
      toSeq(): this;

      /**
       * Returns a new Seq with other collections concatenated to this one.
       *
       * All entries will be present in the resulting Seq, even if they
       * have the same key.
       */
      concat<KC, VC>(...collections: Array<Iterable<[KC, VC]>>): Seq.Keyed<K | KC, V | VC>;
      concat<C>(...collections: Array<{[key: string]: C}>): Seq.Keyed<K | string, V | C>;

      /**
       * Returns a new Seq.Keyed with values passed through a
       * `mapper` function.
       *
       * ```js
       * const { Seq } = require('immutable')
       * Seq.Keyed({ a: 1, b: 2 }).map(x => 10 * x)
       * // Seq { "a": 10, "b": 20 }
       * ```
       *
       * Note: `map()` always returns a new instance, even if it produced the
       * same value at every step.
       */
      map<M>(
        mapper: (value: V, key: K, iter: this) => M,
        context?: any
      ): Seq.Keyed<K, M>;

      /**
       * @see Collection.Keyed.mapKeys
       */
      mapKeys<M>(
        mapper: (key: K, value: V, iter: this) => M,
        context?: any
      ): Seq.Keyed<M, V>;

      /**
       * @see Collection.Keyed.mapEntries
       */
      mapEntries<KM, VM>(
        mapper: (entry: [K, V], index: number, iter: this) => [KM, VM],
        context?: any
      ): Seq.Keyed<KM, VM>;

      /**
       * Flat-maps the Seq, returning a Seq of the same type.
       *
       * Similar to `seq.map(...).flatten(true)`.
       */
      flatMap<KM, VM>(
        mapper: (value: V, key: K, iter: this) => Iterable<[KM, VM]>,
        context?: any
      ): Seq.Keyed<KM, VM>;

      /**
       * Returns a new Seq with only the entries for which the `predicate`
       * function returns true.
       *
       * Note: `filter()` always returns a new instance, even if it results in
       * not filtering out any values.
       */
      filter<F extends V>(
        predicate: (value: V, key: K, iter: this) => value is F,
        context?: any
      ): Seq.Keyed<K, F>;
      filter(
        predicate: (value: V, key: K, iter: this) => any,
        context?: any
      ): this;

      /**
       * @see Collection.Keyed.flip
       */
      flip(): Seq.Keyed<V, K>;
    }


    /**
     * `Seq` which represents an ordered indexed list of values.
     */
    module Indexed {

      /**
       * Provides an Seq.Indexed of the values provided.
       */
      function of<T>(...values: Array<T>): Seq.Indexed<T>;
    }

    /**
     * Always returns Seq.Indexed, discarding associated keys and
     * supplying incrementing indices.
     *
     * Note: `Seq.Indexed` is a conversion function and not a class, and does
     * not use the `new` keyword during construction.
     */
    export function Indexed(): Seq.Indexed<any>;
    export function Indexed<T>(): Seq.Indexed<T>;
    export function Indexed<T>(collection: Iterable<T>): Seq.Indexed<T>;

    export interface Indexed<T> extends Seq<number, T>, Collection$1.Indexed<T> {
      /**
       * Deeply converts this Indexed Seq to equivalent native JavaScript Array.
       */
      toJS(): Array<any>;

      /**
       * Shallowly converts this Indexed Seq to equivalent native JavaScript Array.
       */
      toJSON(): Array<T>;

      /**
       * Shallowly converts this collection to an Array.
       */
      toArray(): Array<T>;

      /**
       * Returns itself
       */
      toSeq(): this

      /**
       * Returns a new Seq with other collections concatenated to this one.
       */
      concat<C>(...valuesOrCollections: Array<Iterable<C> | C>): Seq.Indexed<T | C>;

      /**
       * Returns a new Seq.Indexed with values passed through a
       * `mapper` function.
       *
       * ```js
       * const { Seq } = require('immutable')
       * Seq.Indexed([ 1, 2 ]).map(x => 10 * x)
       * // Seq [ 10, 20 ]
       * ```
       *
       * Note: `map()` always returns a new instance, even if it produced the
       * same value at every step.
       */
      map<M>(
        mapper: (value: T, key: number, iter: this) => M,
        context?: any
      ): Seq.Indexed<M>;

      /**
       * Flat-maps the Seq, returning a a Seq of the same type.
       *
       * Similar to `seq.map(...).flatten(true)`.
       */
      flatMap<M>(
        mapper: (value: T, key: number, iter: this) => Iterable<M>,
        context?: any
      ): Seq.Indexed<M>;

      /**
       * Returns a new Seq with only the values for which the `predicate`
       * function returns true.
       *
       * Note: `filter()` always returns a new instance, even if it results in
       * not filtering out any values.
       */
      filter<F extends T>(
        predicate: (value: T, index: number, iter: this) => value is F,
        context?: any
      ): Seq.Indexed<F>;
      filter(
        predicate: (value: T, index: number, iter: this) => any,
        context?: any
      ): this;

      /**
       * Returns a Seq "zipped" with the provided collections.
       *
       * Like `zipWith`, but using the default `zipper`: creating an `Array`.
       *
       * ```js
       * const a = Seq([ 1, 2, 3 ]);
       * const b = Seq([ 4, 5, 6 ]);
       * const c = a.zip(b); // Seq [ [ 1, 4 ], [ 2, 5 ], [ 3, 6 ] ]
       * ```
       */
      zip<U>(other: Collection$1<any, U>): Seq.Indexed<[T,U]>;
      zip<U,V>(other: Collection$1<any, U>, other2: Collection$1<any, V>): Seq.Indexed<[T,U,V]>;
      zip(...collections: Array<Collection$1<any, any>>): Seq.Indexed<any>;

      /**
       * Returns a Seq "zipped" with the provided collections.
       *
       * Unlike `zip`, `zipAll` continues zipping until the longest collection is
       * exhausted. Missing values from shorter collections are filled with `undefined`.
       *
       * ```js
       * const a = Seq([ 1, 2 ]);
       * const b = Seq([ 3, 4, 5 ]);
       * const c = a.zipAll(b); // Seq [ [ 1, 3 ], [ 2, 4 ], [ undefined, 5 ] ]
       * ```
       */
      zipAll<U>(other: Collection$1<any, U>): Seq.Indexed<[T,U]>;
      zipAll<U,V>(other: Collection$1<any, U>, other2: Collection$1<any, V>): Seq.Indexed<[T,U,V]>;
      zipAll(...collections: Array<Collection$1<any, any>>): Seq.Indexed<any>;

      /**
       * Returns a Seq "zipped" with the provided collections by using a
       * custom `zipper` function.
       *
       * ```js
       * const a = Seq([ 1, 2, 3 ]);
       * const b = Seq([ 4, 5, 6 ]);
       * const c = a.zipWith((a, b) => a + b, b);
       * // Seq [ 5, 7, 9 ]
       * ```
       */
      zipWith<U, Z>(
        zipper: (value: T, otherValue: U) => Z,
        otherCollection: Collection$1<any, U>
      ): Seq.Indexed<Z>;
      zipWith<U, V, Z>(
        zipper: (value: T, otherValue: U, thirdValue: V) => Z,
        otherCollection: Collection$1<any, U>,
        thirdCollection: Collection$1<any, V>
      ): Seq.Indexed<Z>;
      zipWith<Z>(
        zipper: (...any: Array<any>) => Z,
        ...collections: Array<Collection$1<any, any>>
      ): Seq.Indexed<Z>;
    }


    /**
     * `Seq` which represents a set of values.
     *
     * Because `Seq` are often lazy, `Seq.Set` does not provide the same guarantee
     * of value uniqueness as the concrete `Set`.
     */
    export module Set {

      /**
       * Returns a Seq.Set of the provided values
       */
      function of<T>(...values: Array<T>): Seq.Set<T>;
    }

    /**
     * Always returns a Seq.Set, discarding associated indices or keys.
     *
     * Note: `Seq.Set` is a conversion function and not a class, and does not
     * use the `new` keyword during construction.
     */
    export function Set(): Seq.Set<any>;
    export function Set<T>(): Seq.Set<T>;
    export function Set<T>(collection: Iterable<T>): Seq.Set<T>;

    export interface Set<T> extends Seq<T, T>, Collection$1.Set<T> {
      /**
       * Deeply converts this Set Seq to equivalent native JavaScript Array.
       */
      toJS(): Array<any>;

      /**
       * Shallowly converts this Set Seq to equivalent native JavaScript Array.
       */
      toJSON(): Array<T>;

      /**
       * Shallowly converts this collection to an Array.
       */
      toArray(): Array<T>;

      /**
       * Returns itself
       */
      toSeq(): this

      /**
       * Returns a new Seq with other collections concatenated to this one.
       *
       * All entries will be present in the resulting Seq, even if they
       * are duplicates.
       */
      concat<U>(...collections: Array<Iterable<U>>): Seq.Set<T | U>;

      /**
       * Returns a new Seq.Set with values passed through a
       * `mapper` function.
       *
       * ```js
       * Seq.Set([ 1, 2 ]).map(x => 10 * x)
       * // Seq { 10, 20 }
       * ```
       *
       * Note: `map()` always returns a new instance, even if it produced the
       * same value at every step.
       */
      map<M>(
        mapper: (value: T, key: T, iter: this) => M,
        context?: any
      ): Seq.Set<M>;

      /**
       * Flat-maps the Seq, returning a Seq of the same type.
       *
       * Similar to `seq.map(...).flatten(true)`.
       */
      flatMap<M>(
        mapper: (value: T, key: T, iter: this) => Iterable<M>,
        context?: any
      ): Seq.Set<M>;

      /**
       * Returns a new Seq with only the values for which the `predicate`
       * function returns true.
       *
       * Note: `filter()` always returns a new instance, even if it results in
       * not filtering out any values.
       */
      filter<F extends T>(
        predicate: (value: T, key: T, iter: this) => value is F,
        context?: any
      ): Seq.Set<F>;
      filter(
        predicate: (value: T, key: T, iter: this) => any,
        context?: any
      ): this;
    }

  }

  /**
   * Creates a Seq.
   *
   * Returns a particular kind of `Seq` based on the input.
   *
   *   * If a `Seq`, that same `Seq`.
   *   * If an `Collection`, a `Seq` of the same kind (Keyed, Indexed, or Set).
   *   * If an Array-like, an `Seq.Indexed`.
   *   * If an Iterable Object, an `Seq.Indexed`.
   *   * If an Object, a `Seq.Keyed`.
   *
   * Note: An Iterator itself will be treated as an object, becoming a `Seq.Keyed`,
   * which is usually not what you want. You should turn your Iterator Object into
   * an iterable object by defining a Symbol.iterator (or @@iterator) method which
   * returns `this`.
   *
   * Note: `Seq` is a conversion function and not a class, and does not use the
   * `new` keyword during construction.
   */
  declare function Seq<S extends Seq<any, any>>(seq: S): S;
  declare function Seq<K, V>(collection: Collection$1.Keyed<K, V>): Seq.Keyed<K, V>;
  declare function Seq<T>(collection: Collection$1.Indexed<T>): Seq.Indexed<T>;
  declare function Seq<T>(collection: Collection$1.Set<T>): Seq.Set<T>;
  declare function Seq<T>(collection: Iterable<T>): Seq.Indexed<T>;
  declare function Seq<V>(obj: {[key: string]: V}): Seq.Keyed<string, V>;
  declare function Seq(): Seq<any, any>;

  interface Seq<K, V> extends Collection$1<K, V> {

    /**
     * Some Seqs can describe their size lazily. When this is the case,
     * size will be an integer. Otherwise it will be undefined.
     *
     * For example, Seqs returned from `map()` or `reverse()`
     * preserve the size of the original `Seq` while `filter()` does not.
     *
     * Note: `Range`, `Repeat` and `Seq`s made from `Array`s and `Object`s will
     * always have a size.
     */
    readonly size: number | undefined;


    // Force evaluation

    /**
     * Because Sequences are lazy and designed to be chained together, they do
     * not cache their results. For example, this map function is called a total
     * of 6 times, as each `join` iterates the Seq of three values.
     *
     *     var squares = Seq([ 1, 2, 3 ]).map(x => x * x)
     *     squares.join() + squares.join()
     *
     * If you know a `Seq` will be used multiple times, it may be more
     * efficient to first cache it in memory. Here, the map function is called
     * only 3 times.
     *
     *     var squares = Seq([ 1, 2, 3 ]).map(x => x * x).cacheResult()
     *     squares.join() + squares.join()
     *
     * Use this method judiciously, as it must fully evaluate a Seq which can be
     * a burden on memory and possibly performance.
     *
     * Note: after calling `cacheResult`, a Seq will always have a `size`.
     */
    cacheResult(): this;

    // Sequence algorithms

    /**
     * Returns a new Seq with values passed through a
     * `mapper` function.
     *
     * ```js
     * const { Seq } = require('immutable')
     * Seq([ 1, 2 ]).map(x => 10 * x)
     * // Seq [ 10, 20 ]
     * ```
     *
     * Note: `map()` always returns a new instance, even if it produced the same
     * value at every step.
     */
    map<M>(
      mapper: (value: V, key: K, iter: this) => M,
      context?: any
    ): Seq<K, M>;

    /**
     * Returns a new Seq with values passed through a
     * `mapper` function.
     *
     * ```js
     * const { Seq } = require('immutable')
     * Seq([ 1, 2 ]).map(x => 10 * x)
     * // Seq [ 10, 20 ]
     * ```
     *
     * Note: `map()` always returns a new instance, even if it produced the same
     * value at every step.
     * Note: used only for sets.
     */
    map<M>(
      mapper: (value: V, key: K, iter: this) => M,
      context?: any
    ): Seq<M, M>;

    /**
     * Flat-maps the Seq, returning a Seq of the same type.
     *
     * Similar to `seq.map(...).flatten(true)`.
     */
    flatMap<M>(
      mapper: (value: V, key: K, iter: this) => Iterable<M>,
      context?: any
    ): Seq<K, M>;

    /**
     * Flat-maps the Seq, returning a Seq of the same type.
     *
     * Similar to `seq.map(...).flatten(true)`.
     * Note: Used only for sets.
     */
    flatMap<M>(
      mapper: (value: V, key: K, iter: this) => Iterable<M>,
      context?: any
    ): Seq<M, M>;

    /**
     * Returns a new Seq with only the values for which the `predicate`
     * function returns true.
     *
     * Note: `filter()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filter<F extends V>(
      predicate: (value: V, key: K, iter: this) => value is F,
      context?: any
    ): Seq<K, F>;
    filter(
      predicate: (value: V, key: K, iter: this) => any,
      context?: any
    ): this;
  }

  /**
   * The `Collection` is a set of (key, value) entries which can be iterated, and
   * is the base class for all collections in `immutable`, allowing them to
   * make use of all the Collection methods (such as `map` and `filter`).
   *
   * Note: A collection is always iterated in the same order, however that order
   * may not always be well defined, as is the case for the `Map` and `Set`.
   *
   * Collection is the abstract base class for concrete data structures. It
   * cannot be constructed directly.
   *
   * Implementations should extend one of the subclasses, `Collection.Keyed`,
   * `Collection.Indexed`, or `Collection.Set`.
   */
  declare module Collection$1 {

    /**
     * @deprecated use `const { isKeyed } = require('immutable')`
     */
    function isKeyed(maybeKeyed: any): maybeKeyed is Collection$1.Keyed<any, any>;

    /**
     * @deprecated use `const { isIndexed } = require('immutable')`
     */
    function isIndexed(maybeIndexed: any): maybeIndexed is Collection$1.Indexed<any>;

    /**
     * @deprecated use `const { isAssociative } = require('immutable')`
     */
    function isAssociative(maybeAssociative: any): maybeAssociative is Collection$1.Keyed<any, any> | Collection$1.Indexed<any>;

    /**
     * @deprecated use `const { isOrdered } = require('immutable')`
     */
    function isOrdered(maybeOrdered: any): boolean;


    /**
     * Keyed Collections have discrete keys tied to each value.
     *
     * When iterating `Collection.Keyed`, each iteration will yield a `[K, V]`
     * tuple, in other words, `Collection#entries` is the default iterator for
     * Keyed Collections.
     */
    export module Keyed {}

    /**
     * Creates a Collection.Keyed
     *
     * Similar to `Collection()`, however it expects collection-likes of [K, V]
     * tuples if not constructed from a Collection.Keyed or JS Object.
     *
     * Note: `Collection.Keyed` is a conversion function and not a class, and
     * does not use the `new` keyword during construction.
     */
    export function Keyed<K, V>(collection: Iterable<[K, V]>): Collection$1.Keyed<K, V>;
    export function Keyed<V>(obj: {[key: string]: V}): Collection$1.Keyed<string, V>;

    export interface Keyed<K, V> extends Collection$1<K, V> {
      /**
       * Deeply converts this Keyed collection to equivalent native JavaScript Object.
       *
       * Converts keys to Strings.
       */
      toJS(): Object;

      /**
       * Shallowly converts this Keyed collection to equivalent native JavaScript Object.
       *
       * Converts keys to Strings.
       */
      toJSON(): { [key: string]: V };

      /**
       * Shallowly converts this collection to an Array.
       */
      toArray(): Array<[K, V]>;

      /**
       * Returns Seq.Keyed.
       * @override
       */
      toSeq(): Seq.Keyed<K, V>;


      // Sequence functions

      /**
       * Returns a new Collection.Keyed of the same type where the keys and values
       * have been flipped.
       *
       * <!-- runkit:activate -->
       * ```js
       * const { Map } = require('immutable')
       * Map({ a: 'z', b: 'y' }).flip()
       * // Map { "z": "a", "y": "b" }
       * ```
       */
      flip(): Collection$1.Keyed<V, K>;

      /**
       * Returns a new Collection with other collections concatenated to this one.
       */
      concat<KC, VC>(...collections: Array<Iterable<[KC, VC]>>): Collection$1.Keyed<K | KC, V | VC>;
      concat<C>(...collections: Array<{[key: string]: C}>): Collection$1.Keyed<K | string, V | C>;

      /**
       * Returns a new Collection.Keyed with values passed through a
       * `mapper` function.
       *
       * ```js
       * const { Collection } = require('immutable')
       * Collection.Keyed({ a: 1, b: 2 }).map(x => 10 * x)
       * // Seq { "a": 10, "b": 20 }
       * ```
       *
       * Note: `map()` always returns a new instance, even if it produced the
       * same value at every step.
       */
      map<M>(
        mapper: (value: V, key: K, iter: this) => M,
        context?: any
      ): Collection$1.Keyed<K, M>;

      /**
       * Returns a new Collection.Keyed of the same type with keys passed through
       * a `mapper` function.
       *
       * <!-- runkit:activate -->
       * ```js
       * const { Map } = require('immutable')
       * Map({ a: 1, b: 2 }).mapKeys(x => x.toUpperCase())
       * // Map { "A": 1, "B": 2 }
       * ```
       *
       * Note: `mapKeys()` always returns a new instance, even if it produced
       * the same key at every step.
       */
      mapKeys<M>(
        mapper: (key: K, value: V, iter: this) => M,
        context?: any
      ): Collection$1.Keyed<M, V>;

      /**
       * Returns a new Collection.Keyed of the same type with entries
       * ([key, value] tuples) passed through a `mapper` function.
       *
       * <!-- runkit:activate -->
       * ```js
       * const { Map } = require('immutable')
       * Map({ a: 1, b: 2 })
       *   .mapEntries(([ k, v ]) => [ k.toUpperCase(), v * 2 ])
       * // Map { "A": 2, "B": 4 }
       * ```
       *
       * Note: `mapEntries()` always returns a new instance, even if it produced
       * the same entry at every step.
       */
      mapEntries<KM, VM>(
        mapper: (entry: [K, V], index: number, iter: this) => [KM, VM],
        context?: any
      ): Collection$1.Keyed<KM, VM>;

      /**
       * Flat-maps the Collection, returning a Collection of the same type.
       *
       * Similar to `collection.map(...).flatten(true)`.
       */
      flatMap<KM, VM>(
        mapper: (value: V, key: K, iter: this) => Iterable<[KM, VM]>,
        context?: any
      ): Collection$1.Keyed<KM, VM>;

      /**
       * Returns a new Collection with only the values for which the `predicate`
       * function returns true.
       *
       * Note: `filter()` always returns a new instance, even if it results in
       * not filtering out any values.
       */
      filter<F extends V>(
        predicate: (value: V, key: K, iter: this) => value is F,
        context?: any
      ): Collection$1.Keyed<K, F>;
      filter(
        predicate: (value: V, key: K, iter: this) => any,
        context?: any
      ): this;

      [Symbol.iterator](): IterableIterator<[K, V]>;
    }


    /**
     * Indexed Collections have incrementing numeric keys. They exhibit
     * slightly different behavior than `Collection.Keyed` for some methods in order
     * to better mirror the behavior of JavaScript's `Array`, and add methods
     * which do not make sense on non-indexed Collections such as `indexOf`.
     *
     * Unlike JavaScript arrays, `Collection.Indexed`s are always dense. "Unset"
     * indices and `undefined` indices are indistinguishable, and all indices from
     * 0 to `size` are visited when iterated.
     *
     * All Collection.Indexed methods return re-indexed Collections. In other words,
     * indices always start at 0 and increment until size. If you wish to
     * preserve indices, using them as keys, convert to a Collection.Keyed by
     * calling `toKeyedSeq`.
     */
    export module Indexed {}

    /**
     * Creates a new Collection.Indexed.
     *
     * Note: `Collection.Indexed` is a conversion function and not a class, and
     * does not use the `new` keyword during construction.
     */
    export function Indexed<T>(collection: Iterable<T>): Collection$1.Indexed<T>;

    export interface Indexed<T> extends Collection$1<number, T> {
      /**
       * Deeply converts this Indexed collection to equivalent native JavaScript Array.
       */
      toJS(): Array<any>;

      /**
       * Shallowly converts this Indexed collection to equivalent native JavaScript Array.
       */
      toJSON(): Array<T>;

      /**
       * Shallowly converts this collection to an Array.
       */
      toArray(): Array<T>;

      // Reading values

      /**
       * Returns the value associated with the provided index, or notSetValue if
       * the index is beyond the bounds of the Collection.
       *
       * `index` may be a negative number, which indexes back from the end of the
       * Collection. `s.get(-1)` gets the last item in the Collection.
       */
      get<NSV>(index: number, notSetValue: NSV): T | NSV;
      get(index: number): T | undefined;


      // Conversion to Seq

      /**
       * Returns Seq.Indexed.
       * @override
       */
      toSeq(): Seq.Indexed<T>;

      /**
       * If this is a collection of [key, value] entry tuples, it will return a
       * Seq.Keyed of those entries.
       */
      fromEntrySeq(): Seq.Keyed<any, any>;


      // Combination

      /**
       * Returns a Collection of the same type with `separator` between each item
       * in this Collection.
       */
      interpose(separator: T): this;

      /**
       * Returns a Collection of the same type with the provided `collections`
       * interleaved into this collection.
       *
       * The resulting Collection includes the first item from each, then the
       * second from each, etc.
       *
       * <!-- runkit:activate
       *      { "preamble": "require('immutable')"}
       * -->
       * ```js
       * const { List } = require('immutable')
       * List([ 1, 2, 3 ]).interleave(List([ 'A', 'B', 'C' ]))
       * // List [ 1, "A", 2, "B", 3, "C"" ]
       * ```
       *
       * The shortest Collection stops interleave.
       *
       * <!-- runkit:activate
       *      { "preamble": "const { List } = require('immutable')" }
       * -->
       * ```js
       * List([ 1, 2, 3 ]).interleave(
       *   List([ 'A', 'B' ]),
       *   List([ 'X', 'Y', 'Z' ])
       * )
       * // List [ 1, "A", "X", 2, "B", "Y"" ]
       * ```
       *
       * Since `interleave()` re-indexes values, it produces a complete copy,
       * which has `O(N)` complexity.
       *
       * Note: `interleave` *cannot* be used in `withMutations`.
       */
      interleave(...collections: Array<Collection$1<any, T>>): this;

      /**
       * Splice returns a new indexed Collection by replacing a region of this
       * Collection with new values. If values are not provided, it only skips the
       * region to be removed.
       *
       * `index` may be a negative number, which indexes back from the end of the
       * Collection. `s.splice(-2)` splices after the second to last item.
       *
       * <!-- runkit:activate -->
       * ```js
       * const { List } = require('immutable')
       * List([ 'a', 'b', 'c', 'd' ]).splice(1, 2, 'q', 'r', 's')
       * // List [ "a", "q", "r", "s", "d" ]
       * ```
       *
       * Since `splice()` re-indexes values, it produces a complete copy, which
       * has `O(N)` complexity.
       *
       * Note: `splice` *cannot* be used in `withMutations`.
       */
      splice(
        index: number,
        removeNum: number,
        ...values: Array<T>
      ): this;

      /**
       * Returns a Collection of the same type "zipped" with the provided
       * collections.
       *
       * Like `zipWith`, but using the default `zipper`: creating an `Array`.
       *
       *
       * <!-- runkit:activate
       *      { "preamble": "const { List } = require('immutable')" }
       * -->
       * ```js
       * const a = List([ 1, 2, 3 ]);
       * const b = List([ 4, 5, 6 ]);
       * const c = a.zip(b); // List [ [ 1, 4 ], [ 2, 5 ], [ 3, 6 ] ]
       * ```
       */
      zip<U>(other: Collection$1<any, U>): Collection$1.Indexed<[T,U]>;
      zip<U,V>(other: Collection$1<any, U>, other2: Collection$1<any, V>): Collection$1.Indexed<[T,U,V]>;
      zip(...collections: Array<Collection$1<any, any>>): Collection$1.Indexed<any>;

      /**
       * Returns a Collection "zipped" with the provided collections.
       *
       * Unlike `zip`, `zipAll` continues zipping until the longest collection is
       * exhausted. Missing values from shorter collections are filled with `undefined`.
       *
       * ```js
       * const a = List([ 1, 2 ]);
       * const b = List([ 3, 4, 5 ]);
       * const c = a.zipAll(b); // List [ [ 1, 3 ], [ 2, 4 ], [ undefined, 5 ] ]
       * ```
       */
      zipAll<U>(other: Collection$1<any, U>): Collection$1.Indexed<[T,U]>;
      zipAll<U,V>(other: Collection$1<any, U>, other2: Collection$1<any, V>): Collection$1.Indexed<[T,U,V]>;
      zipAll(...collections: Array<Collection$1<any, any>>): Collection$1.Indexed<any>;

      /**
       * Returns a Collection of the same type "zipped" with the provided
       * collections by using a custom `zipper` function.
       *
       * <!-- runkit:activate
       *      { "preamble": "const { List } = require('immutable')" }
       * -->
       * ```js
       * const a = List([ 1, 2, 3 ]);
       * const b = List([ 4, 5, 6 ]);
       * const c = a.zipWith((a, b) => a + b, b);
       * // List [ 5, 7, 9 ]
       * ```
       */
      zipWith<U, Z>(
        zipper: (value: T, otherValue: U) => Z,
        otherCollection: Collection$1<any, U>
      ): Collection$1.Indexed<Z>;
      zipWith<U, V, Z>(
        zipper: (value: T, otherValue: U, thirdValue: V) => Z,
        otherCollection: Collection$1<any, U>,
        thirdCollection: Collection$1<any, V>
      ): Collection$1.Indexed<Z>;
      zipWith<Z>(
        zipper: (...any: Array<any>) => Z,
        ...collections: Array<Collection$1<any, any>>
      ): Collection$1.Indexed<Z>;


      // Search for value

      /**
       * Returns the first index at which a given value can be found in the
       * Collection, or -1 if it is not present.
       */
      indexOf(searchValue: T): number;

      /**
       * Returns the last index at which a given value can be found in the
       * Collection, or -1 if it is not present.
       */
      lastIndexOf(searchValue: T): number;

      /**
       * Returns the first index in the Collection where a value satisfies the
       * provided predicate function. Otherwise -1 is returned.
       */
      findIndex(
        predicate: (value: T, index: number, iter: this) => boolean,
        context?: any
      ): number;

      /**
       * Returns the last index in the Collection where a value satisfies the
       * provided predicate function. Otherwise -1 is returned.
       */
      findLastIndex(
        predicate: (value: T, index: number, iter: this) => boolean,
        context?: any
      ): number;

      // Sequence algorithms

      /**
       * Returns a new Collection with other collections concatenated to this one.
       */
      concat<C>(...valuesOrCollections: Array<Iterable<C> | C>): Collection$1.Indexed<T | C>;

      /**
       * Returns a new Collection.Indexed with values passed through a
       * `mapper` function.
       *
       * ```js
       * const { Collection } = require('immutable')
       * Collection.Indexed([1,2]).map(x => 10 * x)
       * // Seq [ 1, 2 ]
       * ```
       *
       * Note: `map()` always returns a new instance, even if it produced the
       * same value at every step.
       */
      map<M>(
        mapper: (value: T, key: number, iter: this) => M,
        context?: any
      ): Collection$1.Indexed<M>;

      /**
       * Flat-maps the Collection, returning a Collection of the same type.
       *
       * Similar to `collection.map(...).flatten(true)`.
       */
      flatMap<M>(
        mapper: (value: T, key: number, iter: this) => Iterable<M>,
        context?: any
      ): Collection$1.Indexed<M>;

      /**
       * Returns a new Collection with only the values for which the `predicate`
       * function returns true.
       *
       * Note: `filter()` always returns a new instance, even if it results in
       * not filtering out any values.
       */
      filter<F extends T>(
        predicate: (value: T, index: number, iter: this) => value is F,
        context?: any
      ): Collection$1.Indexed<F>;
      filter(
        predicate: (value: T, index: number, iter: this) => any,
        context?: any
      ): this;

      [Symbol.iterator](): IterableIterator<T>;
    }


    /**
     * Set Collections only represent values. They have no associated keys or
     * indices. Duplicate values are possible in the lazy `Seq.Set`s, however
     * the concrete `Set` Collection does not allow duplicate values.
     *
     * Collection methods on Collection.Set such as `map` and `forEach` will provide
     * the value as both the first and second arguments to the provided function.
     *
     * ```js
     * const { Collection } = require('immutable')
     * const seq = Collection.Set([ 'A', 'B', 'C' ])
     * // Seq { "A", "B", "C" }
     * seq.forEach((v, k) =>
     *  assert.equal(v, k)
     * )
     * ```
     */
    export module Set {}

    /**
     * Similar to `Collection()`, but always returns a Collection.Set.
     *
     * Note: `Collection.Set` is a factory function and not a class, and does
     * not use the `new` keyword during construction.
     */
    export function Set<T>(collection: Iterable<T>): Collection$1.Set<T>;

    export interface Set<T> extends Collection$1<T, T> {
      /**
       * Deeply converts this Set collection to equivalent native JavaScript Array.
       */
      toJS(): Array<any>;

      /**
       * Shallowly converts this Set collection to equivalent native JavaScript Array.
       */
      toJSON(): Array<T>;

      /**
       * Shallowly converts this collection to an Array.
       */
      toArray(): Array<T>;

      /**
       * Returns Seq.Set.
       * @override
       */
      toSeq(): Seq.Set<T>;

      // Sequence algorithms

      /**
       * Returns a new Collection with other collections concatenated to this one.
       */
      concat<U>(...collections: Array<Iterable<U>>): Collection$1.Set<T | U>;

      /**
       * Returns a new Collection.Set with values passed through a
       * `mapper` function.
       *
       * ```
       * Collection.Set([ 1, 2 ]).map(x => 10 * x)
       * // Seq { 1, 2 }
       * ```
       *
       * Note: `map()` always returns a new instance, even if it produced the
       * same value at every step.
       */
      map<M>(
        mapper: (value: T, key: T, iter: this) => M,
        context?: any
      ): Collection$1.Set<M>;

      /**
       * Flat-maps the Collection, returning a Collection of the same type.
       *
       * Similar to `collection.map(...).flatten(true)`.
       */
      flatMap<M>(
        mapper: (value: T, key: T, iter: this) => Iterable<M>,
        context?: any
      ): Collection$1.Set<M>;

      /**
       * Returns a new Collection with only the values for which the `predicate`
       * function returns true.
       *
       * Note: `filter()` always returns a new instance, even if it results in
       * not filtering out any values.
       */
      filter<F extends T>(
        predicate: (value: T, key: T, iter: this) => value is F,
        context?: any
      ): Collection$1.Set<F>;
      filter(
        predicate: (value: T, key: T, iter: this) => any,
        context?: any
      ): this;

      [Symbol.iterator](): IterableIterator<T>;
    }

  }

  /**
   * Creates a Collection.
   *
   * The type of Collection created is based on the input.
   *
   *   * If an `Collection`, that same `Collection`.
   *   * If an Array-like, an `Collection.Indexed`.
   *   * If an Object with an Iterator defined, an `Collection.Indexed`.
   *   * If an Object, an `Collection.Keyed`.
   *
   * This methods forces the conversion of Objects and Strings to Collections.
   * If you want to ensure that a Collection of one item is returned, use
   * `Seq.of`.
   *
   * Note: An Iterator itself will be treated as an object, becoming a `Seq.Keyed`,
   * which is usually not what you want. You should turn your Iterator Object into
   * an iterable object by defining a Symbol.iterator (or @@iterator) method which
   * returns `this`.
   *
   * Note: `Collection` is a conversion function and not a class, and does not
   * use the `new` keyword during construction.
   */
  declare function Collection$1<I extends Collection$1<any, any>>(collection: I): I;
  declare function Collection$1<T>(collection: Iterable<T>): Collection$1.Indexed<T>;
  declare function Collection$1<V>(obj: {[key: string]: V}): Collection$1.Keyed<string, V>;

  interface Collection$1<K, V> extends ValueObject {

    // Value equality

    /**
     * True if this and the other Collection have value equality, as defined
     * by `Immutable.is()`.
     *
     * Note: This is equivalent to `Immutable.is(this, other)`, but provided to
     * allow for chained expressions.
     */
    equals(other: any): boolean;

    /**
     * Computes and returns the hashed identity for this Collection.
     *
     * The `hashCode` of a Collection is used to determine potential equality,
     * and is used when adding this to a `Set` or as a key in a `Map`, enabling
     * lookup via a different instance.
     *
     * <!-- runkit:activate
     *      { "preamble": "const { Set,  List } = require('immutable')" }
     * -->
     * ```js
     * const a = List([ 1, 2, 3 ]);
     * const b = List([ 1, 2, 3 ]);
     * assert.notStrictEqual(a, b); // different instances
     * const set = Set([ a ]);
     * assert.equal(set.has(b), true);
     * ```
     *
     * If two values have the same `hashCode`, they are [not guaranteed
     * to be equal][Hash Collision]. If two values have different `hashCode`s,
     * they must not be equal.
     *
     * [Hash Collision]: http://en.wikipedia.org/wiki/Collision_(computer_science)
     */
    hashCode(): number;


    // Reading values

    /**
     * Returns the value associated with the provided key, or notSetValue if
     * the Collection does not contain this key.
     *
     * Note: it is possible a key may be associated with an `undefined` value,
     * so if `notSetValue` is not provided and this method returns `undefined`,
     * that does not guarantee the key was not found.
     */
    get<NSV>(key: K, notSetValue: NSV): V | NSV;
    get(key: K): V | undefined;

    /**
     * True if a key exists within this `Collection`, using `Immutable.is`
     * to determine equality
     */
    has(key: K): boolean;

    /**
     * True if a value exists within this `Collection`, using `Immutable.is`
     * to determine equality
     * @alias contains
     */
    includes(value: V): boolean;
    contains(value: V): boolean;

    /**
     * In case the `Collection` is not empty returns the first element of the
     * `Collection`.
     * In case the `Collection` is empty returns the optional default
     * value if provided, if no default value is provided returns undefined.
     */
    first<NSV>(notSetValue?: NSV): V | NSV;

    /**
     * In case the `Collection` is not empty returns the last element of the
     * `Collection`.
     * In case the `Collection` is empty returns the optional default
     * value if provided, if no default value is provided returns undefined.
     */
    last<NSV>(notSetValue?: NSV): V | NSV;

    // Reading deep values

    /**
     * Returns the value found by following a path of keys or indices through
     * nested Collections.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map, List } = require('immutable')
     * const deepData = Map({ x: List([ Map({ y: 123 }) ]) });
     * deepData.getIn(['x', 0, 'y']) // 123
     * ```
     *
     * Plain JavaScript Object or Arrays may be nested within an Immutable.js
     * Collection, and getIn() can access those values as well:
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map, List } = require('immutable')
     * const deepData = Map({ x: [ { y: 123 } ] });
     * deepData.getIn(['x', 0, 'y']) // 123
     * ```
     */
    getIn(searchKeyPath: Iterable<any>, notSetValue?: any): any;

    /**
     * True if the result of following a path of keys or indices through nested
     * Collections results in a set value.
     */
    hasIn(searchKeyPath: Iterable<any>): boolean;

    // Persistent changes

    /**
     * This can be very useful as a way to "chain" a normal function into a
     * sequence of methods. RxJS calls this "let" and lodash calls it "thru".
     *
     * For example, to sum a Seq after mapping and filtering:
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Seq } = require('immutable')
     *
     * function sum(collection) {
     *   return collection.reduce((sum, x) => sum + x, 0)
     * }
     *
     * Seq([ 1, 2, 3 ])
     *   .map(x => x + 1)
     *   .filter(x => x % 2 === 0)
     *   .update(sum)
     * // 6
     * ```
     */
    update<R>(updater: (value: this) => R): R;


    // Conversion to JavaScript types

    /**
     * Deeply converts this Collection to equivalent native JavaScript Array or Object.
     *
     * `Collection.Indexed`, and `Collection.Set` become `Array`, while
     * `Collection.Keyed` become `Object`, converting keys to Strings.
     */
    toJS(): Array<any> | { [key: string]: any };

    /**
     * Shallowly converts this Collection to equivalent native JavaScript Array or Object.
     *
     * `Collection.Indexed`, and `Collection.Set` become `Array`, while
     * `Collection.Keyed` become `Object`, converting keys to Strings.
     */
    toJSON(): Array<V> | { [key: string]: V };

    /**
     * Shallowly converts this collection to an Array.
     *
     * `Collection.Indexed`, and `Collection.Set` produce an Array of values.
     * `Collection.Keyed` produce an Array of [key, value] tuples.
     */
    toArray(): Array<V> | Array<[K, V]>;

    /**
     * Shallowly converts this Collection to an Object.
     *
     * Converts keys to Strings.
     */
    toObject(): { [key: string]: V };


    // Conversion to Collections

    /**
     * Converts this Collection to a Map, Throws if keys are not hashable.
     *
     * Note: This is equivalent to `Map(this.toKeyedSeq())`, but provided
     * for convenience and to allow for chained expressions.
     */
    toMap(): Map$1<K, V>;

    /**
     * Converts this Collection to a Map, maintaining the order of iteration.
     *
     * Note: This is equivalent to `OrderedMap(this.toKeyedSeq())`, but
     * provided for convenience and to allow for chained expressions.
     */
    toOrderedMap(): OrderedMap<K, V>;

    /**
     * Converts this Collection to a Set, discarding keys. Throws if values
     * are not hashable.
     *
     * Note: This is equivalent to `Set(this)`, but provided to allow for
     * chained expressions.
     */
    toSet(): Set$1<V>;

    /**
     * Converts this Collection to a Set, maintaining the order of iteration and
     * discarding keys.
     *
     * Note: This is equivalent to `OrderedSet(this.valueSeq())`, but provided
     * for convenience and to allow for chained expressions.
     */
    toOrderedSet(): OrderedSet<V>;

    /**
     * Converts this Collection to a List, discarding keys.
     *
     * This is similar to `List(collection)`, but provided to allow for chained
     * expressions. However, when called on `Map` or other keyed collections,
     * `collection.toList()` discards the keys and creates a list of only the
     * values, whereas `List(collection)` creates a list of entry tuples.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map, List } = require('immutable')
     * var myMap = Map({ a: 'Apple', b: 'Banana' })
     * List(myMap) // List [ [ "a", "Apple" ], [ "b", "Banana" ] ]
     * myMap.toList() // List [ "Apple", "Banana" ]
     * ```
     */
    toList(): List<V>;

    /**
     * Converts this Collection to a Stack, discarding keys. Throws if values
     * are not hashable.
     *
     * Note: This is equivalent to `Stack(this)`, but provided to allow for
     * chained expressions.
     */
    toStack(): Stack<V>;


    // Conversion to Seq

    /**
     * Converts this Collection to a Seq of the same kind (indexed,
     * keyed, or set).
     */
    toSeq(): Seq<K, V>;

    /**
     * Returns a Seq.Keyed from this Collection where indices are treated as keys.
     *
     * This is useful if you want to operate on an
     * Collection.Indexed and preserve the [index, value] pairs.
     *
     * The returned Seq will have identical iteration order as
     * this Collection.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Seq } = require('immutable')
     * const indexedSeq = Seq([ 'A', 'B', 'C' ])
     * // Seq [ "A", "B", "C" ]
     * indexedSeq.filter(v => v === 'B')
     * // Seq [ "B" ]
     * const keyedSeq = indexedSeq.toKeyedSeq()
     * // Seq { 0: "A", 1: "B", 2: "C" }
     * keyedSeq.filter(v => v === 'B')
     * // Seq { 1: "B" }
     * ```
     */
    toKeyedSeq(): Seq.Keyed<K, V>;

    /**
     * Returns an Seq.Indexed of the values of this Collection, discarding keys.
     */
    toIndexedSeq(): Seq.Indexed<V>;

    /**
     * Returns a Seq.Set of the values of this Collection, discarding keys.
     */
    toSetSeq(): Seq.Set<V>;


    // Iterators

    /**
     * An iterator of this `Collection`'s keys.
     *
     * Note: this will return an ES6 iterator which does not support
     * Immutable.js sequence algorithms. Use `keySeq` instead, if this is
     * what you want.
     */
    keys(): IterableIterator<K>;

    /**
     * An iterator of this `Collection`'s values.
     *
     * Note: this will return an ES6 iterator which does not support
     * Immutable.js sequence algorithms. Use `valueSeq` instead, if this is
     * what you want.
     */
    values(): IterableIterator<V>;

    /**
     * An iterator of this `Collection`'s entries as `[ key, value ]` tuples.
     *
     * Note: this will return an ES6 iterator which does not support
     * Immutable.js sequence algorithms. Use `entrySeq` instead, if this is
     * what you want.
     */
    entries(): IterableIterator<[K, V]>;


    // Collections (Seq)

    /**
     * Returns a new Seq.Indexed of the keys of this Collection,
     * discarding values.
     */
    keySeq(): Seq.Indexed<K>;

    /**
     * Returns an Seq.Indexed of the values of this Collection, discarding keys.
     */
    valueSeq(): Seq.Indexed<V>;

    /**
     * Returns a new Seq.Indexed of [key, value] tuples.
     */
    entrySeq(): Seq.Indexed<[K, V]>;


    // Sequence algorithms

    /**
     * Returns a new Collection of the same type with values passed through a
     * `mapper` function.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Collection } = require('immutable')
     * Collection({ a: 1, b: 2 }).map(x => 10 * x)
     * // Seq { "a": 10, "b": 20 }
     * ```
     *
     * Note: `map()` always returns a new instance, even if it produced the same
     * value at every step.
     */
    map<M>(
      mapper: (value: V, key: K, iter: this) => M,
      context?: any
    ): Collection$1<K, M>;

    /**
     * Note: used only for sets, which return Collection<M, M> but are otherwise
     * identical to normal `map()`.
     *
     * @ignore
     */
    map<M>(...args: never[]): any;

    /**
     * Returns a new Collection of the same type with only the entries for which
     * the `predicate` function returns true.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * Map({ a: 1, b: 2, c: 3, d: 4}).filter(x => x % 2 === 0)
     * // Map { "b": 2, "d": 4 }
     * ```
     *
     * Note: `filter()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filter<F extends V>(
      predicate: (value: V, key: K, iter: this) => value is F,
      context?: any
    ): Collection$1<K, F>;
    filter(
      predicate: (value: V, key: K, iter: this) => any,
      context?: any
    ): this;

    /**
     * Returns a new Collection of the same type with only the entries for which
     * the `predicate` function returns false.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * Map({ a: 1, b: 2, c: 3, d: 4}).filterNot(x => x % 2 === 0)
     * // Map { "a": 1, "c": 3 }
     * ```
     *
     * Note: `filterNot()` always returns a new instance, even if it results in
     * not filtering out any values.
     */
    filterNot(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): this;

    /**
     * Returns a new Collection of the same type in reverse order.
     */
    reverse(): this;

    /**
     * Returns a new Collection of the same type which includes the same entries,
     * stably sorted by using a `comparator`.
     *
     * If a `comparator` is not provided, a default comparator uses `<` and `>`.
     *
     * `comparator(valueA, valueB)`:
     *
     *   * Returns `0` if the elements should not be swapped.
     *   * Returns `-1` (or any negative number) if `valueA` comes before `valueB`
     *   * Returns `1` (or any positive number) if `valueA` comes after `valueB`
     *   * Is pure, i.e. it must always return the same value for the same pair
     *     of values.
     *
     * When sorting collections which have no defined order, their ordered
     * equivalents will be returned. e.g. `map.sort()` returns OrderedMap.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { Map } = require('immutable')
     * Map({ "c": 3, "a": 1, "b": 2 }).sort((a, b) => {
     *   if (a < b) { return -1; }
     *   if (a > b) { return 1; }
     *   if (a === b) { return 0; }
     * });
     * // OrderedMap { "a": 1, "b": 2, "c": 3 }
     * ```
     *
     * Note: `sort()` Always returns a new instance, even if the original was
     * already sorted.
     *
     * Note: This is always an eager operation.
     */
    sort(comparator?: (valueA: V, valueB: V) => number): this;

    /**
     * Like `sort`, but also accepts a `comparatorValueMapper` which allows for
     * sorting by more sophisticated means:
     *
     *     hitters.sortBy(hitter => hitter.avgHits)
     *
     * Note: `sortBy()` Always returns a new instance, even if the original was
     * already sorted.
     *
     * Note: This is always an eager operation.
     */
    sortBy<C>(
      comparatorValueMapper: (value: V, key: K, iter: this) => C,
      comparator?: (valueA: C, valueB: C) => number
    ): this;

    /**
     * Returns a `Collection.Keyed` of `Collection.Keyeds`, grouped by the return
     * value of the `grouper` function.
     *
     * Note: This is always an eager operation.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List, Map } = require('immutable')
     * const listOfMaps = List([
     *   Map({ v: 0 }),
     *   Map({ v: 1 }),
     *   Map({ v: 1 }),
     *   Map({ v: 0 }),
     *   Map({ v: 2 })
     * ])
     * const groupsOfMaps = listOfMaps.groupBy(x => x.get('v'))
     * // Map {
     * //   0: List [ Map{ "v": 0 }, Map { "v": 0 } ],
     * //   1: List [ Map{ "v": 1 }, Map { "v": 1 } ],
     * //   2: List [ Map{ "v": 2 } ],
     * // }
     * ```
     */
    groupBy<G>(
      grouper: (value: V, key: K, iter: this) => G,
      context?: any
    ): /*Map*/Seq.Keyed<G, /*this*/Collection$1<K, V>>;


    // Side effects

    /**
     * The `sideEffect` is executed for every entry in the Collection.
     *
     * Unlike `Array#forEach`, if any call of `sideEffect` returns
     * `false`, the iteration will stop. Returns the number of entries iterated
     * (including the last iteration which returned false).
     */
    forEach(
      sideEffect: (value: V, key: K, iter: this) => any,
      context?: any
    ): number;


    // Creating subsets

    /**
     * Returns a new Collection of the same type representing a portion of this
     * Collection from start up to but not including end.
     *
     * If begin is negative, it is offset from the end of the Collection. e.g.
     * `slice(-2)` returns a Collection of the last two entries. If it is not
     * provided the new Collection will begin at the beginning of this Collection.
     *
     * If end is negative, it is offset from the end of the Collection. e.g.
     * `slice(0, -1)` returns a Collection of everything but the last entry. If
     * it is not provided, the new Collection will continue through the end of
     * this Collection.
     *
     * If the requested slice is equivalent to the current Collection, then it
     * will return itself.
     */
    slice(begin?: number, end?: number): this;

    /**
     * Returns a new Collection of the same type containing all entries except
     * the first.
     */
    rest(): this;

    /**
     * Returns a new Collection of the same type containing all entries except
     * the last.
     */
    butLast(): this;

    /**
     * Returns a new Collection of the same type which excludes the first `amount`
     * entries from this Collection.
     */
    skip(amount: number): this;

    /**
     * Returns a new Collection of the same type which excludes the last `amount`
     * entries from this Collection.
     */
    skipLast(amount: number): this;

    /**
     * Returns a new Collection of the same type which includes entries starting
     * from when `predicate` first returns false.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable')
     * List([ 'dog', 'frog', 'cat', 'hat', 'god' ])
     *   .skipWhile(x => x.match(/g/))
     * // List [ "cat", "hat", "god"" ]
     * ```
     */
    skipWhile(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): this;

    /**
     * Returns a new Collection of the same type which includes entries starting
     * from when `predicate` first returns true.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable')
     * List([ 'dog', 'frog', 'cat', 'hat', 'god' ])
     *   .skipUntil(x => x.match(/hat/))
     * // List [ "hat", "god"" ]
     * ```
     */
    skipUntil(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): this;

    /**
     * Returns a new Collection of the same type which includes the first `amount`
     * entries from this Collection.
     */
    take(amount: number): this;

    /**
     * Returns a new Collection of the same type which includes the last `amount`
     * entries from this Collection.
     */
    takeLast(amount: number): this;

    /**
     * Returns a new Collection of the same type which includes entries from this
     * Collection as long as the `predicate` returns true.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable')
     * List([ 'dog', 'frog', 'cat', 'hat', 'god' ])
     *   .takeWhile(x => x.match(/o/))
     * // List [ "dog", "frog" ]
     * ```
     */
    takeWhile(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): this;

    /**
     * Returns a new Collection of the same type which includes entries from this
     * Collection as long as the `predicate` returns false.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List } = require('immutable')
     * List([ 'dog', 'frog', 'cat', 'hat', 'god' ])
     *   .takeUntil(x => x.match(/at/))
     * // List [ "dog", "frog" ]
     * ```
     */
    takeUntil(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): this;


    // Combination

    /**
     * Returns a new Collection of the same type with other values and
     * collection-like concatenated to this one.
     *
     * For Seqs, all entries will be present in the resulting Seq, even if they
     * have the same key.
     */
    concat(...valuesOrCollections: Array<any>): Collection$1<any, any>;

    /**
     * Flattens nested Collections.
     *
     * Will deeply flatten the Collection by default, returning a Collection of the
     * same type, but a `depth` can be provided in the form of a number or
     * boolean (where true means to shallowly flatten one level). A depth of 0
     * (or shallow: false) will deeply flatten.
     *
     * Flattens only others Collection, not Arrays or Objects.
     *
     * Note: `flatten(true)` operates on Collection<any, Collection<K, V>> and
     * returns Collection<K, V>
     */
    flatten(depth?: number): Collection$1<any, any>;
    flatten(shallow?: boolean): Collection$1<any, any>;

    /**
     * Flat-maps the Collection, returning a Collection of the same type.
     *
     * Similar to `collection.map(...).flatten(true)`.
     */
    flatMap<M>(
      mapper: (value: V, key: K, iter: this) => Iterable<M>,
      context?: any
    ): Collection$1<K, M>;

    /**
     * Flat-maps the Collection, returning a Collection of the same type.
     *
     * Similar to `collection.map(...).flatten(true)`.
     * Used for Dictionaries only.
     */
    flatMap<KM, VM>(
      mapper: (value: V, key: K, iter: this) => Iterable<[KM, VM]>,
      context?: any
    ): Collection$1<KM, VM>;

    // Reducing a value

    /**
     * Reduces the Collection to a value by calling the `reducer` for every entry
     * in the Collection and passing along the reduced value.
     *
     * If `initialReduction` is not provided, the first item in the
     * Collection will be used.
     *
     * @see `Array#reduce`.
     */
    reduce<R>(
      reducer: (reduction: R, value: V, key: K, iter: this) => R,
      initialReduction: R,
      context?: any
    ): R;
    reduce<R>(
      reducer: (reduction: V | R, value: V, key: K, iter: this) => R
    ): R;

    /**
     * Reduces the Collection in reverse (from the right side).
     *
     * Note: Similar to this.reverse().reduce(), and provided for parity
     * with `Array#reduceRight`.
     */
    reduceRight<R>(
      reducer: (reduction: R, value: V, key: K, iter: this) => R,
      initialReduction: R,
      context?: any
    ): R;
    reduceRight<R>(
      reducer: (reduction: V | R, value: V, key: K, iter: this) => R
    ): R;

    /**
     * True if `predicate` returns true for all entries in the Collection.
     */
    every(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): boolean;

    /**
     * True if `predicate` returns true for any entry in the Collection.
     */
    some(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): boolean;

    /**
     * Joins values together as a string, inserting a separator between each.
     * The default separator is `","`.
     */
    join(separator?: string): string;

    /**
     * Returns true if this Collection includes no values.
     *
     * For some lazy `Seq`, `isEmpty` might need to iterate to determine
     * emptiness. At most one iteration will occur.
     */
    isEmpty(): boolean;

    /**
     * Returns the size of this Collection.
     *
     * Regardless of if this Collection can describe its size lazily (some Seqs
     * cannot), this method will always return the correct size. E.g. it
     * evaluates a lazy `Seq` if necessary.
     *
     * If `predicate` is provided, then this returns the count of entries in the
     * Collection for which the `predicate` returns true.
     */
    count(): number;
    count(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): number;

    /**
     * Returns a `Seq.Keyed` of counts, grouped by the return value of
     * the `grouper` function.
     *
     * Note: This is not a lazy operation.
     */
    countBy<G>(
      grouper: (value: V, key: K, iter: this) => G,
      context?: any
    ): Map$1<G, number>;


    // Search for value

    /**
     * Returns the first value for which the `predicate` returns true.
     */
    find(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any,
      notSetValue?: V
    ): V | undefined;

    /**
     * Returns the last value for which the `predicate` returns true.
     *
     * Note: `predicate` will be called for each entry in reverse.
     */
    findLast(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any,
      notSetValue?: V
    ): V | undefined;

    /**
     * Returns the first [key, value] entry for which the `predicate` returns true.
     */
    findEntry(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any,
      notSetValue?: V
    ): [K, V] | undefined;

    /**
     * Returns the last [key, value] entry for which the `predicate`
     * returns true.
     *
     * Note: `predicate` will be called for each entry in reverse.
     */
    findLastEntry(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any,
      notSetValue?: V
    ): [K, V] | undefined;

    /**
     * Returns the key for which the `predicate` returns true.
     */
    findKey(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): K | undefined;

    /**
     * Returns the last key for which the `predicate` returns true.
     *
     * Note: `predicate` will be called for each entry in reverse.
     */
    findLastKey(
      predicate: (value: V, key: K, iter: this) => boolean,
      context?: any
    ): K | undefined;

    /**
     * Returns the key associated with the search value, or undefined.
     */
    keyOf(searchValue: V): K | undefined;

    /**
     * Returns the last key associated with the search value, or undefined.
     */
    lastKeyOf(searchValue: V): K | undefined;

    /**
     * Returns the maximum value in this collection. If any values are
     * comparatively equivalent, the first one found will be returned.
     *
     * The `comparator` is used in the same way as `Collection#sort`. If it is not
     * provided, the default comparator is `>`.
     *
     * When two values are considered equivalent, the first encountered will be
     * returned. Otherwise, `max` will operate independent of the order of input
     * as long as the comparator is commutative. The default comparator `>` is
     * commutative *only* when types do not differ.
     *
     * If `comparator` returns 0 and either value is NaN, undefined, or null,
     * that value will be returned.
     */
    max(comparator?: (valueA: V, valueB: V) => number): V | undefined;

    /**
     * Like `max`, but also accepts a `comparatorValueMapper` which allows for
     * comparing by more sophisticated means:
     *
     *     hitters.maxBy(hitter => hitter.avgHits);
     *
     */
    maxBy<C>(
      comparatorValueMapper: (value: V, key: K, iter: this) => C,
      comparator?: (valueA: C, valueB: C) => number
    ): V | undefined;

    /**
     * Returns the minimum value in this collection. If any values are
     * comparatively equivalent, the first one found will be returned.
     *
     * The `comparator` is used in the same way as `Collection#sort`. If it is not
     * provided, the default comparator is `<`.
     *
     * When two values are considered equivalent, the first encountered will be
     * returned. Otherwise, `min` will operate independent of the order of input
     * as long as the comparator is commutative. The default comparator `<` is
     * commutative *only* when types do not differ.
     *
     * If `comparator` returns 0 and either value is NaN, undefined, or null,
     * that value will be returned.
     */
    min(comparator?: (valueA: V, valueB: V) => number): V | undefined;

    /**
     * Like `min`, but also accepts a `comparatorValueMapper` which allows for
     * comparing by more sophisticated means:
     *
     *     hitters.minBy(hitter => hitter.avgHits);
     *
     */
    minBy<C>(
      comparatorValueMapper: (value: V, key: K, iter: this) => C,
      comparator?: (valueA: C, valueB: C) => number
    ): V | undefined;


    // Comparison

    /**
     * True if `iter` includes every value in this Collection.
     */
    isSubset(iter: Iterable<V>): boolean;

    /**
     * True if this Collection includes every value in `iter`.
     */
    isSuperset(iter: Iterable<V>): boolean;
  }

  /**
   * The interface to fulfill to qualify as a Value Object.
   */
  interface ValueObject {
    /**
     * True if this and the other Collection have value equality, as defined
     * by `Immutable.is()`.
     *
     * Note: This is equivalent to `Immutable.is(this, other)`, but provided to
     * allow for chained expressions.
     */
    equals(other: any): boolean;

    /**
     * Computes and returns the hashed identity for this Collection.
     *
     * The `hashCode` of a Collection is used to determine potential equality,
     * and is used when adding this to a `Set` or as a key in a `Map`, enabling
     * lookup via a different instance.
     *
     * <!-- runkit:activate -->
     * ```js
     * const { List, Set } = require('immutable');
     * const a = List([ 1, 2, 3 ]);
     * const b = List([ 1, 2, 3 ]);
     * assert.notStrictEqual(a, b); // different instances
     * const set = Set([ a ]);
     * assert.equal(set.has(b), true);
     * ```
     *
     * Note: hashCode() MUST return a Uint32 number. The easiest way to
     * guarantee this is to return `myHash | 0` from a custom implementation.
     *
     * If two values have the same `hashCode`, they are [not guaranteed
     * to be equal][Hash Collision]. If two values have different `hashCode`s,
     * they must not be equal.
     *
     * Note: `hashCode()` is not guaranteed to always be called before
     * `equals()`. Most but not all Immutable.js collections use hash codes to
     * organize their internal data structures, while all Immutable.js
     * collections use equality during lookups.
     *
     * [Hash Collision]: http://en.wikipedia.org/wiki/Collision_(computer_science)
     */
    hashCode(): number;
  }

type IFunction<T = void> = (...args: Array<any>) => T;
type IObject$1 = Record<string, any>;
type Without<T, U> = {
    [P in Exclude<keyof T, keyof U>]?: never;
};
type XOR<T, U> = T | U extends object ? (Without<T, U> & U) | (Without<U, T> & T) : T | U;
type Class<T> = new (...args: Array<any>) => T;

declare const TransformationMatrix_base: Record$1.Factory<{
    a: number;
    b: number;
    c: number;
    d: number;
    e: number;
    f: number;
}>;
declare class TransformationMatrix extends TransformationMatrix_base {
    a: number;
    b: number;
    c: number;
    d: number;
    e: number;
    f: number;
    static defaultValues: IObject$1;
    static IDENTITY: TransformationMatrix;
    translate({ x: tx, y: ty }: {
        x: number;
        y: number;
    }): TransformationMatrix;
    translateX(tx: number): TransformationMatrix;
    translateY(ty: number): TransformationMatrix;
    scale(sx: number, sy?: number): TransformationMatrix;
    transform(a2: number, b2: number, c2: number, d2: number, e2: number, f2: number): TransformationMatrix;
    rotate(degCW: number): TransformationMatrix;
    rotateRad(a: number): TransformationMatrix;
    inverse(): TransformationMatrix;
    toCssValue(): string;
    applyToPoint([x, y]: [number, number]): [number, number];
    applyToVector([x, y]: [number, number]): [number, number];
}

interface PointCtorProps {
    x?: number;
    y?: number;
    [k: string]: unknown;
}
declare const Point_base: Record$1.Factory<PointCtorProps>;
declare class Point extends Point_base {
    x: number;
    y: number;
    static defaultValues: IObject$1;
    constructor(options?: PointCtorProps);
    scale(sx: number, sy?: number): this;
    translate({ x: tx, y: ty }: {
        x: number;
        y: number;
    }): this;
    translateX(tx: number): this;
    translateY(ty: number): this;
    distance(other: this): number;
    rotate(deg: number): this;
    apply(matrix: TransformationMatrix): this;
}

interface IDrawingPoint extends PointCtorProps {
    intensity?: number;
}
declare class DrawingPoint extends Point {
    intensity: number;
    static defaultValues: IObject$1;
    constructor(options?: IDrawingPoint);
}

interface ISize {
    width: number;
    height: number;
}
declare const Size_base: Record$1.Factory<ISize>;
declare class Size$1 extends Size_base {
    scale(factor: number): Size$1;
    ceil(): Size$1;
    floor(): Size$1;
    swapDimensions(): Size$1;
    apply(matrix: TransformationMatrix): Size$1;
}

interface IRect {
    left?: number;
    top?: number;
    width?: number;
    height?: number;
}
declare const Rect_base: Record$1.Factory<IRect>;
declare class Rect$2 extends Rect_base {
    left: number;
    top: number;
    width: number;
    height: number;
    static defaultValues: IObject$1;
    constructor(options?: IRect);
    get right(): number;
    get bottom(): number;
    static fromClientRect({ top, left, width, height }: ClientRect): Rect$2;
    static union(rects: List<Rect$2>): Rect$2;
    static getCenteredRect(inner: Size$1, outer: Size$1): Rect$2;
    static fromInset(inset: Inset): Rect$2;
    static fromPoints(...points: Point[]): Rect$2;
    expandToIncludePoints(...points: Point[]): Rect$2;
    static areRectsCloserThan(a: Rect$2, b: Rect$2, distance: number): boolean;
    static areVerticallyAligned(a: Rect$2, b: Rect$2, thresholdDistance: number): boolean;
    translate({ x: tx, y: ty }: Point): Rect$2;
    translateX(tx: number): Rect$2;
    translateY(ty: number): Rect$2;
    scale(sx: number, sy?: number): Rect$2;
    grow(growth: number): Rect$2;
    getLocation(): Point;
    getSize(): Size$1;
    getCenter(): Point;
    setLocation(location: Point): Rect$2;
    roundOverlap(): Rect$2;
    round(): Rect$2;
    isPointInside(point: Point): boolean;
    isRectInside(other: Rect$2): boolean;
    isRectOverlapping(other: Rect$2): boolean;
    normalize(): Rect$2;
    apply(matrix: TransformationMatrix): Rect$2;
}

interface IInset {
    left: number;
    top: number;
    right: number;
    bottom: number;
}
declare const Inset_base: Record$1.Factory<IInset>;
declare class Inset extends Inset_base {
    static applyToRect(inset: Inset, rect: Rect$2): Rect$2;
    static fromRect(rect: Rect$2): Inset;
    static fromValue(insetValue: number): Inset;
    apply(matrix: TransformationMatrix): Inset;
    setScale(scale: number): Inset;
}
type InsetJSON = [left: number, top: number, right: number, bottom: number];

type ActionCtorProps = {
    subActions?: List<Action> | null;
};
declare abstract class Action extends InheritableImmutableRecord<ActionCtorProps> {
    subActions?: List<Action> | null | undefined;
    protected constructor(args?: ActionCtorProps);
}

type ActionTriggerEventType = 'onPointerEnter' | 'onPointerLeave' | 'onPointerDown' | 'onPointerUp' | 'onPageOpen' | 'onPageClose' | 'onPageVisible' | 'onPageHidden';

interface IGoToAction extends ActionCtorProps {
    pageIndex?: number;
}
declare class GoToAction extends Action {
    pageIndex: number;
    static defaultValues: IObject$1;
    constructor(args?: IGoToAction);
}

interface IGoToEmbeddedAction extends ActionCtorProps {
    newWindow?: boolean;
    relativePath?: string;
    targetType?: 'parent' | 'child';
}
declare class GoToEmbeddedAction extends Action {
    newWindow: boolean;
    relativePath: string;
    targetType: 'parent' | 'child';
    static defaultValues: IObject$1;
    constructor(args?: IGoToEmbeddedAction);
}

interface IGoToRemoteAction extends ActionCtorProps {
    relativePath?: string;
    namedDestination?: string;
}
declare class GoToRemoteAction extends Action {
    relativePath: string;
    namedDestination: string;
    static defaultValues: IObject$1;
    constructor(args?: IGoToRemoteAction);
}

type AnnotationReference = {
    fieldName: string;
} | {
    pdfObjectId: number;
};
interface IHideAction extends ActionCtorProps {
    hide?: boolean;
    annotationReferences?: List<AnnotationReference>;
}
declare class HideAction extends Action {
    hide: boolean;
    annotationReferences: List<AnnotationReference>;
    static defaultValues: IObject$1;
    constructor(args?: IHideAction);
}

interface IJavaScriptAction extends ActionCtorProps {
    script?: string;
}
declare class JavaScriptAction extends Action {
    script: string;
    static defaultValues: IObject$1;
    constructor(args?: IJavaScriptAction);
}

interface ILaunchAction extends ActionCtorProps {
    filePath?: string;
}
declare class LaunchAction extends Action {
    filePath: string;
    static defaultValues: IObject$1;
    constructor(args?: ILaunchAction);
}

interface INamedAction extends ActionCtorProps {
    action?: string;
}
declare class NamedAction extends Action {
    action: string;
    static defaultValues: IObject$1;
    constructor(args?: INamedAction);
}

interface IResetFormAction extends ActionCtorProps {
    fields?: List<string> | null | undefined;
    includeExclude?: boolean;
}
declare class ResetFormAction extends Action {
    fields: List<string> | null | undefined;
    includeExclude: boolean;
    static defaultValues: IObject$1;
    constructor(args?: IResetFormAction);
}

interface ISubmitFormAction extends ActionCtorProps {
    uri?: string;
    fields?: List<string>;
    includeExclude?: boolean;
    includeNoValueFields?: boolean;
    exportFormat?: boolean;
    getMethod?: boolean;
    submitCoordinated?: boolean;
    xfdf?: boolean;
    includeAppendSaves?: boolean;
    includeAnnotations?: boolean;
    submitPDF?: boolean;
    canonicalFormat?: boolean;
    excludeNonUserAnnotations?: boolean;
    excludeFKey?: boolean;
    embedForm?: boolean;
}
declare class SubmitFormAction extends Action {
    uri: string;
    fields: List<string> | null | undefined;
    includeExclude: boolean;
    includeNoValueFields: boolean;
    exportFormat: boolean;
    getMethod: boolean;
    submitCoordinated: boolean;
    xfdf: boolean;
    includeAppendSaves: boolean;
    includeAnnotations: boolean;
    submitPDF: boolean;
    canonicalFormat: boolean;
    excludeNonUserAnnotations: boolean;
    excludeFKey: boolean;
    embedForm: boolean;
    static defaultValues: IObject$1;
    constructor(args?: ISubmitFormAction);
}

interface IURIAction extends ActionCtorProps {
    uri?: string;
}
declare class URIAction extends Action {
    uri: string;
    static defaultValues: IObject$1;
    constructor(args?: IURIAction);
}

declare const Color_base: Record$1.Factory<{
    r: number;
    g: number;
    b: number;
    transparent: boolean;
}>;
declare class Color$2 extends Color_base {
    static BLACK: Color$2;
    static GREY: Color$2;
    static WHITE: Color$2;
    static DARK_BLUE: Color$2;
    static RED: Color$2;
    static PURPLE: Color$2;
    static PINK: Color$2;
    static GREEN: Color$2;
    static ORANGE: Color$2;
    static YELLOW: Color$2;
    static LIGHT_BLUE: Color$2;
    static LIGHT_RED: Color$2;
    static LIGHT_GREEN: Color$2;
    static LIGHT_YELLOW: Color$2;
    static BLUE: Color$2;
    static LIGHT_ORANGE: Color$2;
    static LIGHT_GREY: Color$2;
    static DARK_GREY: Color$2;
    static MAUVE: Color$2;
    static TRANSPARENT: Color$2;
    static fromHex: (hexColor: string) => Color$2;
    constructor(args: {
        r?: number;
        g?: number;
        b?: number;
        transparent?: boolean;
    });
    lighter(percent: number): Color$2;
    darker(percent: number): Color$2;
    equals(color: Color$2 | {
        r: number;
        g: number;
        b: number;
        transparent: boolean;
    }): boolean;
    saturate(percent: number): Color$2;
    sRGBToRGBComponent(RGBComponent: number): number;
    relativeLuminance(): number;
    contrastRatio(color: Color$2): number;
    toCSSValue(): string;
    toHex(): string;
}

declare const BlendMode: {
    readonly normal: "normal";
    readonly multiply: "multiply";
    readonly screen: "screen";
    readonly overlay: "overlay";
    readonly darken: "darken";
    readonly lighten: "lighten";
    readonly colorDodge: "colorDodge";
    readonly colorBurn: "colorBurn";
    readonly hardLight: "hardLight";
    readonly softLight: "softLight";
    readonly difference: "difference";
    readonly exclusion: "exclusion";
};
type IBlendMode = (typeof BlendMode)[keyof typeof BlendMode];

interface ITextMarkupAnnotation extends AnnotationProperties {
    rects: List<Rect$2>;
    color: Color$2;
    blendMode: IBlendMode;
}
declare class TextMarkupAnnotation<T extends ITextMarkupAnnotation = ITextMarkupAnnotation> extends Annotation<T> {
    rects: List<Rect$2>;
    color: Color$2;
    blendMode: IBlendMode;
    static defaultValues: IObject$1;
    static readableName: string;
}

interface IHighlightAnnotation extends ITextMarkupAnnotation {
    color: Color$2;
    blendMode: IBlendMode | 'multiply';
}
declare class HighlightAnnotation<T extends IHighlightAnnotation = IHighlightAnnotation> extends TextMarkupAnnotation<T> {
    blendMode: IBlendMode;
    static className: string;
    static readableName: string;
    static defaultValues: IObject$1;
}

interface IImageAnnotation extends AnnotationProperties {
    description: string | null;
    fileName: string | null;
    contentType: string | null;
    imageAttachmentId: string | null;
    isSignature: boolean;
    xfdfAppearanceStream: string | null;
    xfdfAppearanceStreamOriginalPageRotation: number | null;
}
declare class ImageAnnotation<T extends IImageAnnotation = IImageAnnotation> extends Annotation<T> {
    description: null | string;
    fileName: null | string;
    contentType: string;
    imageAttachmentId: string;
    isSignature: boolean;
    xfdfAppearanceStream: null | string;
    xfdfAppearanceStreamOriginalPageRotation: null | number;
    static defaultValues: IObject$1;
    static readableName: string;
}

interface IInkAnnotation extends AnnotationProperties {
    lines: List<List<DrawingPoint>>;
    lineWidth: number | null;
    strokeColor: Color$2 | null;
    backgroundColor: Color$2 | null;
    isDrawnNaturally: boolean;
    isSignature: boolean;
}
declare class InkAnnotation<T extends IInkAnnotation = IInkAnnotation> extends Annotation<T> {
    lines: List<List<DrawingPoint>>;
    lineWidth: number;
    strokeColor: Color$2 | null;
    backgroundColor: Color$2 | null;
    isDrawnNaturally: boolean;
    isSignature: boolean;
    static defaultValues: IObject$1;
    static readableName: string;
}

declare const MeasurementPrecision: {
    readonly WHOLE: "whole";
    readonly ONE: "oneDp";
    readonly TWO: "twoDp";
    readonly THREE: "threeDp";
    readonly FOUR: "fourDp";
    readonly HALVES: "1/2";
    readonly QUARTERS: "1/4";
    readonly EIGHTHS: "1/8";
    readonly SIXTEENTHS: "1/16";
};
type IMeasurementPrecision = (typeof MeasurementPrecision)[keyof typeof MeasurementPrecision];

declare const MeasurementScaleUnitFrom: {
    readonly INCHES: "in";
    readonly MILLIMETERS: "mm";
    readonly CENTIMETERS: "cm";
    readonly POINTS: "pt";
};
type IMeasurementScaleUnitFrom = (typeof MeasurementScaleUnitFrom)[keyof typeof MeasurementScaleUnitFrom];

declare const MeasurementScaleUnitTo: {
    readonly INCHES: "in";
    readonly MILLIMETERS: "mm";
    readonly CENTIMETERS: "cm";
    readonly POINTS: "pt";
    readonly FEET: "ft";
    readonly METERS: "m";
    readonly YARDS: "yd";
    readonly KILOMETERS: "km";
    readonly MILES: "mi";
};
type IMeasurementScaleUnitTo = (typeof MeasurementScaleUnitTo)[keyof typeof MeasurementScaleUnitTo];

interface IMeasurementScale {
    unitFrom: IMeasurementScaleUnitFrom;
    unitTo: IMeasurementScaleUnitTo;
    fromValue: number;
    toValue: number;
}
declare const MeasurementScale_base: Record$1.Factory<IMeasurementScale>;
declare class MeasurementScale extends MeasurementScale_base {
}

interface IShapeAnnotation extends AnnotationProperties {
    strokeDashArray: [number, number] | null;
    strokeWidth: number | null;
    strokeColor: Color$2 | null;
    fillColor: Color$2 | null;
    measurementScale: MeasurementScale | null;
    measurementPrecision: IMeasurementPrecision | null;
}
declare abstract class ShapeAnnotation<T extends IShapeAnnotation = IShapeAnnotation> extends Annotation<T> {
    strokeDashArray: null | [number, number];
    strokeWidth: number;
    strokeColor: null | Color$2;
    fillColor: null | Color$2;
    measurementPrecision: null | IMeasurementPrecision;
    measurementScale: null | MeasurementScale;
    static readableName: string;
    static defaultValues: IObject$1;
    isMeasurement(): boolean;
    getMeasurementDetails: () => {
        value: number;
        label: string;
    };
}

declare const LineCap: {
    readonly square: "square";
    readonly circle: "circle";
    readonly diamond: "diamond";
    readonly openArrow: "openArrow";
    readonly closedArrow: "closedArrow";
    readonly butt: "butt";
    readonly reverseOpenArrow: "reverseOpenArrow";
    readonly reverseClosedArrow: "reverseClosedArrow";
    readonly slash: "slash";
};
type ILineCap = (typeof LineCap)[keyof typeof LineCap];
type LineCapsType = {
    start?: ILineCap | null;
    end?: ILineCap | null;
};

interface ILineAnnotation extends IShapeAnnotation {
    startPoint: Point | null;
    endPoint: Point | null;
    lineCaps: LineCapsType | null;
    points: List<Point> | null;
}
declare class LineAnnotation<T extends ILineAnnotation = ILineAnnotation> extends ShapeAnnotation<T> {
    startPoint: Point;
    endPoint: Point;
    lineCaps: LineCapsType | null;
    points: List<Point> | null;
    static defaultValues: IObject$1;
    static readableName: string;
}

interface IRectangleAnnotation extends IShapeAnnotation {
    cloudyBorderIntensity?: number | null;
    cloudyBorderInset?: Inset | null;
}
declare class RectangleAnnotation<T extends IRectangleAnnotation = IRectangleAnnotation> extends ShapeAnnotation<T> {
    cloudyBorderIntensity: null | number;
    cloudyBorderInset: null | Inset;
    measurementBBox: null | Rect$2;
    static defaultValues: IObject$1;
    static readableName: string;
    constructor(options?: Partial<T>);
}

interface IEllipseAnnotation extends IShapeAnnotation {
    cloudyBorderIntensity?: number | null;
    cloudyBorderInset?: Inset | null;
}
declare class EllipseAnnotation<T extends IEllipseAnnotation = IEllipseAnnotation> extends ShapeAnnotation<T> {
    cloudyBorderIntensity: null | number;
    cloudyBorderInset: null | Inset;
    measurementBBox: null | Rect$2;
    static defaultValues: IObject$1;
    static readableName: string;
    constructor(options?: Partial<T>);
}

interface IPolygonAnnotation extends IShapeAnnotation {
    points: List<Point> | null;
    cloudyBorderIntensity: number | null;
}
declare class PolygonAnnotation<T extends IPolygonAnnotation = IPolygonAnnotation> extends ShapeAnnotation<T> {
    points: List<Point>;
    cloudyBorderIntensity: null | number;
    static defaultValues: IObject$1;
    static readableName: string;
}

interface IPolyLineAnnotation extends IShapeAnnotation {
    points: List<Point> | null;
    lineCaps: LineCapsType | null;
}
declare class PolylineAnnotation<T extends IPolyLineAnnotation = IPolyLineAnnotation> extends ShapeAnnotation<T> {
    points: List<Point>;
    lineCaps: null | LineCapsType;
    static defaultValues: IObject$1;
    static readableName: string;
}

declare const BorderStyle: {
    readonly solid: "solid";
    readonly dashed: "dashed";
    readonly beveled: "beveled";
    readonly inset: "inset";
    readonly underline: "underline";
};
type IBorderStyle = (typeof BorderStyle)[keyof typeof BorderStyle];

interface ILinkAnnotation extends AnnotationProperties {
    action: Action | null;
    borderColor: Color$2 | null;
    borderStyle: IBorderStyle | null;
    borderWidth: number | null;
}
declare class LinkAnnotation<T extends ILinkAnnotation = ILinkAnnotation> extends Annotation<T> {
    action: Action;
    borderColor: null | Color$2;
    borderStyle: null | IBorderStyle;
    borderWidth: null | number;
    static readableName: string;
    static defaultValues: IObject$1;
}

declare const NoteIcon: {
    readonly COMMENT: "COMMENT";
    readonly RIGHT_POINTER: "RIGHT_POINTER";
    readonly RIGHT_ARROW: "RIGHT_ARROW";
    readonly CHECK: "CHECK";
    readonly CIRCLE: "CIRCLE";
    readonly CROSS: "CROSS";
    readonly INSERT: "INSERT";
    readonly NEW_PARAGRAPH: "NEW_PARAGRAPH";
    readonly NOTE: "NOTE";
    readonly PARAGRAPH: "PARAGRAPH";
    readonly HELP: "HELP";
    readonly STAR: "STAR";
    readonly KEY: "KEY";
};
type INoteIcon = (typeof NoteIcon)[keyof typeof NoteIcon];

interface INoteAnnotation extends AnnotationProperties {
    text: {
        format: 'plain';
        value: string;
    };
    icon: string | INoteIcon;
    color: Color$2;
}
declare class NoteAnnotation<T extends INoteAnnotation = INoteAnnotation> extends Annotation<T> {
    text: {
        format: 'plain';
        value: string;
    };
    icon: INoteIcon;
    color: Color$2;
    static isEditable: boolean;
    static readableName: string;
    static defaultValues: IObject$1;
}

interface ISquiggleAnnotation extends ITextMarkupAnnotation {
    color: Color$2;
}
declare class SquiggleAnnotation<T extends ISquiggleAnnotation = ISquiggleAnnotation> extends TextMarkupAnnotation<T> {
    static className: string;
    static readableName: string;
    static defaultValues: IObject$1;
}

interface IStrikeOutAnnotation extends ITextMarkupAnnotation {
    color: Color$2;
}
declare class StrikeOutAnnotation<T extends IStrikeOutAnnotation = IStrikeOutAnnotation> extends TextMarkupAnnotation<T> {
    static className: string;
    static readableName: string;
    static defaultValues: IObject$1;
}

type ICallout = {
    start: Point | null;
    knee: Point | null;
    end: Point | null;
    cap: ILineCap | null;
    innerRectInset: Inset | null;
};
declare class Callout extends InheritableImmutableRecord<ICallout> {
    start: Point | null;
    knee: Point | null;
    end: Point | null;
    cap: ILineCap | null;
    innerRectInset: Inset | null;
    static defaultValues: {
        start: null;
        knee: null;
        end: null;
        cap: null;
        innerRectInset: null;
    };
}

interface ITextAnnotation extends AnnotationProperties {
    text: {
        format: 'plain' | 'xhtml';
        value: string | null;
    };
    fontColor: Color$2 | null;
    backgroundColor: Color$2 | null;
    font: string;
    fontSize: number | null;
    isBold: boolean | null;
    isItalic: boolean | null;
    horizontalAlign: 'left' | 'center' | 'right';
    verticalAlign: 'top' | 'center' | 'bottom';
    callout: Callout | null;
    borderStyle: IBorderStyle | null;
    borderWidth: number | null;
    borderColor: Color$2 | null;
    isFitting: boolean;
    lineHeightFactor: number | null;
}
declare class TextAnnotation<T extends ITextAnnotation = ITextAnnotation> extends Annotation<T> {
    text: {
        format: 'plain' | 'xhtml';
        value: string;
    };
    fontColor: null | Color$2;
    backgroundColor: null | Color$2;
    font: string;
    fontSize: number;
    isBold: boolean;
    isItalic: boolean;
    horizontalAlign: 'left' | 'center' | 'right';
    verticalAlign: 'top' | 'center' | 'bottom';
    isFitting: boolean;
    callout: null | Callout;
    borderStyle: null | IBorderStyle;
    borderWidth: null | number;
    borderColor: Color$2 | null;
    lineHeightFactor: null | number;
    static defaultValues: IObject$1;
    static readonly isEditable = true;
    static readonly readableName = "Text";
    static readonly fontSizePresets: readonly number[];
}

interface IUnderlineAnnotation extends ITextMarkupAnnotation {
    color: Color$2;
}
declare class UnderlineAnnotation<T extends IUnderlineAnnotation = IUnderlineAnnotation> extends TextMarkupAnnotation<T> {
    static className: string;
    static readableName: string;
    static defaultValues: IObject$1;
}

declare class UnknownAnnotation extends Annotation {
}

type FontSize = 'auto' | number;
type WidgetActionTriggerEventType = ActionTriggerEventType | 'onFocus' | 'onBlur';
type WidgetAnnotationAdditionalActionsType = {
    onFocus?: JavaScriptAction;
    onBlur?: JavaScriptAction;
    onChange?: JavaScriptAction;
    onFormat?: JavaScriptAction;
    onInput?: JavaScriptAction;
    onPointerDown?: Action;
    onPointerUp?: Action;
    onPointerEnter?: Action;
    onPointerLeave?: Action;
};
interface IWidgetAnnotation extends AnnotationProperties {
    formFieldName: string | null;
    borderColor: Color$2 | null;
    borderStyle: IBorderStyle | null;
    borderDashArray: number[] | null;
    borderWidth: number | null;
    backgroundColor: Color$2 | null;
    fontSize: FontSize | null;
    font: string | null;
    fontColor: Color$2 | null;
    isBold: boolean | null;
    isItalic: boolean | null;
    horizontalAlign: 'left' | 'center' | 'right' | null;
    verticalAlign: 'top' | 'center' | 'bottom' | null;
    additionalActions: WidgetAnnotationAdditionalActionsType | null;
    rotation: number;
    lineHeightFactor: number | null;
    buttonIconUpdatedAt: number | null;
}
declare class WidgetAnnotation<T extends IWidgetAnnotation = IWidgetAnnotation> extends Annotation<T> {
    formFieldName: string;
    borderColor: null | Color$2;
    borderStyle: null | IBorderStyle;
    borderDashArray: null | number[];
    borderWidth: null | number;
    backgroundColor: null | Color$2;
    fontSize: null | FontSize;
    font: null | string;
    fontColor: null | Color$2;
    isBold: boolean;
    isItalic: boolean;
    horizontalAlign: 'left' | 'center' | 'right' | null;
    verticalAlign: 'top' | 'center' | 'bottom' | null;
    additionalActions: null | WidgetAnnotationAdditionalActionsType;
    rotation: number;
    lineHeightFactor: null | number;
    action: null | Action;
    buttonIconUpdatedAt: null | number;
    static defaultValues: IObject$1;
    static readableName: string;
}

declare class CommentMarkerAnnotation extends Annotation {
    static readableName: string;
}

interface IRedactionAnnotation extends ITextMarkupAnnotation {
    color: Color$2;
    fillColor: null | Color$2;
    overlayText: null | string;
    repeatOverlayText: null | boolean;
    outlineColor: null | Color$2;
}
declare class RedactionAnnotation<T extends IRedactionAnnotation = IRedactionAnnotation> extends TextMarkupAnnotation<T> {
    fillColor: null | Color$2;
    overlayText: null | string;
    repeatOverlayText: null | boolean;
    outlineColor: null | Color$2;
    color: Color$2;
    static readableName: string;
    static defaultValues: IObject$1;
}

interface IMediaAnnotation extends AnnotationProperties {
    description: null | string;
    fileName: null | string;
    contentType: string | null;
    mediaAttachmentId: string | null;
}
declare class MediaAnnotation<T extends IMediaAnnotation = IMediaAnnotation> extends Annotation<T> {
    description: null | string;
    fileName: null | string;
    contentType: string | null;
    mediaAttachmentId: string | null;
    static defaultValues: IObject$1;
    static readableName: string;
}

type SignatureInfo = {
    type: 'pspdfkit/signature-info';
    signatureType?: SignatureTypeType | null | undefined;
    signerName: string | null | undefined;
    creationDate: Date | null | undefined;
    signatureReason: string | null | undefined;
    signatureLocation: string | null | undefined;
    documentIntegrityStatus: DocumentIntegrityStatusType;
    certificateChainValidationStatus: CertificateChainValidationStatusType;
    signatureValidationStatus: SignatureValidationStatusType;
    isTrusted: boolean;
    isSelfSigned: boolean;
    isExpired: boolean;
    documentModifiedSinceSignature: boolean;
    signatureFormFQN: string;
    PAdESSignatureLevel?: PAdESLevelType | null;
    validFrom: string | null | undefined;
    validUntil: string | null | undefined;
    timestampInfo: {
        type: 'pspdfkit/timestamp-info';
        signerName: string | null | undefined;
    };
    ltv: boolean;
};
declare const DocumentIntegrityStatus: {
    readonly ok: "ok";
    readonly tampered_document: "tampered_document";
    readonly failed_to_retrieve_signature_contents: "failed_to_retrieve_signature_contents";
    readonly failed_to_retrieve_byterange: "failed_to_retrieve_byterange";
    readonly failed_to_compute_digest: "failed_to_compute_digest";
    readonly failed_retrieve_signing_certificate: "failed_retrieve_signing_certificate";
    readonly failed_retrieve_public_key: "failed_retrieve_public_key";
    readonly failed_encryption_padding: "failed_encryption_padding";
    readonly tampered_or_invalid_timestamp: "tampered_or_invalid_timestamp";
    readonly general_failure: "general_failure";
};
type DocumentIntegrityStatusType = (typeof DocumentIntegrityStatus)[keyof typeof DocumentIntegrityStatus];
declare const CertificateChainValidationStatus: {
    readonly ok: "ok";
    readonly ok_but_self_signed: "ok_but_self_signed";
    readonly ok_but_could_not_check_revocation: "ok_but_could_not_check_revocation";
    readonly untrusted: "untrusted";
    readonly expired: "expired";
    readonly not_yet_valid: "not_yet_valid";
    readonly invalid: "invalid";
    readonly revoked: "revoked";
    readonly failed_to_retrieve_signature_contents: "failed_to_retrieve_signature_contents";
    readonly general_validation_problem: "general_validation_problem";
};
type CertificateChainValidationStatusType = (typeof CertificateChainValidationStatus)[keyof typeof CertificateChainValidationStatus];
declare const SignatureValidationStatus: {
    readonly valid: "valid";
    readonly warning: "warning";
    readonly error: "error";
};
type SignatureValidationStatusType = (typeof SignatureValidationStatus)[keyof typeof SignatureValidationStatus];
declare const SignatureType: {
    CMS: string;
    CAdES: string;
};
type SignatureTypeType = (typeof SignatureType)[keyof typeof SignatureType];
declare const PAdESLevel: {
    readonly b_b: "b-b";
    readonly b_t: "b-t";
    readonly b_lt: "b-lt";
};
type PAdESLevelType = (typeof PAdESLevel)[keyof typeof PAdESLevel];

type SignaturesInfo = {
    status: DocumentValidationStatusType;
    checkedAt: Date;
    signatures?: Array<SignatureInfo>;
    documentModifiedSinceSignature?: boolean;
};
declare const DocumentValidationStatus: {
    valid: string;
    warning: string;
    error: string;
    not_signed: string;
};
type DocumentValidationStatusType = keyof typeof DocumentValidationStatus;

type Glyph = {
    c: string;
    rect: Rect$2;
};

type IAnnotationJSON = Omit<BaseAnnotationJSON, 'id' | 'group' | 'permissions'>;
declare class AnnotationSerializer {
    static VERSION: number;
    annotation: AnnotationsUnion;
    constructor(annotation: AnnotationsUnion);
    toJSON(): Omit<BaseAnnotationJSON, 'type'>;
    static fromJSON(id: ID | null, json: IAnnotationJSON, options?: ICollaboratorPermissionsOptions): {
        group?: string | null | undefined;
        canSetGroup?: boolean | undefined;
        isEditable?: boolean | undefined;
        isDeletable?: boolean | undefined;
        blendMode?: IBlendMode | undefined;
        id: string | null;
        name: string | null;
        subject: string | null;
        pdfObjectId: number | null;
        pageIndex: number;
        opacity: number;
        boundingBox: Rect$2;
        noPrint: boolean;
        noZoom: boolean;
        noRotate: boolean;
        noView: boolean;
        hidden: boolean;
        locked: boolean;
        lockedContents: boolean;
        readOnly: boolean;
        action: Action | null | undefined;
        note: string | null;
        createdAt: Date;
        updatedAt: Date;
        creatorName: string | null;
        customData: Record<string, unknown> | null;
        isCommentThreadRoot: boolean;
        isAnonymous: boolean;
    };
    static blendModeObjectForAnnotation(json: IAnnotationJSON): {
        blendMode: IBlendMode;
    } | null;
    serializeFlags(): ("noView" | "noPrint" | "locked" | "lockedContents" | "readOnly" | "hidden" | "noZoom" | "noRotate")[] | null;
}

declare class InkAnnotationSerializer extends AnnotationSerializer {
    annotation: InkAnnotation;
    constructor(annotation: InkAnnotation);
    toJSON(): InkAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<InkAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): InkAnnotation;
    _linesToJSON(): {
        points: [number, number][][];
        intensities: number[][];
    };
    static _JSONToLines(linesJSON: {
        points: Array<Array<[number, number]>>;
        intensities: Array<Array<number>>;
    }): List<List<DrawingPoint>>;
}

declare abstract class ShapeAnnotationSerializer extends AnnotationSerializer {
    annotation: ShapeAnnotationsUnion;
    toJSON(): ShapeAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<ShapeAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): {
        strokeWidth: number | null;
        strokeColor: Color$2 | null;
        fillColor: Color$2 | null;
        strokeDashArray: [number, number] | null | undefined;
        measurementPrecision: IMeasurementPrecision | null | undefined;
        measurementScale: MeasurementScale | null;
        group?: string | null | undefined;
        canSetGroup?: boolean | undefined;
        isEditable?: boolean | undefined;
        isDeletable?: boolean | undefined;
        blendMode?: IBlendMode | undefined;
        id: string | null;
        name: string | null;
        subject: string | null;
        pdfObjectId: number | null;
        pageIndex: number;
        opacity: number;
        boundingBox: Rect$2;
        noPrint: boolean;
        noZoom: boolean;
        noRotate: boolean;
        noView: boolean;
        hidden: boolean;
        locked: boolean;
        lockedContents: boolean;
        readOnly: boolean;
        action: Action | null | undefined;
        note: string | null;
        createdAt: Date;
        updatedAt: Date;
        creatorName: string | null;
        customData: Record<string, unknown> | null;
        isCommentThreadRoot: boolean;
        isAnonymous: boolean;
    };
    _pointsToJSON(): Array<[number, number]>;
    static _JSONToPoints(pointsJSON: Array<[number, number]>): List<Point>;
    static _JSONLinesToPoints(linesJSON: {
        points: Array<Array<[number, number]>>;
        intensities: Array<Array<number>>;
    }): List<Point>;
}
type MeasurementScaleJSON = {
    unitFrom: IMeasurementScaleUnitFrom;
    unitTo: IMeasurementScaleUnitTo;
    from: number;
    to: number;
};

declare class LineAnnotationSerializer extends ShapeAnnotationSerializer {
    annotation: LineAnnotation;
    toJSON(): LineAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<LineAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): LineAnnotation;
}

declare class RectangleAnnotationSerializer extends ShapeAnnotationSerializer {
    annotation: RectangleAnnotation;
    toJSON(): RectangleAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<RectangleAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): RectangleAnnotation;
}

declare class EllipseAnnotationSerializer extends ShapeAnnotationSerializer {
    annotation: EllipseAnnotation;
    toJSON(): EllipseAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<EllipseAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): EllipseAnnotation;
}

declare class PolygonAnnotationSerializer extends ShapeAnnotationSerializer {
    annotation: PolygonAnnotation;
    toJSON(): PolygonAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<PolygonAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: IObject$1): PolygonAnnotation;
}

declare class PolylineAnnotationSerializer extends ShapeAnnotationSerializer {
    annotation: PolylineAnnotation;
    toJSON(): PolylineAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<PolylineAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): PolylineAnnotation;
}

declare class LinkAnnotationSerializer extends AnnotationSerializer {
    annotation: LinkAnnotation;
    constructor(annotation: LinkAnnotation);
    toJSON(): LinkAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<LinkAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): LinkAnnotation;
}

declare abstract class BaseTextMarkupSerializer extends AnnotationSerializer {
    annotation: RedactionAnnotation | TextMarkupAnnotation;
    constructor(annotation: RedactionAnnotation | TextMarkupAnnotation);
    toJSON(): BaseTextMarkupAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<BaseTextMarkupAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): {
        rects: List<Rect$2>;
        group?: string | null | undefined;
        canSetGroup?: boolean | undefined;
        isEditable?: boolean | undefined;
        isDeletable?: boolean | undefined;
        blendMode?: IBlendMode | undefined;
        id: string | null;
        name: string | null;
        subject: string | null;
        pdfObjectId: number | null;
        pageIndex: number;
        opacity: number;
        boundingBox: Rect$2;
        noPrint: boolean;
        noZoom: boolean;
        noRotate: boolean;
        noView: boolean;
        hidden: boolean;
        locked: boolean;
        lockedContents: boolean;
        readOnly: boolean;
        action: Action | null | undefined;
        note: string | null;
        createdAt: Date;
        updatedAt: Date;
        creatorName: string | null;
        customData: Record<string, unknown> | null;
        isCommentThreadRoot: boolean;
        isAnonymous: boolean;
    };
}

declare class TextMarkupAnnotationSerializer extends BaseTextMarkupSerializer {
    annotation: TextMarkupAnnotationsUnion;
    constructor(annotation: TextMarkupAnnotationsUnion);
    toJSON(): TextMarkupAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<TextMarkupAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): TextMarkupAnnotationsUnion;
    typeForAnnotation(): "pspdfkit/markup/highlight" | "pspdfkit/markup/squiggly" | "pspdfkit/markup/strikeout" | "pspdfkit/markup/underline" | "pspdfkit/markup/redaction";
}

declare class RedactionAnnotationSerializer extends BaseTextMarkupSerializer {
    annotation: RedactionAnnotation;
    constructor(annotation: RedactionAnnotation);
    toJSON(): RedactionAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<RedactionAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): RedactionAnnotation;
}

declare class TextAnnotationSerializer extends AnnotationSerializer {
    annotation: TextAnnotation;
    constructor(annotation: TextAnnotation);
    toJSON(): TextAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<TextAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): TextAnnotation;
    _calloutToJSON(): {
        start: [number, number];
        knee: [number, number] | null;
        end: [number, number];
        cap: ILineCap | null;
        innerRectInset: InsetJSON | null;
    } | null;
    static _JSONToCallout(calloutJSON: TextAnnotationJSON['callout']): Callout | null | undefined;
}

declare class NoteAnnotationSerializer extends AnnotationSerializer {
    annotation: NoteAnnotation;
    constructor(annotation: NoteAnnotation);
    toJSON(): NoteAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<NoteAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): NoteAnnotation;
}

declare class ImageAnnotationSerializer extends AnnotationSerializer {
    annotation: ImageAnnotation;
    constructor(annotation: ImageAnnotation);
    toJSON(): ImageAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<ImageAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): ImageAnnotation;
}

declare class StampAnnotationSerializer extends AnnotationSerializer {
    annotation: StampAnnotation;
    constructor(annotation: StampAnnotation);
    toJSON(): StampAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<StampAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): StampAnnotation;
}

declare class WidgetAnnotationSerializer extends AnnotationSerializer {
    annotation: WidgetAnnotation;
    constructor(annotation: WidgetAnnotation);
    toJSON(): WidgetAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<WidgetAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): WidgetAnnotation;
}

type InstantID = string;
declare function generateInstantId(): InstantID;

declare class CommentMarkerAnnotationSerializer extends AnnotationSerializer {
    annotation: CommentMarkerAnnotation;
    constructor(annotation: CommentMarkerAnnotation);
    toJSON(): CommentMarkerAnnotationJSON;
    static fromJSON(id: InstantID | null, json: Omit<CommentMarkerAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): CommentMarkerAnnotation;
}

declare class UnknownAnnotationSerializer extends AnnotationSerializer {
    annotation: UnknownAnnotation;
    constructor(annotation: UnknownAnnotation);
    toJSON(): UnknownAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<UnknownAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): UnknownAnnotation;
}

declare class MediaAnnotationSerializer extends AnnotationSerializer {
    annotation: MediaAnnotation;
    constructor(annotation: MediaAnnotation);
    toJSON(): MediaAnnotationJSON;
    static fromJSON(id: ID | null, json: Omit<MediaAnnotationJSON, 'id' | 'group' | 'permissions'>, options?: ICollaboratorPermissionsOptions): MediaAnnotation;
}

type AnnotationSerializerTypeMap = {
    'pspdfkit/ink': {
        serializer: InkAnnotationSerializer;
        annotation: InkAnnotation;
        json: InkAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<InkAnnotationJSON>;
    };
    'pspdfkit/shape/line': {
        serializer: LineAnnotationSerializer;
        annotation: LineAnnotation;
        json: LineAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<LineAnnotationJSON>;
    };
    'pspdfkit/shape/rectangle': {
        serializer: RectangleAnnotationSerializer;
        annotation: RectangleAnnotation;
        json: RectangleAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<RectangleAnnotationJSON>;
    };
    'pspdfkit/shape/ellipse': {
        serializer: EllipseAnnotationSerializer;
        annotation: EllipseAnnotation;
        json: EllipseAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<EllipseAnnotationJSON>;
    };
    'pspdfkit/shape/polygon': {
        serializer: PolygonAnnotationSerializer;
        annotation: PolygonAnnotation;
        json: PolygonAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<PolygonAnnotationJSON>;
    };
    'pspdfkit/shape/polyline': {
        serializer: PolylineAnnotationSerializer;
        annotation: PolylineAnnotation;
        json: PolylineAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<PolylineAnnotationJSON>;
    };
    'pspdfkit/link': {
        serializer: LinkAnnotationSerializer;
        annotation: LinkAnnotation;
        json: LinkAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<LinkAnnotationJSON>;
    };
    'pspdfkit/markup/highlight': {
        serializer: TextMarkupAnnotationSerializer;
        annotation: HighlightAnnotation;
        json: TextMarkupAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<TextMarkupAnnotationJSON>;
    };
    'pspdfkit/markup/squiggly': {
        serializer: TextMarkupAnnotationSerializer;
        annotation: SquiggleAnnotation;
        json: TextMarkupAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<TextMarkupAnnotationJSON>;
    };
    'pspdfkit/markup/strikeout': {
        serializer: TextMarkupAnnotationSerializer;
        annotation: StrikeOutAnnotation;
        json: TextMarkupAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<TextMarkupAnnotationJSON>;
    };
    'pspdfkit/markup/underline': {
        serializer: TextMarkupAnnotationSerializer;
        annotation: UnderlineAnnotation;
        json: TextMarkupAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<TextMarkupAnnotationJSON>;
    };
    'pspdfkit/markup/redaction': {
        serializer: RedactionAnnotationSerializer;
        annotation: RedactionAnnotation;
        json: RedactionAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<RedactionAnnotationJSON>;
    };
    'pspdfkit/text': {
        serializer: TextAnnotationSerializer;
        annotation: TextAnnotation;
        json: TextAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<TextAnnotationJSON>;
    };
    'pspdfkit/note': {
        serializer: NoteAnnotationSerializer;
        annotation: NoteAnnotation;
        json: NoteAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<NoteAnnotationJSON>;
    };
    'pspdfkit/image': {
        serializer: ImageAnnotationSerializer;
        annotation: ImageAnnotation;
        json: ImageAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<ImageAnnotationJSON>;
    };
    'pspdfkit/stamp': {
        serializer: StampAnnotationSerializer;
        annotation: StampAnnotation;
        json: StampAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<StampAnnotationJSON, 'color'>;
    };
    'pspdfkit/widget': {
        serializer: WidgetAnnotationSerializer;
        annotation: WidgetAnnotation;
        json: WidgetAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<WidgetAnnotationJSON>;
    };
    'pspdfkit/comment-marker': {
        serializer: CommentMarkerAnnotationSerializer;
        annotation: CommentMarkerAnnotation;
        json: CommentMarkerAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<CommentMarkerAnnotationJSON>;
    };
    'pspdfkit/unknown': {
        serializer: UnknownAnnotationSerializer;
        annotation: UnknownAnnotation;
        json: UnknownAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<UnknownAnnotationJSON>;
    };
    'pspdfkit/media': {
        serializer: MediaAnnotationSerializer;
        annotation: MediaAnnotation;
        json: MediaAnnotationJSON;
        jsonForBackend: AnnotationBackendJSON<MediaAnnotationJSON>;
    };
};
type GetTypeFromAnnotationJSON<T extends {
    type: keyof AnnotationSerializerTypeMap;
}> = T extends {
    type: infer U;
} ? U : never;
type AnnotationJSONToAnnotation<T extends {
    type: keyof AnnotationSerializerTypeMap;
}> = AnnotationSerializerTypeMap[GetTypeFromAnnotationJSON<T>]['annotation'];
type Intersection<T, U> = T extends U ? T : never;
type BackendRequiredKeys = 'id' | 'v' | 'pageIndex' | 'type' | 'bbox';
type AnnotationBackendJSON<K extends BaseAnnotationJSON = AnnotationJSONUnion, R extends string = never> = {
    [P in keyof K]?: NonNullable<K[P]>;
} & {
    [P in Intersection<keyof K, BackendRequiredKeys | R>]-?: Exclude<NonNullable<K[P]>, undefined>;
};
type AnnotationsUnion = {
    [K in keyof AnnotationSerializerTypeMap]: AnnotationSerializerTypeMap[K]['annotation'];
}[keyof AnnotationSerializerTypeMap];
type AnnotationsUnionClass = {
    [K in keyof AnnotationSerializerTypeMap]: Class<AnnotationSerializerTypeMap[K]['annotation']>;
}[keyof AnnotationSerializerTypeMap];
type ShapeAnnotationsUnion = PolylineAnnotation | PolygonAnnotation | LineAnnotation | RectangleAnnotation | EllipseAnnotation;
type AnnotationsBackendJSONUnion = {
    [K in keyof AnnotationSerializerTypeMap]: AnnotationSerializerTypeMap[K]['jsonForBackend'];
}[keyof AnnotationSerializerTypeMap];
type TextMarkupAnnotationsUnion = HighlightAnnotation | UnderlineAnnotation | StrikeOutAnnotation | SquiggleAnnotation | RedactionAnnotation;

type CommentJSON = {
    id?: string | null;
    type: 'pspdfkit/comment';
    v: 2;
    rootId: string | number | null;
    pageIndex: number | null;
    pdfObjectId?: number | null;
    creatorName?: string | null;
    name?: string | null;
    createdAt: string | Date;
    updatedAt: string | Date;
    text: {
        value: string | null;
        format: 'xhtml' | 'plain';
    };
    customData?: {
        [key: string]: unknown;
    } | null;
    isAnonymous?: boolean | null;
} & ICollaboratorPermissionsOptions;

type CommentProps$1 = {
    id: InstantID | null;
    rootId: InstantID | null;
    pageIndex: null | number;
    pdfObjectId: number | null;
    creatorName: string | null;
    createdAt: Date;
    updatedAt: Date;
    text: {
        format: 'plain' | 'xhtml';
        value: string | null;
    };
    customData: Record<string, unknown> | null;
    group?: string | null;
    isEditable?: boolean;
    isDeletable?: boolean;
    canSetGroup?: boolean;
    isAnonymous?: boolean | null;
};
declare const Comment_base: Record$1.Factory<CommentProps$1>;
declare class Comment$1 extends Comment_base {
    static toSerializableObject: (comment: Comment$1) => CommentJSON;
    static fromSerializableObject: (comment: CommentJSON) => Comment$1;
    getMentionedUserIds(): Set$1<string>;
}
type MentionableUser$1 = {
    id: string;
    name: string;
    avatarUrl?: string;
    displayName: string;
    description?: string;
};

type IGroup = string | null | undefined;
type IPermissions = {
    edit: boolean;
    delete: boolean;
    setGroup: boolean;
    fill?: boolean;
    reply?: boolean;
};

type ICollaboratorPermissionsOptions = {
    group?: IGroup;
    permissions?: IPermissions;
};

type SerializedAdditionalActionsType = {
    [key in ActionTriggerEventType | FormFieldEventTriggerType | FormFieldInputEventTriggerType | WidgetActionTriggerEventType]?: {
        type: string;
        [key: string]: unknown;
    };
};

type FontSubstitution = {
    pattern: string;
    target: string;
};

declare const ComparisonOperationType: {
    readonly TEXT: "text";
};
type IComparisonOperationType = (typeof ComparisonOperationType)[keyof typeof ComparisonOperationType];

declare global {
    interface SymbolConstructor {
        readonly observable: symbol;
    }
}

type OnCommentCreationStartCallback = (comment: Comment$1) => Comment$1;

interface ITextLine {
    id: number | null;
    pageIndex: number | null;
    boundingBox: Rect$2;
    contents: string;
    hints: Hints | null;
}
declare const TextLine_base: Record$1.Factory<ITextLine>;
declare class TextLine extends TextLine_base {
}

type CustomOverlayItemID = string;
interface ICustomOverlayItem {
    disableAutoZoom?: boolean;
    id: CustomOverlayItemID | null;
    node: Node | null;
    noRotate?: boolean;
    pageIndex: number;
    position: Point;
    onAppear?: null | ((...args: Array<any>) => any);
    onDisappear?: null | ((...args: Array<any>) => any);
}
declare const CustomOverlayItem_base: Record$1.Factory<ICustomOverlayItem>;
declare class CustomOverlayItem extends CustomOverlayItem_base {
    disableAutoZoom: boolean;
    id: CustomOverlayItemID;
    node: Node;
    noRotate: boolean;
    pageIndex: number;
    position: Point;
    onAppear?: ((...args: Array<any>) => any) | null;
    onDisappear?: ((...args: Array<any>) => any) | null;
    constructor(args: ICustomOverlayItem);
}

declare class PageInfo {
    index: number;
    label: string;
    height: number;
    width: number;
    rotation: number;
    rawPdfBoxes: RawPdfBoxes;
}

declare const ViewportPadding_base: Record$1.Factory<{
    horizontal: number;
    vertical: number;
}>;
declare class ViewportPadding extends ViewportPadding_base {
}

declare const LayoutMode: {
    readonly SINGLE: "SINGLE";
    readonly DOUBLE: "DOUBLE";
    readonly AUTO: "AUTO";
};
type ILayoutMode = (typeof LayoutMode)[keyof typeof LayoutMode];

declare const ScrollMode: {
    readonly CONTINUOUS: "CONTINUOUS";
    readonly PER_SPREAD: "PER_SPREAD";
    readonly DISABLED: "DISABLED";
};
type IScrollMode = (typeof ScrollMode)[keyof typeof ScrollMode];

declare const SidebarPlacement: {
    readonly START: "START";
    readonly END: "END";
};
type ISidebarPlacement = (typeof SidebarPlacement)[keyof typeof SidebarPlacement];

declare const ZoomMode: {
    readonly AUTO: "AUTO";
    readonly FIT_TO_WIDTH: "FIT_TO_WIDTH";
    readonly FIT_TO_VIEWPORT: "FIT_TO_VIEWPORT";
    readonly CUSTOM: "CUSTOM";
};
type IZoomMode = (typeof ZoomMode)[keyof typeof ZoomMode];

declare const ShowSignatureValidationStatusMode: {
    readonly IF_SIGNED: "IF_SIGNED";
    readonly HAS_WARNINGS: "HAS_WARNINGS";
    readonly HAS_ERRORS: "HAS_ERRORS";
    readonly NEVER: "NEVER";
};
type IShowSignatureValidationStatusMode = (typeof ShowSignatureValidationStatusMode)[keyof typeof ShowSignatureValidationStatusMode];

declare const InteractionMode: {
    readonly TEXT_HIGHLIGHTER: "TEXT_HIGHLIGHTER";
    readonly INK: "INK";
    readonly INK_SIGNATURE: "INK_SIGNATURE";
    readonly SIGNATURE: "SIGNATURE";
    readonly STAMP_PICKER: "STAMP_PICKER";
    readonly STAMP_CUSTOM: "STAMP_CUSTOM";
    readonly SHAPE_LINE: "SHAPE_LINE";
    readonly SHAPE_RECTANGLE: "SHAPE_RECTANGLE";
    readonly SHAPE_ELLIPSE: "SHAPE_ELLIPSE";
    readonly SHAPE_POLYGON: "SHAPE_POLYGON";
    readonly SHAPE_POLYLINE: "SHAPE_POLYLINE";
    readonly INK_ERASER: "INK_ERASER";
    readonly NOTE: "NOTE";
    readonly COMMENT_MARKER: "COMMENT_MARKER";
    readonly TEXT: "TEXT";
    readonly CALLOUT: "CALLOUT";
    readonly PAN: "PAN";
    readonly SEARCH: "SEARCH";
    readonly DOCUMENT_EDITOR: "DOCUMENT_EDITOR";
    readonly MARQUEE_ZOOM: "MARQUEE_ZOOM";
    readonly REDACT_TEXT_HIGHLIGHTER: "REDACT_TEXT_HIGHLIGHTER";
    readonly REDACT_SHAPE_RECTANGLE: "REDACT_SHAPE_RECTANGLE";
    readonly DOCUMENT_CROP: "DOCUMENT_CROP";
    readonly BUTTON_WIDGET: "BUTTON_WIDGET";
    readonly TEXT_WIDGET: "TEXT_WIDGET";
    readonly RADIO_BUTTON_WIDGET: "RADIO_BUTTON_WIDGET";
    readonly CHECKBOX_WIDGET: "CHECKBOX_WIDGET";
    readonly COMBO_BOX_WIDGET: "COMBO_BOX_WIDGET";
    readonly LIST_BOX_WIDGET: "LIST_BOX_WIDGET";
    readonly SIGNATURE_WIDGET: "SIGNATURE_WIDGET";
    readonly DATE_WIDGET: "DATE_WIDGET";
    readonly FORM_CREATOR: "FORM_CREATOR";
    readonly LINK: "LINK";
    readonly DISTANCE: "DISTANCE";
    readonly PERIMETER: "PERIMETER";
    readonly RECTANGLE_AREA: "RECTANGLE_AREA";
    readonly ELLIPSE_AREA: "ELLIPSE_AREA";
    readonly POLYGON_AREA: "POLYGON_AREA";
    readonly CONTENT_EDITOR: "CONTENT_EDITOR";
    readonly MULTI_ANNOTATIONS_SELECTION: "MULTI_ANNOTATIONS_SELECTION";
    readonly MEASUREMENT: "MEASUREMENT";
    readonly MEASUREMENT_SETTINGS: "MEASUREMENT_SETTINGS";
};
type IInteractionMode = (typeof InteractionMode)[keyof typeof InteractionMode];

declare const SidebarMode: {
    readonly ANNOTATIONS: "ANNOTATIONS";
    readonly BOOKMARKS: "BOOKMARKS";
    readonly DOCUMENT_OUTLINE: "DOCUMENT_OUTLINE";
    readonly THUMBNAILS: "THUMBNAILS";
    readonly SIGNATURES: "SIGNATURES";
    readonly LAYERS: "LAYERS";
    readonly CUSTOM: "CUSTOM";
};
type ISidebarMode = (typeof SidebarMode)[keyof typeof SidebarMode];

declare const Alignment: {
    readonly START: "START";
    readonly END: "END";
};
type IAlignment = (typeof Alignment)[keyof typeof Alignment];

type SidebarOptions<T extends AnnotationsSidebarOptions | LayersSidebarOptions> = T extends AnnotationsSidebarOptions ? {
    [SidebarMode.ANNOTATIONS]: AnnotationsSidebarOptions;
} : {
    [SidebarMode.LAYERS]: LayersSidebarOptions;
};
type AnnotationsSidebarOptions = {
    includeContent: Array<AnnotationsUnionClass | Class<Comment$1>>;
};
type LayersSidebarOptions = {
    lockedLayers: number[];
    iconsAlignment: IAlignment;
};

type Rotation$1 = 0 | 90 | 180 | 270;
interface IViewState {
    allowPrinting: boolean;
    allowExport: boolean;
    currentPageIndex: number;
    instance: Instance | null;
    interactionMode: IInteractionMode | null;
    keepFirstSpreadAsSinglePage: boolean;
    layoutMode: ILayoutMode;
    pageSpacing: number;
    pagesRotation: Rotation$1;
    readOnly: boolean;
    scrollMode: IScrollMode;
    showAnnotations: boolean;
    showComments: boolean;
    showAnnotationNotes: boolean;
    showToolbar: boolean;
    enableAnnotationToolbar: boolean;
    sidebarMode: ISidebarMode | null | undefined;
    sidebarOptions: SidebarOptions<AnnotationsSidebarOptions | LayersSidebarOptions>;
    sidebarPlacement: ISidebarPlacement;
    spreadSpacing: number;
    viewportPadding: ViewportPadding;
    zoom: IZoomMode | number;
    zoomStep: number;
    formDesignMode: boolean;
    showSignatureValidationStatus: IShowSignatureValidationStatusMode;
    previewRedactionMode: boolean;
    canScrollWhileDrawing: boolean;
    keepSelectedTool: boolean;
    resolvedLayoutMode: ILayoutMode;
    sidebarWidth: number;
    disablePointSnapping: boolean;
    enableAlwaysScrollToZoom: boolean;
    forceRenderWidgetsInAnnotationsOrder: boolean;
    prerenderedPageSpreads: number | null;
}
declare const ViewState_base: Record$1.Factory<IViewState>;
declare class ViewState extends ViewState_base {
    zoomIn(): ViewState;
    zoomOut(): ViewState;
    rotateLeft(): ViewState;
    rotateRight(): ViewState;
    goToNextPage(): ViewState;
    goToPreviousPage(): ViewState;
}

declare class InstantClient {
    clientId: string;
    userId: string | null | undefined;
}

declare const PrintMode: {
    readonly DOM: "DOM";
    readonly EXPORT_PDF: "EXPORT_PDF";
};
type IPrintMode = (typeof PrintMode)[keyof typeof PrintMode];

declare function toJSON(bookmark: Bookmark): BookmarkJSON;

type ID$1 = string;
type BookmarkProps = {
    id: ID$1 | null;
    pdfBookmarkId: ID$1 | null;
    name: string | null;
    sortKey: number | null;
    action: Action | null;
};
declare const Bookmark_base: Record$1.Factory<BookmarkProps>;
declare class Bookmark extends Bookmark_base {
    id: ID$1;
    action: Action;
    static toSerializableObject: typeof toJSON;
    static fromSerializableObject: (bookmark: BookmarkJSON) => Bookmark;
}

declare const allowedToolbarTypes: ("link" | "ellipse" | "image" | "line" | "polygon" | "polyline" | "text" | "spacer" | "distance" | "note" | "comment" | "zoom-in" | "zoom-out" | "callout" | "search" | "arrow" | "highlighter" | "undo" | "redo" | "custom" | "measure" | "print" | "rectangle" | "ink" | "stamp" | "cloudy-rectangle" | "dashed-rectangle" | "cloudy-ellipse" | "dashed-ellipse" | "cloudy-polygon" | "dashed-polygon" | "text-highlighter" | "perimeter" | "ellipse-area" | "rectangle-area" | "polygon-area" | "multi-annotations-selection" | "annotate" | "responsive-group" | "layout-config" | "marquee-zoom" | "redact-text-highlighter" | "redact-rectangle" | "document-comparison" | "form-creator" | "content-editor" | "document-assistant" | "sidebar-thumbnails" | "sidebar-document-outline" | "sidebar-annotations" | "sidebar-bookmarks" | "sidebar-signatures" | "sidebar-layers" | "pager" | "pan" | "zoom-mode" | "ink-eraser" | "signature" | "document-editor" | "document-crop" | "export-pdf" | "debug")[];

type ToolItemType = 'custom';
type ToolItem = {
    type: ToolItemType;
    node?: Node;
    id?: string;
    title?: string;
    className?: string;
    icon?: string;
    onPress?: IFunction;
    selected?: boolean;
    disabled?: boolean;
};

type AnnotationPreset$1 = Record<string, any>;
type AnnotationPresetID$1 = string;

type ToolbarItemType = ToolItemType | (typeof allowedToolbarTypes)[number];
type ToolbarItem = Omit<ToolItem, 'type'> & {
    type: ToolbarItemType;
    mediaQueries?: string[];
    responsiveGroup?: string;
    dropdownGroup?: string;
    preset?: AnnotationPresetID$1;
    onKeyPress?: (...args: Array<any>) => any;
};

type OutlineElementProps = {
    children: List<OutlineElement>;
    title: string;
    color: Color$2 | null;
    isBold: boolean;
    isItalic: boolean;
    isExpanded: boolean;
    action: Action | null;
};
declare const OutlineElement_base: Record$1.Factory<OutlineElementProps>;
declare class OutlineElement extends OutlineElement_base {
}

type BaseFormFieldJSON = {
    v: 1;
    pdfObjectId?: number | null;
    annotationIds: Array<string>;
    name: string;
    label: string;
    flags?: FormFieldFlags;
    id: string;
    additionalActions?: SerializedAdditionalActionsType;
    group?: IGroup;
    permissions?: IPermissions;
};
type ChoiceFormFieldJSON = BaseFormFieldJSON & {
    type: 'pspdfkit/form-field/listbox' | 'pspdfkit/form-field/combobox';
    options: Array<FormOptionJSON>;
    multiSelect: boolean;
    commitOnChange: boolean;
    defaultValues: Array<string>;
};
type ListBoxFormFieldJSON = ChoiceFormFieldJSON & {
    type: 'pspdfkit/form-field/listbox';
};
type DoNotSpellCheckPropertyPair = XOR<Record<'doNotSpellCheck', boolean>, Record<'doNotSpellcheck', boolean>>;
type ComboBoxFormFieldJSON = ChoiceFormFieldJSON & {
    type: 'pspdfkit/form-field/combobox';
    edit: boolean;
} & DoNotSpellCheckPropertyPair;
type CheckBoxFormFieldJSON = BaseFormFieldJSON & {
    type: 'pspdfkit/form-field/checkbox';
    options: Array<FormOptionJSON>;
    defaultValues: Array<string>;
};
type RadioButtonFormFieldJSON = BaseFormFieldJSON & {
    type: 'pspdfkit/form-field/radio';
    options: Array<FormOptionJSON>;
    noToggleToOff: boolean;
    radiosInUnison: boolean;
    defaultValue: string;
};
type TextFormFieldJSON = BaseFormFieldJSON & {
    type: 'pspdfkit/form-field/text';
    password: boolean;
    maxLength?: number | null;
    doNotScroll: boolean;
    multiLine: boolean;
    defaultValue: string;
    comb: boolean;
} & DoNotSpellCheckPropertyPair;
type ButtonFormFieldJSON = BaseFormFieldJSON & {
    type: 'pspdfkit/form-field/button';
    buttonLabel: string | null;
};
type SignatureFormFieldJSON = BaseFormFieldJSON & {
    type: 'pspdfkit/form-field/signature';
};
type FormFieldJSON = ListBoxFormFieldJSON | ComboBoxFormFieldJSON | RadioButtonFormFieldJSON | CheckBoxFormFieldJSON | TextFormFieldJSON | ButtonFormFieldJSON | SignatureFormFieldJSON;

type SerializedJSON = {
    skippedPdfObjectIds?: number[];
    annotations?: AnnotationJSONUnion[];
    formFields?: FormFieldJSON[];
    skippedPdfFormFieldIds?: number[];
    formFieldValues?: Record<string, any>[];
    comments?: Record<string, any>[];
    skippedComments?: number[];
    attachments?: Record<string, {
        binary: string;
        contentType: string;
    }>;
    skippedPdfBookmarkIds?: string[];
    bookmarks?: BookmarkJSON[];
};
type InstantJSON = SerializedJSON & {
    format: 'https://pspdfkit.com/instant-json/v1';
    pdfId?: {
        permanent: string;
        changing: string;
    };
};

declare const SearchResult_base: Record$1.Factory<{
    pageIndex: number | null;
    previewText: string;
    locationInPreview: number | null;
    lengthInPreview: number | null;
    rectsOnPage: List<Rect$2>;
    isAnnotation: boolean | null;
    annotationRect?: Rect$2 | null | undefined;
}>;
declare class SearchResult extends SearchResult_base {
}

declare const SearchState_base: Record$1.Factory<ISearchState>;
declare class SearchState extends SearchState_base {
}
interface ISearchState {
    isFocused: boolean;
    isLoading: boolean;
    term: string;
    focusedResultIndex: number;
    results: List<SearchResult>;
    minSearchQueryLength: number;
}

type IsEditableAnnotationCallback = (annotation: AnnotationsUnion) => boolean;

type Rotation = 0 | 90 | 180 | 270;
type AddPageConfiguration = {
    backgroundColor: Color$2;
    pageWidth: number;
    pageHeight: number;
    rotateBy: Rotation;
    insets?: Rect$2;
};
type OperationAttachment = string | File | Blob;
type min = number;
type max = number;
type Range$1 = [min, max];
type ImportPageIndex = Array<number | Range$1>;
type DocumentMetadata = {
    title?: string;
    author?: string;
};
type NonSerializableDocumentOperationsCommon = {
    type: 'removePages';
    pageIndexes: Array<number>;
} | {
    type: 'duplicatePages';
    pageIndexes: Array<number>;
} | {
    type: 'movePages';
    pageIndexes: Array<number>;
    afterPageIndex: number;
} | {
    type: 'movePages';
    pageIndexes: Array<number>;
    beforePageIndex: number;
} | {
    type: 'rotatePages';
    pageIndexes: Array<number>;
    rotateBy: Rotation;
} | {
    type: 'keepPages';
    pageIndexes: Array<number>;
} | {
    type: 'importDocument';
    afterPageIndex: number;
    treatImportedDocumentAsOnePage?: boolean;
    document: OperationAttachment;
    importedPageIndexes?: ImportPageIndex;
} | {
    type: 'importDocument';
    beforePageIndex: number;
    treatImportedDocumentAsOnePage?: boolean;
    document: OperationAttachment;
    importedPageIndexes?: ImportPageIndex;
} | {
    type: 'setPageLabel';
    pageIndexes?: Array<number>;
    pageLabel?: string;
} | {
    type: 'performOcr';
    pageIndexes?: Array<number> | 'all';
    language: string;
} | {
    type: 'applyRedactions';
} | {
    type: 'updateMetadata';
    metadata: DocumentMetadata;
};
type NonSerializableDocumentOperations = {
    type: 'applyInstantJson';
    instantJson: Record<string, any>;
    dataFilePath?: OperationAttachment;
} | {
    type: 'applyXfdf';
    xfdf: string;
    ignorePageRotation?: boolean;
    dataFilePath?: OperationAttachment;
} | {
    type: 'flattenAnnotations';
    pageIndexes?: Array<number>;
    annotationIds?: string[];
    noteAnnotationBackgroundColor?: Color$2;
    noteAnnotationOpacity?: number;
};
type DocumentOperation = (AddPageConfiguration & {
    type: 'addPage';
    afterPageIndex: number;
}) | (AddPageConfiguration & {
    type: 'addPage';
    beforePageIndex: number;
}) | {
    type: 'cropPages';
    pageIndexes?: Array<number>;
    cropBox: Rect$2;
} | {
    type: 'addPageMargins';
    pageIndexes?: Array<number>;
    margins: Inset;
} | NonSerializableDocumentOperationsCommon | NonSerializableDocumentOperations;

type RendererConfiguration = {
    node: Node;
    append?: boolean | null;
    noZoom?: boolean | null;
    onDisappear?: ((arg0: Node | null) => void) | null;
};

type CustomRenderers = {
    Annotation?: (arg0: {
        annotation: AnnotationsUnion;
    }) => RendererConfiguration | null | undefined;
    CommentAvatar?: (arg0: {
        comment: Comment$1;
    }) => RendererConfiguration | null | undefined;
};

type IsEditableCommentCallback = (comment: Comment$1) => boolean;

type OcspResponse = {
    serialNumber: string;
    body: ArrayBuffer;
};
type SignatureCallbackResponsePkcs7 = {
    pkcs7: ArrayBuffer;
    ocspResponses?: OcspResponse[];
};
type SignatureCallbackResponseRaw = {
    certificates: ArrayBuffer[] | string[];
    signedData: ArrayBuffer;
    timestampResponse?: ArrayBuffer;
    ocspResponses?: OcspResponse[];
};
type TwoStepSignatureCallback = (arg0: {
    hash: string;
    fileContents: ArrayBuffer | null;
    dataToBeSigned: ArrayBuffer;
}) => Promise<ArrayBuffer | SignatureCallbackResponsePkcs7 | SignatureCallbackResponseRaw>;

type SignatureMetadata = {
    signerName?: string;
    signatureReason?: string;
    signatureLocation?: string;
};

type SignaturePosition = {
    pageIndex: number;
    boundingBox: Rect$2;
};

declare const SignatureAppearanceMode: {
    readonly signatureOnly: "signatureOnly";
    readonly signatureAndDescription: "signatureAndDescription";
    readonly descriptionOnly: "descriptionOnly";
};
type ISignatureAppearanceMode = (typeof SignatureAppearanceMode)[keyof typeof SignatureAppearanceMode];

type SignatureAppearance = {
    mode?: ISignatureAppearanceMode;
    showSigner?: boolean;
    showSignDate?: boolean;
    showReason?: boolean;
    showLocation?: boolean;
    showWatermark?: boolean;
    showDateTimezone?: boolean;
    watermarkImage?: Blob | File;
    graphicImage?: Blob | File;
};

type SignatureContainerType = 'raw' | 'pkcs7';
type TimestampType = {
    url: string;
    username?: string;
    password?: string;
};
type SigningData = {
    certificates?: ArrayBuffer[] | string[];
    signatureType: SignatureTypeType;
    privateKey?: string;
    signatureContainer?: SignatureContainerType;
    timestamp?: TimestampType;
    ltv?: boolean;
};
type SignaturePreparationData = {
    placeholderSize?: number;
    flatten?: boolean;
    formFieldName?: string;
    position?: SignaturePosition;
    appearance?: SignatureAppearance;
};
type SignatureCreationData = SignaturePreparationData & {
    signatureMetadata?: SignatureMetadata;
} & {
    signingData?: SigningData;
};

type SigningServiceData = {
    signingToken: string;
};

type RedactionAnnotationPreset = {
    fillColor?: Color$2;
    overlayText?: string;
    repeatOverlayText?: boolean;
    color?: Color$2;
    outlineColor?: Color$2;
    creatorName?: string;
};

type Change = AnnotationsUnion | Bookmark | FormField | FormFieldValue | Comment$1;

type DocumentEditorUIHandler = {
    setOperations: (callback: (stagedDocumentOperations: List<DocumentOperation | List<DocumentOperation>>) => List<DocumentOperation | List<DocumentOperation>>, clearPagesSelection?: boolean) => void | Promise<void>;
    getSelectedPageIndexes: () => number[];
};

type BuiltInDocumentEditorFooterItem = 'cancel' | 'spacer' | 'save-as' | 'save' | 'selected-pages' | 'loading-indicator';
type BasicDocumentEditorFooterItem = {
    type: BuiltInDocumentEditorFooterItem | 'custom';
    node?: Node;
    className?: string;
    id?: string;
    onPress?: (e: MouseEvent, documentEditorUIHandler?: DocumentEditorUIHandler, id?: string) => void;
};
type DocumentEditorFooterItem = Omit<BasicDocumentEditorFooterItem, 'type' | 'onPress'> & ({
    type: BuiltInDocumentEditorFooterItem;
    onPress?: (e: Event) => void;
} | {
    type: 'custom';
    onPress?: (e: Event, documentEditorUIHandler: DocumentEditorUIHandler, id?: string) => void;
});

type BuiltInDocumentEditorToolbarItem = 'add' | 'remove' | 'duplicate' | 'rotate-left' | 'rotate-right' | 'move' | 'move-left' | 'move-right' | 'import-document' | 'spacer' | 'undo' | 'redo' | 'select-all' | 'select-none';
type DocumentEditorBuiltinToolbarItem = Omit<ToolItem, 'type' | 'onPress'> & {
    type: BuiltInDocumentEditorToolbarItem;
    onPress?: (e: Event) => void;
};
type DocumentEditorToolbarItem = Omit<ToolItem, 'type' | 'onPress'> & (DocumentEditorBuiltinToolbarItem | {
    type: 'custom';
    onPress?: (e: Event, documentEditorUIHandler: DocumentEditorUIHandler, id?: string) => void;
});

type AnnotationsResizeEvent = {
    annotation: AnnotationsUnion;
    isShiftPressed: boolean;
    resizeAnchor: ResizeAnchor;
};
type ResizeAnchor = 'TOP' | 'BOTTOM' | 'LEFT' | 'RIGHT' | 'TOP_LEFT' | 'TOP_RIGHT' | 'BOTTOM_RIGHT' | 'BOTTOM_LEFT';

type AnnotationResizeStartCallbackConfiguration = {
    maintainAspectRatio?: boolean;
    minWidth?: number | undefined;
    minHeight?: number | undefined;
    maxWidth?: number | undefined;
    maxHeight?: number | undefined;
};
type AnnotationResizeStartCallback = (event: AnnotationsResizeEvent) => AnnotationResizeStartCallbackConfiguration;

declare const UIElement: {
    readonly Sidebar: "Sidebar";
};
type IUIElement = (typeof UIElement)[keyof typeof UIElement];

type RendererProps = {
    containerNode: Node;
    items?: List<any> | null;
};
type ItemRendererProps = {
    itemContainerNode: Node;
    item: any;
};
type ItemCustomRenderer = (itemRendererProps: ItemRendererProps) => void;
type UIRendererConfiguration = {
    node: Node;
    onRenderItem?: ItemCustomRenderer;
};
type Renderer$1 = (rendererProps: RendererProps) => UIRendererConfiguration;
type CustomUISidebarConfiguration = Partial<{
    [K in ISidebarMode]: Renderer$1;
}>;
type CustomUIElementConfiguration = CustomUISidebarConfiguration;
type CustomUI = Partial<Record<IUIElement, CustomUIElementConfiguration>>;

declare const SearchType: {
    readonly TEXT: "text";
    readonly PRESET: "preset";
    readonly REGEX: "regex";
};
type ISearchType = (typeof SearchType)[keyof typeof SearchType];

declare const SearchPattern: {
    readonly CREDIT_CARD_NUMBER: "credit_card_number";
    readonly DATE: "date";
    readonly TIME: "time";
    readonly EMAIL_ADDRESS: "email_address";
    readonly INTERNATIONAL_PHONE_NUMBER: "international_phone_number";
    readonly IP_V4: "ipv4";
    readonly IP_V6: "ipv6";
    readonly MAC_ADDRESS: "mac_address";
    readonly NORTH_AMERICAN_PHONE_NUMBER: "north_american_phone_number";
    readonly SOCIAL_SECURITY_NUMBER: "social_security_number";
    readonly URL: "url";
    readonly US_ZIP_CODE: "us_zip_code";
    readonly VIN: "vin";
};
type ISearchPattern = (typeof SearchPattern)[keyof typeof SearchPattern];

type AnnotationsPressEvent = {
    annotation: AnnotationsUnion;
    nativeEvent: Event;
    preventDefault?: () => void;
    selected: boolean;
};

declare enum AnnotationsWillChangeReason {
    DRAW_START = "DRAW_START",
    DRAW_END = "DRAW_END",
    TEXT_EDIT_START = "TEXT_EDIT_START",
    TEXT_EDIT_END = "TEXT_EDIT_END",
    SELECT_START = "SELECT_START",
    SELECT_END = "SELECT_END",
    MOVE_START = "MOVE_START",
    MOVE_END = "MOVE_END",
    RESIZE_START = "RESIZE_START",
    RESIZE_END = "RESIZE_END",
    ROTATE_START = "ROTATE_START",
    ROTATE_END = "ROTATE_END",
    DELETE_START = "DELETE_START",
    DELETE_END = "DELETE_END",
    PROPERTY_CHANGE = "PROPERTY_CHANGE"
}

type PagePressEvent = {
    pageIndex: number;
    point: Point;
    nativeEvent: Event;
};

type AnnotationPresetsUpdateEvent = {
    preventDefault: () => boolean;
    currentPreset: AnnotationPresetID;
    currentPresetProperties: AnnotationPreset;
    newPresetProperties: AnnotationPreset;
};

type AnnotationsFocusEvent = {
    annotation: AnnotationsUnion;
    nativeEvent: FocusEvent;
};
type AnnotationsBlurEvent = {
    annotation: AnnotationsUnion;
    nativeEvent: FocusEvent;
};

type SaveStateChangeEvent = {
    hasUnsavedChanges: boolean;
};

type SearchTermChangeEvent = {
    term: string;
    preventDefault: () => void;
};

type TextLinePressEvent = {
    textLine: TextLine;
    point: Point;
    nativeEvent: Event;
};

type CropAreaChangeStartEvent = {
    cropBox: Rect$2;
    pageIndex: number;
};
type CropAreaChangeStopEvent = {
    cropBox: Rect$2;
    pageIndex: number;
};

type AnnotationsTransformEvent = {
    annotation: AnnotationsUnion;
};

type AnnotationsCopyEvent = {
    annotation: AnnotationsUnion;
};

type AnnotationsCutEvent = {
    annotation: AnnotationsUnion;
};

type AnnotationsDuplicateEvent = {
    annotations: AnnotationsUnion[];
    formFields?: FormField[];
    originalAnnotations: AnnotationsUnion[];
    originalFormFields?: Map$1<string, FormField>;
};

type AnnotationsPasteEvent = AnnotationsDuplicateEvent & {
    previousAction: 'COPY' | 'CUT';
};

interface ITextSelection$1 {
    startNestedContentBlockId: string | null;
    startTextLineId: number | null;
    startPageIndex: number | null;
    startNode: Text | null;
    startOffset: number | null;
    endNestedContentBlockId: string | null;
    endTextLineId: number | null;
    endPageIndex: number | null;
    endNode: Text | null;
    endOffset: number | null;
    getText: (() => Promise<string>) | null;
    getSelectedTextLines: (() => Promise<List<TextLine>>) | null;
    getBoundingClientRect: (() => Promise<Rect$2 | null>) | null;
    getSelectedRectsPerPage: (() => Promise<List<{
        pageIndex: number;
        rects: List<Rect$2>;
    }>>) | null;
}
declare const PublicTextSelection_base: Record$1.Factory<ITextSelection$1>;
declare class PublicTextSelection extends PublicTextSelection_base {
    startTextLineId: number;
    startNestedContentBlockId: string;
    startPageIndex: number;
    startNode: Text;
    startOffset: number;
    endTextLineId: number;
    endNestedContentBlockId: string;
    endPageIndex: number;
    endNode: Text;
    endOffset: number;
    getText: () => Promise<string>;
    getSelectedTextLines: () => Promise<List<TextLine>>;
    getBoundingClientRect: () => Promise<Rect$2 | null>;
    getSelectedRectsPerPage: () => Promise<List<{
        pageIndex: number;
        rects: List<Rect$2>;
    }>>;
}

interface AnnotationNoteProps extends INoteAnnotation {
    parentAnnotation: AnnotationsUnion | null;
    position: Point;
    notePosition?: Point;
}
declare class AnnotationNote<T extends AnnotationNoteProps = AnnotationNoteProps> extends NoteAnnotation<T> {
    parentAnnotation?: AnnotationsUnion;
    position: Point;
    notePosition?: Point;
    static defaultValues: IObject$1;
}

type AnnotationNotePressEvent = {
    preventDefault: () => boolean;
    annotationNote?: AnnotationNote | null;
};

type AnnotationNoteHoverEvent = {
    preventDefault: () => boolean;
    annotationNote?: AnnotationNote | null;
};

declare const DocumentComparisonSourceType: {
    readonly USE_OPEN_DOCUMENT: "USE_OPEN_DOCUMENT";
    readonly USE_FILE_DIALOG: "USE_FILE_DIALOG";
};
type IDocumentComparisonSourceType = (typeof DocumentComparisonSourceType)[keyof typeof DocumentComparisonSourceType];

type DocumentComparisonSource = {
    source: IDocumentComparisonSourceType | string | ArrayBuffer | Promise<string | ArrayBuffer>;
    pageIndex?: number;
};

type DocumentComparisonStrokeColors = {
    documentA?: Color$2;
    documentB?: Color$2;
};

type DocumentComparisonConfiguration = {
    documentA: DocumentComparisonSource;
    documentB: DocumentComparisonSource;
    strokeColors?: DocumentComparisonStrokeColors;
    blendMode?: IBlendMode;
    autoCompare: boolean;
};

type DocumentComparisonUIStartEvent = DocumentComparisonConfiguration;

type CommentsMentionEvent = {
    comment: Comment$1;
    modifications: List<{
        userId: string;
        action: 'ADDED' | 'REMOVED';
    }>;
};

type Signature = InkAnnotation | ImageAnnotation;
interface HistoryEvent<T> {
    action: T;
    before: AnnotationsUnion;
    after: AnnotationsUnion;
}
interface EventMap {
    'viewState.change': (viewState: ViewState, previousViewState: ViewState) => void;
    'viewState.currentPageIndex.change': (pageIndex: number) => void;
    'viewState.zoom.change': (zoom: number) => void;
    'annotationPresets.update': (event: AnnotationPresetsUpdateEvent) => void;
    'annotations.blur': (event: AnnotationsBlurEvent) => void;
    'annotations.change': () => void;
    'annotations.create': (annotations: List<AnnotationsUnion>) => void;
    'annotations.delete': (annotations: List<AnnotationsUnion>) => void;
    'annotations.didSave': () => void;
    'annotations.focus': (event: AnnotationsFocusEvent) => void;
    'annotations.load': (annotations: List<AnnotationsUnion>) => void;
    'annotations.press': (event: AnnotationsPressEvent) => void;
    'annotations.update': (annotations: List<AnnotationsUnion>) => void;
    'annotations.willChange': (event: {
        reason: AnnotationsWillChangeReason;
        annotations: List<AnnotationsUnion>;
    }) => void;
    'annotations.willSave': () => void;
    'annotationSelection.change': (annotation?: AnnotationsUnion) => void;
    'annotations.transform': (event: AnnotationsTransformEvent) => void;
    'annotations.copy': (event: AnnotationsCopyEvent) => void;
    'annotations.cut': (event: AnnotationsCutEvent) => void;
    'annotations.paste': (event: AnnotationsPasteEvent) => void;
    'annotations.duplicate': (event: AnnotationsDuplicateEvent) => void;
    'bookmarks.change': () => void;
    'bookmarks.create': (bookmarks: List<Bookmark>) => void;
    'bookmarks.update': (bookmarks: List<Bookmark>) => void;
    'bookmarks.delete': (bookmarks: List<Bookmark>) => void;
    'bookmarks.load': (bookmarks: List<Bookmark>) => void;
    'bookmarks.didSave': () => void;
    'bookmarks.willSave': () => void;
    'comments.change': () => void;
    'comments.create': (comments: List<Comment$1>) => void;
    'comments.delete': (comments: List<Comment$1>) => void;
    'comments.update': (comments: List<Comment$1>) => void;
    'comments.load': () => void;
    'comments.willSave': () => void;
    'comments.didSave': () => void;
    'instant.connectedClients.change': (clients: Map$1<string, InstantClient>) => void;
    'document.change': (operations: DocumentOperation[]) => void;
    'document.saveStateChange': (event: SaveStateChangeEvent) => void;
    'formFieldValues.update': (formFields: List<FormField>) => void;
    'formFieldValues.willSave': () => void;
    'formFieldValues.didSave': (res: {
        response: Response;
        error: Error;
    }) => void;
    'forms.willSubmit': (event: {
        preventDefault: () => void;
    }) => void;
    'forms.didSubmit': (event: {
        preventDefault: () => void;
    }) => void;
    'formFields.change': () => void;
    'formFields.create': (formFields: List<FormField>) => void;
    'formFields.delete': (formFields: List<FormField>) => void;
    'formFields.didSave': () => void;
    'formFields.load': (formFields: List<FormField>) => void;
    'formFields.update': (formFields: List<FormField>) => void;
    'formFields.willSave': () => void;
    'search.stateChange': (searchState: SearchState) => void;
    'search.termChange': (event: SearchTermChangeEvent) => void;
    'storedSignatures.change': () => void;
    'storedSignatures.create': (signature: Signature) => void;
    'storedSignatures.delete': (signature: Signature) => void;
    'storedSignatures.update': (signatures: List<Signature>) => void;
    'textLine.press': (event: TextLinePressEvent) => void;
    'textSelection.change': (selection: PublicTextSelection | null) => void;
    'history.change': (event: HistoryEvent<'undo' | 'redo'>) => void;
    'history.willChange': (event: {
        type: 'create' | 'update' | 'delete';
        annotation: Annotation;
        preventDefault: () => void;
    }) => void;
    'history.clear': () => void;
    'history.redo': (event: HistoryEvent<'redo'>) => void;
    'history.undo': (event: HistoryEvent<'undo'>) => void;
    'page.press': (event: PagePressEvent) => void;
    'inkSignatures.create': (signature: Signature) => void;
    'inkSignatures.delete': (signature: Signature) => void;
    'inkSignatures.update': (signatures: Signature[]) => void;
    'inkSignatures.change': () => void;
    'cropArea.changeStart': (opts: CropAreaChangeStartEvent) => void;
    'cropArea.changeStop': (opts: CropAreaChangeStopEvent) => void;
    'documentComparisonUI.start': (opts: DocumentComparisonUIStartEvent) => void;
    'documentComparisonUI.end': () => void;
    'annotationNote.press': (event: AnnotationNotePressEvent) => void;
    'annotationNote.hover': (event: AnnotationNoteHoverEvent) => void;
    'comments.mention': (event: CommentsMentionEvent) => void;
}

interface IEmbeddedFile {
    id: ID;
    attachmentId: string;
    description: null | string;
    fileName: null | string;
    fileSize: null | number;
    updatedAt: null | Date;
}
declare const EmbeddedFile_base: Record$1.Factory<IEmbeddedFile>;
declare class EmbeddedFile extends EmbeddedFile_base {
}

type IAnnotationToolbarType = 'stroke-color' | 'fill-color' | 'background-color' | 'opacity' | 'line-width' | 'blend-mode' | 'spacer' | 'delete' | 'annotation-note' | 'border-color' | 'border-width' | 'border-style' | 'color' | 'linecaps-dasharray' | 'line-style' | 'font' | 'overlay-text' | 'outline-color' | 'apply-redactions' | 'measurementType' | 'measurementScale' | 'back';
type BuiltInAnnotationToolbarItem = {
    type: IAnnotationToolbarType;
};
type Shared = Omit<ToolItem, 'selected' | 'type'> & {
    onPress?: (nativeEvent: MouseEvent, id?: string) => void;
    iconClassName?: string;
    onIconPress?: (nativeEvent: MouseEvent, id?: string) => void;
};
type AnnotationToolbarItem = (Omit<Shared, 'node'> & {
    type: IAnnotationToolbarType;
}) | (Omit<Shared, 'icon'> & {
    id: string;
    type: 'custom';
    icon?: string | Node;
    node?: Node;
});

type AnnotationToolbarItemsCallback = (annotation: AnnotationsUnion, options: {
    defaultAnnotationToolbarItems: BuiltInAnnotationToolbarItem[];
    hasDesktopLayout: boolean;
}) => AnnotationToolbarItem[];

type OnWidgetAnnotationCreationStartCallback = (annotation: WidgetAnnotation, formField: FormField) => {
    annotation?: WidgetAnnotation;
    formField?: FormField;
};

interface ITextRange {
    startNode: Text | null;
    startOffset: number | null;
    endNode: Text | null;
    endOffset: number | null;
}
declare const TextRange_base: Record$1.Factory<ITextRange>;
declare class TextRange extends TextRange_base {
    startNode: Text;
    startOffset: number;
    endNode: Text;
    endOffset: number;
    startAndEndIds(): {
        startTextLineId: number;
        endTextLineId: number;
        startNestedContentBlockId: string;
        endNestedContentBlockId: string;
        startPageIndex: number;
        endPageIndex: number;
    } | null;
}

interface ITextSelection {
    textRange: TextRange | null;
    startTextLineId: number | null;
    endTextLineId: number | null;
    startNestedContentBlockId: string | null;
    endNestedContentBlockId: string | null;
    startPageIndex: number | null;
    endPageIndex: number | null;
}
declare const TextSelection_base: Record$1.Factory<ITextSelection>;
declare class TextSelection extends TextSelection_base {
}

declare const builtInItems: readonly ["highlight", "strikeout", "underline", "squiggle", "redact-text-highlighter", "comment", "document-assistant"];
type InlineToolbarType = (typeof builtInItems)[number];
type InlineTextSelectionToolbarItem = Omit<ToolItem, 'type'> & {
    type: InlineToolbarType | 'custom';
};
type InlineTextSelectionToolbarItemsCallback = (options: {
    defaultItems: InlineTextSelectionToolbarItem[];
    hasDesktopLayout: boolean;
}, selection: TextSelection) => InlineTextSelectionToolbarItem[];

type MeasurementValueConfiguration = {
    name?: string;
    scale: IMeasurementScale;
    precision: IMeasurementPrecision;
    selected?: boolean;
};
type MeasurementValueConfigurationCallback = (configuration: MeasurementValueConfiguration[]) => MeasurementValueConfiguration[];

declare const DocumentPermissionsEnum: {
    readonly annotationsAndForms: "annotationsAndForms";
    readonly assemble: "assemble";
    readonly extract: "extract";
    readonly extractAccessibility: "extractAccessibility";
    readonly fillForms: "fillForms";
    readonly modification: "modification";
    readonly printHighQuality: "printHighQuality";
    readonly printing: "printing";
};
type IDocumentPermissions = (typeof DocumentPermissionsEnum)[keyof typeof DocumentPermissionsEnum];

type OCGLayer = {
    name: string;
    ocgId: number;
    radioGroup?: number;
};
type OCGCollection = {
    name?: string;
    ocgId?: number;
    layers: OCGLayer[];
};
type OCG = OCGLayer | OCGCollection;
type OCGLayers = OCG[];
type OCGLayersVisibilityState = {
    visibleLayerIds: number[];
};

type ViewStateSetter = (currentState: ViewState) => ViewState;
type ToolbarItemsSetter = (currentState: ToolbarItem[]) => ToolbarItem[];
type StoredSignaturesSetter = (annotations: List<InkAnnotation | ImageAnnotation>) => List<InkAnnotation | ImageAnnotation>;
type SearchStateSetter = (currentState: SearchState) => SearchState;
type AnnotationPresetsSetter = (currentState: Record<string, AnnotationPreset$1>) => Record<string, AnnotationPreset$1>;
type StampAnnotationTemplatesSetter = (currentState: Array<StampAnnotation | ImageAnnotation>) => Array<StampAnnotation | ImageAnnotation>;
type SetDocumentEditorFooterFunction = (currentState: DocumentEditorFooterItem[]) => DocumentEditorFooterItem[];
type SetDocumentEditorToolbarFunction = (currentState: DocumentEditorToolbarItem[]) => DocumentEditorToolbarItem[];
declare class Instance {
    totalPageCount: number;
    pageInfoForIndex: (pageIndex: number) => PageInfo | null | undefined;
    textLinesForPageIndex: (pageIndex: number) => Promise<List<TextLine>>;
    getMarkupAnnotationText: (annotation: TextMarkupAnnotationsUnion) => Promise<string>;
    getTextFromRects: (pageIndex: number, rects: List<Rect$2>) => Promise<string>;
    getDocumentPermissions: () => Promise<Record<IDocumentPermissions, boolean>>;
    currentZoomLevel: number;
    maximumZoomLevel: number;
    minimumZoomLevel: number;
    zoomStep: number;
    disablePointSnapping: boolean;
    connectedClients: Map$1<string, InstantClient>;
    addEventListener: <K extends keyof EventMap>(action: K, listener: EventMap[K]) => void;
    removeEventListener: <K extends keyof EventMap>(action: K, listener: EventMap[K]) => void;
    jumpToRect: (pageIndex: number, rect: Rect$2) => void;
    jumpAndZoomToRect: (pageIndex: number, rect: Rect$2) => void;
    transformContentClientToPageSpace: <T extends Rect$2 | Point>(rectOrPoint: T, pageIndex: number) => T;
    transformContentPageToClientSpace: <T extends Rect$2 | Point>(rectOrPoint: T, pageIndex: number) => T;
    transformClientToPageSpace: <T extends Rect$2 | Point>(rectOrPoint: T, pageIndex: number) => T;
    transformPageToClientSpace: <T extends Rect$2 | Point>(rectOrPoint: T, pageIndex: number) => T;
    transformRawToPageSpace: (rawInset: InsetJSON | Inset, pageIndex: number) => Rect$2;
    transformPageToRawSpace: (rect: Rect$2, pageIndex: number) => Inset;
    exportOffice: (options: ExportOfficeFlags) => Promise<ArrayBuffer>;
    exportPDF: (flags?: ExportPDFFlags) => Promise<ArrayBuffer>;
    exportXFDF: () => Promise<string>;
    exportInstantJSON: (version?: number) => Promise<InstantJSON>;
    renderPageAsArrayBuffer: (options: {
        width: number;
    } | {
        height: number;
    }, pageIndex: number) => Promise<ArrayBuffer>;
    renderPageAsImageURL: (options: {
        width: number;
    } | {
        height: number;
    }, pageIndex: number) => Promise<string>;
    print: (printMode?: IPrintMode | {
        mode?: IPrintMode;
        excludeAnnotations?: boolean;
    }) => void;
    abortPrint: () => void;
    setCustomRenderers: (customRenderers: CustomRenderers) => void;
    setCustomUIConfiguration: (customUIConfigurationOrConfigurationSetter: CustomUI | ((customUI: CustomUI | null) => CustomUI)) => void;
    getDocumentOutline: () => Promise<List<OutlineElement>>;
    setDocumentOutline: (outline: List<OutlineElement>) => void;
    getPageGlyphs: (pageIndex: number) => Promise<List<Glyph>>;
    setAnnotationCreatorName: (annotationCreatorName?: string | null) => void;
    setOnWidgetAnnotationCreationStart: (callback: OnWidgetAnnotationCreationStartCallback) => void;
    setOnCommentCreationStart: (callback: OnCommentCreationStartCallback) => void;
    getLayers: () => Promise<OCGLayers>;
    getLayersVisibilityState: () => Promise<OCGLayersVisibilityState>;
    setLayersVisibilityState: (visibilityState: OCGLayersVisibilityState) => Promise<void>;
    contentWindow: Window;
    contentDocument: Document | ShadowRoot;
    readonly viewState: ViewState;
    setViewState: (stateOrFunction: ViewStateSetter | ViewState) => void;
    readonly toolbarItems: ToolbarItem[];
    setToolbarItems: (stateOrFunction: ToolbarItemsSetter | ToolbarItem[]) => void;
    setAnnotationToolbarItems: (annotationToolbarItemsCallback: AnnotationToolbarItemsCallback) => void;
    setInlineTextSelectionToolbarItems: (InlineTextSelectionToolbarItemsCallback: InlineTextSelectionToolbarItemsCallback) => void;
    annotationPresets: Record<AnnotationPresetID$1, AnnotationPreset$1>;
    setAnnotationPresets: (stateOrFunction: AnnotationPresetsSetter | Record<AnnotationPresetID$1, AnnotationPreset$1>) => void;
    setCurrentAnnotationPreset: (annotationPresetID?: string | null) => void;
    readonly currentAnnotationPreset: string | null | undefined;
    readonly stampAnnotationTemplates: Array<StampAnnotation | ImageAnnotation>;
    setStampAnnotationTemplates: (stateOrFunction: StampAnnotationTemplatesSetter | Array<StampAnnotation | ImageAnnotation>) => void;
    getAnnotations: (pageIndex: number) => Promise<List<AnnotationsUnion>>;
    createAttachment: (file: Blob) => Promise<string>;
    getAttachment: (attachmentId: string) => Promise<Blob>;
    calculateFittingTextAnnotationBoundingBox: (annotation: TextAnnotation) => TextAnnotation;
    setOnAnnotationResizeStart: (callback: AnnotationResizeStartCallback) => void;
    getOverlappingAnnotations: (annotationOrFormField: AnnotationsUnion | FormField) => Promise<List<AnnotationsUnion>>;
    getBookmarks: () => Promise<List<Bookmark>>;
    getFormFields: () => Promise<List<FormField>>;
    getFormFieldValues: () => Record<string, null | string | Array<string>>;
    setFormFieldValues: (formFieldValues: Record<string, null | string | Array<string>>) => void;
    getTextSelection: () => Record<string, any> | null | undefined;
    getSelectedAnnotation: () => AnnotationsUnion | null | undefined;
    getSelectedAnnotations: () => List<AnnotationsUnion> | null | undefined;
    getAnnotationsGroups: () => Map$1<string, {
        groupKey: string;
        annotationsIds: Set$1<ID>;
    }> | null | undefined;
    setSelectedAnnotation: (annotationOrAnnotationId?: (AnnotationsUnion | ID) | null) => void;
    setSelectedAnnotations: (annotationsOrAnnotationsId?: List<AnnotationsUnion | ID> | null) => void;
    groupAnnotations: (annotationsOrAnnotationsId?: List<AnnotationsUnion | ID>) => void;
    deleteAnnotationsGroup: (annotationGroupId?: string) => void;
    setEditingAnnotation: (annotationOrAnnotationId?: (AnnotationsUnion | ID) | null, autoSelectText?: boolean | null) => void;
    setCustomOverlayItem: (item: CustomOverlayItem) => void;
    removeCustomOverlayItem: (id: CustomOverlayItemID) => void;
    readonly locale: string;
    setLocale: (arg0: string) => Promise<void>;
    getInkSignatures: () => Promise<List<InkAnnotation | ImageAnnotation>>;
    getStoredSignatures: () => Promise<List<InkAnnotation | ImageAnnotation>>;
    setInkSignatures: (stateOrFunction: StoredSignaturesSetter | List<InkAnnotation | ImageAnnotation>) => Promise<void>;
    setStoredSignatures: (stateOrFunction: StoredSignaturesSetter | List<InkAnnotation | ImageAnnotation>) => Promise<void>;
    search: (term: string, options?: {
        startPageIndex?: number;
        endPageIndex?: number;
        searchType?: ISearchType;
        searchInAnnotations?: boolean;
        caseSensitive?: boolean;
    }) => Promise<List<SearchResult>>;
    startUISearch: (term: string) => void;
    readonly searchState: SearchState;
    setSearchState: (stateOrFunction: SearchStateSetter | SearchState) => void;
    readonly editableAnnotationTypes: Array<Class<AnnotationsUnion>>;
    setEditableAnnotationTypes: (arg0: Array<Class<AnnotationsUnion>>) => void;
    setIsEditableAnnotation: (arg0: IsEditableAnnotationCallback) => void;
    setIsEditableComment: (arg0: IsEditableCommentCallback) => void;
    setGroup: (group: string) => void;
    resetGroup: () => void;
    setMentionableUsers: (users: MentionableUser$1[]) => void;
    setMaxMentionSuggestions: (maxSuggestions: number) => void;
    getComments: () => Promise<List<Comment$1>>;
    setDocumentEditorFooterItems: (stateOrFunction: DocumentEditorFooterItem[] | SetDocumentEditorFooterFunction) => void;
    setDocumentEditorToolbarItems: (stateOrFunction: DocumentEditorToolbarItem[] | SetDocumentEditorToolbarFunction) => void;
    getSignaturesInfo: () => Promise<SignaturesInfo>;
    signDocument: (arg0: SignatureCreationData | null, arg1?: TwoStepSignatureCallback | SigningServiceData) => Promise<void>;
    setSignaturesLTV: (certificates?: ArrayBuffer[] | string[]) => Promise<SignaturesInfo>;
    applyOperations: (operations: Array<DocumentOperation>) => Promise<void>;
    exportPDFWithOperations: (arg0: Array<DocumentOperation>) => Promise<ArrayBuffer>;
    applyRedactions: () => Promise<void>;
    save: () => Promise<void>;
    hasUnsavedChanges: () => boolean;
    ensureChangesSaved: (changes: Change | Array<Change>) => Promise<Array<Change>>;
    create: (changes: Change | Array<Change> | List<Change>) => Promise<Array<Change>>;
    update: (changes: Change | Array<Change> | List<Change>) => Promise<Array<Change>>;
    delete: (ids: InstantID | Change | Array<InstantID | Change> | List<InstantID | Change>) => Promise<Array<Change>>;
    toggleClipboardActions: (enable: boolean) => void;
    setMeasurementSnapping: (enable: boolean) => void;
    setMeasurementPrecision: (precision: IMeasurementPrecision) => void;
    setMeasurementScale: (scale: MeasurementScale) => void;
    setMeasurementValueConfiguration: (configurationCallback: MeasurementValueConfigurationCallback) => void;
    createRedactionsBySearch: (term: string | ISearchPattern, options?: {
        searchType?: ISearchType;
        searchInAnnotations?: boolean;
        caseSensitive?: boolean;
        annotationPreset?: RedactionAnnotationPreset;
    }) => Promise<List<string>>;
    history: {
        undo: () => Promise<boolean>;
        redo: () => Promise<boolean>;
        clear: () => void;
        enable: () => void;
        disable: () => void;
        canUndo: () => boolean;
        canRedo: () => boolean;
    };
    setDocumentComparisonMode: (documentComparisonConfiguration: DocumentComparisonConfiguration | null) => void;
    compareDocuments: (comparisonDocuments: ComparisonDocuments, operations: ComparisonOperation) => Promise<DocumentComparisonResult>;
    getEmbeddedFiles: () => Promise<List<EmbeddedFile>>;
    getPageTabOrder: (pageIndex: number) => Promise<ID[]>;
    setPageTabOrder: (pageIndex: number, annotationIdsSortCallback: (tabOrderedAnnotations: AnnotationsUnion[]) => ID[]) => Promise<void>;
    enableAlwaysScrollToZoom: boolean;
}

type ComparisonDocuments = {
    originalDocument: DocumentDescriptor;
    changedDocument: DocumentDescriptor;
};
type Range = {
    position: number;
    length: number;
};
type TextBlock = {
    range: Range;
    rects: number[][];
};
type Operation = {
    type: 'insert' | 'delete' | 'equal';
    text: string;
    originalTextBlocks: TextBlock;
    changedTextBlocks: TextBlock;
};
type Hunk = {
    originalRange: Range;
    changedRange: Range;
    operations: Operation[];
};
type ComparisonResult = {
    type: 'text';
    hunks: Hunk[];
};
type PageComparisonResult = {
    originalPageIndex: number;
    changedPageIndex: number;
    comparisonResults: ComparisonResult[];
};
type DocumentComparisonResult = {
    documentComparisonResults: PageComparisonResult[];
} | null;

declare const ProductId: {
    SharePoint: string;
    Salesforce: string;
    Maui_Android: string;
    Maui_iOS: string;
    Maui_MacCatalyst: string;
    Maui_Windows: string;
    FlutterForWeb: string;
    Electron: string;
};
type IProductId = (typeof ProductId)[keyof typeof ProductId];

type ActionFlags = 'includeExclude' | 'includeNoValueFields' | 'exportFormat' | 'getMethod' | 'submitCoordinated' | 'xfdf' | 'includeAppendSaves' | 'includeAnnotations' | 'submitPDF' | 'canonicalFormat' | 'excludeNonUserAnnotations' | 'excludeFKey' | 'embedForm';
type ActionJSON = {
    type: 'uri';
    uri: string;
    subactions?: Array<ActionJSON>;
} | {
    type: 'goTo';
    pageIndex: number;
    subactions?: Array<ActionJSON>;
} | {
    type: 'goToEmbedded';
    newWindow: boolean;
    relativePath: string;
    targetType: 'parent' | 'child';
    subactions?: Array<ActionJSON>;
} | {
    type: 'goToRemote';
    relativePath: string;
    namedDestination: string;
    subactions?: Array<ActionJSON>;
} | {
    type: 'hide';
    hide: boolean;
    annotationReferences: Array<AnnotationReference>;
    subactions?: Array<ActionJSON>;
} | {
    type: 'resetForm';
    fields: Array<string> | null;
    flags: string | null;
    subactions?: Array<ActionJSON>;
} | {
    type: 'submitForm';
    uri: string;
    fields: Array<string> | null;
    flags: Array<ActionFlags> | null;
    subactions?: Array<ActionJSON>;
} | {
    type: 'launch';
    filePath: string;
    subactions?: Array<ActionJSON>;
} | {
    type: 'named';
    action: string;
    subactions?: Array<ActionJSON>;
} | {
    type: 'javaScript';
    script: string;
    subactions?: Array<ActionJSON>;
};
type BookmarkJSON = {
    v: 1;
    type: 'pspdfkit/bookmark';
    id: string;
    name: string | null;
    sortKey: number | null;
    action: ActionJSON;
    pdfBookmarkId: string | null;
};
type RawPdfBoxes = {
    bleedBox: null | IRectJSON;
    cropBox: null | IRectJSON;
    mediaBox: null | IRectJSON;
    trimBox: null | IRectJSON;
};
type Hints = {
    glyphs: Array<number | string>;
};

type IRectJSON = [left: number, top: number, width: number, height: number];

type BaseAnnotationJSON = {
    v: number;
    type?: 'pspdfkit/ink' | 'pspdfkit/shape/line' | 'pspdfkit/shape/rectangle' | 'pspdfkit/shape/ellipse' | 'pspdfkit/shape/polygon' | 'pspdfkit/shape/polyline' | 'pspdfkit/link' | 'pspdfkit/markup/highlight' | 'pspdfkit/markup/squiggly' | 'pspdfkit/markup/strikeout' | 'pspdfkit/markup/underline' | 'pspdfkit/markup/redaction' | 'pspdfkit/stamp' | 'pspdfkit/text' | 'pspdfkit/note' | 'pspdfkit/image' | 'pspdfkit/media' | 'pspdfkit/widget' | 'pspdfkit/comment-marker' | 'pspdfkit/unknown';
    name?: string | null;
    id: string;
    subject?: string | null;
    pdfObjectId?: number | null;
    pageIndex: number;
    bbox: IRectJSON;
    opacity?: number;
    flags?: ('noPrint' | 'noZoom' | 'noRotate' | 'noView' | 'hidden' | 'locked' | 'lockedContents' | 'readOnly')[] | null;
    action?: ActionJSON | null;
    note?: string | null;
    createdAt?: string | Date;
    updatedAt?: string | Date;
    creatorName?: string | null;
    customData?: Record<string, unknown> | null;
    isCommentThreadRoot?: boolean;
    isAnonymous?: boolean;
    APStreamCache?: {
        cache: string;
    } | {
        attach: string;
    };
    blendMode?: IBlendMode | null;
} & ICollaboratorPermissionsOptions;
type ImageAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    type: 'pspdfkit/image';
    description?: string | null;
    fileName?: string | null;
    contentType: string;
    imageAttachmentId: string;
    rotation: number;
    isSignature?: boolean;
    xfdfAppearanceStream?: string;
    xfdfAppearanceStreamOriginalPageRotation?: number;
};
type ShapeAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    strokeWidth: number;
    strokeColor: string | null;
    fillColor: string | null;
    strokeDashArray?: [number, number] | null;
    measurementPrecision?: IMeasurementPrecision | null;
    measurementScale?: MeasurementScaleJSON | null;
    lineWidth?: number | null;
};
type EllipseAnnotationJSON = ShapeAnnotationJSON & {
    type: 'pspdfkit/shape/ellipse';
    cloudyBorderIntensity: number | null;
    cloudyBorderInset: InsetJSON | null;
    measurementBBox: IRectJSON | null;
};
type LineAnnotationJSON = ShapeAnnotationJSON & {
    type: 'pspdfkit/shape/line';
    startPoint: [number, number];
    endPoint: [number, number];
    lineCaps?: LineCapsType | null;
    lines?: {
        points: [number, number][][];
        intensities: number[][];
    };
};
type PolygonAnnotationJSON = ShapeAnnotationJSON & {
    type: 'pspdfkit/shape/polygon';
    points: [number, number][];
    cloudyBorderIntensity: number | null;
    lines?: {
        points: [number, number][][];
        intensities: number[][];
    };
};
type PolylineAnnotationJSON = ShapeAnnotationJSON & {
    type: 'pspdfkit/shape/polyline';
    points: [number, number][];
    lineCaps?: LineCapsType | null;
    lines?: {
        points: [number, number][][];
        intensities: number[][];
    };
};
type RectangleAnnotationJSON = ShapeAnnotationJSON & {
    type: 'pspdfkit/shape/rectangle';
    cloudyBorderIntensity: number | null;
    cloudyBorderInset?: InsetJSON | null;
    measurementBBox: IRectJSON | null;
};
type InkAnnotationJSON = BaseAnnotationJSON & {
    type: 'pspdfkit/ink';
    lines: {
        points: [number, number][][];
        intensities: number[][];
    };
    lineWidth: number;
    strokeColor: string | null;
    backgroundColor: string | null;
    isDrawnNaturally: boolean;
    isSignature: boolean;
};
type LinkAnnotationJSON = BaseAnnotationJSON & {
    type: 'pspdfkit/link';
    borderColor?: string | null;
    borderWidth?: number | null;
    borderStyle?: IBorderStyle | null;
};
type NoteAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    type: 'pspdfkit/note';
    text?: {
        format: 'plain';
        value: string;
    };
    icon?: string;
    color?: string;
};
type MediaAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    type: 'pspdfkit/media';
    description: string | null;
    fileName: string | null;
    contentType: string | null;
    mediaAttachmentId: string | null;
};
type BaseTextMarkupAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    rects: [number, number, number, number][];
};
type TextMarkupAnnotationJSON = BaseTextMarkupAnnotationJSON & {
    type: 'pspdfkit/markup/highlight' | 'pspdfkit/markup/squiggly' | 'pspdfkit/markup/strikeout' | 'pspdfkit/markup/underline' | 'pspdfkit/markup/redaction';
    color: string | null;
};
type RedactionAnnotationJSON = BaseTextMarkupAnnotationJSON & {
    type: 'pspdfkit/markup/redaction';
    fillColor?: string | null;
    outlineColor?: string | null;
    overlayText?: string | null;
    repeatOverlayText?: boolean | null;
    rotation?: number;
    color?: string | null;
};
type StampAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    type: 'pspdfkit/stamp';
    stampType: StampKind;
    title: string | null;
    color?: string | null;
    subTitle?: string | null;
    subtitle: string | null;
    rotation: number | null;
    xfdfAppearanceStream?: string;
    xfdfAppearanceStreamOriginalPageRotation?: number;
    kind?: StampKind;
};
type TextAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    type: 'pspdfkit/text';
    text: {
        format: 'xhtml' | 'plain';
        value: string;
    };
    fontColor?: string | null;
    backgroundColor?: string | null;
    font?: string | null;
    rotation?: number | null;
    fontSize?: number | null;
    fontStyle?: string[] | null;
    horizontalAlign?: 'left' | 'center' | 'right';
    verticalAlign?: 'top' | 'center' | 'bottom';
    callout?: {
        start: [number, number];
        knee?: [number, number] | null;
        end: [number, number];
        cap?: ILineCap | null;
        innerRectInset?: InsetJSON | null;
    } | null;
    borderStyle?: IBorderStyle | null;
    borderWidth?: number | null;
    borderColor?: string | null;
    isFitting?: boolean;
    lineHeightFactor?: number | null;
};
type UnknownAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    type: 'pspdfkit/unknown';
};
type WidgetAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    type: 'pspdfkit/widget';
    formFieldName: string;
    borderColor?: string | null;
    borderStyle?: IBorderStyle | null;
    borderDashArray?: number[] | null;
    borderWidth?: number | null;
    font?: string | null;
    fontSize?: 'auto' | number | null;
    fontColor?: string | null;
    backgroundColor?: string | null;
    horizontalAlign?: 'left' | 'center' | 'right' | null;
    verticalAlign?: 'top' | 'center' | 'bottom' | null;
    fontStyle?: string[] | null | undefined;
    rotation?: number;
    additionalActions?: SerializedAdditionalActionsType | null;
    lineHeightFactor?: number | null;
};
type CommentMarkerAnnotationJSON = Omit<BaseAnnotationJSON, 'type'> & {
    type: 'pspdfkit/comment-marker';
};
type AnnotationJSONUnion = TextMarkupAnnotationJSON | TextAnnotationJSON | WidgetAnnotationJSON | RedactionAnnotationJSON | StampAnnotationJSON | NoteAnnotationJSON | LinkAnnotationJSON | InkAnnotationJSON | RectangleAnnotationJSON | PolylineAnnotationJSON | PolygonAnnotationJSON | LineAnnotationJSON | EllipseAnnotationJSON | ImageAnnotationJSON | UnknownAnnotationJSON | MediaAnnotationJSON | CommentMarkerAnnotationJSON;

declare const Theme$1: {
    readonly LIGHT: "LIGHT";
    readonly DARK: "DARK";
    readonly AUTO: "AUTO";
};
type ITheme = (typeof Theme$1)[keyof typeof Theme$1];

interface SVGRProps {
    title?: string;
    titleId?: string;
    size?: CSSProperties["width"];
}
declare type IconProps = SVGRProps & SVGProps<SVGSVGElement>;

interface ResolvedDateTimeFormatOptions extends Intl.ResolvedDateTimeFormatOptions {
    hourCycle?: Intl.DateTimeFormatOptions['hourCycle'];
}
interface DateRangeFormatPart extends Intl.DateTimeFormatPart {
    source: 'startRange' | 'endRange' | 'shared';
}
/** A wrapper around Intl.DateTimeFormat that fixes various browser bugs, and polyfills new features. */
declare class DateFormatter implements Intl.DateTimeFormat {
    constructor(locale: string, options?: Intl.DateTimeFormatOptions);
    /** Formats a date as a string according to the locale and format options passed to the constructor. */
    format(value: Date): string;
    /** Formats a date to an array of parts such as separators, numbers, punctuation, and more. */
    formatToParts(value: Date): Intl.DateTimeFormatPart[];
    /** Formats a date range as a string. */
    formatRange(start: Date, end: Date): string;
    /** Formats a date range as an array of parts. */
    formatRangeToParts(start: Date, end: Date): DateRangeFormatPart[];
    /** Returns the resolved formatting options based on the values passed to the constructor. */
    resolvedOptions(): ResolvedDateTimeFormatOptions;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface AriaLabelingProps {
  /**
   * Defines a string value that labels the current element.
   */
  'aria-label'?: string,

  /**
   * Identifies the element (or elements) that labels the current element.
   */
  'aria-labelledby'?: string,

  /**
   * Identifies the element (or elements) that describes the object.
   */
  'aria-describedby'?: string,

  /**
   * Identifies the element (or elements) that provide a detailed, extended description for the object.
   */
  'aria-details'?: string
}

interface AriaValidationProps {
  // https://www.w3.org/TR/wai-aria-1.2/#aria-errormessage
  /**
   * Identifies the element that provides an error message for the object.
   */
  'aria-errormessage'?: string
}

// A set of common DOM props that are allowed on any component
// Ensure this is synced with DOMPropNames in filterDOMProps
interface DOMProps {
  /**
   * The element's unique identifier. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/id).
   */
  id?: string
}

interface FocusableDOMProps extends DOMProps {
  /**
   * Whether to exclude the element from the sequential tab order. If true,
   * the element will not be focusable via the keyboard by tabbing. This should
   * be avoided except in rare scenarios where an alternative means of accessing
   * the element or its functionality via the keyboard is available.
   */
  excludeFromTabOrder?: boolean
}


interface TextInputDOMEvents {
  // Clipboard events
  /**
   * Handler that is called when the user copies text. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/oncopy).
   */
   onCopy?: ClipboardEventHandler<HTMLInputElement>,

   /**
    * Handler that is called when the user cuts text. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/oncut).
    */
   onCut?: ClipboardEventHandler<HTMLInputElement>,

   /**
    * Handler that is called when the user pastes text. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/onpaste).
    */
   onPaste?: ClipboardEventHandler<HTMLInputElement>,

   // Composition events
   /**
    * Handler that is called when a text composition system starts a new text composition session. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/Element/compositionstart_event).
    */
   onCompositionStart?: CompositionEventHandler<HTMLInputElement>,

   /**
    * Handler that is called when a text composition system completes or cancels the current text composition session. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/Element/compositionend_event).
    */
   onCompositionEnd?: CompositionEventHandler<HTMLInputElement>,

   /**
    * Handler that is called when a new character is received in the current text composition session. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/Element/compositionupdate_event).
    */
   onCompositionUpdate?: CompositionEventHandler<HTMLInputElement>,

   // Selection events
   /**
    * Handler that is called when text in the input is selected. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/Element/select_event).
    */
   onSelect?: ReactEventHandler<HTMLInputElement>,

   // Input events
   /**
    * Handler that is called when the input value is about to be modified. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/beforeinput_event).
    */
   onBeforeInput?: FormEventHandler<HTMLInputElement>,
   /**
    * Handler that is called when the input value is modified. See [MDN](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/input_event).
    */
   onInput?: FormEventHandler<HTMLInputElement>
}

interface InputDOMProps {
  /**
   * The name of the input element, used when submitting an HTML form. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefname).
   */
  name?: string
}

// DOM props that apply to all text inputs
// Ensure this is synced with useTextField
interface TextInputDOMProps extends DOMProps, InputDOMProps, TextInputDOMEvents {
  /**
   * Describes the type of autocomplete functionality the input should provide if any. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefautocomplete).
   */
  autoComplete?: string,

  /**
   * The maximum number of characters supported by the input. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefmaxlength).
   */
  maxLength?: number,

  /**
   * The minimum number of characters required by the input. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefminlength).
   */
  minLength?: number,

  /**
   * Regex pattern that the value of the input must match to be valid. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefpattern).
   */
  pattern?: string,

  /**
   * Content that appears in the input when it is empty. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefplaceholder).
   */
  placeholder?: string,

  /**
   * The type of input to render. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdeftype).
   */
  type?: 'text' | 'search' | 'url' | 'tel' | 'email' | 'password' | (string & {}),

  /**
   * Hints at the type of data that might be entered by the user while editing the element or its contents. See [MDN](https://html.spec.whatwg.org/multipage/interaction.html#input-modalities:-the-inputmode-attribute).
   */
  inputMode?: 'none' | 'text' | 'tel' | 'url' | 'email' | 'numeric' | 'decimal' | 'search'
}

/**
 * This type allows configuring link props with router options and type-safe URLs via TS module augmentation.
 * By default, this is an empty type. Extend with `href` and `routerOptions` properties to configure your router.
 */
interface RouterConfig {}

type Href = RouterConfig extends {href: infer H} ? H : string;
type RouterOptions = RouterConfig extends {routerOptions: infer O} ? O : never;

// Make sure to update filterDOMProps.ts when updating this.
interface LinkDOMProps {
  /** A URL to link to. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#href). */
  href?: Href,
  /** Hints at the human language of the linked URL. See[MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#hreflang). */
  hrefLang?: string,
  /** The target window for the link. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#target). */
  target?: HTMLAttributeAnchorTarget,
  /** The relationship between the linked resource and the current page. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/rel). */
  rel?: string,
  /** Causes the browser to download the linked URL. A string may be provided to suggest a file name. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#download). */
  download?: boolean | string,
  /** A space-separated list of URLs to ping when the link is followed. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#ping). */
  ping?: string,
  /** How much of the referrer to send when following the link. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/a#referrerpolicy). */
  referrerPolicy?: HTMLAttributeReferrerPolicy,
  /** Options for the configured client side router. */
  routerOptions?: RouterOptions
}

/** Any focusable element, including both HTML and SVG elements. */
interface FocusableElement extends Element, HTMLOrSVGElement {}

/** All DOM attributes supported across both HTML and SVG elements. */
interface DOMAttributes<T = FocusableElement> extends AriaAttributes, DOMAttributes$1<T> {
  id?: string | undefined,
  role?: AriaRole | undefined,
  tabIndex?: number | undefined,
  style?: CSSProperties | undefined,
  className?: string | undefined
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



type ValidationState = 'valid' | 'invalid';

type ValidationError = string | string[];

interface Validation<T = unknown> {
  /** Whether user input is required on the input before form submission. */
  isRequired?: boolean,
  /** Whether the input value is invalid. */
  isInvalid?: boolean,
  /** @deprecated Use `isInvalid` instead. */
  validationState?: ValidationState,
  /**
   * Whether to use native HTML form validation to prevent form submission
   * when the value is missing or invalid, or mark the field as required
   * or invalid via ARIA.
   * @default 'aria'
   */
  validationBehavior?: 'aria' | 'native',
  /**
   * A function that returns an error message if a given value is invalid.
   * Validation errors are displayed to the user when the form is submitted
   * if `validationBehavior="native"`. For realtime validation, use the `isInvalid`
   * prop instead.
   */
  validate?: (value: T) => ValidationError | true | null | undefined
}

interface ValidationResult {
  /** Whether the input value is invalid. */
  isInvalid: boolean,
  /** The current error messages for the input if it is invalid, otherwise an empty array. */
  validationErrors: string[],
  /** The native validation details for the input. */
  validationDetails: ValidityState
}

interface InputBase {
  /** Whether the input is disabled. */
  isDisabled?: boolean,
  /** Whether the input can be selected but not changed by the user. */
  isReadOnly?: boolean
}

interface ValueBase<T, C = T> {
  /** The current value (controlled). */
  value?: T,
  /** The default value (uncontrolled). */
  defaultValue?: T,
  /** Handler that is called when the value changes. */
  onChange?: (value: C) => void
}

interface TextInputBase {
  /** Temporary text that occupies the text input when it is empty. */
  placeholder?: string
}

interface RangeInputBase<T> {
  /** The smallest value allowed for the input. */
  minValue?: T,
  /** The largest value allowed for the input. */
  maxValue?: T,
  /** The amount that the input value changes with each increment or decrement "tick". */
  step?: T // ??
}

interface HelpTextProps {
  /** A description for the field. Provides a hint such as specific requirements for what to choose. */
  description?: ReactNode,
  /** An error message for the field. */
  errorMessage?: ReactNode | ((v: ValidationResult) => ReactNode)
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface SingleSelection {
  /** Whether the collection allows empty selection. */
  disallowEmptySelection?: boolean,
  /** The currently selected key in the collection (controlled). */
  selectedKey?: Key$1 | null,
  /** The initial selected key in the collection (uncontrolled). */
  defaultSelectedKey?: Key$1,
  /** Handler that is called when the selection changes. */
  onSelectionChange?: (key: Key$1) => void
}

type SelectionMode = 'none' | 'single' | 'multiple';
type SelectionBehavior = 'toggle' | 'replace';
type Selection$1 = 'all' | Set<Key$1>;
interface MultipleSelection {
  /** The type of selection that is allowed in the collection. */
  selectionMode?: SelectionMode,
  /** Whether the collection allows empty selection. */
  disallowEmptySelection?: boolean,
  /** The currently selected keys in the collection (controlled). */
  selectedKeys?: 'all' | Iterable<Key$1>,
  /** The initial selected keys in the collection (uncontrolled). */
  defaultSelectedKeys?: 'all' | Iterable<Key$1>,
  /** Handler that is called when the selection changes. */
  onSelectionChange?: (keys: Selection$1) => void,
  /** The currently disabled keys in the collection (controlled). */
  disabledKeys?: Iterable<Key$1>
}

type FocusStrategy = 'first' | 'last';
type DisabledBehavior = 'selection' | 'all';

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface DragDropEvent {
  /** The x coordinate of the event, relative to the target element. */
  x: number,
  /** The y coordinate of the event, relative to the target element. */
  y: number
}

type DropOperation = 'copy' | 'link' | 'move' | 'cancel';

interface DragItem {
  [type: string]: string
}

interface DragStartEvent extends DragDropEvent {
  /** The event type. */
  type: 'dragstart'
}

interface DragMoveEvent extends DragDropEvent {
  /** The event type. */
  type: 'dragmove'
}

interface DragEndEvent extends DragDropEvent {
  /** The event type. */
  type: 'dragend',
  /** The drop operation that occurred. */
  dropOperation: DropOperation
}

interface DropEnterEvent extends DragDropEvent {
  /** The event type. */
  type: 'dropenter'
}

interface DropMoveEvent extends DragDropEvent {
  /** The event type. */
  type: 'dropmove'
}

interface DropActivateEvent extends DragDropEvent {
  /** The event type. */
  type: 'dropactivate'
}

interface DropExitEvent extends DragDropEvent {
  /** The event type. */
  type: 'dropexit'
}

interface TextDropItem {
  /** The item kind. */
  kind: 'text',
  /**
   * The drag types available for this item.
   * These are often mime types, but may be custom app-specific types.
   */
  types: Set<string>,
  /** Returns the data for the given type as a string. */
  getText(type: string): Promise<string>
}

interface FileDropItem {
  /** The item kind. */
  kind: 'file',
  /** The file type (usually a mime type). */
  type: string,
  /** The file name. */
  name: string,
  /** Returns the contents of the file as a blob. */
  getFile(): Promise<File>,
  /** Returns the contents of the file as a string. */
  getText(): Promise<string>
}

interface DirectoryDropItem {
  /** The item kind. */
  kind: 'directory',
  /** The directory name. */
  name: string,
  /** Returns the entries contained within the directory. */
  getEntries(): AsyncIterable<FileDropItem | DirectoryDropItem>
}

type DropItem = TextDropItem | FileDropItem | DirectoryDropItem;

interface DropEvent extends DragDropEvent {
  /** The event type. */
  type: 'drop',
  /** The drop operation that should occur. */
  dropOperation: DropOperation,
  /** The dropped items. */
  items: DropItem[]
}

type DropPosition = 'on' | 'before' | 'after';
interface RootDropTarget {
  /** The event type. */
  type: 'root'
}

interface ItemDropTarget {
  /** The drop target type. */
  type: 'item',
  /** The item key. */
  key: Key$1,
  /** The drop position relative to the item. */
  dropPosition: DropPosition
}

type DropTarget = RootDropTarget | ItemDropTarget;

interface DroppableCollectionEnterEvent extends DropEnterEvent {
  /** The drop target. */
  target: DropTarget
}

interface DroppableCollectionActivateEvent extends DropActivateEvent {
  /** The drop target. */
  target: DropTarget
}

interface DroppableCollectionExitEvent extends DropExitEvent {
  /** The drop target. */
  target: DropTarget
}

interface DroppableCollectionDropEvent extends DropEvent {
  /** The drop target. */
  target: DropTarget
}

interface DroppableCollectionInsertDropEvent {
  /** The dropped items. */
  items: DropItem[],
  /** The drop operation that should occur. */
  dropOperation: DropOperation,
   /** The drop target. */
  target: ItemDropTarget
}

interface DroppableCollectionRootDropEvent {
  /** The dropped items. */
  items: DropItem[],
  /** The drop operation that should occur. */
  dropOperation: DropOperation
}

interface DroppableCollectionOnItemDropEvent {
  /** The dropped items. */
  items: DropItem[],
  /** The drop operation that should occur. */
  dropOperation: DropOperation,
  /** Whether the drag originated within the same collection as the drop. */
  isInternal: boolean,
  /** The drop target. */
  target: ItemDropTarget
}

interface DroppableCollectionReorderEvent {
  /** The keys of the items that were reordered. */
  keys: Set<Key$1>,
  /** The drop operation that should occur. */
  dropOperation: DropOperation,
  /** The drop target. */
  target: ItemDropTarget
}

interface DragTypes {
  /** Returns whether the drag contains data of the given type. */
  has(type: string | symbol): boolean
}

interface DropTargetDelegate {
  /**
   * Returns a drop target within a collection for the given x and y coordinates.
   * The point is provided relative to the top left corner of the collection container.
   * A drop target can be checked to see if it is valid using the provided `isValidDropTarget` function.
   */
  getDropTargetFromPoint(x: number, y: number, isValidDropTarget: (target: DropTarget) => boolean): DropTarget | null
}

interface DroppableCollectionUtilityOptions {
  /**
   * The drag types that the droppable collection accepts. If the collection accepts directories, include `DIRECTORY_DRAG_TYPE` in your array of allowed types.
   * @default 'all'
   */
  acceptedDragTypes?: 'all' | Array<string | symbol>,
  /**
   * Handler that is called when external items are dropped "between" items.
   */
  onInsert?: (e: DroppableCollectionInsertDropEvent) => void,
  /**
   * Handler that is called when external items are dropped on the droppable collection's root.
   */
  onRootDrop?: (e: DroppableCollectionRootDropEvent) => void,
  /**
   * Handler that is called when items are dropped "on" an item.
   */
  onItemDrop?: (e: DroppableCollectionOnItemDropEvent) => void,
  /**
   * Handler that is called when items are reordered via drag in the source collection.
   */
  onReorder?: (e: DroppableCollectionReorderEvent) => void,
  /**
   * A function returning whether a given target in the droppable collection is a valid "on" drop target for the current drag types.
   */
  shouldAcceptItemDrop?: (target: ItemDropTarget, types: DragTypes) => boolean
}

interface DroppableCollectionBaseProps {
  /** Handler that is called when a valid drag enters a drop target. */
  onDropEnter?: (e: DroppableCollectionEnterEvent) => void,
  /**
   * Handler that is called after a valid drag is held over a drop target for a period of time.
   * @private
   */
  onDropActivate?: (e: DroppableCollectionActivateEvent) => void,
  /** Handler that is called when a valid drag exits a drop target. */
  onDropExit?: (e: DroppableCollectionExitEvent) => void,
  /**
   * Handler that is called when a valid drag is dropped on a drop target. When defined, this overrides other
   * drop handlers such as `onInsert`, and `onItemDrop`.
   */
  onDrop?: (e: DroppableCollectionDropEvent) => void,
  /**
   * A function returning the drop operation to be performed when items matching the given types are dropped
   * on the drop target.
   */
  getDropOperation?: (target: DropTarget, types: DragTypes, allowedOperations: DropOperation[]) => DropOperation
}

interface DroppableCollectionProps extends DroppableCollectionUtilityOptions, DroppableCollectionBaseProps {}

interface DraggableCollectionStartEvent extends DragStartEvent {
  /** The keys of the items that were dragged. */
  keys: Set<Key$1>
}

interface DraggableCollectionMoveEvent extends DragMoveEvent {
  /** The keys of the items that were dragged. */
  keys: Set<Key$1>
}

interface DraggableCollectionEndEvent extends DragEndEvent {
  /** The keys of the items that were dragged. */
  keys: Set<Key$1>,
  /** Whether the drop ended within the same collection as it originated. */
  isInternal: boolean
}

type DragPreviewRenderer = (items: DragItem[], callback: (node: HTMLElement) => void) => void;

interface DraggableCollectionProps {
  /** Handler that is called when a drag operation is started. */
  onDragStart?: (e: DraggableCollectionStartEvent) => void,
  /** Handler that is called when the drag is moved. */
  onDragMove?: (e: DraggableCollectionMoveEvent) => void,
  /** Handler that is called when the drag operation is ended, either as a result of a drop or a cancellation. */
  onDragEnd?: (e: DraggableCollectionEndEvent) => void,
  /** A function that returns the items being dragged. */
  getItems: (keys: Set<Key$1>) => DragItem[],
  /** The ref of the element that will be rendered as the drag preview while dragging. */
  preview?: RefObject<DragPreviewRenderer | null>,
  /** Function that returns the drop operations that are allowed for the dragged items. If not provided, all drop operations are allowed. */
  getAllowedDropOperations?: () => DropOperation[]
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface ItemProps$1<T> extends LinkDOMProps {
  /** Rendered contents of the item or child items. */
  children: ReactNode,
  /** Rendered contents of the item if `children` contains child items. */
  title?: ReactNode, // label?? contents?
  /** A string representation of the item's contents, used for features like typeahead. */
  textValue?: string,
  /** An accessibility label for this item. */
  'aria-label'?: string,
  /** A list of child item objects. Used for dynamic collections. */
  childItems?: Iterable<T>,
  /** Whether this item has children, even if not loaded yet. */
  hasChildItems?: boolean
}

type ItemElement<T> = ReactElement<ItemProps$1<T>>;
type ItemRenderer<T> = (item: T) => ItemElement<T>;

interface SectionProps<T> {
  /** Rendered contents of the section, e.g. a header. */
  title?: ReactNode,
  /** An accessibility label for the section. */
  'aria-label'?: string,
  /** Static child items or a function to render children. */
  children: ItemElement<T> | ItemElement<T>[] | ItemRenderer<T>,
  /** Item objects in the section. */
  items?: Iterable<T>
}

type SectionElement<T> = ReactElement<SectionProps<T>>;

type CollectionElement<T> = SectionElement<T> | ItemElement<T>;
type CollectionChildren<T> = CollectionElement<T> | CollectionElement<T>[] | ((item: T) => CollectionElement<T>);
interface CollectionBase<T> {
  /** The contents of the collection. */
  children: CollectionChildren<T>,
  /** Item objects in the collection. */
  items?: Iterable<T>,
  /** The item keys that are disabled. These items cannot be selected, focused, or otherwise interacted with. */
  disabledKeys?: Iterable<Key$1>
}

interface CollectionStateBase<T, C extends Collection<Node$1<T>> = Collection<Node$1<T>>> extends Partial<CollectionBase<T>> {
  /** A pre-constructed collection to use instead of building one from items and children. */
  collection?: C
}

interface KeyboardDelegate {
  /** Returns the key visually below the given one, or `null` for none. */
  getKeyBelow?(key: Key$1): Key$1 | null,

  /** Returns the key visually above the given one, or `null` for none. */
  getKeyAbove?(key: Key$1): Key$1 | null,

  /** Returns the key visually to the left of the given one, or `null` for none. */
  getKeyLeftOf?(key: Key$1): Key$1 | null,

  /** Returns the key visually to the right of the given one, or `null` for none. */
  getKeyRightOf?(key: Key$1): Key$1 | null,

  /** Returns the key visually one page below the given one, or `null` for none. */
  getKeyPageBelow?(key: Key$1): Key$1 | null,

  /** Returns the key visually one page above the given one, or `null` for none. */
  getKeyPageAbove?(key: Key$1): Key$1 | null,

  /** Returns the first key, or `null` for none. */
  getFirstKey?(key?: Key$1, global?: boolean): Key$1 | null,

  /** Returns the last key, or `null` for none. */
  getLastKey?(key?: Key$1, global?: boolean): Key$1 | null,

  /** Returns the next key after `fromKey` that matches the given search string, or `null` for none. */
  getKeyForSearch?(search: string, fromKey?: Key$1): Key$1 | null
}

interface Rect$1 {
  x: number,
  y: number,
  width: number,
  height: number
}

interface Size {
  width: number,
  height: number
}

/** A LayoutDelegate provides layout information for collection items. */
interface LayoutDelegate {
  /** Returns a rectangle for the item with the given key. */
  getItemRect(key: Key$1): Rect$1 | null,
  /** Returns the visible rectangle of the collection. */
  getVisibleRect(): Rect$1,
  /** Returns the size of the scrollable content in the collection. */
  getContentSize(): Size
}

/**
 * A generic interface to access a readonly sequential
 * collection of unique keyed items.
 */
interface Collection<T> extends Iterable<T> {
  /** The number of items in the collection. */
  readonly size: number,

  /** Iterate over all keys in the collection. */
  getKeys(): Iterable<Key$1>,

  /** Get an item by its key. */
  getItem(key: Key$1): T | null,

  /** Get an item by the index of its key. */
  at(idx: number): T | null,

  /** Get the key that comes before the given key in the collection. */
  getKeyBefore(key: Key$1): Key$1 | null,

  /** Get the key that comes after the given key in the collection. */
  getKeyAfter(key: Key$1): Key$1 | null,

  /** Get the first key in the collection. */
  getFirstKey(): Key$1 | null,

  /** Get the last key in the collection. */
  getLastKey(): Key$1 | null,

  /** Iterate over the child items of the given key. */
  getChildren?(key: Key$1): Iterable<T>,

  /** Returns a string representation of the item's contents. */
  getTextValue?(key: Key$1): string
}

interface Node$1<T> {
  /** The type of item this node represents. */
  type: string,
  /** A unique key for the node. */
  key: Key$1,
  /** The object value the node was created from. */
  value: T | null,
  /** The level of depth this node is at in the hierarchy. */
  level: number,
  /** Whether this item has children, even if not loaded yet. */
  hasChildNodes: boolean,
  /**
   * The loaded children of this node.
   * @deprecated Use `collection.getChildren(node.key)` instead.
   */
  childNodes: Iterable<Node$1<T>>,
  /** The rendered contents of this node (e.g. JSX). */
  rendered: ReactNode,
  /** A string value for this node, used for features like typeahead. */
  textValue: string,
  /** An accessibility label for this node. */
  'aria-label'?: string,
  /** The index of this node within its parent. */
  index?: number,
  /** A function that should be called to wrap the rendered node. */
  wrapper?: (element: ReactElement) => ReactElement,
  /** The key of the parent node. */
  parentKey?: Key$1 | null,
  /** The key of the node before this node. */
  prevKey?: Key$1 | null,
  /** The key of the node after this node. */
  nextKey?: Key$1 | null,
  /** Additional properties specific to a particular node type. */
  props?: any,
  /** @private */
  shouldInvalidate?: (context: unknown) => boolean,
  /** A function that renders this node to a React Element in the DOM. */
  render?: (node: Node$1<any>) => ReactElement
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



// Event bubbling can be problematic in real-world applications, so the default for React Spectrum components
// is not to propagate. This can be overridden by calling continuePropagation() on the event.
type BaseEvent<T extends SyntheticEvent> = T & {
  /**
   * Use continuePropagation.
   * @deprecated */
  stopPropagation(): void,
  continuePropagation(): void
}

type KeyboardEvent$1 = BaseEvent<KeyboardEvent$2<any>>;

type PointerType = 'mouse' | 'pen' | 'touch' | 'keyboard' | 'virtual';

interface PressEvent {
  /** The type of press event being fired. */
  type: 'pressstart' | 'pressend' | 'pressup' | 'press',
  /** The pointer type that triggered the press event. */
  pointerType: PointerType,
  /** The target element of the press event. */
  target: Element,
  /** Whether the shift keyboard modifier was held during the press event. */
  shiftKey: boolean,
  /** Whether the ctrl keyboard modifier was held during the press event. */
  ctrlKey: boolean,
  /** Whether the meta keyboard modifier was held during the press event. */
  metaKey: boolean,
  /** Whether the alt keyboard modifier was held during the press event. */
  altKey: boolean,
  /** X position relative to the target. */
  x: number,
  /** Y position relative to the target. */
  y: number,
  /**
   * By default, press events stop propagation to parent elements.
   * In cases where a handler decides not to handle a specific event,
   * it can call `continuePropagation()` to allow a parent to handle it.
   */
  continuePropagation(): void
}

interface LongPressEvent extends Omit<PressEvent, 'type' | 'continuePropagation'> {
  /** The type of long press event being fired. */
  type: 'longpressstart' | 'longpressend' | 'longpress'
}

interface KeyboardEvents {
  /** Handler that is called when a key is pressed. */
  onKeyDown?: (e: KeyboardEvent$1) => void,
  /** Handler that is called when a key is released. */
  onKeyUp?: (e: KeyboardEvent$1) => void
}

interface FocusEvents<Target = Element> {
  /** Handler that is called when the element receives focus. */
  onFocus?: (e: FocusEvent$1<Target>) => void,
  /** Handler that is called when the element loses focus. */
  onBlur?: (e: FocusEvent$1<Target>) => void,
  /** Handler that is called when the element's focus status changes. */
  onFocusChange?: (isFocused: boolean) => void
}

interface PressEvents {
  /** Handler that is called when the press is released over the target. */
  onPress?: (e: PressEvent) => void,
  /** Handler that is called when a press interaction starts. */
  onPressStart?: (e: PressEvent) => void,
  /**
   * Handler that is called when a press interaction ends, either
   * over the target or when the pointer leaves the target.
   */
  onPressEnd?: (e: PressEvent) => void,
  /** Handler that is called when the press state changes. */
  onPressChange?: (isPressed: boolean) => void,
  /**
   * Handler that is called when a press is released over the target, regardless of
   * whether it started on the target or not.
   */
  onPressUp?: (e: PressEvent) => void
}

interface FocusableProps<Target = Element> extends FocusEvents<Target>, KeyboardEvents {
  /** Whether the element should receive focus on render. */
  autoFocus?: boolean
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface RefObject<T> {
  current: T
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface LabelableProps {
  /** The content to display as the label. */
  label?: ReactNode
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

type Orientation = 'horizontal' | 'vertical';

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

type Direction = 'ltr' | 'rtl';

/*
 * Copyright 2023 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

type Key$1 = string | number;

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface ToggleProps extends InputBase, Validation<boolean>, FocusableProps {
  /**
   * The label for the element.
   */
  children?: ReactNode,
  /**
   * Whether the element should be selected (uncontrolled).
   */
  defaultSelected?: boolean,
  /**
   * Whether the element should be selected (controlled).
   */
  isSelected?: boolean,
  /**
   * Handler that is called when the element's selection state changes.
   */
  onChange?: (isSelected: boolean) => void,
  /**
   * The value of the input element, used when submitting an HTML form. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefvalue).
   */
  value?: string
}

interface AriaToggleProps extends ToggleProps, FocusableDOMProps, AriaLabelingProps, AriaValidationProps, InputDOMProps {
  /**
   * Identifies the element (or elements) whose contents or presence are controlled by the current element.
   */
  'aria-controls'?: string
}

interface CheckboxProps$1 extends ToggleProps {
  /**
   * Indeterminism is presentational only.
   * The indeterminate visual representation remains regardless of user interaction.
   */
  isIndeterminate?: boolean
}

interface AriaCheckboxProps extends CheckboxProps$1, AriaToggleProps {}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



type MenuTriggerAction = 'focus' | 'input' | 'manual';

interface ComboBoxValidationValue {
  /** The selected key in the ComboBox. */
  selectedKey: Key$1,
  /** The value of the ComboBox input. */
  inputValue: string
}

interface ComboBoxProps$1<T> extends CollectionBase<T>, Omit<SingleSelection, 'disallowEmptySelection' | 'onSelectionChange'>, InputBase, TextInputBase, Validation<ComboBoxValidationValue>, FocusableProps<HTMLInputElement>, LabelableProps, HelpTextProps {
  /** The list of ComboBox items (uncontrolled). */
  defaultItems?: Iterable<T>,
  /** The list of ComboBox items (controlled). */
  items?: Iterable<T>,
  /** Method that is called when the open state of the menu changes. Returns the new open state and the action that caused the opening of the menu. */
  onOpenChange?: (isOpen: boolean, menuTrigger?: MenuTriggerAction) => void,
  /** Handler that is called when the selection changes. */
  onSelectionChange?: (key: Key$1 | null) => void,
  /** The value of the ComboBox input (controlled). */
  inputValue?: string,
  /** The default value of the ComboBox input (uncontrolled). */
  defaultInputValue?: string,
  /** Handler that is called when the ComboBox input value changes. */
  onInputChange?: (value: string) => void,
  /** Whether the ComboBox allows a non-item matching input value to be set. */
  allowsCustomValue?: boolean,
  // /**
  //  * Whether the Combobox should only suggest matching options or autocomplete the field with the nearest matching option.
  //  * @default 'suggest'
  //  */
  // completionMode?: 'suggest' | 'complete',
 /**
  * The interaction required to display the ComboBox menu.
  * @default 'input'
  */
  menuTrigger?: MenuTriggerAction
}

interface AriaComboBoxProps<T> extends ComboBoxProps$1<T>, DOMProps, InputDOMProps, AriaLabelingProps {
  /** Whether keyboard navigation is circular. */
  shouldFocusWrap?: boolean
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



type Placement = 'bottom' | 'bottom left' | 'bottom right' | 'bottom start' | 'bottom end' |
    'top' | 'top left' | 'top right' | 'top start' | 'top end' |
    'left' | 'left top' | 'left bottom' | 'start' | 'start top' | 'start bottom' |
    'right' | 'right top' | 'right bottom' | 'end' | 'end top' | 'end bottom';

interface PositionProps {
  /**
   * The placement of the element with respect to its anchor element.
   * @default 'bottom'
   */
  placement?: Placement,
  /**
   * The placement padding that should be applied between the element and its
   * surrounding container.
   * @default 12
   */
  containerPadding?: number,
  /**
   * The additional offset applied along the main axis between the element and its
   * anchor element.
   * @default 0
   */
  offset?: number,
  /**
   * The additional offset applied along the cross axis between the element and its
   * anchor element.
   * @default 0
   */
  crossOffset?: number,
  /**
   * Whether the element should flip its orientation (e.g. top to bottom or left to right) when
   * there is insufficient room for it to render completely.
   * @default true
   */
  shouldFlip?: boolean,
  // /**
  //  * The element that should be used as the bounding container when calculating container offset
  //  * or whether it should flip.
  //  */
  // boundaryElement?: Element,
  /** Whether the element is rendered. */
  isOpen?: boolean
}

interface OverlayTriggerProps$1 {
  /** Whether the overlay is open by default (controlled). */
  isOpen?: boolean,
  /** Whether the overlay is open by default (uncontrolled). */
  defaultOpen?: boolean,
  /** Handler that is called when the overlay's open state changes. */
  onOpenChange?: (isOpen: boolean) => void
}

interface OverlayTriggerState {
    /** Whether the overlay is currently open. */
    readonly isOpen: boolean;
    /** Sets whether the overlay is open. */
    setOpen(isOpen: boolean): void;
    /** Opens the overlay. */
    open(): void;
    /** Closes the overlay. */
    close(): void;
    /** Toggles the overlay's visibility. */
    toggle(): void;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface SelectProps$1<T> extends CollectionBase<T>, Omit<InputBase, 'isReadOnly'>, Validation<Key$1>, HelpTextProps, LabelableProps, TextInputBase, Omit<SingleSelection, 'disallowEmptySelection'>, FocusableProps {
  /** Sets the open state of the menu. */
  isOpen?: boolean,
  /** Sets the default open state of the menu. */
  defaultOpen?: boolean,
  /** Method that is called when the open state of the menu changes. */
  onOpenChange?: (isOpen: boolean) => void
}

interface AriaSelectProps<T> extends SelectProps$1<T>, DOMProps, AriaLabelingProps, FocusableDOMProps {
  /**
   * Describes the type of autocomplete functionality the input should provide if any. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefautocomplete).
   */
  autoComplete?: string,
  /**
   * The name of the input, used when submitting an HTML form.
   */
  name?: string
}

interface FocusState {
    /** Whether the collection is currently focused. */
    readonly isFocused: boolean;
    /** Sets whether the collection is focused. */
    setFocused(isFocused: boolean): void;
    /** The current focused key in the collection. */
    readonly focusedKey: Key$1;
    /** Whether the first or last child of the focused key should receive focus. */
    readonly childFocusStrategy: FocusStrategy;
    /** Sets the focused key, and optionally, whether the first or last child of that key should receive focus. */
    setFocusedKey(key: Key$1, child?: FocusStrategy): void;
}
interface MultipleSelectionState extends FocusState {
    /** The type of selection that is allowed in the collection. */
    readonly selectionMode: SelectionMode;
    /** The selection behavior for the collection. */
    readonly selectionBehavior: SelectionBehavior;
    /** Sets the selection behavior for the collection. */
    setSelectionBehavior(selectionBehavior: SelectionBehavior): void;
    /** Whether the collection allows empty selection. */
    readonly disallowEmptySelection: boolean;
    /** The currently selected keys in the collection. */
    readonly selectedKeys: Selection$1;
    /** Sets the selected keys in the collection. */
    setSelectedKeys(keys: Selection$1): void;
    /** The currently disabled keys in the collection. */
    readonly disabledKeys: Set<Key$1>;
    /** Whether `disabledKeys` applies to selection, actions, or both. */
    readonly disabledBehavior: DisabledBehavior;
}
interface MultipleSelectionManager extends FocusState {
    /** The type of selection that is allowed in the collection. */
    readonly selectionMode: SelectionMode;
    /** The selection behavior for the collection. */
    readonly selectionBehavior: SelectionBehavior;
    /** Whether the collection allows empty selection. */
    readonly disallowEmptySelection?: boolean;
    /** The currently selected keys in the collection. */
    readonly selectedKeys: Set<Key$1>;
    /** Whether the selection is empty. */
    readonly isEmpty: boolean;
    /** Whether all items in the collection are selected. */
    readonly isSelectAll: boolean;
    /** The first selected key in the collection. */
    readonly firstSelectedKey: Key$1 | null;
    /** The last selected key in the collection. */
    readonly lastSelectedKey: Key$1 | null;
    /** The currently disabled keys in the collection. */
    readonly disabledKeys: Set<Key$1>;
    /** Whether `disabledKeys` applies to selection, actions, or both. */
    readonly disabledBehavior: DisabledBehavior;
    /** Returns whether a key is selected. */
    isSelected(key: Key$1): boolean;
    /** Returns whether the current selection is equal to the given selection. */
    isSelectionEqual(selection: Set<Key$1>): boolean;
    /** Extends the selection to the given key. */
    extendSelection(toKey: Key$1): void;
    /** Toggles whether the given key is selected. */
    toggleSelection(key: Key$1): void;
    /** Replaces the selection with only the given key. */
    replaceSelection(key: Key$1): void;
    /** Replaces the selection with the given keys. */
    setSelectedKeys(keys: Iterable<Key$1>): void;
    /** Selects all items in the collection. */
    selectAll(): void;
    /** Removes all keys from the selection. */
    clearSelection(): void;
    /** Toggles between select all and an empty selection. */
    toggleSelectAll(): void;
    /**
     * Toggles, replaces, or extends selection to the given key depending
     * on the pointer event and collection's selection mode.
     */
    select(key: Key$1, e?: PressEvent | LongPressEvent | PointerEvent): void;
    /** Returns whether the given key can be selected. */
    canSelectItem(key: Key$1): boolean;
    /** Returns whether the given key is non-interactive, i.e. both selection and actions are disabled. */
    isDisabled(key: Key$1): boolean;
    /** Sets the selection behavior for the collection. */
    setSelectionBehavior(selectionBehavior: SelectionBehavior): void;
    /** Returns whether the given key is a hyperlink. */
    isLink(key: Key$1): boolean;
    /** Returns the props for the given item. */
    getItemProps(key: Key$1): any;
}
interface MultipleSelectionStateProps extends MultipleSelection {
    /** How multiple selection should behave in the collection. */
    selectionBehavior?: SelectionBehavior;
    /** Whether onSelectionChange should fire even if the new set of keys is the same as the last. */
    allowDuplicateSelectionEvents?: boolean;
    /** Whether `disabledKeys` applies to all interactions, or only selection. */
    disabledBehavior?: DisabledBehavior;
}
interface SelectionManagerOptions {
    allowsCellSelection?: boolean;
}
/**
 * An interface for reading and updating multiple selection state.
 */
declare class SelectionManager implements MultipleSelectionManager {
    constructor(collection: Collection<Node$1<unknown>>, state: MultipleSelectionState, options?: SelectionManagerOptions);
    /**
     * The type of selection that is allowed in the collection.
     */
    get selectionMode(): SelectionMode;
    /**
     * Whether the collection allows empty selection.
     */
    get disallowEmptySelection(): boolean;
    /**
     * The selection behavior for the collection.
     */
    get selectionBehavior(): SelectionBehavior;
    /**
     * Sets the selection behavior for the collection.
     */
    setSelectionBehavior(selectionBehavior: SelectionBehavior): void;
    /**
     * Whether the collection is currently focused.
     */
    get isFocused(): boolean;
    /**
     * Sets whether the collection is focused.
     */
    setFocused(isFocused: boolean): void;
    /**
     * The current focused key in the collection.
     */
    get focusedKey(): Key$1;
    /** Whether the first or last child of the focused key should receive focus. */
    get childFocusStrategy(): FocusStrategy;
    /**
     * Sets the focused key.
     */
    setFocusedKey(key: Key$1 | null, childFocusStrategy?: FocusStrategy): void;
    /**
     * The currently selected keys in the collection.
     */
    get selectedKeys(): Set<Key$1>;
    /**
     * The raw selection value for the collection.
     * Either 'all' for select all, or a set of keys.
     */
    get rawSelection(): Selection$1;
    /**
     * Returns whether a key is selected.
     */
    isSelected(key: Key$1): boolean;
    /**
     * Whether the selection is empty.
     */
    get isEmpty(): boolean;
    /**
     * Whether all items in the collection are selected.
     */
    get isSelectAll(): boolean;
    get firstSelectedKey(): Key$1 | null;
    get lastSelectedKey(): Key$1 | null;
    get disabledKeys(): Set<Key$1>;
    get disabledBehavior(): DisabledBehavior;
    /**
     * Extends the selection to the given key.
     */
    extendSelection(toKey: Key$1): void;
    /**
     * Toggles whether the given key is selected.
     */
    toggleSelection(key: Key$1): void;
    /**
     * Replaces the selection with only the given key.
     */
    replaceSelection(key: Key$1): void;
    /**
     * Replaces the selection with the given keys.
     */
    setSelectedKeys(keys: Iterable<Key$1>): void;
    /**
     * Selects all items in the collection.
     */
    selectAll(): void;
    /**
     * Removes all keys from the selection.
     */
    clearSelection(): void;
    /**
     * Toggles between select all and an empty selection.
     */
    toggleSelectAll(): void;
    select(key: Key$1, e?: PressEvent | LongPressEvent | PointerEvent): void;
    /**
     * Returns whether the current selection is equal to the given selection.
     */
    isSelectionEqual(selection: Set<Key$1>): boolean;
    canSelectItem(key: Key$1): boolean;
    isDisabled(key: Key$1): boolean;
    isLink(key: Key$1): boolean;
    getItemProps(key: Key$1): any;
}

interface ListProps<T> extends CollectionStateBase<T>, MultipleSelectionStateProps {
    /** Filter function to generate a filtered list of nodes. */
    filter?: (nodes: Iterable<Node$1<T>>) => Iterable<Node$1<T>>;
    /** @private */
    suppressTextValueWarning?: boolean;
}
interface ListState<T> {
    /** A collection of items in the list. */
    collection: Collection<Node$1<T>>;
    /** A set of items that are disabled. */
    disabledKeys: Set<Key$1>;
    /** A selection manager to read and update multiple selection state. */
    selectionManager: SelectionManager;
}

interface SelectStateOptions<T> extends Omit<SelectProps$1<T>, 'children'>, CollectionStateBase<T> {
}

type FilterFn = (textValue: string, inputValue: string) => boolean;
interface ComboBoxStateOptions<T> extends Omit<ComboBoxProps$1<T>, 'children'>, CollectionStateBase<T> {
    /** The filter function used to determine if a option should be included in the combo box list. */
    defaultFilter?: FilterFn;
    /** Whether the combo box allows the menu to be open when the collection is empty. */
    allowsEmptyCollection?: boolean;
    /** Whether the combo box menu should close on blur. */
    shouldCloseOnBlur?: boolean;
}

interface DraggableCollectionStateOptions extends DraggableCollectionProps {
    /** A collection of items. */
    collection: Collection<Node$1<unknown>>;
    /** An interface for reading and updating multiple selection state. */
    selectionManager: MultipleSelectionManager;
    /** Whether the drag events should be disabled. */
    isDisabled?: boolean;
}
interface DroppableCollectionStateOptions extends Omit<DroppableCollectionProps, 'onDropMove' | 'onDropActivate'> {
    /** A collection of items. */
    collection: Collection<Node$1<unknown>>;
    /** An interface for reading and updating multiple selection state. */
    selectionManager: MultipleSelectionManager;
    /** Whether the drop events should be disabled. */
    isDisabled?: boolean;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



type MenuTriggerType = 'press' | 'longPress';

interface MenuTriggerProps extends OverlayTriggerProps$1 {
  /**
   * How the menu is triggered.
   * @default 'press'
   */
  trigger?: MenuTriggerType
}

interface MenuProps$1<T> extends CollectionBase<T>, MultipleSelection {
  /** Where the focus should be set. */
  autoFocus?: boolean | FocusStrategy,
  /** Whether keyboard navigation is circular. */
  shouldFocusWrap?: boolean,
  /** Handler that is called when an item is selected. */
  onAction?: (key: Key$1) => void,
  /** Handler that is called when the menu should close after selecting an item. */
  onClose?: () => void
}

interface AriaMenuProps<T> extends MenuProps$1<T>, DOMProps, AriaLabelingProps {}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface RadioGroupProps$1 extends ValueBase<string|null, string>, InputBase, InputDOMProps, Validation<string | null>, LabelableProps, HelpTextProps, FocusEvents {
  /**
   * The axis the Radio Button(s) should align with.
   * @default 'vertical'
   */
  orientation?: Orientation
}

interface AriaRadioGroupProps extends RadioGroupProps$1, DOMProps, AriaLabelingProps, AriaValidationProps {}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface TextFieldProps extends InputBase, Validation<string>, HelpTextProps, FocusableProps, TextInputBase, ValueBase<string>, LabelableProps {}

interface AriaTextFieldProps extends TextFieldProps, AriaLabelingProps, FocusableDOMProps, TextInputDOMProps, AriaValidationProps {
  // https://www.w3.org/TR/wai-aria-1.2/#textbox
  /** Identifies the currently active element when DOM focus is on a composite widget, textbox, group, or application. */
  'aria-activedescendant'?: string,
  /**
   * Indicates whether inputting text could trigger display of one or more predictions of the user's intended value for an input and specifies how predictions would be
   * presented if they are made.
   */
  'aria-autocomplete'?: 'none' | 'inline' | 'list' | 'both',
  /** Indicates the availability and type of interactive popup element, such as menu or dialog, that can be triggered by an element. */
  'aria-haspopup'?: boolean | 'false' | 'true' | 'menu' | 'listbox' | 'tree' | 'grid' | 'dialog'
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface SearchFieldProps extends TextFieldProps {
  /** Handler that is called when the SearchField is submitted. */
  onSubmit?: (value: string) => void,

  /** Handler that is called when the clear button is pressed. */
  onClear?: () => void
}

interface AriaSearchFieldProps extends SearchFieldProps, AriaTextFieldProps {}

interface SliderProps$1<T = number | number[]> extends RangeInputBase<number>, ValueBase<T>, LabelableProps {
  /**
   * The orientation of the Slider.
   * @default 'horizontal'
   */
  orientation?: Orientation,
  /** Whether the whole Slider is disabled. */
  isDisabled?: boolean,
  /** Fired when the slider stops moving, due to being let go. */
  onChangeEnd?: (value: T) => void,
  // These are duplicated from ValueBase to define defaults for the docs.
  /**
   * The slider's minimum value.
   * @default 0
   */
  minValue?: number,
  /**
   * The slider's maximum value.
   * @default 100
   */
  maxValue?: number,
  /**
   * The slider's step value.
   * @default 1
   */
  step?: number
}

interface AriaSliderProps<T = number | number[]> extends SliderProps$1<T>, DOMProps, AriaLabelingProps {}

interface SliderStateOptions<T> extends SliderProps$1<T> {
    numberFormatter: Intl.NumberFormat;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface NumberFieldProps extends InputBase, Validation<number>, FocusableProps, TextInputBase, ValueBase<number>, RangeInputBase<number>, LabelableProps, HelpTextProps {
  /**
   * Formatting options for the value displayed in the number field.
   * This also affects what characters are allowed to be typed by the user.
   */
  formatOptions?: Intl.NumberFormatOptions
}

interface AriaNumberFieldProps extends NumberFieldProps, DOMProps, AriaLabelingProps, TextInputDOMEvents {
  /** A custom aria-label for the decrement button. If not provided, the localized string "Decrement" is used. */
  decrementAriaLabel?: string,
  /** A custom aria-label for the increment button. If not provided, the localized string "Increment" is used. */
  incrementAriaLabel?: string,
  /**
   * Enables or disables changing the value with scroll.
   */
  isWheelDisabled?: boolean
}

interface NumberFieldStateOptions extends NumberFieldProps {
    /**
     * The locale that should be used for parsing.
     * @default 'en-US'
     */
    locale: string;
}

declare let Item$2: <T>(props: ItemProps$1<T>) => JSX$1.Element;

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface TabListProps$1<T> extends CollectionBase<T>, Omit<SingleSelection, 'disallowEmptySelection'> {
  /**
   * Whether the TabList is disabled.
   * Shows that a selection exists, but is not available in that circumstance.
   */
  isDisabled?: boolean
}

interface AriaTabListBase$1 extends AriaLabelingProps {
  /**
   * Whether tabs are activated automatically on focus or manually.
   * @default 'automatic'
   */
  keyboardActivation?: 'automatic' | 'manual',
  /**
   * The orientation of the tabs.
   * @default 'horizontal'
   */
  orientation?: Orientation
}

interface AriaTabListProps$1<T> extends TabListProps$1<T>, AriaTabListBase$1, DOMProps, AriaLabelingProps {}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface TooltipTriggerProps extends OverlayTriggerProps$1 {
  /**
   * Whether the tooltip should be disabled, independent from the trigger.
   */
  isDisabled?: boolean,

  /**
   * The delay time for the tooltip to show up. [See guidelines](https://spectrum.adobe.com/page/tooltip/#Immediate-or-delayed-appearance).
   * @default 1500
   */
  delay?: number,

  /**
   * The delay time for the tooltip to close. [See guidelines](https://spectrum.adobe.com/page/tooltip/#Warmup-and-cooldown).
   * @default 500
   */
  closeDelay?: number,

  /**
   * By default, opens for both focus and hover. Can be made to open only for focus.
   */
  trigger?: 'focus'
}

interface TooltipProps$1 {
  isOpen?: boolean
}

interface AriaTooltipProps extends TooltipProps$1, DOMProps, AriaLabelingProps {}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface LinkProps$1 extends PressEvents, FocusableProps {}

interface AriaLinkProps extends LinkProps$1, LinkDOMProps, AriaLabelingProps { }

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface ButtonProps extends PressEvents, FocusableProps {
  /** Whether the button is disabled. */
  isDisabled?: boolean,
  /** The content to display in the button. */
  children?: ReactNode
}

interface ToggleButtonProps$1 extends ButtonProps {
  /** Whether the element should be selected (controlled). */
  isSelected?: boolean,
  /** Whether the element should be selected (uncontrolled). */
  defaultSelected?: boolean,
  /** Handler that is called when the element's selection state changes. */
  onChange?: (isSelected: boolean) => void
}

interface AriaButtonElementTypeProps<T extends ElementType = 'button'> {
  /**
   * The HTML element or React element used to render the button, e.g. 'div', 'a', or `RouterLink`.
   * @default 'button'
   */
  elementType?: T | JSXElementConstructor<any>
}

interface LinkButtonProps<T extends ElementType = 'button'> extends AriaButtonElementTypeProps<T> {
  /** A URL to link to if elementType="a". */
  href?: string,
  /** The target window for the link. */
  target?: string,
  /** The relationship between the linked resource and the current page. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/rel). */
  rel?: string
}

interface AriaBaseButtonProps extends FocusableDOMProps, AriaLabelingProps {
  /** Indicates whether the element, or another grouping element it controls, is currently expanded or collapsed. */
  'aria-expanded'?: boolean | 'true' | 'false',
  /** Indicates the availability and type of interactive popup element, such as menu or dialog, that can be triggered by an element. */
  'aria-haspopup'?: boolean | 'menu' | 'listbox' | 'tree' | 'grid' | 'dialog' | 'true' | 'false',
  /** Identifies the element (or elements) whose contents or presence are controlled by the current element. */
  'aria-controls'?: string,
  /** Indicates the current "pressed" state of toggle buttons. */
  'aria-pressed'?: boolean | 'true' | 'false' | 'mixed',
  /**
   * The behavior of the button when used in an HTML form.
   * @default 'button'
   */
  type?: 'button' | 'submit' | 'reset'
}

interface AriaButtonProps<T extends ElementType = 'button'> extends ButtonProps, LinkButtonProps<T>, AriaBaseButtonProps {}
interface AriaToggleButtonProps<T extends ElementType = 'button'> extends ToggleButtonProps$1, AriaBaseButtonProps, AriaButtonElementTypeProps<T> {}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface ListBoxProps$1<T> extends CollectionBase<T>, MultipleSelection, FocusEvents {
  /** Whether to auto focus the listbox or an option. */
  autoFocus?: boolean | FocusStrategy,
  /** Whether focus should wrap around when the end/start is reached. */
  shouldFocusWrap?: boolean
}

interface AriaListBoxPropsBase<T> extends ListBoxProps$1<T>, DOMProps, AriaLabelingProps {}
interface AriaListBoxProps<T> extends AriaListBoxPropsBase<T> {
  /**
   * An optional visual label for the listbox.
   */
  label?: ReactNode,
  /** How multiple selection should behave in the collection. */
  selectionBehavior?: SelectionBehavior,
  /**
   * Handler that is called when a user performs an action on an item. The exact user event depends on
   * the collection's `selectionBehavior` prop and the interaction modality.
   */
  onAction?: (key: Key$1) => void
}

interface SelectableItemStates {
    /** Whether the item is currently in a pressed state. */
    isPressed: boolean;
    /** Whether the item is currently selected. */
    isSelected: boolean;
    /** Whether the item is currently focused. */
    isFocused: boolean;
    /**
     * Whether the item is non-interactive, i.e. both selection and actions are disabled and the item may
     * not be focused. Dependent on `disabledKeys` and `disabledBehavior`.
     */
    isDisabled: boolean;
    /**
     * Whether the item may be selected, dependent on `selectionMode`, `disabledKeys`, and `disabledBehavior`.
     */
    allowsSelection: boolean;
    /**
     * Whether the item has an action, dependent on `onAction`, `disabledKeys`,
     * and `disabledBehavior`. It may also change depending on the current selection state
     * of the list (e.g. when selection is primary). This can be used to enable or disable hover
     * styles or other visual indications of interactivity.
     */
    hasAction: boolean;
}

interface AriaListBoxOptions<T> extends Omit<AriaListBoxProps<T>, 'children'> {
    /** Whether the listbox uses virtual scrolling. */
    isVirtualized?: boolean;
    /**
     * An optional keyboard delegate implementation for type to select,
     * to override the default.
     */
    keyboardDelegate?: KeyboardDelegate;
    /**
     * A delegate object that provides layout information for items in the collection.
     * By default this uses the DOM, but this can be overridden to implement things like
     * virtualized scrolling.
     */
    layoutDelegate?: LayoutDelegate;
    /**
     * Whether the listbox items should use virtual focus instead of being focused directly.
     */
    shouldUseVirtualFocus?: boolean;
    /** Whether selection should occur on press up instead of press down. */
    shouldSelectOnPressUp?: boolean;
    /** Whether options should be focused when the user hovers over them. */
    shouldFocusOnHover?: boolean;
    /**
     * The behavior of links in the collection.
     * - 'action': link behaves like onAction.
     * - 'selection': link follows selection interactions (e.g. if URL drives selection).
     * - 'override': links override all other interactions (link items are not selectable).
     * @default 'override'
     */
    linkBehavior?: 'action' | 'selection' | 'override';
}
interface OptionAria extends SelectableItemStates {
    /** Props for the option element. */
    optionProps: DOMAttributes;
    /** Props for the main text element inside the option. */
    labelProps: DOMAttributes;
    /** Props for the description text element inside the option, if any. */
    descriptionProps: DOMAttributes;
    /** Whether the option is currently focused. */
    isFocused: boolean;
    /** Whether the option is keyboard focused. */
    isFocusVisible: boolean;
}

interface AriaComboBoxOptions<T> extends Omit<AriaComboBoxProps<T>, 'children'> {
    /** The ref for the input element. */
    inputRef: RefObject<HTMLInputElement | null>;
    /** The ref for the list box popover. */
    popoverRef: RefObject<Element | null>;
    /** The ref for the list box. */
    listBoxRef: RefObject<HTMLElement | null>;
    /** The ref for the optional list box popup trigger button.  */
    buttonRef?: RefObject<Element | null>;
    /** An optional keyboard delegate implementation, to override the default. */
    keyboardDelegate?: KeyboardDelegate;
    /**
     * A delegate object that provides layout information for items in the collection.
     * By default this uses the DOM, but this can be overridden to implement things like
     * virtualized scrolling.
     */
    layoutDelegate?: LayoutDelegate;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface AriaDialogProps extends DOMProps, AriaLabelingProps {
  /**
   * The accessibility role for the dialog.
   * @default 'dialog'
   */
  role?: 'dialog' | 'alertdialog'
}

interface DropOptions {
    /** A ref for the droppable element. */
    ref: RefObject<HTMLElement | null>;
    /**
     * A function returning the drop operation to be performed when items matching the given types are dropped
     * on the drop target.
     */
    getDropOperation?: (types: DragTypes, allowedOperations: DropOperation[]) => DropOperation;
    /** A function that returns the drop operation for a specific point within the target. */
    getDropOperationForPoint?: (types: DragTypes, allowedOperations: DropOperation[], x: number, y: number) => DropOperation;
    /** Handler that is called when a valid drag enters the drop target. */
    onDropEnter?: (e: DropEnterEvent) => void;
    /** Handler that is called when a valid drag is moved within the drop target. */
    onDropMove?: (e: DropMoveEvent) => void;
    /**
     * Handler that is called after a valid drag is held over the drop target for a period of time.
     * This typically opens the item so that the user can drop within it.
     * @private
     */
    onDropActivate?: (e: DropActivateEvent) => void;
    /** Handler that is called when a valid drag exits the drop target. */
    onDropExit?: (e: DropExitEvent) => void;
    /** Handler that is called when a valid drag is dropped on the drop target. */
    onDrop?: (e: DropEvent) => void;
    /**
     * Whether the item has an explicit focusable drop affordance to initiate accessible drag and drop mode.
     * If true, the dropProps will omit these event handlers, and they will be applied to dropButtonProps instead.
     */
    hasDropButton?: boolean;
    /**
     * Whether the drop target is disabled. If true, the drop target will not accept any drops.
     */
    isDisabled?: boolean;
}
interface DroppableCollectionOptions extends DroppableCollectionProps {
    /** A delegate object that implements behavior for keyboard focus movement. */
    keyboardDelegate: KeyboardDelegate;
    /** A delegate object that provides drop targets for pointer coordinates within the collection. */
    dropTargetDelegate: DropTargetDelegate;
}
interface DraggableCollectionOptions {
}
interface ClipboardProps {
    /** A function that returns the items to copy. */
    getItems?: () => DragItem[];
    /** Handler that is called when the user triggers a copy interaction. */
    onCopy?: () => void;
    /** Handler that is called when the user triggers a cut interaction. */
    onCut?: () => void;
    /** Handler that is called when the user triggers a paste interaction. */
    onPaste?: (items: DropItem[]) => void;
    /** Whether the clipboard is disabled. */
    isDisabled?: boolean;
}

declare global {
    namespace FormatjsIntl {
        interface Message {
        }
        interface IntlConfig {
        }
        interface Formats {
        }
    }
}

type LocalizedStrings = {
    [lang: string]: {
        [key: string]: string;
    };
};

interface NumberFormatOptions extends Intl.NumberFormatOptions {
    /** Overrides default numbering system for the current locale. */
    numberingSystem?: string;
}

interface Locale {
    /** The [BCP47](https://www.ietf.org/rfc/bcp/bcp47.txt) language code for the locale. */
    locale: string;
    /** The writing direction for the locale. */
    direction: Direction;
}
interface I18nProviderProps$1 {
    /** Contents that should have the locale applied. */
    children: ReactNode;
    /** The locale to apply to the children. */
    locale?: string;
}
interface DateFormatterOptions extends Intl.DateTimeFormatOptions {
    calendar?: string;
}
/**
 * Provides localized date formatting for the current locale. Automatically updates when the locale changes,
 * and handles caching of the date formatter for performance.
 * @param options - Formatting options.
 */
declare function useDateFormatter(options?: DateFormatterOptions): DateFormatter;
/**
 * Provides localized number formatting for the current locale. Automatically updates when the locale changes,
 * and handles caching of the number formatter for performance.
 * @param options - Formatting options.
 */
declare function useNumberFormatter(options?: NumberFormatOptions): Intl.NumberFormat;
/**
 * Provides localized string collation for the current locale. Automatically updates when the locale changes,
 * and handles caching of the collator for performance.
 * @param options - Collator options.
 */
declare function useCollator(options?: Intl.CollatorOptions): Intl.Collator;

interface PressProps extends PressEvents {
    /** Whether the target is in a controlled press state (e.g. an overlay it triggers is open). */
    isPressed?: boolean;
    /** Whether the press events should be disabled. */
    isDisabled?: boolean;
    /** Whether the target should not receive focus on press. */
    preventFocusOnPress?: boolean;
    /**
     * Whether press events should be canceled when the pointer leaves the target while pressed.
     * By default, this is `false`, which means if the pointer returns back over the target while
     * still pressed, onPressStart will be fired again. If set to `true`, the press is canceled
     * when the pointer leaves the target and onPressStart will not be fired if the pointer returns.
     */
    shouldCancelOnPointerExit?: boolean;
    /** Whether text selection should be enabled on the pressable element. */
    allowTextSelectionOnPress?: boolean;
}
interface KeyboardProps extends KeyboardEvents {
    /** Whether the keyboard events should be disabled. */
    isDisabled?: boolean;
}

interface GridListProps$1<T> extends CollectionBase<T>, MultipleSelection {
    /**
     * Handler that is called when a user performs an action on an item. The exact user event depends on
     * the collection's `selectionBehavior` prop and the interaction modality.
     */
    onAction?: (key: Key$1) => void;
    /** Whether `disabledKeys` applies to all interactions, or only selection. */
    disabledBehavior?: DisabledBehavior;
}
interface AriaGridListProps<T> extends GridListProps$1<T>, DOMProps, AriaLabelingProps {
    /**
     * Whether keyboard navigation to focusable elements within grid list items is
     * via the left/right arrow keys or the tab key.
     * @default 'arrow'
     */
    keyboardNavigationBehavior?: 'arrow' | 'tab';
}
interface AriaGridListOptions<T> extends Omit<AriaGridListProps<T>, 'children'> {
    /** Whether the list uses virtual scrolling. */
    isVirtualized?: boolean;
    /**
     * An optional keyboard delegate implementation for type to select,
     * to override the default.
     */
    keyboardDelegate?: KeyboardDelegate;
    /**
     * A delegate object that provides layout information for items in the collection.
     * By default this uses the DOM, but this can be overridden to implement things like
     * virtualized scrolling.
     */
    layoutDelegate?: LayoutDelegate;
    /**
     * Whether focus should wrap around when the end/start is reached.
     * @default false
     */
    shouldFocusWrap?: boolean;
    /**
     * The behavior of links in the collection.
     * - 'action': link behaves like onAction.
     * - 'selection': link follows selection interactions (e.g. if URL drives selection).
     * - 'override': links override all other interactions (link items are not selectable).
     * @default 'action'
     */
    linkBehavior?: 'action' | 'selection' | 'override';
}
interface GridListItemAria extends SelectableItemStates {
    /** Props for the list row element. */
    rowProps: DOMAttributes;
    /** Props for the grid cell element within the list row. */
    gridCellProps: DOMAttributes;
    /** Props for the list item description element, if any. */
    descriptionProps: DOMAttributes;
}

interface AriaLinkOptions extends AriaLinkProps {
    /** Whether the link is disabled. */
    isDisabled?: boolean;
    /**
     * The HTML element used to render the link, e.g. 'a', or 'span'.
     * @default 'a'
     */
    elementType?: string;
}

interface AriaPositionProps extends PositionProps {
    /**
     * Cross size of the overlay arrow in pixels.
     * @default 0
     */
    arrowSize?: number;
    /**
     * Element that that serves as the positioning boundary.
     * @default document.body
     */
    boundaryElement?: Element;
    /**
     * The ref for the element which the overlay positions itself with respect to.
     */
    targetRef: RefObject<Element | null>;
    /**
     * The ref for the overlay element.
     */
    overlayRef: RefObject<Element | null>;
    /**
     * A ref for the scrollable region within the overlay.
     * @default overlayRef
     */
    scrollRef?: RefObject<Element | null>;
    /**
     * Whether the overlay should update its position automatically.
     * @default true
     */
    shouldUpdatePosition?: boolean;
    /** Handler that is called when the overlay should close. */
    onClose?: () => void;
    /**
     * The maxHeight specified for the overlay element.
     * By default, it will take all space up to the current viewport height.
     */
    maxHeight?: number;
    /**
     * The minimum distance the arrow's edge should be from the edge of the overlay element.
     * @default 0
     */
    arrowBoundaryOffset?: number;
}
interface AriaOverlayProps {
    /** Whether the overlay is currently open. */
    isOpen?: boolean;
    /** Handler that is called when the overlay should close. */
    onClose?: () => void;
    /**
     * Whether to close the overlay when the user interacts outside it.
     * @default false
     */
    isDismissable?: boolean;
    /** Whether the overlay should close when focus is lost or moves outside it. */
    shouldCloseOnBlur?: boolean;
    /**
     * Whether pressing the escape key to close the overlay should be disabled.
     * @default false
     */
    isKeyboardDismissDisabled?: boolean;
    /**
     * When user interacts with the argument element outside of the overlay ref,
     * return true if onClose should be called.  This gives you a chance to filter
     * out interaction with elements that should not dismiss the overlay.
     * By default, onClose will always be called on interaction outside the overlay ref.
     */
    shouldCloseOnInteractOutside?: (element: Element) => boolean;
}
interface OverlayTriggerProps {
    /** Type of overlay that is opened by the trigger. */
    type: 'dialog' | 'menu' | 'listbox' | 'tree' | 'grid';
}
interface OverlayTriggerAria {
    /** Props for the trigger element. */
    triggerProps: AriaButtonProps;
    /** Props for the overlay container element. */
    overlayProps: DOMProps;
}
/**
 * Handles the behavior and accessibility for an overlay trigger, e.g. a button
 * that opens a popover, menu, or other overlay that is positioned relative to the trigger.
 */
declare function useOverlayTrigger(props: OverlayTriggerProps, state: OverlayTriggerState, ref?: RefObject<Element | null>): OverlayTriggerAria;
interface AriaPopoverProps extends Omit<AriaPositionProps, 'isOpen' | 'onClose' | 'targetRef' | 'overlayRef'> {
    /**
     * The ref for the element which the popover positions itself with respect to.
     */
    triggerRef: RefObject<Element | null>;
    /**
     * The ref for the popover element.
     */
    popoverRef: RefObject<Element | null>;
    /**
     * Whether the popover is non-modal, i.e. elements outside the popover may be
     * interacted with by assistive technologies.
     *
     * Most popovers should not use this option as it may negatively impact the screen
     * reader experience. Only use with components such as combobox, which are designed
     * to handle this situation carefully.
     */
    isNonModal?: boolean;
    /**
     * Whether pressing the escape key to close the popover should be disabled.
     *
     * Most popovers should not use this option. When set to true, an alternative
     * way to close the popover with a keyboard must be provided.
     *
     * @default false
     */
    isKeyboardDismissDisabled?: boolean;
    /**
     * When user interacts with the argument element outside of the popover ref,
     * return true if onClose should be called. This gives you a chance to filter
     * out interaction with elements that should not dismiss the popover.
     * By default, onClose will always be called on interaction outside the popover ref.
     */
    shouldCloseOnInteractOutside?: (element: Element) => boolean;
}
interface OverlayProps {
    /**
     * The container element in which the overlay portal will be placed.
     * @default document.body
     */
    portalContainer?: Element;
    /** The overlay to render in the portal. */
    children: ReactNode;
    /**
     * Disables default focus management for the overlay, including containment and restoration.
     * This option should be used very carefully. When focus management is disabled, you must
     * implement focus containment and restoration to ensure the overlay is keyboard accessible.
     */
    disableFocusManagement?: boolean;
    /**
     * Whether the overlay is currently performing an exit animation. When true,
     * focus is allowed to move outside.
     */
    isExiting?: boolean;
}
interface AriaModalOverlayProps extends Pick<AriaOverlayProps, 'shouldCloseOnInteractOutside'> {
    /**
     * Whether to close the modal when the user interacts outside it.
     * @default false
     */
    isDismissable?: boolean;
    /**
     * Whether pressing the escape key to close the modal should be disabled.
     * @default false
     */
    isKeyboardDismissDisabled?: boolean;
}

interface AriaMenuTriggerProps {
    /** The type of menu that the menu trigger opens. */
    type?: 'menu' | 'listbox';
    /** Whether menu trigger is disabled. */
    isDisabled?: boolean;
    /** How menu is triggered. */
    trigger?: MenuTriggerType;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface ProgressBaseProps {
  /**
   * The current value (controlled).
   * @default 0
   */
  value?: number,
  /**
   * The smallest value allowed for the input.
   * @default 0
   */
  minValue?: number,
  /**
   * The largest value allowed for the input.
   * @default 100
   */
  maxValue?: number
}

interface ProgressBarBaseProps extends ProgressBaseProps {
  /** The content to display as the label. */
  label?: ReactNode,
  /**
   * The display format of the value label.
   * @default {style: 'percent'}
   */
  formatOptions?: Intl.NumberFormatOptions,
  /** The content to display as the value's label (e.g. 1 of 4). */
  valueLabel?: ReactNode
}

interface ProgressBarProps$1 extends ProgressBarBaseProps {
  /**
   * Whether presentation is indeterminate when progress isn't known.
   */
  isIndeterminate?: boolean
}

interface AriaProgressBarProps extends ProgressBarProps$1, DOMProps, AriaLabelingProps {}

interface AriaSelectOptions<T> extends Omit<AriaSelectProps<T>, 'children'> {
    /**
     * An optional keyboard delegate implementation for type to select,
     * to override the default.
     */
    keyboardDelegate?: KeyboardDelegate;
}

interface SeparatorProps$1 extends DOMProps, AriaLabelingProps {
    /**
     * The orientation of the separator.
     * @default 'horizontal'
     */
    orientation?: Orientation;
    /** The HTML element type that will be used to render the separator. */
    elementType?: string;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface SwitchBase extends InputBase, FocusableProps {
  /**
   * The content to render as the Switch's label.
   */
  children?: ReactNode,
  /**
   * Whether the Switch should be selected (uncontrolled).
   */
  defaultSelected?: boolean,
  /**
   * Whether the Switch should be selected (controlled).
   */
  isSelected?: boolean,
  /**
   * Handler that is called when the Switch's selection state changes.
   */
  onChange?: (isSelected: boolean) => void,
  /**
   * The value of the input element, used when submitting an HTML form. See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#htmlattrdefvalue).
   */
  value?: string
}
interface SwitchProps$1 extends SwitchBase {}
interface AriaSwitchBase extends SwitchBase, FocusableDOMProps, InputDOMProps, AriaLabelingProps {
  /**
   * Identifies the element (or elements) whose contents or presence are controlled by the current element.
   */
  'aria-controls'?: string
}
interface AriaSwitchProps extends SwitchProps$1, AriaSwitchBase {}

interface AriaTagGroupProps<T> extends CollectionBase<T>, MultipleSelection, DOMProps, LabelableProps, AriaLabelingProps, Omit<HelpTextProps, 'errorMessage'> {
    /** How multiple selection should behave in the collection. */
    selectionBehavior?: SelectionBehavior;
    /** Handler that is called when a user deletes a tag.  */
    onRemove?: (keys: Set<Key$1>) => void;
    /** An error message for the field. */
    errorMessage?: ReactNode;
}

/**
 * A map of HTML element names and their interface types.
 * For example `'a'` -> `HTMLAnchorElement`.
 */
type IntrinsicHTMLElements = {
    [K in keyof IntrinsicHTMLAttributes]: IntrinsicHTMLAttributes[K] extends HTMLAttributes<infer T> ? T : never;
};
/**
 * A map of HTML element names and their attribute interface types.
 * For example `'a'` -> `AnchorHTMLAttributes<HTMLAnchorElement>`.
 */
type IntrinsicHTMLAttributes = JSX$1.IntrinsicElements;
/**
 * The intrinsic HTML element names that `useTextField` supports; e.g. `input`,
 * `textarea`.
 */
type TextFieldIntrinsicElements = keyof Pick<IntrinsicHTMLElements, 'input' | 'textarea'>;
interface AriaTextFieldOptions<T extends TextFieldIntrinsicElements> extends AriaTextFieldProps {
    /**
     * The HTML element used to render the input, e.g. 'input', or 'textarea'.
     * It determines whether certain HTML attributes will be included in `inputProps`.
     * For example, [`type`](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input#attr-type).
     * @default 'input'
     */
    inputElementType?: T;
    /**
     * Controls whether inputted text is automatically capitalized and, if so, in what manner.
     * See [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/autocapitalize).
     */
    autoCapitalize?: 'off' | 'none' | 'on' | 'sentences' | 'words' | 'characters';
}

interface ResponsiveArray<Length extends number, Value> extends ReadonlyArray<Value> {
    0: Value;
    length: Length;
}

declare const createLayerSprinkles: (layer: string) => ((props: {
    paddingBlockStart?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingInlineEnd?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingBlockEnd?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingInlineStart?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginBlockStart?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginInlineEnd?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginBlockEnd?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginInlineStart?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    gap?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingLeft?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingRight?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingTop?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingBottom?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    padding?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingX?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    paddingY?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginLeft?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginRight?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginTop?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginBottom?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    margin?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginX?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
    marginY?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "3xl" | "4xl" | "5xl" | "6xl" | "7xl" | "8xl" | "9xl" | null>;
} & {
    borderStartStartRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
    borderStartEndRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
    borderEndStartRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
    borderEndEndRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
    borderRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
    borderTopRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
    borderBottomRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
    borderLeftRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
    borderRightRadius?: ("none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | {
        mobile?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        tablet?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
        desktop?: "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "xs" | "sm" | "md" | "lg" | "xl" | "2xl" | "full" | null>;
} & {
    boxShadow?: "medium" | "low" | undefined;
} & {
    opacity?: "none" | "medium" | "low" | "high" | undefined;
} & {
    typography?: "heading.h6.regular" | "heading.h6.medium" | "heading.h6.semibold" | "heading.h5.regular" | "heading.h5.medium" | "heading.h5.semibold" | "heading.h4.regular" | "heading.h4.medium" | "heading.h4.semibold" | "heading.h3.regular" | "heading.h3.medium" | "heading.h3.semibold" | "heading.h2.regular" | "heading.h2.medium" | "heading.h2.semibold" | "heading.h1.regular" | "heading.h1.medium" | "heading.h1.semibold" | "label.sm.regular" | "label.sm.medium" | "label.sm.semibold" | "label.md.regular" | "label.md.medium" | "label.md.semibold" | "label.lg.regular" | "label.lg.medium" | "label.lg.semibold" | "body.sm.regular" | "body.sm.medium" | "body.sm.semibold" | "body.md.regular" | "body.md.medium" | "body.md.semibold" | "body.lg.regular" | "body.lg.medium" | "body.lg.semibold" | undefined;
} & {
    color?: "text.primary" | "text.secondary" | "text.helper" | "text.placeholder" | "text.disabled" | "text.inverse" | "icon.primary" | "icon.secondary" | "icon.disabled" | "icon.inverse" | "icon.interactive" | "interactive.primary" | "interactive.hovered" | "interactive.active" | "interactive.visited" | "interactive.disabled" | "support.error.subtler" | "support.error.subtle" | "support.error.medium" | "support.error.strong" | "support.success.subtler" | "support.success.subtle" | "support.success.medium" | "support.success.strong" | "support.warning.subtler" | "support.warning.subtle" | "support.warning.medium" | "support.warning.strong" | "support.info.subtler" | "support.info.subtle" | "support.info.medium" | "support.info.strong" | "focused.primary" | "focused.inset" | undefined;
    backgroundColor?: "background.primary.subtle" | "background.primary.medium" | "background.primary.strong" | "background.secondary.subtle" | "background.secondary.medium" | "background.secondary.strong" | "background.inverse.subtle" | "background.inverse.medium" | "background.inverse.strong" | "background.overlay.subtle" | "background.overlay.medium" | "positive.subtle" | "positive.medium" | "positive.strong" | "positive.interactive" | "support.error.subtler" | "support.error.subtle" | "support.error.medium" | "support.error.strong" | "support.success.subtler" | "support.success.subtle" | "support.success.medium" | "support.success.strong" | "support.warning.subtler" | "support.warning.subtle" | "support.warning.medium" | "support.warning.strong" | "support.info.subtler" | "support.info.subtle" | "support.info.medium" | "support.info.strong" | "focused.primary" | "focused.inset" | undefined;
    borderColor?: "transparent" | "support.error.subtler" | "support.error.subtle" | "support.error.medium" | "support.error.strong" | "support.success.subtler" | "support.success.subtle" | "support.success.medium" | "support.success.strong" | "support.warning.subtler" | "support.warning.subtle" | "support.warning.medium" | "support.warning.strong" | "support.info.subtler" | "support.info.subtle" | "support.info.medium" | "support.info.strong" | "focused.primary" | "focused.inset" | "border.subtle" | "border.medium" | "border.strong" | "border.inverse" | "border.interactive" | "border.disabled" | undefined;
    outlineColor?: "transparent" | "support.error.subtler" | "support.error.subtle" | "support.error.medium" | "support.error.strong" | "support.success.subtler" | "support.success.subtle" | "support.success.medium" | "support.success.strong" | "support.warning.subtler" | "support.warning.subtle" | "support.warning.medium" | "support.warning.strong" | "support.info.subtler" | "support.info.subtle" | "support.info.medium" | "support.info.strong" | "focused.primary" | "focused.inset" | "border.subtle" | "border.medium" | "border.strong" | "border.inverse" | "border.interactive" | "border.disabled" | undefined;
} & {
    position?: ("fixed" | "absolute" | "relative" | {
        mobile?: "fixed" | "absolute" | "relative" | undefined;
        tablet?: "fixed" | "absolute" | "relative" | undefined;
        desktop?: "fixed" | "absolute" | "relative" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "fixed" | "absolute" | "relative" | null>;
    display?: ("none" | "block" | "flex" | "grid" | "inline-flex" | {
        mobile?: "none" | "block" | "flex" | "grid" | "inline-flex" | undefined;
        tablet?: "none" | "block" | "flex" | "grid" | "inline-flex" | undefined;
        desktop?: "none" | "block" | "flex" | "grid" | "inline-flex" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "none" | "block" | "flex" | "grid" | "inline-flex" | null>;
    flexDirection?: ("column" | "column-reverse" | "row" | "row-reverse" | {
        mobile?: "column" | "column-reverse" | "row" | "row-reverse" | undefined;
        tablet?: "column" | "column-reverse" | "row" | "row-reverse" | undefined;
        desktop?: "column" | "column-reverse" | "row" | "row-reverse" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "column" | "column-reverse" | "row" | "row-reverse" | null>;
    alignItems?: ("stretch" | "center" | "flex-end" | "flex-start" | {
        mobile?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        tablet?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        desktop?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "stretch" | "center" | "flex-end" | "flex-start" | null>;
    alignSelf?: ("stretch" | "center" | "flex-end" | "flex-start" | {
        mobile?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        tablet?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        desktop?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "stretch" | "center" | "flex-end" | "flex-start" | null>;
    flex?: (number | {
        mobile?: number | undefined;
        tablet?: number | undefined;
        desktop?: number | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, number | null>;
    justifyContent?: ("space-between" | "stretch" | "center" | "flex-end" | "flex-start" | {
        mobile?: "space-between" | "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        tablet?: "space-between" | "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        desktop?: "space-between" | "stretch" | "center" | "flex-end" | "flex-start" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "space-between" | "stretch" | "center" | "flex-end" | "flex-start" | null>;
    justifySelf?: ("stretch" | "center" | "flex-end" | "flex-start" | {
        mobile?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        tablet?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        desktop?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "stretch" | "center" | "flex-end" | "flex-start" | null>;
    placeItems?: ("stretch" | "center" | "flex-end" | "flex-start" | {
        mobile?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        tablet?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
        desktop?: "stretch" | "center" | "flex-end" | "flex-start" | undefined;
    } | undefined) | ResponsiveArray<1 | 2 | 3, "stretch" | "center" | "flex-end" | "flex-start" | null>;
} & {
    transition?: "transition.medium" | undefined;
    transitionProperty?: "opacity" | "transform" | "background-color" | undefined;
} & {
    readonly flexWrap?: "nowrap" | "wrap" | undefined;
    readonly insetBlockStart?: 0 | undefined;
    readonly insetBlockEnd?: 0 | undefined;
    readonly insetInlineStart?: 0 | undefined;
    readonly insetInlineEnd?: 0 | undefined;
    readonly flexShrink?: 0 | undefined;
    readonly flexGrow?: 0 | 1 | undefined;
    readonly width?: "full" | undefined;
    readonly height?: "full" | undefined;
    readonly cursor?: "pointer" | undefined;
    inset?: 0 | undefined;
} & {
    outline?: "none" | undefined;
    outlineStyle?: "solid" | undefined;
    outlineWidth?: number | undefined;
    outlineOffset?: number | undefined;
    border?: "none" | undefined;
    borderStyle?: "dashed" | "solid" | undefined;
    borderWidth?: number | undefined;
}) => string) & {
    properties: Set<"alignItems" | "alignSelf" | "backgroundColor" | "borderEndEndRadius" | "borderEndStartRadius" | "borderStartEndRadius" | "borderStartStartRadius" | "boxShadow" | "color" | "cursor" | "display" | "flexDirection" | "flexGrow" | "flexShrink" | "flexWrap" | "height" | "insetBlockEnd" | "insetBlockStart" | "insetInlineEnd" | "insetInlineStart" | "justifyContent" | "justifySelf" | "marginBlockEnd" | "marginBlockStart" | "marginBottom" | "marginInlineEnd" | "marginInlineStart" | "marginLeft" | "marginRight" | "marginTop" | "opacity" | "outlineColor" | "outlineOffset" | "outlineStyle" | "outlineWidth" | "paddingBlockEnd" | "paddingBlockStart" | "paddingBottom" | "paddingInlineEnd" | "paddingInlineStart" | "paddingLeft" | "paddingRight" | "paddingTop" | "position" | "transitionProperty" | "width" | "border" | "borderColor" | "borderRadius" | "borderStyle" | "borderWidth" | "flex" | "gap" | "inset" | "margin" | "outline" | "padding" | "placeItems" | "transition" | "typography" | "paddingX" | "paddingY" | "marginX" | "marginY" | "borderTopRadius" | "borderBottomRadius" | "borderLeftRadius" | "borderRightRadius">;
};
declare type BaseSprinkles = ReturnType<typeof createLayerSprinkles>;

interface Theme {
  elevation: Elevation;
  opacity: Opacity;
  rounded: Rounded;
  spacing: Spacing;
  color: Color$1;
  typography: Typography;
}
interface Typography {
  heading: Heading;
  label: Label;
  body: Label;
}
interface Label {
  sm: H6;
  md: H6;
  lg: H6;
}
interface Heading {
  h6: H6;
  h5: H6;
  h4: H6;
  h3: H6;
  h2: H6;
  h1: H6;
}
interface H6 {
  regular: Regular;
  medium: Regular;
  semibold: Regular;
}
interface Regular {
  font: string;
  letterSpacing: string;
}
interface Color$1 {
  support: Support;
  focused: Focused;
  background: Background;
  text: Text$2;
  icon: Icon;
  border: Border;
  interactive: Interactive;
  positive: Positive;
}
interface Positive {
  subtle: string;
  medium: string;
  strong: string;
  interactive: string;
}
interface Interactive {
  primary: string;
  hovered: string;
  active: string;
  visited: string;
  disabled: string;
}
interface Border {
  subtle: string;
  medium: string;
  strong: string;
  inverse: string;
  interactive: string;
  disabled: string;
}
interface Icon {
  primary: string;
  secondary: string;
  inverse: string;
  interactive: string;
  disabled: string;
}
interface Text$2 {
  primary: string;
  secondary: string;
  helper: string;
  placeholder: string;
  inverse: string;
  disabled: string;
}
interface Background {
  primary: Primary;
  inverse: Primary;
  secondary: Primary;
  overlay: Overlay;
}
interface Overlay {
  subtle: string;
  medium: string;
  interactive: string;
}
interface Primary {
  subtle: string;
  medium: string;
  strong: string;
}
interface Focused {
  primary: string;
  inset: string;
}
interface Support {
  error: Error$1;
  success: Error$1;
  warning: Error$1;
  info: Error$1;
}
interface Error$1 {
  subtler: string;
  subtle: string;
  medium: string;
  strong: string;
}
interface Spacing {
  xs: string;
  sm: string;
  md: string;
  lg: string;
  xl: string;
  '2xl': string;
  '3xl': string;
  '4xl': string;
  '5xl': string;
  '6xl': string;
  '7xl': string;
  '8xl': string;
  '9xl': string;
}
interface Rounded {
  xs: string;
  sm: string;
  md: string;
  lg: string;
  xl: string;
  '2xl': string;
  full: string;
}
interface Opacity {
  none: string;
  low: string;
  medium: string;
  high: string;
}
interface Elevation {
  low: string;
  medium: string;
}

interface Sprinkles extends BaseSprinkles {
    base: BaseSprinkles;
    variants: BaseSprinkles;
    compoundVariants: BaseSprinkles;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



interface TabListProps<T> extends CollectionBase<T>, Omit<SingleSelection, 'disallowEmptySelection'> {
  /**
   * Whether the TabList is disabled.
   * Shows that a selection exists, but is not available in that circumstance.
   */
  isDisabled?: boolean
}

interface AriaTabListBase extends AriaLabelingProps {
  /**
   * Whether tabs are activated automatically on focus or manually.
   * @default 'automatic'
   */
  keyboardActivation?: 'automatic' | 'manual',
  /**
   * The orientation of the tabs.
   * @default 'horizontal'
   */
  orientation?: Orientation
}

interface AriaTabListProps<T> extends TabListProps<T>, AriaTabListBase, DOMProps, AriaLabelingProps {}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



/** A list of supported color formats. */
type ColorFormat = 'hex' | 'hexa' | 'rgb' | 'rgba' | 'hsl' | 'hsla' | 'hsb' | 'hsba';

type ColorSpace = 'rgb' | 'hsl' | 'hsb';

/** A list of color channels. */
type ColorChannel = 'hue' | 'saturation' | 'brightness' | 'lightness' | 'red' | 'green' | 'blue' | 'alpha';

type ColorAxes = {xChannel: ColorChannel, yChannel: ColorChannel, zChannel: ColorChannel};

interface ColorChannelRange {
  /** The minimum value of the color channel. */
  minValue: number,
  /** The maximum value of the color channel. */
  maxValue: number,
  /** The step value of the color channel, used when incrementing and decrementing. */
  step: number,
  /** The page step value of the color channel, used when incrementing and decrementing. */
  pageSize: number
}

/** Represents a color value. */
interface Color {
  /** Converts the color to the given color format, and returns a new Color object. */
  toFormat(format: ColorFormat): Color,
  /** Converts the color to a string in the given format. */
  toString(format?: ColorFormat | 'css'): string,
  /** Returns a duplicate of the color value. */
  clone(): Color,
  /** Converts the color to hex, and returns an integer representation. */
  toHexInt(): number,
  /**
   * Returns the numeric value for a given channel.
   * Throws an error if the channel is unsupported in the current color format.
   */
  getChannelValue(channel: ColorChannel): number,
  /**
   * Sets the numeric value for a given channel, and returns a new Color object.
   * Throws an error if the channel is unsupported in the current color format.
   */
  withChannelValue(channel: ColorChannel, value: number): Color,
  /**
   * Returns the minimum, maximum, and step values for a given channel.
   */
  getChannelRange(channel: ColorChannel): ColorChannelRange,
  /**
   * Returns a localized color channel name for a given channel and locale,
   * for use in visual or accessibility labels.
   */
  getChannelName(channel: ColorChannel, locale: string): string,
  /**
   * Returns the number formatting options for the given channel.
   */
  getChannelFormatOptions(channel: ColorChannel): Intl.NumberFormatOptions,
  /**
   * Formats the numeric value for a given channel for display according to the provided locale.
   */
  formatChannelValue(channel: ColorChannel, locale: string): string,
  /**
   * Returns the color space, 'rgb', 'hsb' or 'hsl', for the current color.
   */
  getColorSpace(): ColorSpace,
  /**
   * Returns the color space axes, xChannel, yChannel, zChannel.
   */
  getColorSpaceAxes(xyChannels: {xChannel?: ColorChannel, yChannel?: ColorChannel}): ColorAxes,
  /**
   * Returns an array of the color channels within the current color space space.
   */
  getColorChannels(): [ColorChannel, ColorChannel, ColorChannel],
  /**
   * Returns a localized name for the color, for use in visual or accessibility labels.
   */
  getColorName(locale: string): string,
  /**
   * Returns a localized name for the hue, for use in visual or accessibility labels.
   */
  getHueName(locale: string): string
}

interface ColorFieldProps extends Omit<ValueBase<string | Color | null>, 'onChange'>, InputBase, Validation<Color | null>, FocusableProps, TextInputBase, LabelableProps, HelpTextProps {
  /** Handler that is called when the value changes. */
  onChange?: (color: Color | null) => void
}

interface IObject<T> {
    [name: string]: T;
}

declare type TargetParam<Param> = Pick<Param, Exclude<keyof Param, keyof EmitterParam>> & Partial<EmitterParam>;
interface EmitterParam<Target = any> {
    type: string;
    currentTarget: Target;
    stop(): void;
}
declare type OnEvent$1<Param, Target = any> = EmitterParam<Target> & Param;
declare type EventHash<Events, Target> = Partial<{
    [Key in keyof Events]: EventListener<Events[Key], Target>;
}>;
declare type EventListener<Param, Target> = (e: OnEvent$1<Param, Target>) => any;

declare class EventEmitter<Events extends {} = {
    [key: string]: {
        [key: string]: any;
    };
}> {
    private _events;
    on<Name extends keyof Events, Param = Events[Name]>(eventName: Name, listener: EventListener<Param, this>): this;
    on(events: EventHash<Events, this>): this;
    off<Name extends keyof Events, Param = Events[Name]>(eventName?: Name, listener?: EventListener<Param, this>): this;
    off(events: EventHash<Events, this>): this;
    once<Name extends keyof Events & string, Param = Events[Name]>(eventName: Name, listener?: EventListener<Param, this>): Promise<OnEvent$1<Param, this>>;
    emit<Name extends keyof Events, Param = Events[Name]>(eventName: {} extends Param ? Name : never): boolean;
    emit<Name extends keyof Events, Param = Events[Name]>(eventName: Name, param: TargetParam<Param>): boolean;
    trigger<Name extends keyof Events, Param = Events[Name]>(eventName: {} extends TargetParam<Param> ? Name : never): boolean;
    trigger<Name extends keyof Events, Param = Events[Name]>(eventName: Name, param: TargetParam<Param>): boolean;
    private _addEvent;
}

/**
 * @typedef
 * @memberof Gesto
 */
interface Client {
    clientX: number;
    clientY: number;
    originalClientX?: number;
    originalClientY?: number;
}
/**
 * @typedef
 * @memberof Gesto
 */
interface Dist {
    distX: number;
    distY: number;
}
/**
 * @typedef
 * @memberof Gesto
 */
interface Delta {
    deltaX: number;
    deltaY: number;
}
/**
 * @typedef
 * @memberof Gesto
 * @extends Gesto.Client
 * @extends Gesto.Dist
 * @extends Gesto.Delta
 */
interface Position extends Client, Dist, Delta {
}
/**
 * @typedef
 * @memberof Gesto
 * @extends Gesto.Position
 * @extends EventEmitter.EmitterParam
 */
interface OnDragStart$1<T = Gesto> extends Position, EmitterParam<T> {
    datas: IObject<any>;
    data: IObject<any>;
    inputEvent: any;
    isTrusted: boolean;
    isMouseEvent: boolean;
    isSecondaryButton: boolean;
    isDouble: boolean;
    preventDefault: () => void;
    preventDrag: () => void;
}
/**
 * @typedef
 * @memberof Gesto
 * @extends Gesto.Position
 * @extends EventEmitter.EmitterParam
 */
interface OnDrag$1<T = Gesto> extends Position, EmitterParam<T> {
    isDrag: boolean;
    isMouseEvent: boolean;
    isSecondaryButton: boolean;
    isPinch: boolean;
    movement: number;
    datas: IObject<any>;
    data: IObject<any>;
    isScroll: boolean;
    isFirstDrag: boolean;
    inputEvent: any;
    isTrusted: boolean;
}
/**
 * @typedef
 * @memberof Gesto
 * @extends Gesto.Position
 * @extends EventEmitter.EmitterParam
 */
interface OnDragEnd$1<T = Gesto> extends Position, EmitterParam<T> {
    isDrag: boolean;
    isClick: boolean;
    isMouseEvent: boolean;
    isSecondaryButton: boolean;
    isDouble: boolean;
    datas: IObject<any>;
    data: IObject<any>;
    inputEvent: any;
    isTrusted: boolean;
}
/**
 * @typedef
 * @memberof Gesto
 * @extends Gesto.Position
 * @extends EventEmitter.EmitterParam
 */
interface OnPinchStart<T = Gesto> extends Position, EmitterParam<T> {
    datas: IObject<any>;
    data: IObject<any>;
    touches: Position[];
    angle: number;
    inputEvent: any;
    isTrusted: boolean;
    preventDefault: () => void;
    preventDrag: () => void;
}
/**
 * @typedef
 * @memberof Gesto
 * @extends Gesto.Position
 * @extends EventEmitter.EmitterParam
 */
interface OnPinch<T = Gesto> extends Position, EmitterParam<T> {
    datas: IObject<any>;
    data: IObject<any>;
    touches: Position[];
    rotation: number;
    angle: number;
    scale: number;
    distance: number;
    movement: number;
    inputEvent: any;
    isTrusted: boolean;
}
/**
 * @typedef
 * @memberof Gesto
 * @extends Gesto.Position
 * @extends EventEmitter.EmitterParam
 */
interface OnPinchEnd<T = Gesto> extends Position, EmitterParam<T> {
    isPinch: boolean;
    datas: IObject<any>;
    data: IObject<any>;
    touches: Position[];
    inputEvent: any;
}
/**
 * @typedef
 * @memberof Gesto
 */
interface GestoOptions {
    container?: Window | Node | Element;
    /**
     * @default ["mouse", "touch"]
     */
    events?: Array<"drag" | "mouse" | "touch">;
    /**
     * Whether to prevent dragging of the right mouse button
     * @default true
     */
    preventRightClick?: boolean;
    /**
     * @default true
     */
    preventWheelClick?: boolean;
    preventDefault?: boolean;
    /**
     * Prevents pinching when the drag is moved more than a certain distance. That distance allowance is pinchThreshold.
     * @default 0
     */
    pinchThreshold?: number;
    /**
     * Whether to keep dragging even when pinch ends
     * @default false
     */
    keepDragging?: boolean;
    /**
     * Prevent click event on drag. (mousemove, touchmove)
     * @default false
     */
    preventClickEventOnDrag?: boolean;
    /**
     * Prevent click event on dragStart. (mousedown, touchstart)
     * @default false
     */
    preventClickEventOnDragStart?: boolean;
    /**
     * Prevent click event according to specific conditions.
     * Returning true allows the click event, returning false prevents it.
     * @default null
     */
    preventClickEventByCondition?: ((e: MouseEvent) => boolean) | null;
    pinchOutside?: boolean;
    /**
     * Prevent dragging of `input`, `textarea`, and contenteditable.
     * @default false
     */
    checkInput?: boolean;
    /**
     * Whether to drag the focused input
     * If `checkInput` is true, this option is not applied.
     * @default false
     */
    dragFocusedInput?: boolean;
    checkWindowBlur?: boolean;
}
/**
 * @typedef
 * @memberof Gesto
 */
type GestoEvents = {
    "dragStart": OnDragStart$1;
    "drag": OnDrag$1;
    "dragEnd": OnDragEnd$1;
    "pinchStart": OnPinchStart;
    "pinch": OnPinch;
    "pinchEnd": OnPinchEnd;
};

/**
 * You can set up drag, pinch events in any browser.
 */
declare class Gesto extends EventEmitter<GestoEvents> {
    options: GestoOptions;
    private flag;
    private pinchFlag;
    private data;
    private isDrag;
    private isPinch;
    private clientStores;
    private targets;
    private prevTime;
    private doubleFlag;
    private _useMouse;
    private _useTouch;
    private _useDrag;
    private _dragFlag;
    private _isTrusted;
    private _isMouseEvent;
    private _isSecondaryButton;
    private _preventMouseEvent;
    private _prevInputEvent;
    private _isDragAPI;
    private _isIdle;
    private _preventMouseEventId;
    private _window;
    /**
     *
     */
    constructor(targets: Array<Element | Window> | Element | Window, options?: GestoOptions);
    /**
     * Stop Gesto's drag events.
     */
    stop(): void;
    /**
     * The total moved distance
     */
    getMovement(clients?: Client[]): number;
    /**
     * Whether to drag
     */
    isDragging(): boolean;
    /**
     * Whether the operation of gesto is finished and is in idle state
     */
    isIdle(): boolean;
    /**
     * Whether to start drag
     */
    isFlag(): boolean;
    /**
     * Whether to start pinch
     */
    isPinchFlag(): boolean;
    /**
     * Whether to start double click
     */
    isDoubleFlag(): boolean;
    /**
     * Whether to pinch
     */
    isPinching(): boolean;
    /**
     * If a scroll event occurs, it is corrected by the scroll distance.
     */
    scrollBy(deltaX: number, deltaY: number, e: any, isCallDrag?: boolean): void;
    /**
     * Create a virtual drag event.
     */
    move([deltaX, deltaY]: number[], inputEvent: any): TargetParam<OnDrag$1>;
    /**
     * The dragStart event is triggered by an external event.
     */
    triggerDragStart(e: any): void;
    /**
     * Set the event data while dragging.
     */
    setEventData(data: IObject<any>): this;
    /**
     * Set the event data while dragging.
     * Use `setEventData`
     * @deprecated
     */
    setEventDatas(data: IObject<any>): this;
    /**
     * Get the current event state while dragging.
     */
    getCurrentEvent(inputEvent?: any): {
        movement: number;
        isDrag: boolean;
        isPinch: boolean;
        isScroll: boolean;
        inputEvent: any;
        distX: number;
        distY: number;
        clientX: number;
        clientY: number;
        originalClientX?: number | undefined;
        originalClientY?: number | undefined;
        deltaX: number;
        deltaY: number;
        data: IObject<any>;
        datas: IObject<any>;
    };
    /**
     * Get & Set the event data while dragging.
     */
    getEventData(): IObject<any>;
    /**
     * Get & Set the event data while dragging.
     * Use getEventData method
     * @depreacated
     */
    getEventDatas(): IObject<any>;
    /**
     * Unset Gesto
     */
    unset(): void;
    onDragStart: (e: any, isTrusted?: boolean) => false | undefined;
    onDrag: (e: any, isScroll?: boolean) => void;
    onDragEnd: (e?: any) => void;
    onPinchStart(e: TouchEvent): void;
    onPinch(e: TouchEvent, clients: Client[]): void;
    onPinchEnd(e: TouchEvent): void;
    private getCurrentStore;
    private moveClients;
    private onBlur;
    private _addStore;
    private _getPosition;
    private _allowClickEvent;
    private _attchDragEvent;
    private _dettachDragEvent;
    private _onClick;
    private _onContextMenu;
    private _allowMouseEvent;
    private _passCallback;
}

declare class CustomGesto {
    private ableName;
    private prevX;
    private prevY;
    private startX;
    private startY;
    private isDrag;
    private isFlag;
    private datas;
    constructor(ableName?: string);
    dragStart(client: number[], e: any): {
        type: string;
        inputEvent: any;
        isDrag: boolean;
        isFirstDrag: boolean;
        datas: IObject<any>;
        originalDatas: IObject<any>;
        parentEvent: boolean;
        parentGesto: CustomGesto;
        clientX: number;
        clientY: number;
        originalClientX?: number | undefined;
        originalClientY?: number | undefined;
        distX: number;
        distY: number;
        deltaX: number;
        deltaY: number;
    };
    drag(client: number[], inputEvent: any): OnCustomDrag;
    move(delta: number[], inputEvent: any): OnCustomDrag;
}

interface MoveableElementMatrixInfo {
    hasZoom: boolean;
    hasFixed: boolean;
    originalRootMatrix: number[];
    rootMatrix: number[];
    beforeMatrix: number[];
    offsetMatrix: number[];
    allMatrix: number[];
    targetMatrix: number[];
    transformOrigin: number[];
    targetOrigin: number[];
    is3d: boolean;
    targetTransform: string;
    inlineTransform: string;
    offsetContainer: HTMLElement | null;
    offsetRootContainer: HTMLElement | null;
    matrixes: MatrixInfo[];
}

interface MoveableElementInfo extends MoveableElementMatrixInfo, MoveablePosition, ElementSizes {
    width: number;
    height: number;
    rotation: number;
}

interface MoveableTargetInfo extends MoveableElementInfo {
    targetClientRect: MoveableClientRect;
    containerClientRect: MoveableClientRect;
    moveableClientRect: MoveableClientRect;
    rootContainerClientRect: MoveableClientRect;
    beforeDirection: 1 | -1;
    beforeOrigin: number[];
    offsetDelta: number[];
    originalBeforeOrigin: number[];
    target: HTMLElement | SVGElement | null | undefined;
    style: Partial<Writable<CSSStyleDeclaration>>;
}

interface MoveableClientRect {
    left: number;
    top: number;
    right: number;
    bottom: number;
    width: number;
    height: number;
    clientLeft?: number;
    clientTop?: number;
    clientWidth?: number;
    clientHeight?: number;
    scrollWidth?: number;
    scrollHeight?: number;
    overflow?: boolean;
}
declare type MoveableManagerProps<T = {}> = {
    cssStyled: any;
    customStyledMap: Record<string, any>;
    wrapperMoveable?: MoveableManagerInterface | null;
    isWrapperMounted?: boolean;
    parentMoveable?: MoveableManagerInterface | null;
    parentPosition?: number[] | null;
    groupable?: boolean;
} & MoveableDefaultOptions & (unknown extends T ? IObject<any> : T);
/**
 * @typedef
 * @memberof Moveable
 */
interface MoveablePosition {
    left: number;
    top: number;
    right: number;
    bottom: number;
    origin: number[];
    pos1: number[];
    pos2: number[];
    pos3: number[];
    pos4: number[];
    direction: 1 | -1;
}
/**
 * @typedef
 * @memberof Moveable
 * @options
 */
interface DefaultOptions {
    /**
     * The target(s) to indicate Moveable Control Box.
     * @default null
     */
    target?: SVGElement | HTMLElement | null;
    /**
     * The external target(s) to drag Moveable target(s)
     * @default target
     */
    dragTarget?: MoveableRefType | null;
    /**
     * If dragTarget is set directly, taget itself cannot be dragged.
     * Whether to drag the target as well.
     * @default false
     */
    dragTargetSelf?: boolean;
    /**
     * Container area where drag works
     * @default window
     */
    dragContainer?: null | Window | MoveableRefType<HTMLElement>;
    /**
     * A container into which Moveables are inserted.
     * Set it only when used within the slot of Web Components or when the container is different.
     * @default parentElement
     */
    container?: SVGElement | HTMLElement | null;
    /**
     * Whether to warp itself to the container itself. Don't set it.
     * @private
     * @default false
     */
    warpSelf?: boolean;
    /**
     * Moveable Root Container (No Transformed Container)
     * @default parentElement
     * @story options--options-root-container
     */
    rootContainer?: MoveableRefType<HTMLElement>;
    /**
     * If you want to set the dragging information to the viewer, refer to the following.
     * @default null
     * @story options--options-view-container
     */
    viewContainer?: MoveableRefType<HTMLElement>;
    /**
     * Whether the target size is detected and updated whenever it changes.
     * It is more effective when used together with `useMutationObserver`.
     * @default false
     * @story options--options-resize-observer
     */
    useResizeObserver?: boolean;
    /**
     * Whether the target size, pos in inline style is detected and updated whenever it changes.
     * It is more effective when used together with `useResizeObserver`.
     * @default false
     * @story options--options-mutation-observer
     */
    useMutationObserver?: boolean;
    /**
     * Zooms in the elements of a moveable.
     * @default 1
     */
    zoom?: number;
    /**
     * The default transformOrigin of the target can be set in advance.
     * @default ""
     */
    transformOrigin?: Array<string | number> | string | "";
    /**
     * You can add your custom able.
     * @default []
     */
    ables?: Able[];
    /**
     * You can specify the className of the moveable controlbox.
     * @default ""
     */
    className?: string;
    /**
     * Minimum distance to pinch.
     * @default 20
     */
    pinchThreshold?: number;
    /**
     * Whether the container containing the target becomes a pinch.
     * @default true
     */
    pinchOutside?: boolean;
    /**
     * Lets generate events of ables at the same time. (like Resizable, Scalable)
     * @default false
     */
    triggerAblesSimultaneously?: boolean;
    /**
     * Checks whether this is an element to input text or contentEditable, and prevents dragging.
     * @default false
     */
    checkInput?: boolean;
    /**
     * add nonce property to style for CSP.
     * @deprecated
     * @default ""
     */
    cspNonce?: string;
    /**
     * You can set the translateZ value of moveable.
     * @default 50
     */
    translateZ?: number | string;
    /**
     * Whether to hide the line corresponding to the rect of the target.
     * @default false
     */
    hideDefaultLines?: boolean;
    /**
     * Whether to prevent bubbling of events like mousedown, touchstart, etc.
     * @default false
     */
    stopPropagation?: boolean;
    /**
     * Whether to call preventDefault on touchstart or mousedown
     * @since 0.44.0
     * @default true
     */
    preventDefault?: boolean;
    /**
     * Whether to prevent dragging using the right mouse button
     * @default true
     */
    preventRightClick?: boolean;
    /**
     * Whether to prevent dragging using the wheel (middle) mouse button
     * @default true
     */
    preventWheelClick?: boolean;
    /**
     * Prevent click event on drag. (mousemove, touchmove)
     * @default true
     */
    preventClickEventOnDrag?: boolean;
    /**
     * Whether to drag the focused input
     * If `checkInput` is true, this option is not applied.
     * @since 0.47.0
     * @story options--options-drag-focused-input
     * @default false
     */
    dragFocusedInput?: boolean;
    /**
     * Prevent click event on dragStart. (mousedown, touchstart)
     * @default false
     */
    preventClickDefault?: boolean;
    /**
     * You can use props in object format or custom props.
     * @default empty object
     */
    props?: Record<string, any>;
    /**
     * Data for first render
     * @default null
     */
    persistData?: PersistRectData | null;
    /**
     * Whether to accurately show the position of a movable control box
     * Because getBoundingClientRect is used, css zoom, transform: rotate between container and rootContainer cannot be used.
     * group is not supported.
     * @default false
     */
    useAccuratePosition?: boolean;
    /**
     * By adding padding to the line, you can increase the area of the line that can be clicked and dragged.
     * @since 0.43.0
     * @story options--options-line-padding
     * @default 0
     */
    linePadding?: number;
    /**
     * By adding padding to the control, you can increase the area of the control that can be clicked and dragged.
     * Either `rotateAroundControls` or `displayAroundControls` can be used.
     * @since 0.44.0
     * @story options--options-control-padding
     * @default 0
     */
    controlPadding?: number;
    /**
     * @private
     * single => group로 변환과정에 도형 유지를 위한 첫 렌더링 state
     */
    firstRenderState?: MoveableManagerState | null;
    /**
     * @private
     */
    requestStyles?: string[];
    /**
     * If you are using React 18's concurrent mode, use `flushSync` for UI sync.
     * @default empty function
     * @example
     * ```jsx
     * import { flushSync } from "react-dom";
     *
     * <Moveable flushSync={flushSync} />
     * ```
     */
    flushSync?: (callback: () => void) => void;
}
/**
 * @typedef
 * @extends Moveable.DefaultOptions
 * @extends Moveable.DragAreaOptions
 * @extends Moveable.OriginOptions
 * @extends Moveable.PaddingOptions
 */
interface MoveableDefaultOptions extends DefaultOptions, DragAreaOptions, OriginOptions, PaddingOptions {
}
interface MatrixInfo {
    type: "offset" | "target" | "zoom";
    target: SVGElement | HTMLElement;
    matrix?: number[];
    origin?: number[];
    zoom?: number;
}
declare type MoveableManagerState<T = {}> = {
    container: SVGElement | HTMLElement | null | undefined;
    disableNativeEvent: boolean;
    gestos: Record<string, Gesto | CustomGesto | null>;
    renderLines: number[][][];
    renderPoses: number[][];
    posDelta: number[];
    style: Partial<Writable<CSSStyleDeclaration>>;
    isPersisted?: boolean;
} & MoveableTargetInfo & T;
/**
 * @typedef
 * @memberof Moveable
 */
interface ElementSizes {
    svg: boolean;
    offsetWidth: number;
    offsetHeight: number;
    clientWidth: number;
    clientHeight: number;
    inlineCSSWidth: number;
    inlineCSSHeight: number;
    cssWidth: number;
    cssHeight: number;
    contentWidth: number;
    contentHeight: number;
    minWidth: number;
    minHeight: number;
    maxWidth: number;
    maxHeight: number;
    minOffsetWidth: number;
    minOffsetHeight: number;
    maxOffsetWidth: number;
    maxOffsetHeight: number;
}
/**
 * @typedef
 * @memberof Moveable
 * @property - left padding
 * @property - top padding
 * @property - right padding
 * @property - bottom padding
 */
interface PaddingBox {
    left?: number;
    top?: number;
    right?: number;
    bottom?: number;
}
interface Renderer {
    createElement(type: any, props?: any, ...children: any[]): any;
}
/**
 * @typedef
 * @memberof Moveable.Snappable
 */
interface SnapGuideline {
    type: "horizontal" | "vertical";
    direction: string;
    hide?: boolean;
    element?: Element | null;
    isStart?: boolean;
    isEnd?: boolean;
    isCenter?: boolean;
    isInner?: boolean;
    grid?: boolean;
    pos: number[];
    size: number;
    className?: string;
    sizes?: number[];
    gap?: number;
    elementDirection?: string;
    elementRect?: SnapElementRect;
    gapRects?: SnapElementRect[];
}
/**
 * @memberof Moveable
 * @typedef
 */
declare type MoveableRefType<T extends Element = HTMLElement | SVGElement> = string | (() => T) | MoveableRefObject<T> | T | null | undefined;
/**
 * @memberof Moveable
 * @typedef
 */
interface MoveableRefObject<T extends Element = HTMLElement | SVGElement> {
    current: T | undefined | null;
}
/**
 * You can make Able that can work in Moveable.
 * @typedef
 * In Able, you can manage drag events, props, state, fire event props, and render elements.
 * @memberof Moveable
 */
interface Able<Props extends IObject<any> = IObject<any>, Events extends IObject<any> = IObject<any>> {
    name: string;
    props?: readonly (keyof Props)[];
    events?: readonly (keyof Events)[];
    always?: boolean;
    ableGroup?: string;
    updateRect?: boolean;
    canPinch?: boolean;
    css?: string[];
    /**
     * You can request style. Specify the name of the style in camel case.
     * You can check it with `moveable.state.style`
     * @exmaple
     * ["borderRadius", "top", "left"]
     */
    requestStyle?(): string[];
    /**
     * If you use group, you can request child style. Specify the name of the style in camel case.
     * You can check it with `moveable.state.style`
     * @exmaple
     * ["borderRadius", "top", "left"]
     */
    requestChildStyle?(): string[];
    /**
     * You can specify the class name to be added to the Moveable control box.
     */
    className?(moveable: any): string;
    /**
     * You can specify the class name to be added to the Moveable View Container
     */
    viewClassName?(moveable: any): string;
    /**
     * Check how related to drag
     */
    dragRelation?: "strong" | "weak" | undefined | null | false;
    /**
     * Fired when the event is cleared
     */
    unset?(moveable: any): any;
    /**
     * Renders the React DOM structure for the able.
     */
    render?(moveable: any, renderer: Renderer): any;
    dragStart?(moveable: any, e: any): any;
    drag?(moveable: any, e: any): any;
    dragEnd?(moveable: any, e: any): any;
    dragAfter?(moveable: any, e: any): any;
    pinchStart?(moveable: any, e: OnPinchStart): any;
    pinch?(moveable: any, e: OnPinch): any;
    pinchEnd?(moveable: any, e: OnPinchEnd): any;
    dragControlCondition?(moveable: any, e: any): boolean;
    dragControlStart?(moveable: any, e: any): any;
    dragControl?(moveable: any, e: any): any;
    dragControlEnd?(moveable: any, e: any): any;
    dragControlAfter?(moveable: any, e: any): any;
    dragGroupCondition?(moveable: any, e: any): boolean;
    dragGroupStart?(moveable: any, e: any): any;
    dragGroup?(moveable: any, e: any): any;
    dragGroupEnd?(moveable: any, e: any): any;
    pinchGroupStart?(moveable: any, e: OnPinchStart): any;
    pinchGroup?(moveable: any, e: OnPinch): any;
    pinchGroupEnd?(moveable: any, e: OnPinchEnd): any;
    dragGroupControlCondition?(moveable: any, e: any): boolean;
    dragGroupControlStart?(moveable: any, e: any): any;
    dragGroupControl?(moveable: any, e: any): any;
    dragGroupControlEnd?(moveable: any, e: any): any;
    mouseEnter?(moveable: any, e: any): any;
    mouseLeave?(moveable: any, e: any): any;
    mouseGroupEnter?(moveable: any, e: any): any;
    mouseGroupLeave?(moveable: any, e: any): any;
    request?(moveable: any): AbleRequester;
}
/**
 * @typedef
 * @memberof Moveable
 */
interface OnEvent {
    /**
     * The Moveable instance
     */
    currentTarget: MoveableManagerInterface<Record<string, any>, Record<string, any>>;
    /**
     * The Moveable instance
     */
    moveable: MoveableManagerInterface<Record<string, any>, Record<string, any>>;
    /**
     * The Moveable's target
     */
    target: HTMLElement | SVGElement;
    /**
     * The horizontal coordinate within the application's client area at which the event occurred.
     */
    clientX: number;
    /**
     * The vertical coordinate within the application's client area at which the event occurred.
     */
    clientY: number;
    /**
     * Whether this is the first drag in the drag event
     */
    isFirstDrag: number;
    /**
     * Objects that can send information to the following events.
     */
    datas: IObject<any>;
    /**
     * The mouse or touch input event that is invoking the moveable event
     */
    inputEvent: any;
    /**
     * Stop the currently working able.
     */
    stopAble(): void;
    /**
     * Calling `stopDrag` in a drag-related event ends the drag.
     */
    stopDrag(): void;
    /**
     * Whether the event did not occur externally
     */
    isTrusted: boolean;
}
/**
 * @typedef
 * @memberof Moveable
 * @extends Moveable.OnEvent
 * @property - This is the last dragged event. No, if you haven't dragged.
 * @property - Whether this moved
 * @property - Whether it is double-click
 */
interface OnEndEvent extends OnEvent {
    lastEvent: any | undefined;
    isDrag: boolean;
    isDouble: boolean;
}
/**
 * @typedef
 * @memberof Moveable
 */
interface OnTransformStartEvent {
    /**
     * Set your original transform.
     * `transformIndex` is the sequence of functions used in the event.
     * If you use `setTransform`, you don't need to use `set` function.
     * @default transform of target's inline style
     */
    setTransform(transform: string | string[], transformIndex?: number): void;
    /**
     * `transformIndex` is the sequence of functions used in the event.
     * @default index with that property in transform or last
     */
    setTransformIndex(transformIndex: number): void;
}
/**
 * @typedef
 * @memberof Moveable
 * @extends Moveable.CSSObject
 */
interface TransformObject extends CSSObject {
    /**
     * a target's next transform string value.
     */
    transform: string;
    /**
     * A transform obtained by the simultaneous occurrence of other events in the current event
     */
    afterTransform: string;
}
/**
 * @typedef
 * @memberof Moveable
 * @extends Moveable.TransformObject
 */
interface OnTransformEvent extends TransformObject {
    /**
     * transform events causes a `drag` event. In some events, there may be no value.
     */
    drag: OnDrag;
}
/**
 * @typedef
 * @memberof Moveable
 */
interface AbleRequestParam {
    /**
     * Run the request instantly. (requestStart, request, requestEnd happen at the same time)
     */
    isInstant?: boolean;
    [key: string]: any;
}
/**
 * @typedef
 * @memberof Moveable
 * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.html#request|Request Method}
 * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.Draggable.html#request|Draggable Requester}
 * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.Resizable.html#request|Resizable Requester}
 * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.Scalable.html#request|Scalable Requester}
 * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.Rotatable.html#request|Rotatable Requester}
 * @property - Continue executing the request.
 * @property - End the request.
 */
interface Requester<RequestParam extends {} = AbleRequestParam> {
    request(param: RequestParam): this;
    requestEnd(): this;
}
interface AbleRequester {
    isControl: boolean;
    requestStart(param: IObject<any>): IObject<any>;
    request(param: IObject<any>): IObject<any>;
    requestEnd(): IObject<any>;
}
/**
 * When the drag starts, the dragStart event is called.
 * @typedef
 * @memberof Moveable.Draggable
 * @extends Moveable.OnEvent
 * @extends Moveable.OnTransformStartEvent
 */
interface OnDragStart extends OnEvent, OnTransformStartEvent {
    /**
     * You can set the start translate value.
     */
    set: (translate: number[]) => void;
}
/**
 * @typedef
 * @memberof Moveable.Draggable
 * @extends Moveable.OnEvent
 * @extends Moveable.CSSObject
 * @property - The delta of [left, top]
 * @property - The distance of [left, top]
 * @property - The position of [left, top]
 * @property - The delta of [translateX, translateY]
 * @property - The distance of [translateX, translateY]
 * @property - The position of [translateX, translateY]
 * @property - a target's transform
 * @property - a target's left
 * @property - a target's top
 * @property - a target's bottom
 * @property - a target's offset width
 * @property - a target's offset height
 * @property - a target's right
 * @property - Whether or not it is being pinched.
 */
interface OnDrag extends OnEvent, CSSObject {
    beforeDelta: number[];
    beforeDist: number[];
    beforeTranslate: number[];
    delta: number[];
    dist: number[];
    translate: number[];
    transform: string;
    left: number;
    top: number;
    bottom: number;
    width: number;
    height: number;
    right: number;
    isPinch: boolean;
}
/**
 * @typedef
 * @memberof Moveable.Draggable
 * @extends Moveable.OnEndEvent
 */
interface OnDragEnd extends OnEndEvent {
}
/**
 * @typedef
 * @memberof Moveable.Resizable
 * @extends Moveable.OnEvent
 * @property - The direction of resize.
 * @property - resize causes a `dragStart` event.
 * @property - You can set the css width, height value.
 * @property - You can set the css min width, min height value. (default: min-width)
 * @property - You can set the css max width, max height value. (default: max-width)
 * @property - You can set the css origin (default: % %)
 * @property - Set a fixed direction to resize. (default: Opposite direction)
 * @property - Set the ratio of width and height. (default: offsetWidth / offsetHeight)
 */
interface OnResizeStart extends OnEvent {
    /**
     * The direction of resize.
     */
    direction: number[];
    /**
     * First set (boundingWidth / boundingHeight) value
     */
    startRatio: number;
    /**
     * resize causes a `dragStart` event.
     */
    dragStart: OnDragStart | false;
    /**
     * You can set the css width, height value.
     */
    set: (size: number[]) => any;
    /**
     * You can set the css min offset width, min offset height value.
     * @default [minOffsetWidth, minOffsetHeight])
     */
    setMin: (minSize: Array<string | number>) => any;
    /**
     * You can set the css max offset width, max offset height value.
     * @default [maxOffsetWidth, maxOffsetHeight])
     */
    setMax: (maxSize: Array<string | number>) => any;
    /**
     * You can set the css origin
     * @default transformOrigin
     */
    setOrigin: (origin: Array<string | number>) => any;
    /**
     * Set a fixed direction to resize.
     * @default Opposite direction
     */
    setFixedDirection: (startDirecition: number[]) => any;
    /**
     * Set a fixed direction to resize.
     * @default Opposite position
     * @private
     */
    setFixedPosition: (startPosition: number[]) => any;
    /**
     * Set the ratio of width and height.
     * @default offsetWidth / offsetHeight
     */
    setRatio: (ratio: number) => any;
}
/**
 * @typedef
 * @memberof Moveable
 */
interface CSSObject {
    /**
     * You can get the style you can get from the event.
     */
    style: Record<string, string>;
    /**
     * You can get it as a cssText that you can get from that event.
     */
    cssText: string;
}
/**
 * @typedef
 * @memberof Moveable.Resizable
 * @extends Moveable.OnEvent
 * @extends Moveable.OnTransformEvent
 */
interface OnResize extends OnEvent, OnTransformEvent {
    /**
     * The direction of resize.
     */
    direction: number[];
    /**
     * a target's cssWidth
     */
    width: number;
    /**
     * a target's cssHeight
     */
    height: number;
    /**
     * a target's offset width as an integer with bounding width
     */
    offsetWidth: number;
    /**
     * a target's offset height as an integer with bounding height
     */
    offsetHeight: number;
    /**
     * a target's bounding width
     */
    boundingWidth: number;
    /**
     * a target's bounding height
     */
    boundingHeight: number;
    /**
     * The distance of [boundingWidth, boundingHeight]
     */
    dist: number[];
    /**
     * The delta of [boundingWidth, boundingHeight]
     */
    delta: number[];
    /**
     * First set (boundingWidth / boundingHeight) value
     */
    startRatio: number;
    /**
     * Whether or not it is being pinched.
     */
    isPinch: boolean;
}
/**
 * @typedef
 * @memberof Moveable.Resizable
 * @extends Moveable.OnEndEvent
 */
interface OnResizeEnd extends OnEndEvent {
}
/**
 * @typedef
 * @memberof Moveable.Rotatable
 * @extends Moveable.OnEvent
 * @extends Moveable.OnTransformStartEvent
 */
interface OnRotateStart extends OnEvent, OnTransformStartEvent {
    /**
     * You can set the start rotate value.
     */
    set: (rotate: number) => void;
    /**
     * Set a fixed direction to rotate.
     * @default target's transformOrigin
     */
    setFixedDirection: (fixedDirection: number[]) => void;
    /**
     * Set a fixed position to rotate.
     * @default target's transformOrigin
     */
    setFixedPosition: (fixedPosition: number[]) => void;
    /**
     * rotate causes a `dragStart` event.
     */
    dragStart: OnDragStart | false;
    /**
     * rotate causes a `resizeStart` event.
     */
    resizeStart: OnResizeStart | false;
}
/**
 * @typedef
 * @description Parameters for the `rotate` event
 * @memberof Moveable.Rotatable
 * @extends Moveable.OnEvent
 * @extends Moveable.OnTransformEvent
 */
interface OnRotate extends OnEvent, OnTransformEvent {
    /**
     * The distance of rotation degree before transform is applied
     */
    beforeDist: number;
    /**
     * The delta of rotation degree before transform is applied
     */
    beforeDelta: number;
    /**
     * The now rotation degree before transform is applied
     * @deprecated
     */
    beforeRotate: number;
    /**
     * The now rotation degree before transform is applied
     */
    beforeRotation: number;
    /**
     * The distance of rotation degree
     */
    dist: number;
    /**
     * The delta of rotation degree
     */
    delta: number;
    /**
     * The now rotation degree
     * @deprecated
     */
    rotate: number;
    /**
     * The now rotation degree
     */
    rotation: number;
    /**
     * The distance of client rotation degree
     */
    absoluteDist: number;
    /**
     * The delta of client rotation degree
     */
    absoluteDelta: number;
    /**
     * The now client rotation degree
     * @deprecated
     */
    absoluteRotate: number;
    /**
     * The now client rotation degree
     */
    absoluteRotation: number;
    /**
     * Whether or not it is being pinched.
     */
    isPinch: boolean;
    /**
     * rotate causes a `resize` event.
     */
    resize?: OnResize;
}
/**
 * @typedef
 * @memberof Moveable.Rotatable
 * @extends Moveable.OnEndEvent
 */
interface OnRotateEnd extends OnEndEvent {
}
/**
 * `renderStart` event occurs at the first start of all events.
 * @typedef
 * @memberof Moveable
 * @extends Moveable.OnEvent
 * @property - Whether or not it is being pinched.
 */
interface OnRenderStart extends OnEvent {
    isPinch: boolean;
}
/**
 * `render` event occurs before the target is drawn on the screen.
 * @typedef
 * @memberof Moveable
 * @extends Moveable.OnEvent
 * @extends Moveable.CSSObject
 */
interface OnRender extends OnEvent, CSSObject {
    /**
     * a target's next transform string value.
     */
    transform: string;
    /**
     * Whether or not it is being pinched.
     */
    isPinch: boolean;
    /**
     * Return transform as a transform object.
     * `rotate` is a number and everything else is an array of numbers.
     */
    transformObject: Record<string, any>;
}
/**
 * @typedef
 * @memberof Moveable
 */
interface PaddingOptions {
    /**
     * Add padding around the target to increase the drag area.
     * @default null
     */
    padding?: PaddingBox | number;
}
/**
 * @typedef
 * @memberof Moveable
 */
interface OriginOptions {
    /**
     * Whether or not the origin control box will be visible or not.
     * @default true
     */
    origin?: boolean;
    /**
     * Sets the transform origin based on the svg target. If not set, it is set as the transform origin based on the owner of svg.
     * @since 0.47.0
     * @default ""
     */
    svgOrigin?: string;
}
/**
 * @typedef
 * @memberof Moveable
 * @extends Moveable.AbleRequestParam
 * @description the Resizable's request parameter
 */
interface AbleRequesters {
    draggable: DraggableRequestParam;
    resizable: ResizableRequestParam;
    scalable: ScalableRequestParam;
    rotatable: RotatableRequestParam;
    [key: string]: AbleRequestParam;
}
/**
 * @typedef
 * @memberof Moveable.Draggable
 * @extends Moveable.AbleRequestParam
 * @description the Draggable's request parameter
 */
interface DraggableRequestParam extends AbleRequestParam {
    /**
     * x position
     */
    x?: number;
    /**
     * y position
     */
    y?: number;
    /**
     * X number to move
     */
    deltaX?: number;
    /**
     * Y number to move
     */
    deltaY?: number;
    /**
     * whether to use with `snappable`
     */
    useSnap?: boolean;
}
/**
 * @typedef
 * @memberof Moveable.Resizable
 * @extends Moveable.AbleRequestParam
 * @description the Resizable's request parameter
 */
interface ResizableRequestParam extends AbleRequestParam {
    /**
     * Direction to resize
     * @default [1, 1]
     */
    direction?: number[];
    /**
     * Whether to force keepRatio to resize
     */
    keepRatio?: boolean;
    /**
     * delta number of width
     */
    deltaWidth?: number;
    /**
     * delta number of height
     */
    deltaHeight?: number;
    /**
     * offset number of width
     */
    offsetWidth?: number;
    /**
     * offset number of height
     */
    offsetHeight?: number;
    /**
     *
     */
    horizontal?: boolean;
    /**
     * whether to use with `snappable`
     */
    useSnap?: boolean;
}
/**
 * @typedef
 * @memberof Moveable.Scalable
 * @extends Moveable.AbleRequestParam
 * @description the Scalable's request parameter
 */
interface ScalableRequestParam extends AbleRequestParam {
    /**
     * Direction to scale
     * @default [1, 1]
     */
    direction?: number[];
    /**
     * Whether to force keepRatio to resize
     */
    keepRatio?: boolean;
    /**
     * delta number of width
     */
    deltaWidth?: number;
    /**
     * delta number of height
     */
    deltaHeight?: number;
    /**
     * whether to use with `snappable`
     */
    useSnap?: boolean;
}
/**
 * @typedef
 * @memberof Moveable.Rotatable
 * @extends Moveable.AbleRequestParam
 * @description the Rotatable's request parameter
 */
interface RotatableRequestParam extends AbleRequestParam {
    /**
     * delta number of rotation
     */
    deltaRotate?: number;
    /**
     * absolute number of moveable's rotation
     */
    rotate?: number;
}
declare type RotationPosition = "top" | "bottom" | "left" | "right" | "top-right" | "top-left" | "bottom-right" | "bottom-left" | "left-top" | "left-bottom" | "right-top" | "right-bottom" | "none";
/**
 * @typedef
 * @memberof Moveable.Snappable
 */
interface SnapDirections {
    /**
     * Whether to snap the top of the element
     * @default true
     */
    left?: boolean;
    /**
     * Whether to snap the left of the element
     * @default true
     */
    top?: boolean;
    /**
     * Whether to snap the right of the element
     * @default true
     */
    right?: boolean;
    /**
     * Whether to snap the bottom of the element
     * @default true
     */
    bottom?: boolean;
    /**
     * Whether to snap the center((left + right) / 2) of the element
     * @default false
     */
    center?: boolean;
    /**
     * Whether to snap the middle((top + bottom) / 2) of the element
     * @default false
     */
    middle?: boolean;
}
/**
 * @typedef
 * @memberof Moveable.Snappable
 * @extends Moveable.Snappable.SnapDirections
 */
interface ElementGuidelineValue extends SnapDirections {
    /**
     * guideline element
     */
    element: Element;
    /**
     * class names to add to guideline
     * @default ""
     * @example
     *
     * ```css
     * .moveable-gap.red {
     *   background: red!important;
     * }
     * ```
     * ```css
     * .moveable-bold.red {
     *   background: red!important;
     * }
     * ```
     * ```css
     * .moveable-dashed.red {
     *   border-top-color: red!important;
     * }
     * ```
     */
    className?: string;
    /**
     * Whether to update the element size at every render
     * @default false
     */
    refresh?: boolean;
}
/**
 * @typedef
 * @memberof Moveable.Snappable
 * @extends Moveable.Snappable.SnapDirections
 */
interface ElementGuidelineValue extends SnapDirections {
    /**
     * guideline element
     */
    element: Element;
    /**
     * class names to add to guideline
     * @default ""
     */
    className?: string;
    /**
     * Whether to update the element size at every render
     * @default false
     */
    refresh?: boolean;
}
/**
 * @typedef
 * @memberof Moveable.Snappable
 */
interface OnSnap {
    /**
     * snapped verticalGuidelines, horizontalGuidelines,
     */
    guidelines: SnapGuideline[];
    /**
     * snapped elements (group by element)
     */
    elements: SnapGuideline[];
    /**
     * gaps is snapped guidelines that became gap snap between elements.
     */
    gaps: SnapGuideline[];
}
interface SnapDirectionPoses {
    left?: number;
    top?: number;
    right?: number;
    bottom?: number;
    center?: number;
    middle?: number;
}
interface SnapElementRect extends ElementGuidelineValue {
    rect: SnapDirectionPoses;
}
/**
 * @typedef
 * @memberof Moveable
 */
interface DragAreaOptions {
    /**
     * Instead of firing an event on the target, we add it to the moveable control element. You can use click and clickGroup events.
     * @default if group, true, else fals
     */
    dragArea?: boolean;
    /**
     * Set `pointerEvents: none;` css to pass events in dragArea.
     * @default false
     */
    passDragArea?: boolean;
}
interface OnCustomDrag extends Position {
    type: string;
    inputEvent: any;
    isDrag: boolean;
    isFirstDrag: boolean;
    datas: IObject<any>;
    originalDatas: IObject<any>;
    parentEvent: boolean;
    parentGesto: CustomGesto;
}
/**
 * @typedef
 * @memberof Moveable
 */
declare type PersistRectData = Omit<Partial<RectInfo>, "children"> & {
    children?: Array<Partial<RectInfo>>;
};
/**
 * @typedef
 * @memberof Moveable
 */
interface RectInfo {
    /**
     * The coordinates of the vertex 1
     */
    pos1: number[];
    /**
     * The coordinates of the vertex 2
     */
    pos2: number[];
    /**
     * The coordinates of the vertex 3
     */
    pos3: number[];
    /**
     * The coordinates of the vertex 4
     */
    pos4: number[];
    /**
     * left position of the target relative to the container
     */
    left: number;
    /**
     * top position of the target relative to the container
     */
    top: number;
    /**
     * The width of moveable element
     */
    width: number;
    /**
     * The height of moveable element
     */
    height: number;
    /**
     * The offset width of the target
     */
    offsetWidth: number;
    /**
     * The offset height of the target
     */
    offsetHeight: number;
    /**
     * The absolute transform origin
     */
    origin: number[];
    /**
     * The absolute transform origin before transformation
     */
    beforeOrigin: number[];
    /**
     * The target transform origin
     */
    transformOrigin: number[];
    /**
     * you can get the absolute rotation value
     */
    rotation: number;
    /**
     * If you use a group, you can get child moveables' rect info
     */
    children?: RectInfo[];
}
/**
 * @typedef
 * @memberof Moveable
 * @property - top position
 * @property - left position
 * @property - target's width
 * @property - target's height
 */
interface HitRect {
    top: number;
    left: number;
    width?: number;
    height?: number;
}
/**
 * @typedef
 * @memberof Moveable
 * @extends Moveable.MoveableInterface
 */
interface MoveableManagerInterface<T = {}, U = {}> extends MoveableInterface {
    moveables?: MoveableManagerInterface[];
    props: MoveableManagerProps<T>;
    state: MoveableManagerState<U>;
    renderState: Record<string, any>;
    rotation: number;
    scale: number[];
    controlGesto: Gesto;
    targetGesto: Gesto;
    enabledAbles: Able[];
    controlAbles: Able[];
    targetAbles: Able[];
    areaElement: HTMLElement;
    controlBox: HTMLElement;
    isUnmounted: boolean;
    useCSS(tag: string, css: string): any;
    getContainer(): HTMLElement | SVGElement;
    getRotation(): number;
    getState(): MoveableManagerState<U>;
    triggerEvent(name: string, params: IObject<any>, isManager?: boolean): any;
}
/**
 * @typedef
 * @memberof Moveable
 */
interface MoveableInterface {
    getManager(): MoveableManagerInterface<any, any>;
    getRect(): RectInfo;
    getAble<T extends Able>(ableName: string): T | undefined;
    isMoveableElement(target: Element): boolean;
    /**
     * If the location or size of the target is changed, call the `.updateRect()` method.
     * Use the `useResizeObserver` and `useMutationObserver` props to update automatically.
     */
    updateRect(type?: "Start" | "" | "End", isTarget?: boolean, isSetState?: boolean): void;
    /**
     * @deprecated
     * Use `.updateRect()` method
     */
    updateTarget(): void;
    /**
     * Request able through a method rather than an event.
     * At the moment of execution, requestStart is executed,
     * and then request and requestEnd can be executed through Requester.
     * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.Draggable.html#request Draggable Requester}
     * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.Resizable.html#request Resizable Requester}
     * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.Scalable.html#request Scalable Requester}
     * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.Rotatable.html#request Rotatable Requester}
     * @see {@link https://daybrush.com/moveable/release/latest/doc/Moveable.OriginDraggable.html#request OriginDraggable Requester}
     * @param - ableName
     * @param - request to be able params.
     * @param - If isInstant is true, request and requestEnd are executed immediately.
     * @return - Able Requester. If there is no request in able, nothing will work.
     * @example
     * import Moveable from "moveable";
     *
     * const moveable = new Moveable(document.body);
     *
     * // Instantly Request (requestStart - request - requestEnd)
     * moveable.request("draggable", { deltaX: 10, deltaY: 10 }, true);
     *
     * // Start move
     * const requester = moveable.request("draggable");
     * requester.request({ deltaX: 10, deltaY: 10 });
     * requester.request({ deltaX: 10, deltaY: 10 });
     * requester.request({ deltaX: 10, deltaY: 10 });
     * requester.requestEnd();
     */
    request<RequestParam extends AbleRequesters[Name], Name extends string = string>(ableName: Name, params?: RequestParam, isInstant?: boolean): Requester<RequestParam>;
    /**
     * moveable is the top level that manages targets
     * `Single`: MoveableManager instance
     * `Group`: MoveableGroup instance
     * `IndividualGroup`: MoveableIndividaulGroup instance
     * Returns leaf target MoveableManagers.
     */
    getMoveables(): MoveableManagerInterface[];
    /**
     * Returns the element of the control box.
     */
    getControlBoxElement(): HTMLElement;
    /**
     * Target element to be dragged in moveable
     */
    getDragElement(): HTMLElement | SVGElement | null | undefined;
    destroy(): void;
    dragStart(e: MouseEvent | TouchEvent, target?: EventTarget | null): void;
    isInside(clientX: number, clientY: number): boolean;
    isDragging(ableName?: string): boolean;
    hitTest(el: Element | HitRect): number;
    setState(state: any, callback?: () => any): any;
    waitToChangeTarget(): Promise<void>;
    forceUpdate(callback?: () => any): any;
    updateSelectors(): void;
    getTargets(): Array<HTMLElement | SVGElement>;
    stopDrag(type?: "target" | "control"): void;
}
declare type Writable<T> = {
    -readonly [key in keyof T]: T[key];
};

interface AriaToolbarProps extends AriaLabelingProps {
    /**
     * The orientation of the entire toolbar.
     * @default 'horizontal'
     */
    orientation?: Orientation;
}

/** These types can be animated */
type Animatable<T = any> = T extends number ? number : T extends string ? string : T extends ReadonlyArray<number | string> ? Array<number | string> extends T ? ReadonlyArray<number | string> : {
    [P in keyof T]: Animatable<T[P]>;
} : never;

/**
 * MIT License
 * Copyright (c) Alec Larson
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/** Ensure each type of `T` is an array */
type Arrify<T> = [T, T] extends [infer T, infer DT] ? DT extends ReadonlyArray<any> ? Array<DT[number]> extends DT ? ReadonlyArray<T extends ReadonlyArray<infer U> ? U : T> : DT : ReadonlyArray<T extends ReadonlyArray<infer U> ? U : T> : never;
/** Override the property types of `A` with `B` and merge any new properties */
type Merge<A, B> = Remap<{
    [P in keyof A]: P extends keyof B ? B[P] : A[P];
} & Omit$1<B, keyof A>>;
/** Better type errors for overloads with generic types */
type Constrain<T, U> = [T] extends [Any$2] ? U : [T] extends [U] ? T : U;
/** Try to simplify `&` out of an object type */
type Remap<T> = {} & {
    [P in keyof T]: T[P];
};
type Pick$1<T, K extends keyof T> = {} & {
    [P in K]: T[P];
};
type Omit$1<T, K> = Pick$1<T, Exclude<keyof T, K>>;
type OneOrMore<T> = T | readonly T[];
type Falsy = false | null | undefined;
interface Lookup<T = any> {
    [key: string]: T;
}
/** Intersected with other object types to allow for unknown properties */
interface UnknownProps extends Lookup<unknown> {
}
/** Use `[T] extends [Any]` to know if a type parameter is `any` */
declare class Any$2 {
    private _;
}

type EasingFunction = (t: number) => number;
type ExtrapolateType = 'identity' | 'clamp' | 'extend';
type InterpolatorArgs<Input = any, Output = any> = [InterpolatorFn<Arrify<Input>, Output>] | [InterpolatorConfig<Output>] | [
    readonly number[],
    readonly Constrain<Output, Animatable>[],
    (ExtrapolateType | undefined)?
];
type InterpolatorFn<Input, Output> = (...inputs: Arrify<Input>) => Output;
type InterpolatorConfig<Output = Animatable> = {
    /**
     * What happens when the spring goes below its target value.
     *
     *  - `extend` continues the interpolation past the target value
     *  - `clamp` limits the interpolation at the max value
     *  - `identity` sets the value to the interpolation input as soon as it hits the boundary
     *
     * @default 'extend'
     */
    extrapolateLeft?: ExtrapolateType;
    /**
     * What happens when the spring exceeds its target value.
     *
     *  - `extend` continues the interpolation past the target value
     *  - `clamp` limits the interpolation at the max value
     *  - `identity` sets the value to the interpolation input as soon as it hits the boundary
     *
     * @default 'extend'
     */
    extrapolateRight?: ExtrapolateType;
    /**
     * What happens when the spring exceeds its target value.
     * Shortcut to set `extrapolateLeft` and `extrapolateRight`.
     *
     *  - `extend` continues the interpolation past the target value
     *  - `clamp` limits the interpolation at the max value
     *  - `identity` sets the value to the interpolation input as soon as it hits the boundary
     *
     * @default 'extend'
     */
    extrapolate?: ExtrapolateType;
    /**
     * Input ranges mapping the interpolation to the output values.
     *
     * @example
     *
     *   range: [0, 0.5, 1], output: ['yellow', 'orange', 'red']
     *
     * @default [0,1]
     */
    range?: readonly number[];
    /**
     * Output values from the interpolation function. Should match the length of the `range` array.
     */
    output: readonly Constrain<Output, Animatable>[];
    /**
     * Transformation to apply to the value before interpolation.
     */
    map?: (value: number) => number;
    /**
     * Custom easing to apply in interpolator.
     */
    easing?: EasingFunction;
};

interface Timeout {
    time: number;
    handler: () => void;
    cancel: () => void;
}

/**
 * MIT License
 * Copyright (c) Alec Larson
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
declare const $get: unique symbol;
declare const $observers: unique symbol;
/** An event sent to `FluidObserver` objects. */
interface FluidEvent<T = any> {
    type: string;
    parent: FluidValue<T>;
}
/**
 * Extend this class for automatic TypeScript support when passing this
 * value to `fluids`-compatible libraries.
 */
declare abstract class FluidValue<T = any, E extends FluidEvent<T> = any> {
    private [$get];
    private [$observers]?;
    constructor(get?: () => T);
    /** Get the current value. */
    protected get?(): T;
    /** Called after an observer is added. */
    protected observerAdded?(count: number, observer: FluidObserver<E>): void;
    /** Called after an observer is removed. */
    protected observerRemoved?(count: number, observer: FluidObserver<E>): void;
}
/** An observer of `FluidValue` objects. */
type FluidObserver<E extends FluidEvent = any> = {
    eventObserved(event: E): void;
} | {
    (event: E): void;
};
/** Add the `FluidValue` type to every property. */
type FluidProps<T> = T extends object ? {
    [P in keyof T]: T[P] | FluidValue<Exclude<T[P], void>>;
} : unknown;

/** An animated number or a native attribute value */
declare class AnimatedValue<T = any> extends Animated {
    protected _value: T;
    done: boolean;
    elapsedTime: number;
    lastPosition: number;
    lastVelocity?: number | null;
    v0?: number | null;
    durationProgress: number;
    constructor(_value: T);
    /** @internal */
    static create(value: any): AnimatedValue<any>;
    getPayload(): Payload;
    getValue(): T;
    setValue(value: T, step?: number): boolean;
    reset(): void;
}
declare abstract class Animated<T = any> {
    /** The cache of animated values */
    protected payload?: Payload;
    constructor();
    /** Get the current value. Pass `true` for only animated values. */
    abstract getValue(animated?: boolean): T;
    /** Set the current value. Returns `true` if the value changed. */
    abstract setValue(value: T): boolean | void;
    /** Reset any animation state. */
    abstract reset(goal?: T): void;
    /** Get every `AnimatedValue` used by this node. */
    getPayload(): Payload;
}
type Payload = readonly AnimatedValue[];

/**
 * An `Interpolation` is a memoized value that's computed whenever one of its
 * `FluidValue` dependencies has its value changed.
 *
 * Other `FrameValue` objects can depend on this. For example, passing an
 * `Interpolation` as the `to` prop of a `useSpring` call will trigger an
 * animation toward the memoized value.
 */
declare class Interpolation<Input = any, Output = any> extends FrameValue<Output> {
    /** The source of input values */
    readonly source: unknown;
    /** Useful for debugging. */
    key?: string;
    /** Equals false when in the frameloop */
    idle: boolean;
    /** The function that maps inputs values to output */
    readonly calc: InterpolatorFn<Input, Output>;
    /** The inputs which are currently animating */
    protected _active: Set<FluidValue<any, any>>;
    constructor(
    /** The source of input values */
    source: unknown, args: InterpolatorArgs<Input, Output>);
    advance(_dt?: number): void;
    protected _get(): Output;
    protected _start(): void;
    protected _attach(): void;
    protected _detach(): void;
    /** @internal */
    eventObserved(event: FrameValue.Event): void;
}

/**
 * A kind of `FluidValue` that manages an `AnimatedValue` node.
 *
 * Its underlying value can be accessed and even observed.
 */
declare abstract class FrameValue<T = any> extends FluidValue<T, FrameValue.Event<T>> {
    readonly id: number;
    abstract key?: string;
    abstract get idle(): boolean;
    protected _priority: number;
    get priority(): number;
    set priority(priority: number);
    /** Get the current value */
    get(): T;
    /** Create a spring that maps our value to another value */
    to<Out>(...args: InterpolatorArgs<T, Out>): Interpolation<T, Out>;
    /** @deprecated Use the `to` method instead. */
    interpolate<Out>(...args: InterpolatorArgs<T, Out>): Interpolation<T, Out>;
    toJSON(): T;
    protected observerAdded(count: number): void;
    protected observerRemoved(count: number): void;
    /** @internal */
    abstract advance(dt: number): void;
    /** @internal */
    abstract eventObserved(_event: FrameValue.Event): void;
    /** Called when the first child is added. */
    protected _attach(): void;
    /** Called when the last child is removed. */
    protected _detach(): void;
    /** Tell our children about our new value */
    protected _onChange(value: T, idle?: boolean): void;
    /** Tell our children about our new priority */
    protected _onPriorityChange(priority: number): void;
}
declare namespace FrameValue {
    /** A parent changed its value */
    interface ChangeEvent<T = any> {
        parent: FrameValue<T>;
        type: 'change';
        value: T;
        idle: boolean;
    }
    /** A parent changed its priority */
    interface PriorityEvent<T = any> {
        parent: FrameValue<T>;
        type: 'priority';
        priority: number;
    }
    /** A parent is done animating */
    interface IdleEvent<T = any> {
        parent: FrameValue<T>;
        type: 'idle';
    }
    /** Events sent to children of `FrameValue` objects */
    type Event<T = any> = ChangeEvent<T> | PriorityEvent<T> | IdleEvent<T>;
}

declare class AnimationConfig {
    /**
     * With higher tension, the spring will resist bouncing and try harder to stop at its end value.
     *
     * When tension is zero, no animation occurs.
     *
     * @default 170
     */
    tension: number;
    /**
     * The damping ratio coefficient, or just the damping ratio when `speed` is defined.
     *
     * When `speed` is defined, this value should be between 0 and 1.
     *
     * Higher friction means the spring will slow down faster.
     *
     * @default 26
     */
    friction: number;
    /**
     * The natural frequency (in seconds), which dictates the number of bounces
     * per second when no damping exists.
     *
     * When defined, `tension` is derived from this, and `friction` is derived
     * from `tension` and `damping`.
     */
    frequency?: number;
    /**
     * The damping ratio, which dictates how the spring slows down.
     *
     * Set to `0` to never slow down. Set to `1` to slow down without bouncing.
     * Between `0` and `1` is for you to explore.
     *
     * Only works when `frequency` is defined.
     *
     * @default 1
     */
    damping: number;
    /**
     * Higher mass means more friction is required to slow down.
     *
     * Defaults to 1, which works fine most of the time.
     *
     * @default 1
     */
    mass: number;
    /**
     * The initial velocity of one or more values.
     *
     * @default 0
     */
    velocity: number | number[];
    /**
     * The smallest velocity before the animation is considered "not moving".
     *
     * When undefined, `precision` is used instead.
     */
    restVelocity?: number;
    /**
     * The smallest distance from a value before that distance is essentially zero.
     *
     * This helps in deciding when a spring is "at rest". The spring must be within
     * this distance from its final value, and its velocity must be lower than this
     * value too (unless `restVelocity` is defined).
     *
     * @default 0.01
     */
    precision?: number;
    /**
     * For `duration` animations only. Note: The `duration` is not affected
     * by this property.
     *
     * Defaults to `0`, which means "start from the beginning".
     *
     * Setting to `1+` makes an immediate animation.
     *
     * Setting to `0.5` means "start from the middle of the easing function".
     *
     * Any number `>= 0` and `<= 1` makes sense here.
     */
    progress?: number;
    /**
     * Animation length in number of milliseconds.
     */
    duration?: number;
    /**
     * The animation curve. Only used when `duration` is defined.
     *
     * Defaults to quadratic ease-in-out.
     */
    easing: EasingFunction;
    /**
     * Avoid overshooting by ending abruptly at the goal value.
     *
     * @default false
     */
    clamp: boolean;
    /**
     * When above zero, the spring will bounce instead of overshooting when
     * exceeding its goal value. Its velocity is multiplied by `-1 + bounce`
     * whenever its current value equals or exceeds its goal. For example,
     * setting `bounce` to `0.5` chops the velocity in half on each bounce,
     * in addition to any friction.
     */
    bounce?: number;
    /**
     * "Decay animations" decelerate without an explicit goal value.
     * Useful for scrolling animations.
     *
     * Use `true` for the default exponential decay factor (`0.998`).
     *
     * When a `number` between `0` and `1` is given, a lower number makes the
     * animation slow down faster. And setting to `1` would make an unending
     * animation.
     *
     * @default false
     */
    decay?: boolean | number;
    /**
     * While animating, round to the nearest multiple of this number.
     * The `from` and `to` values are never rounded, as well as any value
     * passed to the `set` method of an animated value.
     */
    round?: number;
    constructor();
}

/** The object type of the `config` prop. */
type SpringConfig = Partial<AnimationConfig>;
/** The object given to the `onRest` prop and `start` promise. */
interface AnimationResult<T extends Readable = any> {
    value: T extends Readable<infer U> ? U : never;
    /** When true, no animation ever started. */
    noop?: boolean;
    /** When true, the animation was neither cancelled nor stopped prematurely. */
    finished?: boolean;
    /** When true, the animation was cancelled before it could finish. */
    cancelled?: boolean;
}
/** The promised result of an animation. */
type AsyncResult<T extends Readable = any> = Promise<AnimationResult<T>>;
/**
 * The set of `SpringValue` objects returned by a `useSpring` call (or similar).
 */
type SpringValues<T extends Lookup = any> = [T] extends [Any$2] ? Lookup<SpringValue<unknown> | undefined> : {
    [P in keyof T]: SpringWrap<T[P]>;
};
type SpringWrap<T> = [
    Exclude<T, FluidValue>,
    Extract<T, readonly any[]>
] extends [object | void, never] ? never : SpringValue<Exclude<T, FluidValue | void>> | Extract<T, void>;

/** @internal */
interface Readable<T = any> {
    get(): T;
}
/** @internal */
type InferState<T extends Readable> = T extends Controller<infer State> ? State : T extends SpringValue<infer U> ? U : unknown;
/** @internal */
type InferProps$1<T extends Readable> = T extends Controller<infer State> ? ControllerUpdate<State> : T extends SpringValue<infer U> ? SpringUpdate<U> : Lookup;
/** @internal */
type InferTarget<T> = T extends object ? T extends ReadonlyArray<number | string> ? SpringValue<T> : Controller<T> : SpringValue<T>;
/** @internal */
interface AnimationTarget<T = any> extends Readable<T> {
    start(props: any): AsyncResult<this>;
    stop: Function;
    item?: unknown;
}
/** @internal */
interface AnimationRange<T> {
    to: T | FluidValue<T> | undefined;
    from: T | FluidValue<T> | undefined;
}
/** @internal */
type AnimationResolver<T extends Readable> = (result: AnimationResult<T> | AsyncResult<T>) => void;
/** @internal */
type EventKey = Exclude<keyof ReservedEventProps, 'onResolve' | 'onDestroyed'>;
/** @internal */
type PickEventFns<T> = {
    [P in Extract<keyof T, EventKey>]?: Extract<T[P], Function>;
};

/** An animation being executed by the frameloop */
declare class Animation<T = any> {
    changed: boolean;
    values: readonly AnimatedValue[];
    toValues: readonly number[] | null;
    fromValues: readonly number[];
    to: T | FluidValue<T>;
    from: T | FluidValue<T>;
    config: AnimationConfig;
    immediate: boolean;
}
interface Animation<T> extends PickEventFns<SpringProps<T>> {
}

type AsyncTo<T> = SpringChain<T> | SpringToFn<T>;
/** @internal */
type RunAsyncProps<T extends AnimationTarget = any> = InferProps$1<T> & {
    callId: number;
    parentId?: number;
    cancel: boolean;
    to?: any;
};
/** @internal */
interface RunAsyncState<T extends AnimationTarget = any> {
    paused: boolean;
    pauseQueue: Set<() => void>;
    resumeQueue: Set<() => void>;
    timeouts: Set<Timeout>;
    delayed?: boolean;
    asyncId?: number;
    asyncTo?: AsyncTo<InferState<T>>;
    promise?: AsyncResult<T>;
    cancelId?: number;
}

interface DefaultSpringProps<T> extends Pick<SpringProps<T>, 'pause' | 'cancel' | 'immediate' | 'config'>, PickEventFns<SpringProps<T>> {
}
/**
 * Only numbers, strings, and arrays of numbers/strings are supported.
 * Non-animatable strings are also supported.
 */
declare class SpringValue<T = any> extends FrameValue<T> {
    /** The property name used when `to` or `from` is an object. Useful when debugging too. */
    key?: string;
    /** The animation state */
    animation: Animation<T>;
    /** The queue of pending props */
    queue?: SpringUpdate<T>[];
    /** Some props have customizable default values */
    defaultProps: DefaultSpringProps<T>;
    /** The state for `runAsync` calls */
    protected _state: RunAsyncState<SpringValue<T>>;
    /** The promise resolvers of pending `start` calls */
    protected _pendingCalls: Set<AnimationResolver<this>>;
    /** The counter for tracking `scheduleProps` calls */
    protected _lastCallId: number;
    /** The last `scheduleProps` call that changed the `to` prop */
    protected _lastToId: number;
    protected _memoizedDuration: number;
    constructor(from: Exclude<T, object>, props?: SpringUpdate<T>);
    constructor(props?: SpringUpdate<T>);
    /** Equals true when not advancing on each frame. */
    get idle(): boolean;
    get goal(): T;
    get velocity(): VelocityProp<T>;
    /**
     * When true, this value has been animated at least once.
     */
    get hasAnimated(): boolean;
    /**
     * When true, this value has an unfinished animation,
     * which is either active or paused.
     */
    get isAnimating(): boolean;
    /**
     * When true, all current and future animations are paused.
     */
    get isPaused(): boolean;
    /**
     *
     *
     */
    get isDelayed(): boolean | undefined;
    /** Advance the current animation by a number of milliseconds */
    advance(dt: number): void;
    /** Set the current value, while stopping the current animation */
    set(value: T | FluidValue<T>): this;
    /**
     * Freeze the active animation in time, as well as any updates merged
     * before `resume` is called.
     */
    pause(): void;
    /** Resume the animation if paused. */
    resume(): void;
    /** Skip to the end of the current animation. */
    finish(): this;
    /** Push props into the pending queue. */
    update(props: SpringUpdate<T>): this;
    /**
     * Update this value's animation using the queue of pending props,
     * and unpause the current animation (if one is frozen).
     *
     * When arguments are passed, a new animation is created, and the
     * queued animations are left alone.
     */
    start(): AsyncResult<this>;
    start(props: SpringUpdate<T>): AsyncResult<this>;
    start(to: T, props?: SpringProps<T>): AsyncResult<this>;
    /**
     * Stop the current animation, and cancel any delayed updates.
     *
     * Pass `true` to call `onRest` with `cancelled: true`.
     */
    stop(cancel?: boolean): this;
    /** Restart the animation. */
    reset(): void;
    /** @internal */
    eventObserved(event: FrameValue.Event): void;
    /**
     * Parse the `to` and `from` range from the given `props` object.
     *
     * This also ensures the initial value is available to animated components
     * during the render phase.
     */
    protected _prepareNode(props: {
        to?: any;
        from?: any;
        reverse?: boolean;
        default?: any;
    }): {
        to: any;
        from: any;
    };
    /** Every update is processed by this method before merging. */
    protected _update({ ...props }: SpringProps<T>, isLoop?: boolean): AsyncResult<SpringValue<T>>;
    /** Merge props into the current animation */
    protected _merge(range: AnimationRange<T>, props: RunAsyncProps<SpringValue<T>>, resolve: AnimationResolver<SpringValue<T>>): void;
    /** Update the `animation.to` value, which might be a `FluidValue` */
    protected _focus(value: T | FluidValue<T>): void;
    protected _attach(): void;
    protected _detach(): void;
    /**
     * Update the current value from outside the frameloop,
     * and return the `Animated` node.
     */
    protected _set(arg: T | FluidValue<T>, idle?: boolean): Animated | undefined;
    protected _onStart(): void;
    protected _onChange(value: T, idle?: boolean): void;
    protected _start(): void;
    protected _resume(): void;
    /**
     * Exit the frameloop and notify `onRest` listeners.
     *
     * Always wrap `_stop` calls with `batchedUpdates`.
     */
    protected _stop(goal?: any, cancel?: boolean): void;
}

/** Queue of pending updates for a `Controller` instance. */
interface ControllerQueue<State extends Lookup = Lookup> extends Array<ControllerUpdate<State, any> & {
    /** The keys affected by this update. When null, all keys are affected. */
    keys: string[] | null;
}> {
}
declare class Controller<State extends Lookup = Lookup> {
    readonly id: number;
    /** The animated values */
    springs: SpringValues<State>;
    /** The queue of props passed to the `update` method. */
    queue: ControllerQueue<State>;
    /**
     * The injected ref. When defined, render-based updates are pushed
     * onto the `queue` instead of being auto-started.
     */
    ref?: SpringRef<State>;
    /** Custom handler for flushing update queues */
    protected _flush?: ControllerFlushFn<this>;
    /** These props are used by all future spring values */
    protected _initialProps?: Lookup;
    /** The counter for tracking `scheduleProps` calls */
    protected _lastAsyncId: number;
    /** The values currently being animated */
    protected _active: Set<FrameValue<any>>;
    /** The values that changed recently */
    protected _changed: Set<FrameValue<any>>;
    /** Equals false when `onStart` listeners can be called */
    protected _started: boolean;
    private _item?;
    /** State used by the `runAsync` function */
    protected _state: RunAsyncState<this>;
    /** The event queues that are flushed once per frame maximum */
    protected _events: {
        onStart: Map<((result: AnimationResult<SpringValue<State>>, ctrl: Controller<State>, item?: any) => void) | ((result: AnimationResult<SpringValue<State>>, ctrl: Controller<State>, item: any) => void), AnimationResult<any>>;
        onChange: Map<((result: AnimationResult<SpringValue<State>>, ctrl: Controller<State>, item?: any) => void) | ((result: AnimationResult<SpringValue<State>>, ctrl: Controller<State>, item: any) => void), AnimationResult<any>>;
        onRest: Map<((result: AnimationResult<SpringValue<State>>, ctrl: Controller<State>, item?: any) => void) | ((result: AnimationResult<SpringValue<State>>, ctrl: Controller<State>, item: any) => void), AnimationResult<any>>;
    };
    constructor(props?: ControllerUpdate<State> | null, flush?: ControllerFlushFn<any>);
    /**
     * Equals `true` when no spring values are in the frameloop, and
     * no async animation is currently active.
     */
    get idle(): boolean;
    get item(): any;
    set item(item: any);
    /** Get the current values of our springs */
    get(): State & UnknownProps;
    /** Set the current values without animating. */
    set(values: Partial<State>): void;
    /** Push an update onto the queue of each value. */
    update(props: ControllerUpdate<State> | Falsy): this;
    /**
     * Start the queued animations for every spring, and resolve the returned
     * promise once all queued animations have finished or been cancelled.
     *
     * When you pass a queue (instead of nothing), that queue is used instead of
     * the queued animations added with the `update` method, which are left alone.
     */
    start(props?: OneOrMore<ControllerUpdate<State>> | null): AsyncResult<this>;
    /** Stop all animations. */
    stop(): this;
    /** Stop animations for the given keys. */
    stop(keys: OneOrMore<string>): this;
    /** Cancel all animations. */
    stop(cancel: boolean): this;
    /** Cancel animations for the given keys. */
    stop(cancel: boolean, keys: OneOrMore<string>): this;
    /** Stop some or all animations. */
    stop(keys?: OneOrMore<string>): this;
    /** Cancel some or all animations. */
    stop(cancel: boolean, keys?: OneOrMore<string>): this;
    /** Freeze the active animation in time */
    pause(keys?: OneOrMore<string>): this;
    /** Resume the animation if paused. */
    resume(keys?: OneOrMore<string>): this;
    /** Call a function once per spring value */
    each(iterator: (spring: SpringValue, key: string) => void): void;
    /** @internal Called at the end of every animation frame */
    protected _onFrame(): void;
    /** @internal */
    eventObserved(event: FrameValue.Event): void;
}
/**
 * For testing whether a type is an object but not an array.
 *
 *     T extends IsPlainObject<T> ? true : false
 *
 * When `any` is passed, the resolved type is `true | false`.
 */
type IsPlainObject<T> = T extends ReadonlyArray<any> ? Any$2 : T extends object ? object : Any$2;
type StringKeys<T> = T extends IsPlainObject<T> ? string & keyof T : string;

/** The flush function that handles `start` calls */
type ControllerFlushFn<T extends Controller<any> = Controller> = (ctrl: T, queue: ControllerQueue<InferState<T>>) => AsyncResult<T>;
/**
 * An async function that can update or stop the animations of a spring.
 * Typically defined as the `to` prop.
 *
 * The `T` parameter can be a set of animated values (as an object type)
 * or a primitive type for a single animated value.
 */
interface SpringToFn<T = any> {
    (start: StartFn<T>, stop: StopFn<T>): Promise<any> | void;
}
type StartFn<T> = InferTarget<T> extends {
    start: infer T;
} ? T : never;
type StopFn<T> = InferTarget<T> extends {
    stop: infer T;
} ? T : never;
type EventHandler<TResult extends Readable = any, TSource = unknown, Item = undefined> = Item extends undefined ? (result: AnimationResult<TResult>, ctrl: TSource, item?: Item) => void : (result: AnimationResult<TResult>, ctrl: TSource, item: Item) => void;
/**
 * Called before the first frame of every animation.
 * From inside the `requestAnimationFrame` callback.
 */
type OnStart<TResult extends Readable, TSource, Item = undefined> = EventHandler<TResult, TSource, Item>;
/** Called when a `SpringValue` changes */
type OnChange<TResult extends Readable, TSource, Item = undefined> = EventHandler<TResult, TSource, Item>;
type OnPause<TResult extends Readable, TSource, Item = undefined> = EventHandler<TResult, TSource, Item>;
type OnResume<TResult extends Readable, TSource, Item = undefined> = EventHandler<TResult, TSource, Item>;
/** Called once the animation comes to a halt */
type OnRest<TResult extends Readable, TSource, Item = undefined> = EventHandler<TResult, TSource, Item>;
type OnResolve<TResult extends Readable, TSource, Item = undefined> = EventHandler<TResult, TSource, Item>;
/**
 * Called after an animation is updated by new props,
 * even if the animation remains idle.
 */
type OnProps<T = unknown> = (props: Readonly<SpringProps<T>>, spring: SpringValue<T>) => void;

declare enum TransitionPhase {
    /** This transition is being mounted */
    MOUNT = "mount",
    /** This transition is entering or has entered */
    ENTER = "enter",
    /** This transition had its animations updated */
    UPDATE = "update",
    /** This transition will expire after animating */
    LEAVE = "leave"
}
type UseTransitionProps<Item = any> = Merge<Omit<ControllerProps<UnknownProps, Item>, 'onResolve'>, {
    from?: TransitionFrom<Item>;
    initial?: TransitionFrom<Item>;
    enter?: TransitionTo<Item>;
    update?: TransitionTo<Item>;
    leave?: TransitionTo<Item>;
    keys?: ItemKeys<Item>;
    sort?: (a: Item, b: Item) => number;
    trail?: number;
    exitBeforeEnter?: boolean;
    /**
     * When `true` or `<= 0`, each item is unmounted immediately after its
     * `leave` animation is finished.
     *
     * When `false`, items are never unmounted.
     *
     * When `> 0`, this prop is used in a `setTimeout` call that forces a
     * rerender if the component that called `useTransition` doesn't rerender
     * on its own after an item's `leave` animation is finished.
     */
    expires?: boolean | number | ((item: Item) => boolean | number);
    config?: SpringConfig | ((item: Item, index: number, state: TransitionPhase) => AnimationProps['config']);
    /**
     * Called after a transition item is unmounted.
     */
    onDestroyed?: (item: Item, key: Key) => void;
    /**
     * Used to access the imperative API.
     *
     * Animations never auto-start when `ref` is defined.
     */
    ref?: SpringRef;
}>;
type Key = string | number;
type ItemKeys<T = any> = OneOrMore<Key> | ((item: T) => Key) | null;
type TransitionFrom<Item> = Falsy | GoalProp<UnknownProps> | ((item: Item, index: number) => GoalProp<UnknownProps> | Falsy);
type TransitionTo<Item, State extends Lookup = Lookup> = Falsy | OneOrMore<ControllerUpdate<State, Item>> | Function | ((item: Item, index: number) => ControllerUpdate<State, Item> | SpringChain<State> | SpringToFn<State> | Falsy);
/**
 * The props of a `useSpring` call or its async `update` function.
 *
 * The `T` parameter can be a set of animated values (as an object type)
 * or a primitive type for a single animated value.
 */
type SpringUpdate<T = any> = ToProps<T> & SpringProps<T>;
/**
 * Use the `SpringUpdate` type if you need the `to` prop to exist.
 * For function types, prefer one overload per possible `to` prop
 * type (for better type inference).
 *
 * The `T` parameter can be a set of animated values (as an object type)
 * or a primitive type for a single animated value.
 */
interface SpringProps<T = any> extends AnimationProps<T> {
    from?: GoalValue<T>;
    loop?: LoopProp<SpringUpdate>;
    /**
     * Called after an animation is updated by new props,
     * even if the animation remains idle.
     */
    onProps?: EventProp<OnProps<T>>;
    /**
     * Called when an animation moves for the first time.
     */
    onStart?: EventProp<OnStart<SpringValue<T>, SpringValue<T>>>;
    /**
     * Called when a spring has its value changed.
     */
    onChange?: EventProp<OnChange<SpringValue<T>, SpringValue<T>>>;
    onPause?: EventProp<OnPause<SpringValue<T>, SpringValue<T>>>;
    onResume?: EventProp<OnResume<SpringValue<T>, SpringValue<T>>>;
    /**
     * Called when all animations come to a stand-still.
     */
    onRest?: EventProp<OnRest<SpringValue<T>, SpringValue<T>>>;
}
/**
 * A union type of all possible `to` prop values.
 *
 * This is not recommended for function types. Instead, you should declare
 * an overload for each `to` type. See `SpringUpdateFn` for an example.
 *
 * The `T` parameter can be a set of animated values (as an object type)
 * or a primitive type for a single animated value.
 */
type ToProps<T = any> = {
    to?: GoalProp<T> | SpringToFn<T> | SpringChain<T>;
} | ([T] extends [IsPlainObject<T>] ? InlineToProps<T> : never);
/**
 * A value or set of values that can be animated from/to.
 *
 * The `T` parameter can be a set of animated values (as an object type)
 * or a primitive type for a single animated value.
 */
type GoalProp<T> = [T] extends [IsPlainObject<T>] ? GoalValues<T> | Falsy : GoalValue<T>;
/** A set of values for a `Controller` to animate from/to. */
type GoalValues<T> = FluidProps<T> extends infer Props ? {
    [P in keyof Props]?: Props[P] | null;
} : never;
/**
 * A value that `SpringValue` objects can animate from/to.
 *
 * The `UnknownProps` type lets you pass in { a: 1 } if the `key`
 * property of `SpringValue` equals "a".
 */
type GoalValue<T> = T | FluidValue<T> | UnknownProps | null | undefined;
/**
 * Where `to` is inferred from non-reserved props
 *
 * The `T` parameter can be a set of animated values (as an object type)
 * or a primitive type for a single animated value.
 */
type InlineToProps<T = any> = Remap<GoalValues<T> & {
    to?: undefined;
}>;
/** A serial queue of spring updates. */
interface SpringChain<T = any> extends Array<[
    T
] extends [IsPlainObject<T>] ? ControllerUpdate<T> : SpringTo<T> | SpringUpdate<T>> {
}
/** A value that any `SpringValue` or `Controller` can animate to. */
type SpringTo<T = any> = ([T] extends [IsPlainObject<T>] ? never : T | FluidValue<T>) | SpringChain<T> | SpringToFn<T> | Falsy;
type ControllerUpdate<State extends Lookup = Lookup, Item = undefined> = unknown & ToProps<State> & ControllerProps<State, Item>;
/**
 * Props for `Controller` methods and constructor.
 */
interface ControllerProps<State extends Lookup = Lookup, Item = undefined> extends AnimationProps<State> {
    ref?: SpringRef<State>;
    from?: GoalValues<State> | Falsy;
    loop?: LoopProp<ControllerUpdate>;
    /**
     * Called when the # of animating values exceeds 0
     *
     * Also accepts an object for per-key events
     */
    onStart?: OnStart<SpringValue<State>, Controller<State>, Item> | {
        [P in keyof State]?: OnStart<SpringValue<State[P]>, Controller<State>, Item>;
    };
    /**
     * Called when the # of animating values hits 0
     *
     * Also accepts an object for per-key events
     */
    onRest?: OnRest<SpringValue<State>, Controller<State>, Item> | {
        [P in keyof State]?: OnRest<SpringValue<State[P]>, Controller<State>, Item>;
    };
    /**
     * Called once per frame when animations are active
     *
     * Also accepts an object for per-key events
     */
    onChange?: OnChange<SpringValue<State>, Controller<State>, Item> | {
        [P in keyof State]?: OnChange<SpringValue<State[P]>, Controller<State>, Item>;
    };
    onPause?: OnPause<SpringValue<State>, Controller<State>, Item> | {
        [P in keyof State]?: OnPause<SpringValue<State[P]>, Controller<State>, Item>;
    };
    onResume?: OnResume<SpringValue<State>, Controller<State>, Item> | {
        [P in keyof State]?: OnResume<SpringValue<State[P]>, Controller<State>, Item>;
    };
    /**
     * Called after an animation is updated by new props.
     * Useful for manipulation
     *
     * Also accepts an object for per-key events
     */
    onProps?: OnProps<State> | {
        [P in keyof State]?: OnProps<State[P]>;
    };
    /**
     * Called when the promise for this update is resolved.
     */
    onResolve?: OnResolve<SpringValue<State>, Controller<State>, Item>;
}
type LoopProp<T extends object> = boolean | T | (() => boolean | T);
type VelocityProp<T = any> = T extends ReadonlyArray<number | string> ? number[] : number;
/** For props that can be set on a per-key basis. */
type MatchProp<T> = boolean | OneOrMore<StringKeys<T>> | ((key: StringKeys<T>) => boolean);
/** Event props can be customized per-key. */
type EventProp<T> = T | Lookup<T | undefined>;
/**
 * Most of the reserved animation props, except `to`, `from`, `loop`,
 * and the event props.
 */
interface AnimationProps<T = any> {
    /**
     * Configure the spring behavior for each key.
     */
    config?: SpringConfig | ((key: StringKeys<T>) => SpringConfig);
    /**
     * Milliseconds to wait before applying the other props.
     */
    delay?: number | ((key: StringKeys<T>) => number);
    /**
     * When true, props jump to their goal values instead of animating.
     */
    immediate?: MatchProp<T>;
    /**
     * Cancel all animations by using `true`, or some animations by using a key
     * or an array of keys.
     */
    cancel?: MatchProp<T>;
    /**
     * Pause all animations by using `true`, or some animations by using a key
     * or an array of keys.
     */
    pause?: MatchProp<T>;
    /**
     * Start the next animations at their values in the `from` prop.
     */
    reset?: MatchProp<T>;
    /**
     * Swap the `to` and `from` props.
     */
    reverse?: boolean;
    /**
     * Override the default props with this update.
     */
    default?: boolean | SpringProps<T>;
}
interface ReservedEventProps {
    onProps?: any;
    onStart?: any;
    onChange?: any;
    onPause?: any;
    onResume?: any;
    onRest?: any;
    onResolve?: any;
    onDestroyed?: any;
}

interface ControllerUpdateFn<State extends Lookup = Lookup> {
    (i: number, ctrl: Controller<State>): ControllerUpdate<State> | Falsy;
}
interface SpringRef<State extends Lookup = Lookup> {
    (props?: ControllerUpdate<State> | ControllerUpdateFn<State>): AsyncResult<Controller<State>>[];
    current: Controller<State>[];
    /** Add a controller to this ref */
    add(ctrl: Controller<State>): void;
    /** Remove a controller from this ref */
    delete(ctrl: Controller<State>): void;
    /** Pause all animations. */
    pause(): this;
    /** Pause animations for the given keys. */
    pause(keys: OneOrMore<string>): this;
    /** Pause some or all animations. */
    pause(keys?: OneOrMore<string>): this;
    /** Resume all animations. */
    resume(): this;
    /** Resume animations for the given keys. */
    resume(keys: OneOrMore<string>): this;
    /** Resume some or all animations. */
    resume(keys?: OneOrMore<string>): this;
    /** Update the state of each controller without animating. */
    set(values: Partial<State>): void;
    /** Update the state of each controller without animating based on their passed state. */
    set(values: (index: number, ctrl: Controller<State>) => Partial<State>): void;
    /** Start the queued animations of each controller. */
    start(): AsyncResult<Controller<State>>[];
    /** Update every controller with the same props. */
    start(props: ControllerUpdate<State>): AsyncResult<Controller<State>>[];
    /** Update controllers based on their state. */
    start(props: ControllerUpdateFn<State>): AsyncResult<Controller<State>>[];
    /** Start animating each controller. */
    start(props?: ControllerUpdate<State> | ControllerUpdateFn<State>): AsyncResult<Controller<State>>[];
    /** Stop all animations. */
    stop(): this;
    /** Stop animations for the given keys. */
    stop(keys: OneOrMore<string>): this;
    /** Cancel all animations. */
    stop(cancel: boolean): this;
    /** Cancel animations for the given keys. */
    stop(cancel: boolean, keys: OneOrMore<string>): this;
    /** Stop some or all animations. */
    stop(keys?: OneOrMore<string>): this;
    /** Cancel some or all animations. */
    stop(cancel: boolean, keys?: OneOrMore<string>): this;
    /** Add the same props to each controller's update queue. */
    update(props: ControllerUpdate<State>): this;
    /** Generate separate props for each controller's update queue. */
    update(props: ControllerUpdateFn<State>): this;
    /** Add props to each controller's update queue. */
    update(props: ControllerUpdate<State> | ControllerUpdateFn<State>): this;
    _getProps(arg: ControllerUpdate<State> | ControllerUpdateFn<State>, ctrl: Controller<State>, index: number): ControllerUpdate<State> | Falsy;
}
declare const SpringRef: <State extends Lookup<any> = Lookup<any>>() => SpringRef<State>;

interface AriaColorSwatchProps extends AriaLabelingProps, DOMProps {
    /** The color value to display in the swatch. */
    color?: string | Color | null;
    /**
     * A localized accessible name for the color.
     * By default, a description is generated from the color value,
     * but this can be overridden if you have a more specific color
     * name (e.g. Pantone colors).
     */
    colorName?: string;
}

/*
 * Copyright 2020 Adobe. All rights reserved.
 * This file is licensed to you under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under
 * the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR REPRESENTATIONS
 * OF ANY KIND, either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */



// Not extending CollectionBase to avoid async loading props
interface ActionGroupProps$1<T> extends MultipleSelection {
  /**
   * The axis the ActionGroup should align with.
   * @default 'horizontal'
   */
  orientation?: Orientation,
  /** An list of `Item` elements or a function. If the latter, a list of items must be provided using the `items` prop. */
  children: ItemElement<T> | ItemElement<T>[] | ItemRenderer<T>,
  /** A list of items to display as children. Must be used with a function as the sole child. */
  items?: Iterable<T>,
  /** A list of keys to disable. */
  disabledKeys?: Iterable<Key$1>,
  /**
   * Whether the ActionGroup is disabled.
   * Shows that a selection exists, but is not available in that circumstance.
   */
  isDisabled?: boolean,
  /**
   * Invoked when an action is taken on a child. Especially useful when `selectionMode` is none.
   * The sole argument `key` is the key for the item.
   */
  onAction?: (key: Key$1) => void
}

interface AriaActionGroupProps<T> extends ActionGroupProps$1<T>, DOMProps, AriaLabelingProps {}

interface ListOption<T = Record<string, any>> {
    id: string;
    label: string;
    description?: string;
    icon?: React__default.FC<IconProps>;
    data?: T;
}
interface ListSection {
    id: string;
    title?: string;
    children: ListOption[];
}
declare type ListItem = ListOption | ListSection;

declare type DragAndDropProps = Omit<DraggableCollectionStateOptions & DraggableCollectionOptions & DroppableCollectionOptions & DroppableCollectionStateOptions, "keyboardDelegate" | "dropTargetDelegate" | "shouldAcceptItemDrop" | "onDropEnter" | "onDropActivate" | "onDropExit" | "getDropOperation" | "collection" | "selectionManager" | "onRootDrop" | "onInsert" | "getAllowedDropOperations" | "getItems" | "onItemDrop"> & Partial<Pick<DraggableCollectionStateOptions, "getItems">> & {
    /**
     * Indicates whether reordering is enabled.
     *
     * @default false
     */
    enableReorder: boolean;
    /**
     * Whether the items are arranged in a stack or grid.
     *
     * @default stack
     */
    layout: "grid" | "stack";
    /**
     * The primary orientation of the items. Usually this is the direction that
     * the collection scrolls.
     *
     * @default vertical
     */
    orientation: Orientation;
};

interface BlockProps$1 {
    /**
     * The unique identifier for the block. This is used to identify the block in
     * the DOM and in the block map. It is added as a data attribute
     * `data-block-id` to the root element of the block if a DOM node is
     * rendered.
     */
    "data-block-id"?: string;
    /**
     * Represents a data block group. This is similar to `data-block-id` but it
     * doesn't have to be unique just like `class`. This is used to group blocks
     * together in the DOM and in the block map. It is added as a data attribute
     * `data-block-class` to the root element of the block if a DOM node is
     * rendered.
     */
    "data-block-class"?: string;
}
interface StylingProps extends BlockProps$1 {
    /** The className applied to the root element of the component. */
    className?: string;
    /** The style applied to the root element of the component. */
    style?: React__default.CSSProperties;
}
interface Rect {
    /** The left position of the rectangle. */
    left: number;
    /** The top position of the rectangle. */
    top: number;
    /** The width of the rectangle. */
    width?: number;
    /** The height of the rectangle. */
    height?: number;
}

/**
 * Check if the selector is inside an overlay content
 *
 * @param selector
 */
declare function isInsideOverlayContent(selector: HTMLElement): Element | null;

/**
 * Check if the given object is a `Rect`.
 *
 * @param rect
 */
declare function isRect(rect: Rect): rect is Rect;

interface ListBoxHandle {
    scrollIntoView: (id: string, options: ScrollIntoViewOptions) => void;
}
interface ListBoxProps extends StylingProps, Omit<AriaListBoxProps<ListItem> & AriaListBoxOptions<ListItem>, "children" | "linkBehavior" | "isVirtualized" | "keyboardDelegate" | "items">, Omit<DragAndDropProps, "preview" | "enableReorder" | "orientation" | "layout">, Partial<Pick<DragAndDropProps, "enableReorder" | "orientation" | "layout">> {
    /**
     * The items to render in the listbox. Items have the following shape:
     *
     * ```ts
     * export type ListOption = {
     *   id: string;
     *   label: string;
     *   description?: string;
     *   icon?: React.FC<IconProps>;
     * };
     *
     * export type ListSection = {
     *   id: string;
     *   title?: string;
     *   type: "section";
     *   children: ListOption[];
     * };
     *
     * type ListItem = ListOption | ListSection;
     * ```
     */
    items?: ListItem[];
    /**
     * The custom render function for the listbox options.
     *
     * @param item Node<ListItem>
     * @param options [OptionAria]()
     * @param ref React.RefObject<HTMLLIElement>
     */
    renderOption?: (item: Node$1<ListItem>, options: OptionAria & {
        showSelectedIcon?: boolean;
        _state: ListState<ListItem>;
    }, ref: React__default.Ref<HTMLLIElement>) => React__default.ReactNode;
    /** Renders the drop indicator component. */
    renderDropIndicator?: (options: {
        dropIndicatorProps: React__default.HTMLAttributes<HTMLLIElement>;
        isDropTarget: boolean;
        isHidden: boolean;
        orientation: Orientation;
    }, ref: React__default.RefObject<HTMLLIElement>) => React__default.ReactNode;
    /**
     * The custom render function for the drag preview. This can be used to render
     * a custom drag preview when dragging items.
     */
    renderDragPreview?: (items: DragItem[]) => React__default.ReactNode;
    /**
     * The custom render function for the listbox sections.
     *
     * @param section ListSection
     * @param ref React.RefObject<HTMLDivElement>
     */
    renderSectionHeader?: (section: Node$1<ListItem>, ref: React__default.Ref<HTMLSpanElement>) => React__default.ReactNode;
    /** Whether to show the selected checkmark icon. */
    showSelectedIcon?: boolean;
    /** The label for the listbox. */
    label?: React__default.ReactNode;
    /** The CSS class name for the drop indicator. */
    dropIndicatorClassName?: string;
    /** The style of the drop indicator used in a component. */
    dropIndicatorStyle?: React__default.CSSProperties;
    /** The CSS class name for the option. */
    optionClassName?: string | ((item: ListOption) => string);
    /** The CSS class name for the section. */
    sectionClassName?: string | ((item: ListSection) => string);
    /** The style of the option. */
    optionStyle?: React__default.CSSProperties | ((item: ListOption) => React__default.CSSProperties);
    /** The style of the section. */
    sectionStyle?: React__default.CSSProperties | ((item: ListSection) => React__default.CSSProperties);
    /**
     * Whether to show each section title when provided.
     *
     * @default false
     */
    showSectionHeader?: boolean;
    /**
     * Wether to add initial padding to section titles if shown.
     *
     * @default false
     */
    withSectionHeaderPadding?: boolean;
    /** Handle to access the listbox methods. */
    listBoxHandle?: React__default.RefObject<ListBoxHandle>;
}

declare const ListBox: React__default.ForwardRefExoticComponent<ListBoxProps & React__default.RefAttributes<HTMLUListElement>>;

interface ActionButtonSharedProps extends Omit<AriaButtonProps<"button" | "div">, "rel" | "href" | "target" | "children">, StylingProps {
    /**
     * There are some cases where we need to use a button that is not interactive.
     * For example, when we need to use a button inside a TabItem component that
     * should remove the tab when clicked. Since this button never needs to be
     * focused directly, we can set this prop to true to avoid the button being
     * focusable. At the same time we also avoid the [nested element
     * rule](https://dequeuniversity.com/rules/axe/html/4.6/nested-interactive)
     */
    UNSAFE_NON_INTERACTIVE?: boolean;
}
interface ActionButtonProps extends ActionButtonSharedProps {
    /**
     * The button's variant.
     *
     * @default primary
     */
    variant?: "primary" | "secondary" | "tertiary" | "ghost" | "popover" | "toolbar" | "success" | "warning" | "error";
    /** The button's label. */
    label: React__default.ReactNode;
    /** The button's icon before the label. */
    iconStart?: React__default.FC<IconProps>;
    /** The button's icon after the label. */
    iconEnd?: React__default.FC<IconProps>;
    /**
     * The button's size.
     *
     * @default sm
     */
    size?: "sm" | "md" | "lg";
    /** Indicates whether focusing should be prevented on press. */
    preventFocusOnPress?: boolean;
}

declare const ActionButton: React__default.ForwardRefExoticComponent<ActionButtonProps & React__default.RefAttributes<HTMLButtonElement>>;

interface DomNodeRendererProps extends StylingProps {
    /** The class name applied to the wrapper `div` component. */
    className?: string;
    /**
     * The DOM node to render inside the component. If this is not provided, the
     * component will render nothing. If this is provided, the component will
     * render the DOM node inside a `div` component with `display` set to
     * `contents`.
     */
    node: Node;
}

declare const DomNodeRenderer: React__default.ForwardRefExoticComponent<DomNodeRendererProps & React__default.RefAttributes<HTMLDivElement>>;

declare type OverlayTriggerProps_ = Parameters<typeof useOverlayTrigger>[0];
interface PopoverProps extends OverlayTriggerProps$1, OverlayTriggerProps_ {
    /** The children of the popover. */
    children: React__default.ReactNode;
}
interface PopoverContentProps extends Omit<AriaPopoverProps, "popoverRef" | "triggerRef" | "arrowSize" | "arrowBoundaryOffset">, StylingProps {
    /**
     * The reference element for the popover position. By default, the popover is
     * positioned relative to the trigger element.
     */
    triggerRef?: React__default.RefObject<HTMLElement>;
    /** The class name for the underlay element. */
    underlayClassName?: string;
    /** The contents of the popover. */
    children: React__default.ReactNode | ((state: OverlayTriggerState) => React__default.ReactNode);
    /**
     * The container element for the popover. By default, the modal is rendered as
     * a child of the body element.
     *
     * @default document.body
     */
    portalContainer?: HTMLElement;
    /**
     * Whether to include an arrow on the popover.
     *
     * @default false
     */
    includeArrow?: boolean;
    /** The class name for the content container element. */
    contentContainerClassName?: string;
    /** The style for the content container element. */
    contentContainerStyle?: React__default.CSSProperties;
}
interface PopoverTriggerProps {
    /** The contents of the popover trigger. */
    children?: React__default.ReactNode | (({ isOpen, triggerProps, triggerRef, }: {
        isOpen: boolean;
        triggerProps: AriaButtonProps<"button">;
        triggerRef: React__default.RefObject<HTMLElement>;
    }) => React__default.ReactNode);
    /**
     * Client rect to be used as anchor for the popover content when no child
     * element is provided.
     */
    anchorRect?: Rect;
}

declare const PopoverContent: React__default.ForwardRefExoticComponent<PopoverContentProps & React__default.RefAttributes<HTMLDivElement>>;

declare const PopoverTrigger: React__default.FC<PopoverTriggerProps>;

declare const Popover: React__default.FC<PopoverProps>;

interface TooltipProps extends StylingProps, TooltipTriggerProps, AriaTooltipProps, Omit<AriaPositionProps, "overlayRef" | "targetRef" | "maxHeight" | "isOpen" | "arrowSize"> {
    /** The content of the tooltip. */
    text?: string;
    /** The tooltip trigger element. */
    children: React__default.ReactNode;
    /**
     * The delay time for the tooltip to show up.
     *
     * @default 1000
     */
    delay?: number;
    /**
     * The delay time for the tooltip to hide.
     *
     * @default 500
     */
    closeDelay?: number;
    /**
     * Represents the size of an element.
     *
     * @default md
     */
    size?: "sm" | "md";
    /**
     * The variant that can be applied to an element.
     *
     * @default default
     */
    variant?: "default" | "inverse";
    /**
     * Indicates whether to include an arrow in the output.
     *
     * @default true
     */
    includeArrow?: boolean;
    /**
     * The container element for the modal. By default, the modal is rendered as a
     * child of the body element.
     *
     * @default document.body
     */
    portalContainer?: HTMLElement;
}

declare const Tooltip: React__default.ForwardRefExoticComponent<TooltipProps & React__default.RefAttributes<HTMLDivElement>>;

interface IconComponentProps {
    /** The icon to show at the start of the color input. */
    icon?: React__default.FC<IconProps>;
    /** The aria label for the color input button. */
    "aria-label": string;
    /** Props to pass to the tooltip component. */
    tooltipProps?: Omit<TooltipProps, "variant" | "size" | "includeArrow" | "children">;
}

interface SliderProps extends Omit<AriaSliderProps<number> & SliderStateOptions<number>, "orientation" | "numberFormatter">, StylingProps {
    /**
     * Whether the slider is read only.
     *
     * @default false
     */
    isReadOnly?: boolean;
    /** The format options passed to `Intl.NumberFormat`. */
    numberFormatOptions?: Intl.NumberFormatOptions & {
        numberingSystem?: string;
    };
    /** The label for the slider. */
    label?: React__default.ReactNode;
    /**
     * Whether to include a number input for the slider.
     *
     * @default false
     */
    includeNumberInput?: boolean;
}
interface IconSliderProps extends Pick<SliderProps, "value" | "minValue" | "maxValue" | "onChange" | "defaultValue" | "includeNumberInput">, StylingProps, IconComponentProps {
    /**
     * A function to format the value of the slider.
     *
     * @default value => value
     */
    formatValue?: (value: number) => string | number;
    /** The props passed to the `PopoverContent` component. */
    popoverContentProps?: Omit<PopoverContentProps, "children">;
}

declare const Slider: React__default.ForwardRefExoticComponent<SliderProps & React__default.RefAttributes<HTMLDivElement>>;

declare const IconSlider: React__default.ForwardRefExoticComponent<IconSliderProps & React__default.RefAttributes<HTMLDivElement>>;

interface SwitchProps extends Omit<AriaSwitchProps, "children">, Omit<ToggleProps, "children" | "isRequired" | "validate" | "validationBehavior" | "validationState" | "isInvalid">, StylingProps {
    /** The label to display next to the switch. */
    label?: React__default.ReactNode;
    /** The position of the label. */
    labelPosition?: "top" | "left";
    /** The status label to display next to the switch. */
    statusLabel?: {
        on: string;
        off: string;
    };
}

declare const Switch: React__default.ForwardRefExoticComponent<SwitchProps & React__default.RefAttributes<HTMLLabelElement>>;

interface SearchInputProps extends Omit<AriaSearchFieldProps, "validationState" | "label" | "aria-autocomplete" | "autoComplete" | "isRequired" | "aria-haspopup" | "isReadOnly" | "aria-activedescendant" | "type" | "validationBehavior" | "validate" | "isInvalid">, StylingProps {
    /**
     * The size of the search input.
     *
     * @default "md"
     */
    size?: "sm" | "md" | "lg";
    /**
     * The variant of the search input.
     *
     * @default "primary"
     */
    variant?: "primary" | "ghost";
    /** Defines a string value that labels the current element. */
    "aria-label": string;
}

declare const SearchInput: React__default.ForwardRefExoticComponent<SearchInputProps & React__default.RefAttributes<HTMLDivElement>>;

interface ThemeProviderProps extends StylingProps {
    /**
     * The theme to use. If not provided, the theme will be inherited from the
     * parent ThemeProvider. If there is no parent ThemeProvider, the default
     * theme will be used.
     *
     * @default "system"
     */
    theme?: Theme | "light" | "dark" | "system";
    /** The children to render. */
    children: React__default.ReactNode;
    /**
     * A function that will be called when the theme changes from any of the child
     * components.
     */
    setTheme?: (theme: Theme | "light" | "dark" | "system") => void;
}

declare const ThemeProvider: React__default.ForwardRefExoticComponent<ThemeProviderProps & React__default.RefAttributes<HTMLDivElement>>;

/**
 * A hook that returns the user preferences for color scheme, contrast, reduced
 * motion, transparency and reduced data. The values are updated when the user
 * changes their preferences.
 *
 * @example
 *   const { colorScheme, highContrast, reducedMotion, transparency } =
 *     useUserPreferences();
 *
 * @param ownerWindow - The window object to use. Defaults to the global window
 *   object.
 * @returns The user preferences.
 */
declare function useUserPreferences(ownerWindow?: Window & typeof globalThis): {
    colorScheme: "dark" | "light";
    highContrast: boolean;
    reducedMotion: boolean;
    transparency: boolean;
};

interface PortalContainerProviderProps {
    /**
     * The container element for the popover. By default, the modal is rendered as
     * a child of the body element.
     *
     * @default undefined
     */
    portalContainer: HTMLElement | undefined;
    /** The children to render. */
    children: React__default.ReactNode;
}

declare const PortalContainerProvider: React__default.FC<PortalContainerProviderProps>;

declare type MenuOption = Omit<ListOption, "description"> & {
    keyboardShortcut?: string;
};
declare type MenuSection = Omit<ListSection, "children"> & {
    children: MenuOption[];
};
declare type MenuItem = MenuOption | MenuSection;
declare type MenuPopoverProps = Pick<PopoverContentProps, "isNonModal" | "placement" | "shouldUpdatePosition" | "shouldFlip" | "boundaryElement" | "crossOffset" | "offset" | "portalContainer">;
interface MenuProps extends MenuPopoverProps, MenuTriggerProps, Omit<AriaMenuProps<MenuItem>, "children">, Omit<AriaMenuTriggerProps, "type"> {
    /** The `className` property assigned to the root element of the component. */
    className?: string;
    /** The `className` property assigned to the content element of the component. */
    contentClassName?: string;
    /** The `className` property assigned to the item element of the component. */
    itemClassName?: string;
    /**
     * A list of items to render in the menu. Items have the following shape:
     *
     * ```ts
     * export type MenuOption = {
     *   id: string;
     *   label: string;
     *   keyboardShortcut?: string;
     *   icon?: React.FC<IconProps>;
     * };
     *
     * export type MenuSection = {
     *   id: string;
     *   title?: string;
     *   type: "section";
     *   children: MenuOption[];
     * };
     *
     * export type MenuItem = MenuOption | MenuSection;
     * ```
     */
    items: MenuItem[];
    /**
     * A function that renders the trigger element of the component. The default
     * implementation renders an `ActionButton` component.
     *
     * ```tsx
     * <Menu renderTrigger={({ buttonProps, ref }) => <ActionButton {...buttonProps} label="Label" ref={ref} />
     * ```
     */
    renderTrigger?: (options: {
        buttonProps: ActionButtonProps;
        ref: React__default.RefObject<HTMLButtonElement>;
    }) => React__default.ReactNode;
    /**
     * The label of the trigger element. This can be a string, a React node, or a
     * function that accepts a boolean indicating whether the menu is open.
     */
    triggerLabel?: React__default.ReactNode | ((isOpen: boolean) => React__default.ReactNode);
}

declare const Menu: React__default.ForwardRefExoticComponent<MenuProps & React__default.RefAttributes<HTMLUListElement>>;

interface LinkProps extends AriaLinkOptions, StylingProps {
    /** The link's style variant. */
    variant?: "default" | "inline";
    /** The children to render. */
    children: React__default.ReactNode;
    /** The size of the link. */
    size?: "sm" | "md" | "lg";
    /**
     * The type of the link.
     *
     * @default "body"
     */
    type?: "body" | "label";
    /** The title of the link, for providing additional information. */
    title?: string;
}

declare const Link: React__default.ForwardRefExoticComponent<LinkProps & React__default.RefAttributes<HTMLElement>>;

interface SharedFileUploadProps extends StylingProps {
    /** Whether the component is inlined. */
    variant?: "default" | "inline";
    /** Whether the component is disabled. */
    isDisabled?: boolean;
    /** The label to display. */
    label: string;
    /** The description to display. */
    description?: string;
    /**
     * The pattern to match the file name against. This is a regular expression,
     * and will be matched against the entire file name.
     */
    accept?: string;
    /** The name of the input. */
    name?: string;
    /** The callback function that is fired when the value changes. */
    onChange?: (event: React__default.ChangeEvent<HTMLInputElement>) => void;
    /**
     * The callback function that is fired when the value changes and the value is
     * valid.
     */
    onValueChange?: (files: File[]) => void;
    /**
     * Identifies the element (or elements) that provide a detailed, extended
     * description for the object.
     */
    "aria-describedby"?: string;
}
interface FileUploadProps extends SharedFileUploadProps {
    /** The Icon to display. */
    icon?: React__default.FC<IconProps>;
    /** Whether to allow multiple files to be uploaded. */
    multiple?: boolean;
}

declare const FileUpload: React__default.ForwardRefExoticComponent<FileUploadProps & React__default.RefAttributes<HTMLLabelElement>>;

interface SeparatorProps extends SeparatorProps$1, StylingProps {
    /**
     * The variant of the separator.
     *
     * @default "primary"
     */
    variant?: "primary" | "secondary";
}

declare const Separator: React__default.ForwardRefExoticComponent<SeparatorProps & React__default.RefAttributes<HTMLDivElement>>;

interface ProgressBarProps extends StylingProps, Omit<AriaProgressBarProps, "isIndeterminate"> {
    /**
     * The variant of the progress bar.
     *
     * @default active
     */
    variant?: "active" | "success" | "error";
    /** The description of the progress bar. */
    description?: string;
    /**
     * Whether the progress bar value should be displayed.
     *
     * @default false
     */
    showValue?: boolean;
    /** The error message of the progress bar. */
    errorMessage?: string;
}

declare const ProgressBar: React__default.ForwardRefExoticComponent<ProgressBarProps & React__default.RefAttributes<HTMLDivElement>>;

interface InputMessage {
    /** The description to display below the input. */
    description?: string;
    /** The error message to display when the input is in an error state. */
    errorMessage?: string;
    /** The warning message to display when the input is in a warning state. */
    warningMessage?: string;
}
interface TextInputProps extends Omit<AriaTextFieldProps, "validationState" | "isInvalid" | "description" | "errorMessage">, StylingProps, InputMessage {
    /** The state of the input. */
    validationState?: "valid" | "error" | "warning";
    /**
     * The variant of the text input.
     *
     * @default primary
     */
    variant?: "primary" | "ghost";
    /**
     * The position of the label relative to the input.
     *
     * @default top
     */
    labelPosition?: "top" | "start";
    /** Whether the input is read only. */
    readOnly?: boolean;
    /** The style object to apply to the input element */
    inputStyle?: React__default.CSSProperties;
    /** The class name to apply to the input element */
    inputClassName?: string;
}

declare const TextInput: React__default.ForwardRefExoticComponent<TextInputProps & React__default.RefAttributes<HTMLInputElement>>;

interface ReactionProps extends Omit<AriaSwitchProps & ToggleProps, "validationState" | "isRequired" | "children" | "isInvalid" | "validationBehavior" | "validate">, StylingProps {
    /** The size of the reaction. */
    size?: "sm" | "md";
    /** The number of reactions. */
    count: number;
    /** The icon to render. */
    icon?: React__default.FC<IconProps>;
}

declare const Reaction: React__default.ForwardRefExoticComponent<ReactionProps & React__default.RefAttributes<HTMLLabelElement>>;

interface PortalProps extends OverlayProps, StylingProps {
}

declare const Portal: React__default.ForwardRefExoticComponent<PortalProps & React__default.RefAttributes<HTMLDivElement>>;

interface ProgressSpinnerProps extends Omit<AriaProgressBarProps, "value" | "isIndeterminate" | "formatOptions" | "maxValue" | "minValue" | "valueLabel">, StylingProps {
    /**
     * The size of the progress circle.
     *
     * @default md
     */
    size?: "sm" | "md";
    /**
     * The progress circle's variant.
     *
     * @default active
     */
    variant?: "active" | "inactive" | "success" | "error";
}

declare const ProgressSpinner: React__default.ForwardRefExoticComponent<ProgressSpinnerProps & React__default.RefAttributes<HTMLDivElement>>;

interface ActionIconButtonProps extends ActionButtonSharedProps {
    /** The icon to display */
    icon: React__default.FC<IconProps>;
    /**
     * The size of the icon button.
     *
     * @default "md"
     */
    size?: "xxs" | "xs" | "sm" | "md" | "lg";
    /**
     * The variant of the icon button.
     *
     * @default "primary"
     */
    variant?: "primary" | "secondary" | "tertiary" | "ghost" | "toolbar" | "popover";
    /** Indicates whether focusing should be prevented on press. */
    preventFocusOnPress?: boolean;
}

declare const ActionIconButton: React__default.ForwardRefExoticComponent<ActionIconButtonProps & React__default.RefAttributes<HTMLButtonElement>>;

interface ToggleButtonProps extends StylingProps, Omit<AriaToggleButtonProps, "children"> {
    /** The text content of the button. */
    label: React__default.ReactNode;
    /** The icon to display before the label. */
    iconStart?: React__default.FC<SVGRProps>;
    /**
     * The size of the button.
     *
     * @default "md"
     */
    size?: "sm" | "md" | "lg";
    /** Indicates whether focusing should be prevented on press. */
    preventFocusOnPress?: boolean;
}

declare const ToggleButton: React__default.ForwardRefExoticComponent<ToggleButtonProps & React__default.RefAttributes<HTMLButtonElement>>;

interface CheckboxProps extends Omit<AriaCheckboxProps, "children" | "validationBehavior" | "validationState" | "validate">, StylingProps {
    /** The checkbox's label. */
    label?: string;
    /**
     * The position of the label
     *
     * @default end
     */
    labelPosition?: "start" | "end";
}

declare const Checkbox: React__default.ForwardRefExoticComponent<CheckboxProps & React__default.RefAttributes<HTMLLabelElement>>;

declare type Excluded = "isRequired" | "locale" | "validationState" | "label" | "formatOptions" | "isInvalid" | "validationBehaviour" | "validate" | "description" | "errorMessage";
interface PaginationProps extends Omit<AriaNumberFieldProps, Excluded>, Omit<NumberFieldStateOptions, Excluded>, StylingProps {
    /** The largest value allowed for the input. */
    maxValue: number;
    /** The label for the pagination component. */
    "aria-label": string;
    /**
     * The variant of the pagination component.
     *
     * @default floating
     */
    variant?: "floating" | "pinned";
    /**
     * The size of the pagination component.
     *
     * @default md
     */
    size?: "xs" | "sm" | "md" | "lg";
    /** Whether to hide the increment and decrement actions. */
    hideActions?: boolean;
}

declare const Pagination: React__default.ForwardRefExoticComponent<PaginationProps & React__default.RefAttributes<HTMLDivElement>>;

interface I18nProviderProps extends I18nProviderProps$1 {
    /** The messages to use for internationalization. */
    messages?: LocalizedStrings;
    /** The children to render. */
    children: React__default.ReactNode;
}

declare const I18nProvider: React__default.FC<I18nProviderProps>;

declare type NumberFormatProps = Parameters<typeof useNumberFormatter>[0] & {
    /** The number to format. */
    value: number;
};

declare const NumberFormat: React__default.FC<NumberFormatProps>;

declare type DateFormatProps = Parameters<typeof useDateFormatter>[0] & {
    /** The date to format. */
    date: Date;
};

declare const DateFormat: React__default.FC<DateFormatProps>;

interface Item$1 {
    id: string;
    label: string;
    variant?: "neutral" | "red" | "green" | "blue" | "high-contrast";
    icon?: React__default.FC<SVGRProps>;
}
interface TagGroupProps extends StylingProps, Omit<AriaTagGroupProps<Item$1>, "children" | "errorMessage" | "description">, Omit<ListProps<Item$1>, "children" | "collection" | "filter"> {
    /**
     * The items to display in the tag group.
     *
     * Type Item = { id: string; label: string; variant?: "neutral" | "red" |
     * "green" | "blue" | "high-contrast"; icon?: React.FC<SVGRProps>; }
     */
    items: Item$1[];
    /**
     * The variant of the tag group.
     *
     * @default "neutral"
     */
    variant?: Item$1["variant"];
    /** The label for the tag group. */
    "aria-label": string;
}

declare const TagGroup: React__default.ForwardRefExoticComponent<TagGroupProps & React__default.RefAttributes<HTMLDivElement>>;

interface AccordionProps extends StylingProps {
    /**
     * This prop controls the expansion mode of the accordion. If set to `single`,
     * only one item can be expanded at a time. If set to `multiple`, multiple
     * items can be expanded at the same time.
     *
     * @default single
     */
    expansionMode?: "single" | "multiple";
    /** This callback is called when the expanded state of the accordion changes. */
    onChange?: (value: Set<Key$2>) => void;
    /** The children of the accordion. Each child should be an `AccordionItem`. */
    children: React__default.ReactNode;
    /** The keys of the items that should be expanded by default. */
    defaultExpandedKeys?: Set<Key$2>;
    /**
     * The keys of the items that should be expanded. If this prop is provided,
     * the accordion will be a controlled component. This means that the expanded
     * state of the accordion will not change unless you update this prop.
     */
    expandedKeys?: Set<Key$2>;
    /**
     * The keys of the items that should be disabled. Disabled items cannot be
     * expanded or collapsed.
     */
    disabledKeys?: Set<Key$2>;
}
interface AccordionItemProps extends StylingProps {
    /** The title of the accordion item. */
    title: string;
    /**
     * The buttons that should be rendered in the accordion item. These buttons
     * will be rendered in the top right corner of the accordion item. They will
     * be rendered in the order they are provided. The buttons will be rendered as
     * `ActionIconButton`s, so you can pass any props that you would pass to an
     * `ActionIconButton`.
     *
     * @example
     *   ```jsx
     *   <AccordionItem
     *    title="Accordion Item"
     *    actions={[{
     *      icon: EditIcon,
     *      label: "Edit",
     *      onPress: () => alert("Edit"),
     *    }]}>Content</AccordionItem>
     *    ```;
     */
    actions?: Omit<ActionIconButtonProps, "size" | "variant" | "excludeFromTabOrder">[];
    /**
     * The action buttons should only be shown when the user hovers over or
     * focuses the accordion item.
     *
     * @default false
     */
    showActionsOnTriggerOnly?: boolean;
    /** The content of the accordion item. */
    children: React__default.ReactNode;
    /** The key of the accordion item. */
    value: Key$2;
    /**
     * The description of the accordion item. This will be rendered at the end of
     * the accordion header.
     */
    subLabel?: string;
}

declare const Accordion: React__default.ForwardRefExoticComponent<AccordionProps & React__default.RefAttributes<HTMLDivElement>>;

declare const AccordionItem: React__default.ForwardRefExoticComponent<AccordionItemProps & React__default.RefAttributes<HTMLHeadingElement>>;

interface SelectProps extends Omit<AriaSelectOptions<ListItem> & SelectStateOptions<ListItem> & PopoverContentProps, "validationState" | "items" | "children" | "isRequired" | "className" | "style" | "triggerRef" | "validate" | "validationBehavior" | "keyboardDelegate">, Pick<ListBoxProps, "items" | "showSelectedIcon" | "optionStyle" | "optionClassName">, StylingProps {
    /**
     * The position of the label.
     *
     * @default "top"
     */
    labelPosition?: "top" | "start";
    /** The description of the select component. */
    description?: string;
    /** The error message of the select component. */
    errorMessage?: string;
    /** The warning message of the select component. */
    warningMessage?: string;
    /** The label of the select component. */
    label?: React__default.ReactNode;
    /** The variant of the select component. */
    variant?: "primary" | "ghost";
    /** The state of the select component. */
    validationState?: "valid" | "error" | "warning";
    /** Whether the select component is read only. */
    isReadOnly?: boolean;
    /**
     * This callback can be used to render the trigger element of the select
     * component. The trigger element is the element that is clicked to open the
     * select menu.
     */
    renderTrigger?: (options: {
        buttonProps: Omit<ActionButtonProps, "variant" | "label" | "elementType" | "iconStart" | "iconEnd">;
        valueProps: DOMAttributes;
        isOpen: boolean;
        selectedValue: ListItem | null;
        ref: React__default.RefObject<HTMLButtonElement>;
    }) => React__default.ReactNode;
}
interface IconSelectProps extends Omit<SelectProps, "renderTrigger" | "aria-label">, IconComponentProps {
}

declare const Select: React__default.ForwardRefExoticComponent<SelectProps & React__default.RefAttributes<HTMLDivElement>>;

declare const IconSelect: React__default.ForwardRefExoticComponent<IconSelectProps & React__default.RefAttributes<HTMLDivElement>>;

interface AvatarProps extends StylingProps {
    /**
     * The size of the component.
     *
     * @default "md"
     */
    size?: "md" | "sm";
    /**
     * The icon to display. This icon has to be imported from the
     * `@baseline-ui/icons` package.
     */
    icon?: React__default.FC<SVGRProps>;
    /** The name of the user. */
    name: string;
    /** The URL of the image to display. */
    imgSrc?: string;
    /**
     * The image loading strategy to use. See
     * [MDN](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/img#attributes:~:text=a%20fallback%20destination.-,loading,-Indicates%20how%20the)
     */
    imgLoading?: "lazy" | "eager";
    /** Whether to show the initials of the user if no image is provided. */
    showInitials?: boolean;
    /** Whether the component is disabled. */
    isDisabled?: boolean;
    /** Whether the user has a notification. */
    hasNotifications?: boolean;
}

declare const Avatar: React__default.ForwardRefExoticComponent<AvatarProps & React__default.RefAttributes<HTMLSpanElement>>;

declare type ItemProps = React__default.ComponentProps<typeof Item$2> & StylingProps & {
    key?: Key$1;
    icon?: React__default.FC<SVGRProps>;
    value?: string;
    title?: string;
    titleClassName?: string;
    titleStyle?: React__default.CSSProperties;
};
declare type TabsProps = Omit<StylingProps & AriaTabListProps<ItemProps>, "orientation" | "children" | "items" | "selectedKey" | "defaultSelectedKey" | "disabledKeys"> & {
    /**
     * The `TabItem` components to render inside the `Tabs` component.
     *
     * ```jsx
     * <Tabs>
     *   <TabItem key="tab1" title="Tab 1">
     *     <div>Tab 1 content</div>
     *   </TabItem>
     *   <TabItem key="tab2" title="Tab 2">
     *     <div>Tab 2 content</div>
     *   </TabItem>
     * </Tabs>;
     * ```
     */
    children: React__default.ReactNode;
    /**
     * The key of the tab that needs to be removed.
     *
     * @param key
     */
    onRemove?: (key: Key$1) => void;
    /** The value of the tab that needs to be selected. */
    selectedValue?: string;
    /** The default value of the tab that needs to be selected. */
    defaultSelectedValue?: string;
    /** The values of the tabs that need to be disabled. */
    disabledValues?: Set<Key$1>;
    /** A list of action buttons to render beside the tab items. */
    actions?: Omit<ActionIconButtonProps, "variant" | "size">[];
    /** The variant of the tabs */
    variant?: "primary" | "ghost";
    /** The className to apply to the tab header container. */
    tabHeaderClassName?: string;
    /** The style to apply to the tab header container. */
    tabHeaderStyle?: React__default.CSSProperties;
};
declare type TabItemProps = StylingProps & {
    /** The title of the tab. This will be displayed in the tab button. */
    title: string;
    /** The icon to display in front of the title. */
    icon?: React__default.FC<SVGRProps>;
    /** The value of the tab. This will be used to identify the tab. */
    value: string;
    /** The className to apply to the tab item. */
    titleClassName?: string;
    /** The style to apply to the tab item. */
    titleStyle?: React__default.CSSProperties;
    /** The contents of the tab item. */
    children: React__default.ReactNode;
};

declare const Tabs: React__default.ForwardRefExoticComponent<Omit<StylingProps & AriaTabListProps$1<ItemProps>, "children" | "items" | "orientation" | "disabledKeys" | "selectedKey" | "defaultSelectedKey"> & {
    children: React__default.ReactNode;
    onRemove?: ((key: Key$1) => void) | undefined;
    selectedValue?: string | undefined;
    defaultSelectedValue?: string | undefined;
    disabledValues?: Set<Key$1> | undefined;
    actions?: Omit<ActionIconButtonProps, "size" | "variant">[] | undefined;
    variant?: "primary" | "ghost" | undefined;
    tabHeaderClassName?: string | undefined;
    tabHeaderStyle?: React__default.CSSProperties | undefined;
} & React__default.RefAttributes<HTMLDivElement>>;

declare const TabItem: React__default.FC<TabItemProps>;

interface InlineAlertProps extends StylingProps {
    /**
     * The variant of the alert.
     *
     * @default "info"
     */
    variant?: "info" | "warning" | "error" | "success";
    /** The title of the alert. */
    title: string;
    /** The description of the alert. */
    description: string;
    /**
     * The label of the action button. When provided, the action button will be
     * rendered.
     */
    actionLabel?: string;
    /** The callback fired when the action button is clicked. */
    onAction?: () => void;
    /**
     * The callback fired when the close button is clicked. When provided, the
     * close button will be rendered.
     */
    onClose?: () => void;
    /**
     * The arrangement of the variable.
     *
     * @default "single"
     * @typedef {"single" | "multi" | "compact"} Arrangement
     */
    arrangement?: "single" | "multi" | "compact";
    /**
     * The size of the alert.
     *
     * @default "md"
     */
    size?: "sm" | "md";
}

declare const InlineAlert: React__default.ForwardRefExoticComponent<InlineAlertProps & React__default.RefAttributes<HTMLDivElement>>;

interface NumberInputProps extends StylingProps, Omit<AriaNumberFieldProps, "validationState" | "isRequired" | "validate" | "validationBehavior"> {
    /**
     * The position of the label relative to the input.
     *
     * @default top
     */
    labelPosition?: "top" | "start";
    /**
     * The variant of the number input.
     *
     * @default primary
     */
    variant?: "primary" | "ghost";
    /**
     * Whether to show the stepper buttons.
     *
     * @default true
     */
    showStepper?: boolean;
    /** The description to display below the input. */
    description?: string;
    /** The error message to display when the input is in an error state. */
    errorMessage?: string;
}

declare const NumberInput: React__default.ForwardRefExoticComponent<NumberInputProps & React__default.RefAttributes<HTMLDivElement>>;

interface DrawerProps extends StylingProps, AriaDialogProps {
    /** The children to render. */
    children: React__default.ReactNode;
    /**
     * The valid values for the background type are "medium" and "subtle".
     *
     * @default "medium"
     */
    background?: "medium" | "subtle";
    /** The title to display in the drawer. */
    title: string;
    /** The props of additional `ActionIconButton` to display in the drawer header. */
    action?: ActionIconButtonProps;
    /** The callback to call when the drawer is requested to be closed. */
    onCloseRequest?: () => void;
}

declare const Drawer: React__default.ForwardRefExoticComponent<DrawerProps & React__default.RefAttributes<HTMLDivElement>>;

interface DialogProps extends AriaDialogProps, StylingProps {
    /** The children to render. */
    children: React__default.ReactNode;
    /**
     * The size of the dialog. This defines the max width of the dialog.
     *
     * @default "sm"
     */
    size?: "sm" | "md" | "lg" | "content";
    /**
     * The variant of the dialog.
     *
     * @default "primary"
     */
    variant?: "primary" | "ghost";
    /**
     * The background color of the drawer.
     *
     * @default "medium"
     */
    drawerBackground?: DrawerProps["background"];
    /**
     * When used inside a modal, this prop will decide whether the dialog should
     * inherit the drawer styles.
     *
     * @default false
     */
    inheritDrawer?: boolean;
}
interface DialogTitleProps extends StylingProps {
    /** The children to render. */
    children: React__default.ReactNode;
}

declare const Dialog: React__default.ForwardRefExoticComponent<DialogProps & React__default.RefAttributes<HTMLDivElement>>;

declare const DialogTitle: React__default.ForwardRefExoticComponent<DialogTitleProps & React__default.RefAttributes<HTMLDivElement>>;

interface ColorPreset$1 {
    label: string;
    color: string;
}
interface ColorInputProps extends OverlayTriggerProps$1, StylingProps, Pick<ColorFieldProps, "onChange" | "defaultValue" | "value" | "label" | "isDisabled">, Pick<AriaLabelingProps, "aria-label">, Pick<PopoverContentProps, "isNonModal" | "placement" | "shouldUpdatePosition" | "shouldFlip" | "boundaryElement" | "crossOffset" | "offset" | "portalContainer"> {
    /**
     * The list of color presets to show in the color picker. The signature of the
     * color presets is:
     *
     * ```ts
     * export interface ColorPreset {
     *   label: string;
     *   color: string;
     * }
     * ```
     *
     * The color can be a hex or hexa value.
     */
    presets?: ColorPreset$1[];
    /**
     * Whether to include the custom color picker. If `false`, the color picker
     * will only show the color presets. The custom color picker allows the user
     * to pick any color. You can set the `allowAlpha` prop to `false` to disable
     * the alpha channel.
     *
     * @default true
     */
    includePicker?: boolean;
    /**
     * Whether to show the color label. The color label is the name of the color
     * that is currently selected. If boolean, the label will be shown only when
     * the color is not `null`. If a function, the function will be called with
     * the current color and should return a string. If `false`, the label will
     * not be shown.
     *
     * @default true
     */
    colorLabel?: boolean | ((color: Color | null) => string);
    /**
     * Whether to allow alpha values in the custom color picker.
     *
     * @default true
     */
    allowAlpha?: boolean;
    /**
     * Whether to allow the removal of the color.
     *
     * @default false
     */
    allowRemoval?: boolean;
    /** The label position of the color input. */
    labelPosition?: "top" | "start";
    /**
     * Whether to show the color picker in active or lazy mode. In active mode,
     * the color picker will be rendered when the color input popover is open and
     * the value of the color input will be updated when the user picks a color.
     * In lazy mode, the color picker will be rendered only when the user clicks
     * on the "Add color" button and the selected color will be added to custom
     * colors when the user clicks on the "Add color" button.
     *
     * @default active
     */
    pickerMode?: "active" | "lazy";
    /**
     * The key to use to store the picked color in the local storage.
     *
     * @default baselinePickedColor
     */
    storePickedColorKey?: string;
    /**
     * The label to show on the add color button.
     *
     * @default Add color
     */
    addColorButtonLabel?: string;
    /**
     * The label to show on the remove color button.
     *
     * @default Remove color
     */
    removeColorButtonLabel?: string;
    /**
     * The label to show on the custom colors section.
     *
     * @default Custom colors
     */
    customColorsLabel?: string;
    /**
     * An optional function to render the custom trigger button.
     *
     * @param options
     */
    renderTriggerButton?: (options: {
        isOpen: boolean;
        color: Color | null;
        ref: React__default.RefObject<HTMLButtonElement>;
        colorName?: string | boolean;
        triggerProps: AriaButtonProps<"button">;
        labelId?: string;
    }) => React__default.ReactElement;
}
interface IconColorInputProps extends StylingProps, Omit<ColorInputProps, "renderTriggerButton" | "colorLabel" | "label" | "labelPosition" | "aria-label">, IconComponentProps {
    /**
     * The variant of the color input button.
     *
     * @default standard
     */
    variant?: "compact" | "standard";
}

declare const ColorInput: React__default.ForwardRefExoticComponent<ColorInputProps & React__default.RefAttributes<HTMLDivElement>>;

declare const IconColorInputButton: React__default.ForwardRefExoticComponent<Omit<ColorInputButtonProps, "children" | "color" | "labelPosition"> & {
    icon: React__default.FC<IconProps>;
    color?: string | null | undefined;
} & React__default.RefAttributes<HTMLButtonElement>>;
interface ColorInputButtonProps extends Omit<ActionButtonProps, "label">, Pick<ColorInputProps, "colorLabel"> {
    isOpen: boolean;
    "aria-label"?: string;
    labelPosition?: "top" | "start";
    color: Color | null;
    colorName?: string | boolean;
}

declare const IconColorInput: React__default.ForwardRefExoticComponent<IconColorInputProps & React__default.RefAttributes<HTMLDivElement>>;

interface FreehandCanvasProps extends StylingProps, AriaLabelingProps {
    /** The className applicable to the SVG canvas. */
    canvasClassName?: string;
    /** The style applicable to the SVG canvas. */
    canvasStyle?: React__default.CSSProperties;
    /** The ref for the canvas. */
    canvasRef?: React__default.RefObject<HTMLDivElement>;
    /** The className applicable to the footer. */
    footerClassName?: string;
    /** The style applicable to the footer. */
    footerStyle?: React__default.CSSProperties;
    /**
     * How much to soften the stroke's edges.
     *
     * @default 0
     */
    smoothing?: number;
    /**
     * How much to streamline the stroke.
     *
     * @default 0
     */
    streamline?: number;
    /** An easing function to apply to each point's pressure. */
    easing?: (pressure: number) => number;
    /**
     * Whether to simulate pressure based on velocity.
     *
     * @default false
     */
    simulatePressure?: boolean;
    /** Cap, taper and easing for the start of the line. */
    start?: {
        cap?: boolean;
        taper?: number | boolean;
        easing?: (distance: number) => number;
    };
    /** Cap, taper and easing for the end of the line. */
    end?: {
        cap?: boolean;
        taper?: number | boolean;
        easing?: (distance: number) => number;
    };
    /**
     * The width of the stroke.
     *
     * @default 2
     */
    strokeWidth?: number;
    /**
     * The color of the stroke.
     *
     * @default #3A87FD
     */
    strokeColor?: string;
    /**
     * Whether to keep the points within the canvas.
     *
     * @default true
     */
    isBound?: boolean;
    /**
     * Callback for when the lines change. This is called on every point change.
     *
     * @param lines
     */
    onChange?: (lines: FreehandCanvasLine[]) => void;
    /**
     * Callback for when the lines change has ended. This is called once after the
     * pointerup event. It is recommended to use this callback for performance
     * reasons.
     *
     * @param lines
     */
    onChangeEnd?: (lines: FreehandCanvasLine[]) => void;
    /** The default lines to render. */
    defaultValue?: FreehandCanvasLine[];
    /**
     * The value of the canvas. This can be used to make the canvas a controlled
     * component.
     */
    value?: FreehandCanvasLine[];
    /** The label for the canvas. */
    label?: string;
    /**
     * Whether to enable history for undo/redo.
     *
     * @default false
     */
    enableHistory?: boolean;
    /**
     * Whether the canvas is inline.
     *
     * @default true
     */
    isInline?: boolean;
    /** Whether the canvas is disabled. */
    isDisabled?: boolean;
    /** The placeholder for the canvas. */
    placeholder?: string;
    /** The label for the clear button. */
    clearLabel?: string;
    /** The label for the undo button. */
    undoLabel?: string;
    /** The label for the redo button. */
    redoLabel?: string;
    /** Description for the canvas. */
    description?: string;
}
/** A line in the freehand canvas. */
declare type FreehandCanvasLine = FreehandCanvasPoint[];
/** A point in the freehand canvas. The pressure is optional. */
declare type FreehandCanvasPoint = [x: number, y: number, pressure?: number];

declare const FreehandCanvas: React__default.ForwardRefExoticComponent<FreehandCanvasProps & React__default.RefAttributes<HTMLDivElement>>;

declare type TextProps<T extends keyof React__default.JSX.IntrinsicElements = "span"> = StylingProps & React__default.ComponentProps<T> & {
    /**
     * The type of text to render.
     *
     * @default "title"
     */
    type?: "subtitle" | "title" | "label" | "value" | "helper" | "body";
    /**
     * The size of the text.
     *
     * @default "md"
     */
    size?: "sm" | "md" | "lg";
    /** The text to render. Can be a string or a number. */
    children: React__default.ReactNode;
    /**
     * The HTML element to render.
     *
     * @default "span"
     */
    elementType?: React__default.ElementType;
};

declare const Text$1: React__default.ForwardRefExoticComponent<Omit<TextProps<"span">, "ref"> & React__default.RefAttributes<HTMLDivElement>>;

interface TransformProps {
    /** The `className` property assigned to the root element of the component. */
    className?: string;
    /** The `style` property assigned to the root element of the component. */
    style?: React__default.CSSProperties;
    /** The children to render. */
    children: (options: {
        style: React__default.CSSProperties;
    }) => React__default.ReactNode;
    /** Whether the transform is draggable. */
    isDraggable?: boolean;
    /** Whether the transform is resizable. */
    isResizable?: boolean;
    /** Whether the transform is rotatable. */
    isRotatable?: boolean;
    /** Whether the transform is snappable. */
    isSnappable?: boolean;
    /** A callback that will be called when the transform changes. */
    onTransform?: (transform: OnRender) => void;
    /** A callback that will be called when the transform starts. */
    onTransformStart?: (transform: OnRenderStart) => void;
    /** A callback that will be called when the transform ends. */
    onTransformEnd?: (transform: OnRender) => void;
    /** A callback that will be called when the target drag starts. */
    onDragStart?: (e: OnDragStart) => void;
    /** A callback that will be called when the target is dragged. */
    onDrag?: (e: OnDrag) => void;
    /** A callback that will be called when the target drag ends. */
    onDragEnd?: (e: OnDragEnd) => void;
    /** A callback that will be called when the target resize starts. */
    onResizeStart?: (e: OnResizeStart) => void;
    /** A callback that will be called when the target is resized. */
    onResize?: (e: OnResize) => void;
    /** A callback that will be called when the target resize ends. */
    onResizeEnd?: (e: OnResizeEnd) => void;
    /** A callback that will be called when the target rotation starts. */
    onRotateStart?: (e: OnRotateStart) => void;
    /** A callback that will be called when the target is rotated. */
    onRotate?: (e: OnRotate) => void;
    /** A callback that will be called when the target rotation ends. */
    onRotateEnd?: (e: OnRotateEnd) => void;
    /** A callback that will be called when the target is snapped. */
    onSnap?: (e: OnSnap) => void;
    /** A callback that will be called when the child is double clicked. */
    onDoubleClick?: (e: React__default.MouseEvent<HTMLDivElement, MouseEvent>) => void;
    /** A list of selectors relative to which the guidelines will be rendered. */
    elementGuidelines?: string[];
    /**
     * Set directions to show the control box.
     *
     * @default false if rotatable, ["n", "nw", "ne", "s", "se", "sw", "e", "w"] otherwise
     */
    renderDirections?: boolean | ("n" | "nw" | "ne" | "s" | "se" | "sw" | "e" | "w")[];
    /**
     * Whether the anchor should be hidden when dragging.
     *
     * @default true
     */
    hideAnchorOnDrag?: boolean;
    /** The `title` property assigned to the root element of the component. */
    title: string;
    /** The position of the rotation anchor. */
    rotationPosition?: RotationPosition;
    /**
     * Degree angles to snap to rotation
     *
     * @default [ ]
     */
    snapRotationDegrees?: number[];
    /**
     * Snap works if `abs(current rotation - snapRotationDegrees) <
     * snapRotationThreshold`.
     *
     * @default 5
     */
    snapRotationThreshold?: number;
    /** Whether the transform should be updated automatically. */
    autoUpdate?: boolean;
    /** The element ref, selector or element to which the transform will be bound. */
    bound?: React__default.RefObject<HTMLElement> | string | HTMLElement;
    /** The element ref, selector or element that is it's scrollable ancestor. */
    scrollableAncestor?: React__default.RefObject<HTMLElement> | string | HTMLElement;
    /**
     * Whether to prevent bubbling of events like mousedown, touchstart, etc.
     *
     * @default false
     */
    stopPropagation?: boolean;
    /**
     * Whether to call preventDefault on touchstart or mousedown
     *
     * @default true
     */
    preventDefault?: boolean;
    /**
     * The default transformOrigin of the target can be set in advance.
     *
     * @default ""
     */
    transformOrigin?: (string | number)[] | string;
}

declare const Transform: React__default.ForwardRefExoticComponent<TransformProps & React__default.RefAttributes<HTMLDivElement>>;

interface ImageDropZoneProps extends Omit<SharedFileUploadProps, "label" | "variant">, Omit<DropOptions, "ref" | "getDropOperation" | "hasDropButton" | "getDropOperationForPoint">, Omit<ClipboardProps, "getItems" | "onCut" | "onCopy">, Pick<FreehandCanvasProps, "footerClassName" | "footerStyle" | "placeholder" | "clearLabel">, AriaLabelingProps {
    /**
     * The initial image to display. This will be used as the `src` attribute of
     * the `<img>` element. You will be able to choose a new image.
     */
    defaultImageSrc?: string | File;
    /**
     * The image to display. This will be used as the `src` attribute of the
     * `<img>` element and will not be able to be changed from within the
     * component. Basically, this makes the component controlled.
     */
    imageSrc?: string | File;
    /**
     * The alt text to display for the image. This will be used as the `alt`
     * attribute of the `<img>`. element. If no `imageAlt` is provided, the
     * `label` will be used.
     */
    imageAlt?: string;
    /** The label to display for the picker button. */
    pickerButtonLabel?: string;
    /**
     * Whether to show the checkered background behind the image. This is useful
     * when the image has transparency.
     *
     * @default true
     */
    includeCheckeredBackground?: boolean;
    /** Whether the component is inlined. */
    isInline?: boolean;
    /** The class name to apply to the picker button. */
    pickerButtonClassName?: string;
    /** The style to apply to the picker button. */
    pickerButtonStyle?: React__default.CSSProperties;
    /** The class name to apply to the label. */
    labelClassName?: string;
    /** The style to apply to the label. */
    labelStyle?: React__default.CSSProperties;
}

declare const ImageDropZone: React__default.ForwardRefExoticComponent<ImageDropZoneProps & React__default.RefAttributes<HTMLDivElement>>;

interface PreviewProps extends StylingProps, AriaLabelingProps {
    /**
     * Whether the preview should be displayed inline or not. This is just a
     * visual change, the component will still behave the same.
     *
     * @default true
     */
    isInline?: boolean;
    /**
     * The image to display in the preview. This can be a URL or a base64 encoded
     * string.
     */
    imageSrc?: string;
    /** The alt text for the image. */
    imageAlt?: string;
    /** The SVG to display in the preview. This can be a React element or a string. */
    svgSrc?: string | React__default.ReactElement;
    /** Whether the preview should be disabled or not. */
    isDisabled?: boolean;
    /** The text to display in the preview. */
    textValue?: string;
    /** The style of the text. */
    textStyle?: React__default.CSSProperties;
    /**
     * This is fired when the `actionButtons` are not passed and you trigger a
     * `click` event on the preview. In case you have passed `actionButtons`, this
     * is not fired as focus shifts to the action buttons.
     */
    onPress?: PressProps["onPress"];
    /**
     * The callback to be called when the delete button is clicked. When passed,
     * the delete button is displayed.
     */
    onDelete?: ActionButtonProps["onPress"];
    /**
     * The label to be used for the delete button. This is used for accessibility
     * purposes.
     */
    deleteAriaLabel?: string;
    /** The callback to be called when the preview is clicked. */
    addAriaLabel?: string;
    /**
     * The accent variable represents the type of accent color used in the
     * application. It can have one of the following values:
     *
     * - "theme": Represents the accent color defined by the current application
     *   theme.
     * - "positive": Represents a positive accent color that doesn't change with
     *   theme.
     *
     * @default "theme"
     */
    accent?: "theme" | "positive";
}

declare const Preview: React__default.ForwardRefExoticComponent<PreviewProps & React__default.RefAttributes<HTMLDivElement>>;

interface RadioGroupProps extends StylingProps, Omit<AriaRadioGroupProps, "description" | "errorMessage" | "isRequired" | "isInvalid" | "validate" | "validationBehavior" | "validationState"> {
    /**
     * The position of the label relative to the input.
     *
     * @default top
     */
    labelPosition?: "top" | "start";
    /**
     * The list of items to render in the radio group. Items have the following
     * shape:
     *
     * ```ts
     * export type ListOption = {
     *   id: string;
     *   label: string;
     *   description?: string;
     *   icon?: React.FC<IconProps>;
     * };
     * ```
     */
    items: ListOption[];
    /**
     * Render item option
     *
     * @typedef {Object} RenderItemOptions
     * @property {boolean} isFocusVisible - Whether the focus ring is visible.
     * @property {boolean} isFocused - Whether the option is focused.
     * @property {boolean} isSelected - Whether the option is selected.
     * @property {boolean} isHovered - Whether the option is hovered.
     * @property {boolean} isDisabled - Whether the option is disabled.
     */
    /**
     * The custom render function for the radio group options.
     *
     * @param {ListOption} item The item to render.
     * @param {RenderItemOptions} options The options for rendering the item.
     */
    renderOption?: (item: ListOption, options: {
        isFocusVisible: boolean;
        isFocused: boolean;
        isSelected: boolean;
        isHovered: boolean;
        isDisabled: boolean;
        optionLabelPosition: "start" | "end";
        isReadOnly?: boolean;
    }) => React__default.ReactNode;
    /** The class name for the group container. */
    optionsContainerClassName?: string;
    /** The style for the group container. */
    optionsContainerStyle?: React__default.CSSProperties;
    /** The ids of the items that are disabled. */
    disabledValues?: string[];
    /**
     * The position of the label of individual radio buttons.
     *
     * @default end
     */
    optionLabelPosition?: "start" | "end";
    /** The className applicable to the label. */
    optionClassName?: string;
    /** The style applicable to the label. */
    optionStyle?: React__default.CSSProperties;
}

declare const RadioGroup: React__default.ForwardRefExoticComponent<RadioGroupProps & React__default.RefAttributes<HTMLDivElement>>;

interface ModalProps extends OverlayTriggerProps$1 {
    /** The contents of the modal. */
    children: React__default.ReactNode;
}
interface ModalContentProps extends Omit<AriaModalOverlayProps, "isDismissable" | "isKeyboardDismissDisabled">, StylingProps {
    /**
     * The contents of the modal. Can be a React node or a function that returns a
     * React node.
     */
    children: React__default.ReactNode | ((props: OverlayTriggerState) => React__default.ReactNode);
    /**
     * The container element for the modal. By default, the modal is rendered as a
     * child of the body element.
     *
     * @default document.body
     */
    portalContainer?: HTMLElement;
    /**
     * Whether to disable the animation for the modal. This is useful when you
     * want to render one modal after another without the animation.
     *
     * @default false
     */
    disableAnimation?: boolean;
    /**
     * Whether the modal should be rendered as a drawer. This will render the
     * modal with a transparent background and no border at the bottom. The `auto`
     * value will render the modal as a drawer on mobile and as a dialog on
     * desktop and tablet.
     *
     * @default false
     */
    enableDrawer?: boolean | "auto";
    /**
     * Specifies if the component is dismissable or not. If the value is a
     * function, it will be called with a boolean indicating if the modal is
     * rendered as a drawer or not.
     *
     * @default true
     */
    isDismissable?: boolean | ((isDrawer: boolean) => boolean);
    /**
     * Determines if keyboard dismissal is disabled. If the value is a function,
     * it will be called with a boolean indicating if the modal is rendered as a
     * drawer or not.
     *
     * @default false
     */
    isKeyboardDismissDisabled?: boolean | ((isDrawer: boolean) => boolean);
}

declare const Modal: React__default.FC<ModalProps>;

declare const ModalContent: React__default.FC<ModalContentProps>;

declare const ModalTrigger: React__default.FC<{
    children: React__default.ReactNode;
}>;

declare const ModalClose: React__default.FC<{
    children: React__default.ReactNode;
}>;

interface MessageFormatProps {
    /**
     * By default `<MessageFormat>` will render the formatted string into a
     * `<React.Fragment>`. If you need to customize rendering, you can either wrap
     * it with another React element (recommended), specify a different tagName
     * (e.g., 'div')
     */
    elementType?: React__default.ElementType | "div" | "span";
    /** The id of the message to format. */
    id: string;
    /** The default message to use if the message id is not found. */
    defaultMessage?: string;
}

declare const MessageFormat: React__default.FC<MessageFormatProps>;

interface GroupProps extends StylingProps, AriaLabelingProps {
    /**
     * The `role` property specifies the accessibility `role` for the group. By
     * default, it is set to 'group'. If the contents of the group are important
     * enough to be included in the page table of contents, use 'region'. If the
     * group is visual only and does not represent a semantic grouping of
     * controls, use 'presentation'.
     *
     * @default group
     */
    role?: "group" | "region" | "presentation";
    /** Whether the group is disabled. */
    isDisabled?: boolean;
    /** The children to render. */
    children: React__default.ReactNode;
}

declare const Group: React__default.ForwardRefExoticComponent<GroupProps & React__default.RefAttributes<HTMLDivElement>>;

declare type SprinkleProps = Parameters<Sprinkles>[0];
interface BoxProps extends Pick<StylingProps, "data-block-id">, Omit<React__default.ComponentPropsWithoutRef<"div">, "color">, SprinkleProps {
    /**
     * The HTML element to use for the box.
     *
     * @default "div"
     */
    elementType?: string;
}

declare const Box: React__default.ForwardRefExoticComponent<BoxProps & React__default.RefAttributes<HTMLDivElement>>;

interface ToolbarProps extends StylingProps, AriaToolbarProps {
    /** The children of the toolbar. */
    children: React__default.ReactNode;
    /**
     * Identifies the element (or elements) whose contents or presence are
     * controlled by the current element.
     */
    "aria-controls"?: string;
    /**
     * This prop is used to determine if the toolbar should be collapsible. If
     * true, the buttons in the toolbar will be hidden behind a menu when the
     * toolbar is too small to display all of the buttons.
     *
     * @default false
     */
    isCollapsible?: boolean;
    /**
     * The props to pass to the Menu component when the toolbar is collapsible.
     * This prop is only relevant when `isCollapsible` is true.
     */
    collapsibleMenuProps?: Omit<MenuProps, "items" | "onAction" | "placement" | "disabledKeys">;
    /**
     * A function that renders a spacer element in the collapsible toolbar between
     * the toolbar buttons and the menu trigger. This prop is only relevant when
     * `isCollapsible` is true.
     */
    renderSpacer?: boolean;
    /** The callback to call when any key is pressed. */
    onKeyDown?: KeyboardProps["onKeyDown"];
}

declare const Toolbar: React__default.ForwardRefExoticComponent<ToolbarProps & React__default.RefAttributes<HTMLDivElement>>;

interface ToggleIconButtonProps extends StylingProps, Omit<ToggleProps, "children">, Omit<ToggleButtonProps, "size" | "iconStart" | "label"> {
    /**
     * The available sizes for a variable.
     *
     * @default "md"
     */
    size?: "xxs" | "xs" | "sm" | "md" | "lg";
    /** Represents the variant of a component. */
    variant?: "primary" | "secondary" | "tertiary" | "toolbar";
    /**
     * The icon to be displayed on the button. You can pass an object with
     * `selected` and `unselected` keys to display different icons based on the
     * state of the button.
     */
    icon: React__default.FC<IconProps> | {
        selected: React__default.FC<IconProps>;
        unselected: React__default.FC<IconProps>;
    };
    /** Indicates whether focusing should be prevented on press. */
    preventFocusOnPress?: boolean;
}

declare const ToggleIconButton: React__default.ForwardRefExoticComponent<ToggleIconButtonProps & React__default.RefAttributes<HTMLButtonElement>>;

interface MentionableUser {
    id: string;
    name: string;
    displayName: string;
    avatar?: AvatarProps;
    description?: string;
}
interface EditorHandle {
    setCaretPosition: (index: number) => void;
    focus: () => void;
}
interface EditorProps extends StylingProps, AriaLabelingProps {
    /**
     * The function to be called when `Enter` key is pressed or Save button is
     * clicked.
     */
    onSave?: (value: string) => void;
    /**
     * The function to be called when Cancel button is clicked. If not provided,
     * the Cancel button will not be rendered.
     */
    onCancel?: () => void;
    /** The default value of the editor. */
    defaultValue?: string;
    /** The value of the editor. */
    value?: string;
    /**
     * The function to be called when the value of the editor changes.
     *
     * @param value
     */
    onChange?: (value: string) => void;
    /** Whether the editor is disabled. This includes both input and save button. */
    isDisabled?: boolean;
    /** Whether saving it disabled. Ths disables only save button. */
    isSaveDisabled?: boolean;
    /**
     * Indicates whether rich text is enabled or not.
     *
     * @default true
     */
    enableRichText?: boolean;
    /** The placeholder text to be displayed when the editor is empty. */
    placeholder?: string;
    /**
     * The variant of the editor.
     *
     * @default default
     */
    variant?: "default" | "minimal";
    /** Whether the editor is inline or not. */
    isInline?: boolean;
    /** Specifies whether spell checking is enabled or disabled. */
    spellCheck?: boolean;
    /**
     * Optional property that represents the ARIA label for the submit button.
     * This property is used to provide an accessible label for the button, which
     * is read by screen readers to assist visually impaired users.
     *
     * @default Save
     */
    submitButtonAriaLabel?: string;
    /**
     * The React node representing the icon to be displayed on the submit button.
     *
     * @default ArrowUpCircleFilledIcon
     */
    submitButtonIcon?: React__default.FC<IconProps>;
    /**
     * Optional property that represents the ARIA label for the cancel button.
     *
     * @default Cancel
     */
    cancelButtonAriaLabel?: string;
    /**
     * The name of the avatar. The avatar is visible when `variant` is set to
     * `minimal` and `enableRichText` is set to `false`.
     */
    avatarName?: string;
    /**
     * The ARIA label for the editable content. In case of rich text, this label
     * is used to provide an accessible label for the contenteditable element,
     * which is read by screen readers to assist visually impaired users. In case
     * of plain text, this label is used to provide an accessible label for the
     * textarea element.
     *
     * @default Editing Area
     */
    editableContentAriaLabel?: string;
    /**
     * An optional array of mentionable users. Mention feature is only available
     * when `enableRichText` is set to `true` and this prop is provided.
     *
     * ```tsx
     * type MentionableUser = {
     *   id: string;
     *   name: string;
     *   displayName: string;
     *   avatar?: AvatarProps;
     *   description?: string;
     * };
     * ```
     */
    mentionableUsers?: MentionableUser[];
    /**
     * The maximum number of mentionable users to be displayed in the suggestions.
     * This prop is only used when `mentionableUsers` is provided.
     *
     * @default 10
     */
    maxMentionableUsersSuggestions?: number;
    /** Footer button represents the buttons that are displayed in the footer. */
    footerButtons?: ({
        type: "action";
        props: Omit<ActionIconButtonProps, "size" | "variant">;
    } | {
        type: "toggle";
        props: Omit<ToggleIconButtonProps, "size" | "variant">;
    })[];
    /** The imperative handle for manipulating editor. */
    editorHandle?: React__default.RefObject<EditorHandle>;
    /**
     * Whether to clear the editor value when the editor is saved.
     *
     * @default false
     */
    clearOnSave?: boolean;
    /**
     * Whether to clear the editor value when the cancel button is clicked.
     *
     * @default false
     */
    clearOnCancel?: boolean;
    /**
     * Whether to autofocus the editor when it is mounted.
     *
     * @default false
     */
    autoFocus?: boolean;
    /**
     * Whether to save the editor value when `Enter` key is pressed.
     *
     * @default false
     */
    saveOnEnter?: boolean;
}

declare const Editor: React__default.MemoExoticComponent<React__default.ForwardRefExoticComponent<EditorProps & React__default.RefAttributes<HTMLDivElement>>>;

interface ScrollControlButtonProps extends StylingProps, Pick<ActionButtonProps, "label"> {
    /**
     * The ref to the scrollable element.
     *
     * @default document.body
     */
    scrollRef?: React__default.RefObject<HTMLElement>;
    /**
     * Indicates whether the button should be hidden when the keyboard is being
     * used.
     *
     * @default false
     */
    hideForKeyboard?: boolean;
    /**
     * The delay in milliseconds before the button is hidden when the scrolling
     * stops.
     *
     * @default 1500
     */
    delay?: number;
    /**
     * Indicates whether the scroll should be smooth.
     *
     * @default true
     */
    smoothScroll?: boolean;
}

declare const ScrollControlButton: React__default.ForwardRefExoticComponent<ScrollControlButtonProps & React__default.RefAttributes<HTMLButtonElement>>;

interface TransitionProps extends StylingProps {
    isMounted: boolean;
    transitions: UseTransitionProps;
    elementType?: keyof React__default.ReactHTML;
    children: React__default.ReactNode;
}

declare const Transition: React__default.ForwardRefExoticComponent<TransitionProps & React__default.RefAttributes<HTMLElement>>;

interface AlertDialogProps extends StylingProps {
    /**
     * The title of the dialog. This is the main heading of the dialog and is
     * displayed at the top of the dialog.
     */
    title: string;
    /**
     * The message of the dialog. This is the main content of the dialog and is
     * displayed below the title.
     */
    message?: React__default.ReactNode;
    /**
     * The label for the cancel button. This is the text that is displayed on the
     * button that allows the user to cancel the dialog.
     */
    cancelLabel?: string;
    /**
     * The label for the confirm button. This is the text that is displayed on the
     * button that allows the user to confirm the dialog.
     */
    primaryActionLabel?: string;
    /**
     * The function that is called when the user confirms the dialog. This is
     * called when the user clicks the primary action button.
     */
    onPrimaryAction?: () => void;
    /**
     * The function that is called when the user cancels the dialog. This is
     * called when the user clicks the cancel button or the close button.
     */
    onCancel?: () => void;
    /** The button that should be focused when the dialog is opened. */
    autoFocusButton?: "primary" | "cancel";
    /** Whether the primary action button should be disabled. */
    isPrimaryActionDisabled?: boolean;
    /**
     * The props that are passed to the text input. If this is provided, a text
     * input will be displayed at the bottom of the dialog.
     */
    textInputProps?: TextInputProps;
    /**
     * The icon that is displayed at the top of the dialog. This is typically used
     * to display an icon that represents the type of alert that is being shown.
     */
    icon?: React__default.FC<IconProps>;
    /** The color of the icon. This is the color that is used to fill the icon. */
    iconColor?: string;
    /**
     * Whether the close button should be displayed in the top end corner of the
     * dialog. If this is `true`, a close button will be displayed. If this is
     * `false`, no close button will be displayed.
     *
     * @default true
     */
    showCloseButton?: boolean;
}

declare const AlertDialog: React__default.ForwardRefExoticComponent<AlertDialogProps & React__default.RefAttributes<HTMLDivElement>>;

interface MarkdownProps extends StylingProps {
    /** The markdown to render. */
    children: string;
    /** The flag to show the caret at the end */
    showCaret?: boolean;
}

declare const Markdown: React__default.ForwardRefExoticComponent<MarkdownProps & React__default.RefAttributes<HTMLDivElement>>;

interface AudioPlayerProps extends StylingProps, AriaLabelingProps {
    /**
     * The sources of the audio file. The `url` is the URL of the audio file, and
     * the `type` is the MIME type of the audio file.
     *
     * ```tsx
     * <AudioPlayer
     *  sources={[
     *    { url: "/sound.mp3", type: "audio/mpeg" },
     *    { url: "/sound.ogg", type: "audio/ogg" },
     *  ]}
     * />
     * ```
     */
    sources: {
        url: string;
        type: string;
    }[];
    /**
     * The size of the audio player.
     *
     * @default "lg"
     */
    size?: "sm" | "lg";
}

declare const AudioPlayer: React__default.ForwardRefExoticComponent<AudioPlayerProps & React__default.RefAttributes<HTMLDivElement>>;

interface Item {
    id: string;
    label: string;
    src?: string;
    alt?: string;
}
interface ImageGalleryProps extends StylingProps, Omit<ListBoxProps, "items" | "layout" | "grid" | "renderOption" | "renderDropIndicator" | "renderDragPreview" | "showSelectedIcon" | "orientation" | "shouldSelectOnPressUp" | "shouldFocusOnHover" | "dropIndicatorClassName" | "dropIndicatorStyle" | "shouldUseVirtualFocus" | "getItems" | "acceptedDragTypes" | "sectionClassName" | "sectionStyle"> {
    /**
     * An array of objects containing the image source, alt text, and label for
     * each image. This property makes the component a controlled component.
     *
     * ```tsx
     * const items = [
     *   {
     *     id: "1",
     *     src: "https://example.com/image1.jpg",
     *     alt: "Image 1",
     *     label: "Image 1",
     *   },
     *   {
     *     id: "2",
     *     src: "https://example.com/image2.jpg",
     *     alt: "Image 2",
     *     label: "Image 2",
     *   },
     * ];
     * ```
     */
    items?: Item[];
    /**
     * An array of objects containing the image source, alt text, and label for
     * each image.
     *
     * ```tsx
     * const items = [
     *   {
     *     id: "1",
     *     src: "https://example.com/image1.jpg",
     *     alt: "Image 1",
     *     label: "Image 1",
     *   },
     *   {
     *     id: "2",
     *     src: "https://example.com/image2.jpg",
     *     alt: "Image 2",
     *     label: "Image 2",
     *   },
     * ];
     * ```
     */
    defaultItems?: Item[];
    /**
     * The width of the images.
     *
     * @default "sm"
     */
    imageWidth?: number | "sm" | "md";
    /**
     * The aspect ratio of the images.
     *
     * @default 16 / 9
     */
    aspectRatio?: number;
    /**
     * The fit of the images.
     *
     * @default "cover"
     */
    fit?: "contain" | "cover";
    /** A callback that is called when the list of images or selection is updated. */
    onListChange?: (items: Item[]) => void;
    /** A callback that is called when an image is deleted. */
    onDelete?: (keys: Selection$1) => void;
    /** A callback that is called when a key is pressed. */
    onKeyDown?: (event: React__default.KeyboardEvent<HTMLDivElement>) => void;
    /** ClassName for each item container */
    imageContainerClassName?: string;
    /** ClassName for each item image */
    imageClassName?: string;
    /** ClassName for each item image label */
    labelClassName?: string;
    /** A callback that is called to render the image. */
    renderImage?: (item: ListOption<Pick<Item, "alt" | "src">>, options: {
        deleteElement?: React__default.ReactElement;
        onDelete?: (id: string) => void;
    }) => React__default.ReactElement;
}

declare const ImageGallery: React__default.ForwardRefExoticComponent<ImageGalleryProps & React__default.RefAttributes<HTMLDivElement>>;

interface ButtonSelectProps extends StylingProps, Pick<SelectProps, "isDisabled" | "validationState" | "onSelectionChange" | "selectedKey" | "defaultSelectedKey" | "items" | "showSelectedIcon" | "isOpen" | "defaultOpen" | "optionClassName" | "optionStyle">, Pick<ToggleButtonProps, "onPress" | "isSelected" | "defaultSelected" | "excludeFromTabOrder">, AriaLabelingProps {
    /**
     * The size of the button.
     *
     * @default "md"
     */
    size?: "md" | "lg";
    /**
     * Whether to hide the label in the button.
     *
     * @default false
     */
    hideLabel?: boolean;
    /** A callback that is called when the button selection changes. */
    onButtonSelectionChange?: (options: {
        isSelected: boolean;
        selectedKey: string;
    }) => void;
    /** The aria-label for the select button. */
    moreAriaLabel?: string;
}

declare const ButtonSelect: React__default.ForwardRefExoticComponent<ButtonSelectProps & React__default.RefAttributes<HTMLDivElement>>;

interface ComboBoxProps extends StylingProps, Omit<AriaComboBoxOptions<ListItem>, "items" | "children" | "inputRef" | "popoverRef" | "listBoxRef" | "buttonRef" | "validationState" | "errorMessage" | "description">, Omit<AriaComboBoxProps<ListItem>, "items" | "children" | "errorMessage" | "description" | "validationState">, Omit<ListBoxProps, "autoFocus" | "onBlur" | "onFocus" | "onSelectionChange" | "selectionBehavior" | "selectionMode" | "selectedKeys" | "defaultSelectedKeys" | "renderDropIndicator" | "renderDragPreview" | "dropIndicatorClassName" | "dropIndicatorStyle" | keyof DragAndDropProps>, AriaValidationProps, Omit<ComboBoxStateOptions<ListItem>, "items" | "children" | "errorMessage" | "description" | "validationState" | "selectedKey">, InputMessage {
    /** ID of the initially selected item */
    defaultSelectedKey?: string;
    /**
     * The state of the combobox input.
     *
     * @default valid
     */
    validationState?: "valid" | "error" | "warning";
    /**
     * The variant of the text input.
     *
     * @default primary
     */
    variant?: "primary" | "ghost";
    /**
     * The position of the label relative to the input.
     *
     * @default top
     */
    labelPosition?: "top" | "start";
    /** Whether the input is read only. */
    readOnly?: boolean;
    /** The style object to apply to the input element */
    inputStyle?: React__default.CSSProperties;
    /** The class name to apply to the input element */
    inputClassName?: string;
    /** Optional label for the toggle button */
    toggleLabel?: string;
    /**
     * The custom render function for the listbox options.
     *
     * @param item Node<ListItem>
     * @param options [OptionAria]()
     * @param ref React.RefObject<HTMLLIElement>
     */
    renderOption?: (item: Node$1<ListItem>, options: OptionAria & {
        showSelectedIcon?: boolean;
        _state: ListState<ListItem>;
    }, ref: React__default.Ref<HTMLLIElement>) => React__default.ReactNode;
    filterItems?: boolean;
    onInputChange?: (value: string) => void;
    showIcon?: boolean;
    /**
     * The items to render in the listbox. Items have the following shape:
     *
     * ```ts
     * export type ListOption = {
     *   id: string;
     *   label: string;
     *   description?: string;
     *   icon?: React.FC<IconProps>;
     * };
     *
     * export type ListSection = {
     *   id: string;
     *   title?: string;
     *   type: "section";
     *   children: ListOption[];
     * };
     *
     * type ListItem = ListOption | ListSection;
     * ```
     */
    items: ListItem[];
    /**
     * Specify which interaction should trigger the dropdown to open
     *
     * @default input
     */
    menuTrigger?: "manual" | "input" | "focus";
    /** Custom filter function to use when filtering items. */
    filter?: ComboBoxStateOptions<ListItem>["defaultFilter"];
    /**
     * The type of the input field
     *
     * @default text
     */
    inputType?: "text" | "number";
    /**
     * The minimum value for the input field. This is only valid when `inputType`
     * is set to `number`.
     */
    minValue?: number;
    /**
     * The maximum value for the input field. This is only valid when `inputType`
     * is set to `number`.
     */
    maxValue?: number;
}

declare const ComboBox: React__default.ForwardRefExoticComponent<ComboBoxProps & React__default.RefAttributes<HTMLDivElement>>;

interface ColorSwatchProps extends Omit<AriaColorSwatchProps, "colorName">, StylingProps, DOMProps, AriaLabelingProps {
    /** Whether the swatch is focused by keyboard. */
    isFocusVisible?: boolean;
    /** Whether the swatch is selected. */
    isSelected?: boolean;
    /**
     * Whether the swatch is interactive. This is used to determine whether to
     * show hover styles.
     */
    isInteractive?: boolean;
    /**
     * Whether the swatch is disabled. This is used to determine whether to show
     * disabled styles.
     */
    isDisabled?: boolean;
}

declare const ColorSwatch: React__default.ForwardRefExoticComponent<ColorSwatchProps & React__default.RefAttributes<HTMLDivElement>>;

declare type GridListProps = Omit<StylingProps & ListProps<ListOption> & AriaGridListOptions<ListOption> & {
    /**
     * The items to render in the grid list. Items have the following shape:
     *
     * ```ts
     * export type ListOption = {
     *   id: string;
     *   label: string;
     *   description?: string;
     *   icon?: React.FC<IconProps>;
     *   data?: T;
     * };
     * ```
     */
    items: ListOption[];
    /**
     * Function to render a grid item. This is called for each item in the
     * list. The function receives the item, props for the grid item, and a
     * ref to the grid item element. It should return a React element
     * representing the item.
     *
     * The `checkBoxProps` property in the options object is used to pass
     * props to the checkbox rendered in the grid item. This property is
     * `undefined` when the grid list is not selectable.
     *
     * ```jsx
     * <GridList
     *   items={items}
     *   renderGridItem={(
     *     item,
     *     { rowProps, isDisabled, gridCellProps, checkBoxProps },
     *     ref,
     *   ) => (
     *     <div
     *       {...rowProps}
     *       ref={ref}
     *       style={{ opacity: isDisabled ? 0.5 : 1 }}
     *     >
     *       <Checkbox {...checkBoxProps} />
     *       <Text>{item.label}</Text>
     *       <ActionButton label="Click Me" />
     *     </div>
     *   )}
     * />;
     * ```
     *
     * @param item
     * @param options
     * @param ref
     */
    renderGridItem: (item: Node$1<ListOption>, options: GridListItemAria & {
        checkBoxProps?: CheckboxProps;
    }, ref: React__default.Ref<HTMLLIElement>) => React__default.ReactElement;
    /**
     * Enables editing mode which disables keydown event capturing, useful
     * when you may want the internal `GridList` from taking away focus from
     * child input due to type-ahead
     *
     * @default false
     */
    isEditing?: boolean;
}, "children" | "suppressTextValueWarning" | "allowDuplicateSelectionEvents" | "linkBehavior" | "keyboardDelegate" | "isVirtualized" | "filter">;

declare const GridList: React__default.ForwardRefExoticComponent<GridListProps & React__default.RefAttributes<HTMLUListElement>>;

interface ActionGroupProps extends StylingProps, Omit<AriaActionGroupProps<ListItem>, "children"> {
    /** The items to display in the action group. */
    items: ListItem[];
    /**
     * Represents the mode of selection for a component. This variable should be
     * assigned one of the following string values: "single" or "multiple".
     */
    selectionMode?: "single" | "multiple";
    /** The props to be passed to the tooltip component. */
    tooltipProps?: IconComponentProps["tooltipProps"] | ((item: ListItem) => IconComponentProps["tooltipProps"]);
    /**
     * A function that renders an action item. This function is called for each
     * item in the action group. By default, an `ActionIconButton` is rendered.
     *
     * @default
     *
     * ```jsx
     * <ActionIconButton
     *   variant="tertiary"
     *   icon={icon}
     *   aria-label={item.textValue}
     *   size="sm"
     *   isDisabled={_isDisabled}
     *   className={"BaselineUI-ActionGroup-Item"}
     * />;
     * ```
     */
    renderActionItem?: (item: Node$1<ListItem>, options: {
        isSelected?: boolean;
        isDisabled?: boolean;
    }) => React__default.ReactNode;
}

declare const ActionGroup: React__default.ForwardRefExoticComponent<ActionGroupProps & React__default.RefAttributes<HTMLDivElement>>;
declare const ActionGroupItem: React__default.FC<ActionGroupItemProps>;
interface ActionGroupItemProps extends Pick<ActionGroupProps, "tooltipProps" | "renderActionItem" | "onAction" | "isDisabled"> {
    item: Node$1<ListItem>;
    state: ListState<ListItem>;
}

interface ColorSwatchPickerProps extends StylingProps, Pick<RadioGroupProps, "optionsContainerClassName" | "labelPosition">, Pick<ListBoxProps, "optionClassName" | "optionStyle">, AriaLabelingProps {
    /**
     * The items to display in the picker. Each item should have a `color` and a
     * `label`.
     *
     * ```js
     * const items = [
     *   { color: "#FF0000", label: "Red" },
     *   { color: "#00FF00", label: "Green" },
     *   { color: "#0000FF", label: "Blue" },
     *   { color: "#FFFF00", label: "Yellow" },
     * ];
     * ```
     */
    items: ColorPreset$1[];
    /** The icon visible at the start of the picker. */
    icon?: React__default.FC<IconProps>;
    /**
     * The value of the selected item. This is useful if you want to control the
     * value of the picker.
     */
    value?: string;
    /** A function that is called when the selected item changes. */
    onChange?: (value: string) => void;
    /** The default value of the picker. */
    defaultValue?: string;
    /** The label for the picker. */
    label?: string;
    /** Whether the picker is disabled. */
    isDisabled?: boolean;
}

declare const ColorSwatchPicker: React__default.ForwardRefExoticComponent<ColorSwatchPickerProps & React__default.RefAttributes<HTMLDivElement>>;

/**
 * A hook that creates an IntersectionObserver and observes a target element.
 *
 * @example
 *   ```tsx
 *   import { useIntersectionObserver } from "@baseline-ui/core";
 *
 *   const MyComponent = () => {
 *    const ref = useRef(null);
 *
 *    const onIntersect = (entry) => {
 *      console.log(entry);
 *    };
 *
 *    useIntersectionObserver({ ref, onIntersect });
 *
 *    return <div ref={ref}>Hello world</div>;
 *   };
 *   ```;
 *
 * @param {IntersectionObserverOptions} options - The options for the
 *   IntersectionObserver.
 * @param {React.RefObject<HTMLElement>} options.ref - The ref of the target
 *   element to observe.
 * @param {(entry: IntersectionObserverEntry) => void} options.onIntersect - The
 *   function to call when the target element intersects the root element.
 * @param {boolean} options.isDisabled - Whether the IntersectionObserver should
 *   be disabled.
 * @param {number | number[]} options.threshold - The threshold(s) at which to
 *   trigger the onIntersect function.
 * @param {string} options.rootMargin - The margin around the root element.
 * @param {React.RefObject<HTMLElement>} options.root - The ref of the root
 *   element to observe.
 */
declare function useIntersectionObserver({ ref, onIntersect, isDisabled, threshold, rootMargin, root, }: IntersectionObserverOptions): void;
interface IntersectionObserverOptions {
    ref: React__default.RefObject<HTMLElement>;
    onIntersect: (entry: IntersectionObserverEntry) => void;
    root?: React__default.RefObject<HTMLElement>;
    rootMargin?: string;
    threshold?: number;
    isDisabled?: boolean;
}

/**
 * A hook that observes the size changes of a DOM element using the
 * ResizeObserver API.
 *
 * @example
 *   ```tsx
 *   const ref = useRef<HTMLDivElement>(null);
 *
 *   const { size } = useResizeObserver({
 *    ref,
 *    onResize: (entry) => {
 *    // do something
 *    },
 *    isDisabled: false,
 *   });
 *   ```;
 *
 * @param {Object} options - The options object.
 * @param {React.RefObject<HTMLElement>} options.ref - The ref object that
 *   points to the observed element.
 * @param {Function} options.onResize - The callback function that is called
 *   when the element is resized.
 * @param {boolean} options.isDisabled - A flag that indicates whether the
 *   observer is disabled.
 * @returns {Object} An object that contains the size of the observed element.
 */
declare function useResizeObserver({ ref, onResize, isDisabled, }: ResizeObserverOptions): {
    size: {
        width: number;
        height: number;
    } | undefined;
};
interface ResizeObserverOptions {
    ref: React__default.RefObject<HTMLElement>;
    onResize?: (entry: ResizeObserverEntry) => void;
    isDisabled?: boolean;
}

/**
 * A custom hook that loads an image and returns its loading state and props.
 *
 * @example
 *   ```tsx
 *   const { isLoaded, hasError, isLoading, imgProps } = useImage({
 *    src: "https://via.placeholder.com/150",
 *    alt: "Placeholder image",
 *   });
 *   ```;
 */
declare function useImage({ src, alt }: {
    src?: string;
    alt: string;
}): {
    isLoaded: boolean;
    hasError: boolean;
    isLoading: boolean;
    imgProps: {
        src?: string;
        alt: string;
    };
};

/**
 * A hook that provides undo and redo functionality for a given state.
 *
 * @example
 *   ```tsx
 *   const { state, push, undo, redo, pastStates, futureStates, canUndo, canRedo } = useUndoRedo({
 *     onAction: (state, action) => {
 *     // do something
 *     },
 *   });
 *
 *   // push a new state
 *   push({ foo: "bar" });
 *
 *   // undo the last state
 *   undo();
 *
 *   // redo the last state
 *   redo();
 *   ```;
 *
 * @template T The type of the state object.
 * @param initialState The initial state object.
 * @param options Additional options for the hook.
 * @param options.isDisabled Whether the undo/redo functionality should be
 *   disabled.
 * @param options.onAction A callback function that is called whenever an undo
 *   or redo action is performed.
 * @returns An object containing the current state, undo and redo functions, and
 *   other related properties.
 */
declare const useUndoRedo: <T>(initialState: T | undefined, { isDisabled, onAction, }: {
    isDisabled?: boolean | undefined;
    onAction?: ((state: T, action: "UNDO" | "REDO") => void) | undefined;
}) => {
    state: T;
    push: (newState: T) => void;
    undo: () => void;
    redo: () => void;
    pastStates: T[];
    futureStates: T[];
    canUndo: boolean;
    canRedo: boolean;
};

/**
 * This hook returns true if the component is being rendered for the first time.
 * This is useful for avoiding side effects on the first render inside of
 * useEffect.
 *
 * @example
 *   ```tsx
 *   const isFirstRender = useIsFirstRender();
 *
 *   useEffect(() => {
 *      if (isFirstRender) return;
 *      // do something
 *      return () => {
 *        // do something on unmount
 *      };
 *   }, [isFirstRender]);
 *   ```;
 *
 * @returns Whether or not the component is being rendered for the first time
 */
declare function useIsFirstRender(): boolean;

/**
 * A hook that provides internationalization functionality. It returns a
 * function to format messages and the current locale. The hook can be used with
 * or without a messages object. If no messages object is provided, the hook
 * will use the messages provided by the `I18nProvider`. If a messages object is
 * provided, the hook will use those messages instead of the ones provided by
 * the `I18nProvider`.
 *
 * @example
 *   ```tsx
 *   import { useI18n } from "@baseline-ui/core";
 *
 *   const messages = {
 *    en: { hello: "Hello" },
 *    fr: { hello: "Bonjour" },
 *   };
 *
 *   const MyComponent = () => {
 *    const { formatMessage, locale } = useI18n(messages);
 *
 *   return (
 *    <div>
 *      <p>{formatMessage("hello")}</p>
 *      <p>{locale}</p>
 *    </div>
 *    )};
 *   ```;
 *
 * @template T - A type that represents the shape of the translation messages.
 * @param {Object<string, T>} [messages] - An optional object containing
 *   translation messages for different languages.
 * @returns {I18nResult<T>} An object containing a function to format messages
 *   and the current locale.
 */
declare function useI18n<T extends Record<string, string>>(messages?: Record<string, T>): I18nResult<T>;
interface I18nResult<T extends LocalizedStrings[keyof LocalizedStrings]> {
    formatMessage: (id: keyof T | MessageDescriptor, values?: Record<string, any>) => string;
    locale: Locale;
}
interface MessageDescriptor {
    id: string;
    defaultMessage?: string;
}
/**
 * Defines a set of messages for use with the `useI18n` hook.
 *
 * @param messages An object containing message descriptors.
 * @returns An object containing the message descriptors with their keys as
 *   property names.
 */
declare const defineMessages: <T extends Record<string, {
    id: string;
}>>(messages: T) => T;

/**
 * A hook that uses a MutationObserver to watch for changes to the DOM.
 *
 * @example
 *   ```tsx
 *   const targetRef = React.useRef<HTMLDivElement>(null);
 *
 *   useMutationObserver(targetRef.current, { attributes: true }, (mutations) => {
 *    console.log(mutations);
 *    // do something with the mutations
 *    // ...
 *   });
 *   ```;
 *
 * @param target The element to observe for changes.
 * @param options The options to pass to the MutationObserver.
 * @param callback The callback to call when a mutation occurs.
 * @see https://developer.mozilla.org/en-US/docs/Web/API/MutationObserver
 */
declare function useMutationObserver(target: Element | null, options: MutationObserverInit, callback: MutationCallback): MutationObserver | null;

declare type SetValue<T> = Dispatch<SetStateAction<T>>;
/**
 * A hook to persist a value to localStorage.
 *
 * @example
 *   ```tsx
 *   function MyComponent() {
 *    const [value, setValue] = useLocalStorage("my-value", "default-value");
 *    return (
 *    <input value={value} onChange={e => setValue(e.target.value)} />
 *    );
 *   }
 *   ```;
 *
 * @param key The localStorage key to use
 * @param initialValue The initial value to use if the key is not found in
 *   localStorage
 */
declare function useLocalStorage<T>(key: string, initialValue: T): [T, SetValue<T>];

/**
 * A hook that provides text selection functionality for a given HTML element.
 *
 * ```jsx
 * function MyComponent() {
 *   const ref = useRef();
 *   const [selection, setSelection] = useState();
 *
 *   useTextSelection({
 *     ref,
 *     onSelectionChange: setSelection,
 *   });
 *
 *   return (
 *     <div ref={ref}>
 *       <p>Lorem ipsum dolor sit amet, consectetur adipiscing elit.</p>
 *     </div>
 *   );
 * }
 * ```
 */
declare function useTextSelection({ ref, onSelectionChange, onSelectionEnd, isDisabled, }: TextSelectionProps): void;
interface TextSelectionProps {
    ref: React__default.RefObject<HTMLElement>;
    onSelectionChange: (selection: Selection) => void;
    onSelectionEnd?: (selection: Selection) => void;
    isDisabled?: boolean;
}

/**
 * Returns the current device type based on the window width.
 *
 * ```jsx
 * const device = useDevice();
 *
 * if (device === "mobile") {
 *   return <MobileComponent />;
 * } else if (device === "tablet") {
 *   return <TabletComponent />;
 * } else {
 *   return <DesktopComponent />;
 * }
 * ```
 *
 * @param element - The element to use to get the owner window. Defaults to
 *   `document.body`.
 */
declare function useDevice(element?: Element | null | undefined): "mobile" | "tablet" | "desktop";
declare type Device = ReturnType<typeof useDevice>;

/**
 * Custom hook for accessing PortalContainerProviderContext. This hook allows
 * components to either use a specified portal container or fallback to a
 * default provided via context.
 *
 * @param _portalContainer - An optional HTML element that can be specified as
 *   an override.
 * @returns The portal container element where components should be rendered,
 *   either the overridden container, one from context or undefined.
 */
declare function usePortalContainer(_portalContainer?: HTMLElement): HTMLElement | undefined;

declare const Core_Accordion: typeof Accordion;
declare const Core_AccordionItem: typeof AccordionItem;
type Core_AccordionItemProps = AccordionItemProps;
type Core_AccordionProps = AccordionProps;
declare const Core_ActionButton: typeof ActionButton;
type Core_ActionButtonProps = ActionButtonProps;
declare const Core_ActionGroup: typeof ActionGroup;
declare const Core_ActionGroupItem: typeof ActionGroupItem;
type Core_ActionGroupProps = ActionGroupProps;
declare const Core_ActionIconButton: typeof ActionIconButton;
type Core_ActionIconButtonProps = ActionIconButtonProps;
declare const Core_AlertDialog: typeof AlertDialog;
type Core_AlertDialogProps = AlertDialogProps;
declare const Core_AudioPlayer: typeof AudioPlayer;
type Core_AudioPlayerProps = AudioPlayerProps;
declare const Core_Avatar: typeof Avatar;
type Core_AvatarProps = AvatarProps;
declare const Core_Box: typeof Box;
type Core_BoxProps = BoxProps;
declare const Core_ButtonSelect: typeof ButtonSelect;
type Core_ButtonSelectProps = ButtonSelectProps;
declare const Core_Checkbox: typeof Checkbox;
type Core_CheckboxProps = CheckboxProps;
declare const Core_ColorInput: typeof ColorInput;
type Core_ColorInputProps = ColorInputProps;
declare const Core_ColorSwatch: typeof ColorSwatch;
declare const Core_ColorSwatchPicker: typeof ColorSwatchPicker;
type Core_ColorSwatchPickerProps = ColorSwatchPickerProps;
type Core_ColorSwatchProps = ColorSwatchProps;
declare const Core_ComboBox: typeof ComboBox;
type Core_ComboBoxProps = ComboBoxProps;
declare const Core_DateFormat: typeof DateFormat;
type Core_DateFormatProps = DateFormatProps;
type Core_Device = Device;
declare const Core_Dialog: typeof Dialog;
type Core_DialogProps = DialogProps;
declare const Core_DialogTitle: typeof DialogTitle;
type Core_DialogTitleProps = DialogTitleProps;
declare const Core_DomNodeRenderer: typeof DomNodeRenderer;
type Core_DomNodeRendererProps = DomNodeRendererProps;
declare const Core_Drawer: typeof Drawer;
type Core_DrawerProps = DrawerProps;
declare const Core_Editor: typeof Editor;
type Core_EditorHandle = EditorHandle;
type Core_EditorProps = EditorProps;
declare const Core_FileUpload: typeof FileUpload;
type Core_FileUploadProps = FileUploadProps;
declare const Core_FreehandCanvas: typeof FreehandCanvas;
type Core_FreehandCanvasProps = FreehandCanvasProps;
declare const Core_GridList: typeof GridList;
type Core_GridListProps = GridListProps;
declare const Core_Group: typeof Group;
type Core_GroupProps = GroupProps;
declare const Core_I18nProvider: typeof I18nProvider;
type Core_I18nResult<T extends LocalizedStrings[keyof LocalizedStrings]> = I18nResult<T>;
declare const Core_IconColorInput: typeof IconColorInput;
declare const Core_IconColorInputButton: typeof IconColorInputButton;
type Core_IconColorInputProps = IconColorInputProps;
declare const Core_IconSelect: typeof IconSelect;
type Core_IconSelectProps = IconSelectProps;
declare const Core_IconSlider: typeof IconSlider;
type Core_IconSliderProps = IconSliderProps;
declare const Core_ImageDropZone: typeof ImageDropZone;
type Core_ImageDropZoneProps = ImageDropZoneProps;
declare const Core_ImageGallery: typeof ImageGallery;
type Core_ImageGalleryProps = ImageGalleryProps;
declare const Core_InlineAlert: typeof InlineAlert;
type Core_InlineAlertProps = InlineAlertProps;
declare const Core_Link: typeof Link;
type Core_LinkProps = LinkProps;
declare const Core_ListBox: typeof ListBox;
type Core_ListBoxProps = ListBoxProps;
type Core_ListOption<T = Record<string, any>> = ListOption<T>;
declare const Core_Markdown: typeof Markdown;
type Core_MarkdownProps = MarkdownProps;
declare const Core_Menu: typeof Menu;
type Core_MenuItem = MenuItem;
type Core_MenuProps = MenuProps;
type Core_MessageDescriptor = MessageDescriptor;
declare const Core_MessageFormat: typeof MessageFormat;
type Core_MessageFormatProps = MessageFormatProps;
declare const Core_Modal: typeof Modal;
declare const Core_ModalClose: typeof ModalClose;
declare const Core_ModalContent: typeof ModalContent;
type Core_ModalContentProps = ModalContentProps;
type Core_ModalProps = ModalProps;
declare const Core_ModalTrigger: typeof ModalTrigger;
declare const Core_NumberFormat: typeof NumberFormat;
type Core_NumberFormatProps = NumberFormatProps;
declare const Core_NumberInput: typeof NumberInput;
type Core_NumberInputProps = NumberInputProps;
declare const Core_Pagination: typeof Pagination;
type Core_PaginationProps = PaginationProps;
declare const Core_Popover: typeof Popover;
declare const Core_PopoverContent: typeof PopoverContent;
type Core_PopoverContentProps = PopoverContentProps;
type Core_PopoverProps = PopoverProps;
declare const Core_PopoverTrigger: typeof PopoverTrigger;
type Core_PopoverTriggerProps = PopoverTriggerProps;
declare const Core_Portal: typeof Portal;
declare const Core_PortalContainerProvider: typeof PortalContainerProvider;
type Core_PortalProps = PortalProps;
declare const Core_Preview: typeof Preview;
type Core_PreviewProps = PreviewProps;
declare const Core_ProgressBar: typeof ProgressBar;
type Core_ProgressBarProps = ProgressBarProps;
declare const Core_ProgressSpinner: typeof ProgressSpinner;
type Core_ProgressSpinnerProps = ProgressSpinnerProps;
declare const Core_RadioGroup: typeof RadioGroup;
type Core_RadioGroupProps = RadioGroupProps;
declare const Core_Reaction: typeof Reaction;
type Core_ReactionProps = ReactionProps;
type Core_Rect = Rect;
declare const Core_ScrollControlButton: typeof ScrollControlButton;
type Core_ScrollControlButtonProps = ScrollControlButtonProps;
declare const Core_SearchInput: typeof SearchInput;
type Core_SearchInputProps = SearchInputProps;
declare const Core_Select: typeof Select;
type Core_SelectProps = SelectProps;
declare const Core_Separator: typeof Separator;
type Core_SeparatorProps = SeparatorProps;
declare const Core_Slider: typeof Slider;
type Core_SliderProps = SliderProps;
type Core_StylingProps = StylingProps;
declare const Core_Switch: typeof Switch;
type Core_SwitchProps = SwitchProps;
declare const Core_TabItem: typeof TabItem;
type Core_TabItemProps = TabItemProps;
declare const Core_Tabs: typeof Tabs;
type Core_TabsProps = TabsProps;
declare const Core_TagGroup: typeof TagGroup;
type Core_TagGroupProps = TagGroupProps;
declare const Core_TextInput: typeof TextInput;
type Core_TextInputProps = TextInputProps;
type Core_TextProps<T extends keyof React__default.JSX.IntrinsicElements = "span"> = TextProps<T>;
declare const Core_ThemeProvider: typeof ThemeProvider;
type Core_ThemeProviderProps = ThemeProviderProps;
declare const Core_ToggleButton: typeof ToggleButton;
type Core_ToggleButtonProps = ToggleButtonProps;
declare const Core_ToggleIconButton: typeof ToggleIconButton;
type Core_ToggleIconButtonProps = ToggleIconButtonProps;
declare const Core_Toolbar: typeof Toolbar;
type Core_ToolbarProps = ToolbarProps;
declare const Core_Tooltip: typeof Tooltip;
type Core_TooltipProps = TooltipProps;
declare const Core_Transform: typeof Transform;
type Core_TransformProps = TransformProps;
declare const Core_Transition: typeof Transition;
type Core_TransitionProps = TransitionProps;
declare const Core_defineMessages: typeof defineMessages;
declare const Core_isInsideOverlayContent: typeof isInsideOverlayContent;
declare const Core_isRect: typeof isRect;
declare const Core_useCollator: typeof useCollator;
declare const Core_useDateFormatter: typeof useDateFormatter;
declare const Core_useDevice: typeof useDevice;
declare const Core_useI18n: typeof useI18n;
declare const Core_useImage: typeof useImage;
declare const Core_useIntersectionObserver: typeof useIntersectionObserver;
declare const Core_useIsFirstRender: typeof useIsFirstRender;
declare const Core_useLocalStorage: typeof useLocalStorage;
declare const Core_useMutationObserver: typeof useMutationObserver;
declare const Core_useNumberFormatter: typeof useNumberFormatter;
declare const Core_usePortalContainer: typeof usePortalContainer;
declare const Core_useResizeObserver: typeof useResizeObserver;
declare const Core_useTextSelection: typeof useTextSelection;
declare const Core_useUndoRedo: typeof useUndoRedo;
declare const Core_useUserPreferences: typeof useUserPreferences;
declare namespace Core {
  export { Core_Accordion as Accordion, Core_AccordionItem as AccordionItem, type Core_AccordionItemProps as AccordionItemProps, type Core_AccordionProps as AccordionProps, Core_ActionButton as ActionButton, type Core_ActionButtonProps as ActionButtonProps, Core_ActionGroup as ActionGroup, Core_ActionGroupItem as ActionGroupItem, type Core_ActionGroupProps as ActionGroupProps, Core_ActionIconButton as ActionIconButton, type Core_ActionIconButtonProps as ActionIconButtonProps, Core_AlertDialog as AlertDialog, type Core_AlertDialogProps as AlertDialogProps, Core_AudioPlayer as AudioPlayer, type Core_AudioPlayerProps as AudioPlayerProps, Core_Avatar as Avatar, type Core_AvatarProps as AvatarProps, type BlockProps$1 as BlockProps, Core_Box as Box, type Core_BoxProps as BoxProps, Core_ButtonSelect as ButtonSelect, type Core_ButtonSelectProps as ButtonSelectProps, Core_Checkbox as Checkbox, type Core_CheckboxProps as CheckboxProps, Core_ColorInput as ColorInput, type Core_ColorInputProps as ColorInputProps, type ColorPreset$1 as ColorPreset, Core_ColorSwatch as ColorSwatch, Core_ColorSwatchPicker as ColorSwatchPicker, type Core_ColorSwatchPickerProps as ColorSwatchPickerProps, type Core_ColorSwatchProps as ColorSwatchProps, Core_ComboBox as ComboBox, type Core_ComboBoxProps as ComboBoxProps, Core_DateFormat as DateFormat, type Core_DateFormatProps as DateFormatProps, type Core_Device as Device, Core_Dialog as Dialog, type Core_DialogProps as DialogProps, Core_DialogTitle as DialogTitle, type Core_DialogTitleProps as DialogTitleProps, Core_DomNodeRenderer as DomNodeRenderer, type Core_DomNodeRendererProps as DomNodeRendererProps, Core_Drawer as Drawer, type Core_DrawerProps as DrawerProps, Core_Editor as Editor, type Core_EditorHandle as EditorHandle, type Core_EditorProps as EditorProps, Core_FileUpload as FileUpload, type Core_FileUploadProps as FileUploadProps, Core_FreehandCanvas as FreehandCanvas, type Core_FreehandCanvasProps as FreehandCanvasProps, Core_GridList as GridList, type Core_GridListProps as GridListProps, Core_Group as Group, type Core_GroupProps as GroupProps, Core_I18nProvider as I18nProvider, type I18nProviderProps$1 as I18nProviderProps, type Core_I18nResult as I18nResult, Core_IconColorInput as IconColorInput, Core_IconColorInputButton as IconColorInputButton, type Core_IconColorInputProps as IconColorInputProps, Core_IconSelect as IconSelect, type Core_IconSelectProps as IconSelectProps, Core_IconSlider as IconSlider, type Core_IconSliderProps as IconSliderProps, Core_ImageDropZone as ImageDropZone, type Core_ImageDropZoneProps as ImageDropZoneProps, Core_ImageGallery as ImageGallery, type Core_ImageGalleryProps as ImageGalleryProps, Core_InlineAlert as InlineAlert, type Core_InlineAlertProps as InlineAlertProps, Core_Link as Link, type Core_LinkProps as LinkProps, Core_ListBox as ListBox, type Core_ListBoxProps as ListBoxProps, type Core_ListOption as ListOption, Core_Markdown as Markdown, type Core_MarkdownProps as MarkdownProps, Core_Menu as Menu, type Core_MenuItem as MenuItem, type Core_MenuProps as MenuProps, type Core_MessageDescriptor as MessageDescriptor, Core_MessageFormat as MessageFormat, type Core_MessageFormatProps as MessageFormatProps, Core_Modal as Modal, Core_ModalClose as ModalClose, Core_ModalContent as ModalContent, type Core_ModalContentProps as ModalContentProps, type Core_ModalProps as ModalProps, Core_ModalTrigger as ModalTrigger, Core_NumberFormat as NumberFormat, type Core_NumberFormatProps as NumberFormatProps, Core_NumberInput as NumberInput, type Core_NumberInputProps as NumberInputProps, Core_Pagination as Pagination, type Core_PaginationProps as PaginationProps, Core_Popover as Popover, Core_PopoverContent as PopoverContent, type Core_PopoverContentProps as PopoverContentProps, type Core_PopoverProps as PopoverProps, Core_PopoverTrigger as PopoverTrigger, type Core_PopoverTriggerProps as PopoverTriggerProps, Core_Portal as Portal, Core_PortalContainerProvider as PortalContainerProvider, type Core_PortalProps as PortalProps, Core_Preview as Preview, type Core_PreviewProps as PreviewProps, Core_ProgressBar as ProgressBar, type Core_ProgressBarProps as ProgressBarProps, Core_ProgressSpinner as ProgressSpinner, type Core_ProgressSpinnerProps as ProgressSpinnerProps, Core_RadioGroup as RadioGroup, type Core_RadioGroupProps as RadioGroupProps, Core_Reaction as Reaction, type Core_ReactionProps as ReactionProps, type Core_Rect as Rect, Core_ScrollControlButton as ScrollControlButton, type Core_ScrollControlButtonProps as ScrollControlButtonProps, Core_SearchInput as SearchInput, type Core_SearchInputProps as SearchInputProps, Core_Select as Select, type Core_SelectProps as SelectProps, Core_Separator as Separator, type Core_SeparatorProps as SeparatorProps, Core_Slider as Slider, type Core_SliderProps as SliderProps, type Core_StylingProps as StylingProps, Core_Switch as Switch, type Core_SwitchProps as SwitchProps, Core_TabItem as TabItem, type Core_TabItemProps as TabItemProps, Core_Tabs as Tabs, type Core_TabsProps as TabsProps, Core_TagGroup as TagGroup, type Core_TagGroupProps as TagGroupProps, Text$1 as Text, Core_TextInput as TextInput, type Core_TextInputProps as TextInputProps, type Core_TextProps as TextProps, Core_ThemeProvider as ThemeProvider, type Core_ThemeProviderProps as ThemeProviderProps, Core_ToggleButton as ToggleButton, type Core_ToggleButtonProps as ToggleButtonProps, Core_ToggleIconButton as ToggleIconButton, type Core_ToggleIconButtonProps as ToggleIconButtonProps, Core_Toolbar as Toolbar, type Core_ToolbarProps as ToolbarProps, Core_Tooltip as Tooltip, type Core_TooltipProps as TooltipProps, Core_Transform as Transform, type Core_TransformProps as TransformProps, Core_Transition as Transition, type Core_TransitionProps as TransitionProps, Core_defineMessages as defineMessages, Core_isInsideOverlayContent as isInsideOverlayContent, Core_isRect as isRect, Core_useCollator as useCollator, Core_useDateFormatter as useDateFormatter, Core_useDevice as useDevice, Core_useI18n as useI18n, Core_useImage as useImage, Core_useIntersectionObserver as useIntersectionObserver, Core_useIsFirstRender as useIsFirstRender, Core_useLocalStorage as useLocalStorage, Core_useMutationObserver as useMutationObserver, Core_useNumberFormatter as useNumberFormatter, Core_usePortalContainer as usePortalContainer, Core_useResizeObserver as useResizeObserver, Core_useTextSelection as useTextSelection, Core_useUndoRedo as useUndoRedo, Core_useUserPreferences as useUserPreferences };
}

declare type Any$1 = any;
declare type IRecord$1 = Record<Any$1, Any$1>;
declare type AnyComponent = React__default.ComponentType<Any$1> | keyof ReactHTML;
declare type Child = Block | string | number | ((...args: Any$1[]) => React__default.ReactElement);
declare type Props<T extends AnyComponent> = Omit<React__default.ComponentPropsWithRef<T>, "children"> & {
    "data-block-id"?: string;
    "data-block-class"?: string;
    children?: Child[];
};
declare type PropsWithoutChildren<T extends AnyComponent> = Omit<Props<T>, "children">;
/**
 * The Block class is a wrapper around React components. It is used to create a
 * virtual DOM tree that can be rendered to the DOM.
 */
declare class Block<T extends AnyComponent = AnyComponent> {
    id: string;
    className?: string;
    private readonly _container;
    private readonly _component;
    private _props;
    private _children;
    private parent?;
    constructor(component: AnyComponent, props: Props<T>, options?: {
        container?: HTMLElement;
        ownerDocument?: Document;
    });
    private _setParent;
    /**
     * This getter returns the props of the block.
     *
     * ```tsx
     * const block = new Block("div", { className: "test" });
     *
     * block.props; // { className: "test" }
     * ```
     */
    get props(): PropsWithoutChildren<T>;
    /**
     * This setter sets the props of the block. This method will replace the
     * existing props with the new props. Do not pass children in the props. Use
     * the `children` setter instead.
     *
     * ```tsx
     * const block = new Block("div", { className: "test" });
     *
     * block.props = { className: "test2" };
     * ```
     *
     * @param props
     */
    set props(props: PropsWithoutChildren<T>);
    /**
     * This method will create a React element from the block and all of its
     * children.
     *
     * ```ts
     * const App = () => {
     *   const block = new Block("div", { className: "test" });
     *   return block.createComponent();
     *   // <div className="test" />
     * };
     * ```
     *
     * @param key The key to use for the component.
     * @returns A React element.
     */
    createComponent(key?: Key$2): React__default.ReactElement;
    /**
     * This getter returns a list of the children of the block.
     *
     * ```ts
     * const block = new Block("div", { className: "test" });
     *
     * block.children = ["test"];
     * block.children; // ["test"]
     * ```
     */
    get children(): Child[];
    /**
     * This setter sets the children of the block. This method will replace the
     * existing children with the new children.
     *
     * ```ts
     * const block = new Block("div", { className: "test" });
     *
     * block.children = ["test"];
     * block.children; // ["test"]
     * ```
     *
     * @param children The new children to use.
     */
    set children(children: Child[]);
    /**
     * Prepends children to the block. This method will prepend the children to
     * the beginning of the existing children.
     *
     * ```ts
     * const child1 = new Block("div", { className: "test1", data-block-id: "test1" });
     * const child2 = new Block("div", { className: "test2", data-block-id: "test2" });
     * const block = new Block("div", { className: "test", children: [
     *  child1,
     *  child2,
     * ]});
     *
     * const child3 = new Block("div", { className: "test3", data-block-id: "test3" });
     * block.prependChildren(child3);
     * block.children; // [child3, child1, child2]
     * ```
     *
     * @param children The children to prepend.
     * @returns The Block instance.
     */
    prependChildren(...children: Child[]): this;
    /**
     * Appends children to the block. This method will append the children to the
     * end of the existing children.
     *
     * ```ts
     * const child1 = new Block("div", { className: "test1", data-block-id: "test1" });
     * const child2 = new Block("div", { className: "test2", data-block-id: "test2" });
     * const block = new Block("div", { className: "test", children: [
     *  child1,
     *  child2,
     * ]});
     *
     * const child3 = new Block("div", { className: "test3", data-block-id: "test3" });
     *
     * block.appendChildren(child3);
     * block.children; // [child1, child2, child3]
     * ```
     *
     * @param children
     * @returns The Block instance.
     */
    appendChildren(...children: Child[]): this;
    /**
     * This method will search the entire tree of children and then inserts
     * children after the child with the given id. This method will insert the
     * children after the first child with the given id. If the child is not
     * found, the children will not be inserted.
     *
     * ```ts
     * const child1 = new Block("div", { className: "test1", data-block-id: "test1" });
     * const child2 = new Block("div", { className: "test2", data-block-id: "test2" });
     *
     * const block = new Block("div", { className: "test", children: [
     *  child1,
     *  child2,
     * ]});
     *
     * const child3 = new Block("div", { className: "test3", data-block-id: "test3" });
     * block.insertAfter(child2.id, child3);
     * block.children; // [child1, child2, child3]
     * ```
     *
     * @param block The id of the child block or the block to insert the children
     *   after.
     * @param children The children to insert.
     * @returns The Block instance.
     */
    insertBefore(block: string | Block, ...children: Child[]): this;
    /**
     * This method will search the entire tree of children and then inserts
     * children after the child with the given id. This method will insert the
     * children after the first child with the given id. If the child is not
     * found, the children will not be inserted.
     *
     * ```ts
     * const child1 = new Block("div", { className: "test1", data-block-id: "test1" });
     * const child2 = new Block("div", { className: "test2", data-block-id: "test2" });
     * const block = new Block("div", { className: "test", children: [child1, child2] });
     *
     * const child3 = new Block("div", { className: "test3", data-block-id: "test3" });
     *
     * block.insertAfter(child1.id, child3);
     * block.children; // [child1, child3, child2]
     * ```
     *
     * @param block
     * @param children The children to insert.
     * @returns The Block instance.
     */
    insertAfter(block: string | Block, ...children: Child[]): this;
    /**
     * This method will search the entire tree of children and then replaces the
     * child with the given id with the new children. This method will replace the
     * first child with the given id. If the child is not found, the children will
     * not be inserted.
     *
     * ```ts
     * const child1 = new Block("div", { className: "test1", data-block-id: "test1" });
     * const child2 = new Block("div", { className: "test2", data-block-id: "test2" });
     * const block = new Block("div", { className: "test", children: [child1, child2] });
     *
     * const child3 = new Block("div", { className: "test3", data-block-id: "test3" });
     *
     * block.replace(child1.id, child3);
     * block.children; // [child3, child2]
     * ```
     *
     * @param block The id of the child block or the block to replace.
     * @param children The children to insert.
     * @returns The Block instance.
     */
    replace(block: string | Block, ...children: Child[]): this;
    /**
     * Finds a child block by its id. This method will search the entire tree of
     * children. If the child is not found, undefined is returned.
     *
     * ```ts
     * const child1 = new Block("div", {
     *   className: "test1",
     *   "data-block-id": "test1",
     * });
     * const child2 = new Block("div", {
     *   className: "test2",
     *   "data-block-id": "test2",
     * });
     * const block = new Block("div", { className: "test" }, child1, child2);
     *
     * block.getBlockById(child2.id); // child2
     * ```
     *
     * @param id The id of the child to find.
     * @returns The child block or undefined.
     */
    getBlockById<T extends AnyComponent>(id: string): Block<T> | undefined;
    /**
     * Finds the blocks by the class. This method will search the entire tree of
     * children. If the child is not found, an empty array is returned.
     *
     * @param group
     */
    getBlocksByClass<T extends AnyComponent>(group: string): Block<T>[];
    /**
     * Finds the blocks by the component. This method will search the entire tree
     * of children. If the child is not found, undefined is returned.
     *
     * ```ts
     * const child1 = new Block("div", {
     *   className: "test1",
     *   "data-block-id": "test1",
     * });
     * const child2 = new Block("div", {
     *   className: "test2",
     *   "data-block-id": "test2",
     * });
     * const slider = new Block(Slider, {
     *   className: "test",
     *   "data-block-id": "test",
     * });
     *
     * const block = new Block(
     *   "div",
     *   { className: "test" },
     *   child1,
     *   child2,
     *   slider,
     * );
     *
     * block.getBlocksByComponent(Slider); // [slider]
     * ```
     *
     * @param component
     * @returns An array of blocks.
     */
    getBlocksByComponent<T extends AnyComponent>(component: T): Block<T>[];
    private _getChild;
    private _removeChild;
    /**
     * Removes a child block by its id. This method will search the entire tree of
     * children. If the child is not found, undefined is returned.
     *
     * ```ts
     * const child1 = new Block("div", { className: "test1", data-block-id: "test1" });
     * const child2 = new Block("div", { className: "test2", data-block-id: "test2" });
     *
     * const block = new Block("div", { className: "test", children: [child1, child2] });
     *
     * block.removeChildById(child2.id);
     * block.children; // [child1]
     * ```
     *
     * @param id
     */
    removeChildById(id: string): void;
    /**
     * Returns a DOM Node that can be added directly to the DOM. This method will
     * create a DOM Node from the block and all of its children.
     */
    createDOMNode(container?: HTMLElement): HTMLElement;
    /**
     * This method will merge the props of the block with the new props.
     *
     * ```ts
     * const block = new Block("div", { className: "test", id: "testid" });
     *
     * block.mergeProps({ className: "test2" });
     * block.props; // { className: "test2", id: "testid" }
     * ```
     *
     * @param props The new props to use.
     * @returns The Block instance.
     */
    mergeProps(props: Partial<Props<T>>): this;
    /**
     * Sets a single prop on the component. This method will not re-render the
     * component.
     *
     * ```ts
     * const block = new Block("div", { className: "test", id: "testid" });
     *
     * block.setProp("className", "test2");
     * block.props; // { className: "test2", id: "testid" }
     * ```
     *
     * @param key
     * @param value
     * @returns The Block instance.
     */
    setProp<U extends keyof Props<T>>(key: U, value: Props<T>[U]): this;
    /**
     * This method will return a JSON representation of the block and all of its
     * children.
     */
    toJS(): IJSON;
}
interface IJSON {
    id: string;
    group?: string;
    component: string;
    props: IRecord$1;
    children: (IJSON | {
        value: string | number | ((...args: Any$1[]) => React__default.ReactElement);
    })[];
}

declare type Any = any;
interface BlockProps {
    "data-block-id"?: string;
    "data-block-class"?: string;
}
/**
 * Creates a new Block instance.
 *
 * @param component The component to render.
 * @param props The props to pass to the component.
 * @param children
 */
declare function createBlock<T extends keyof React__default.ReactHTML>(component: T, props?: Omit<React__default.ComponentPropsWithRef<T>, "children"> & BlockProps, ...children: Child[]): Block<T>;
declare function createBlock<T extends React__default.ForwardRefExoticComponent<Any>>(component: T, props?: Omit<React__default.ComponentPropsWithRef<T>, "children"> & BlockProps, ...children: Child[]): Block<T>;
declare function createBlock<T extends (...args: Any[]) => Any>(component: T, props?: Omit<Parameters<T>[0], "children"> & BlockProps, ...children: Child[]): Block<T>;

declare type Messages = {
    action: string;
    addBookmark: string;
    addLink: string;
    addSignature: string;
    anonymous: string;
    backgroundColor: string;
    black: string;
    blue: string;
    bold: string;
    bookmarks: string;
    cancel: string;
    chat: string;
    chooseColor: string;
    clearSignature: string;
    close: string;
    color: string;
    comment: string;
    commentAction: string;
    commentEditorLabel: string;
    commentOptions: string;
    comments: string;
    commentsCount: string;
    commentThread: string;
    copy: string;
    createStamp: string;
    customStamp: string;
    cut: string;
    darkBlue: string;
    date: string;
    delete: string;
    deleteBookmarkConfirmAccessibilityLabel: string;
    deleteBookmarkConfirmMessage: string;
    deleteComment: string;
    deleteCommentConfirmMessage: string;
    discardChanges: string;
    docEditorMoveBeginningHint: string;
    documentEditor: string;
    documentEditorDesc: string;
    documentMergedHere: string;
    done: string;
    draw: string;
    edit: string;
    editContent: string;
    editLink: string;
    fontColor: string;
    fonts: string;
    help: string;
    image: string;
    italic: string;
    linkAnnotation: string;
    loading: string;
    mention: string;
    nMoreComments: string;
    noBookmarks: string;
    none: string;
    openHelpDialog: string;
    pagesSelected: string;
    paste: string;
    pause: string;
    play: string;
    preview: string;
    readMore: string;
    redo: string;
    removeLink: string;
    save: string;
    saveAs: string;
    saveSignature: string;
    selectAll: string;
    selectDragImage: string;
    shortcut: string;
    showMore: string;
    signature: string;
    signatures: string;
    signHere: string;
    stampAnnotationTemplatesDialog: string;
    stampText: string;
    storeSignature: string;
    submit: string;
    time: string;
    toolbarFocus: string;
    type: string;
    typeSomething: string;
    underline: string;
    undo: string;
    useAnExistingStampDesign: string;
    viewMoreResults: string;
};

interface SignatureTextInputProps extends StylingProps, Omit<AriaTextFieldOptions<"input">, "isRequired" | "isReadOnly" | "inputElementType" | "validationState" | "isInvalid" | "description" | "errorMessage" | keyof AriaLabelingProps> {
    /** The style object to apply to the input element. */
    inputStyle?: React__default.CSSProperties;
    /** The class name to apply to the input element. */
    inputClassName?: string;
    /** The clear button label. */
    clearLabel: string;
    /**
     * Whether the input is inline. This affects the styling of the input.
     *
     * @default true
     */
    isInline?: boolean;
    /** The label for the input. */
    label: string;
    /** The class name to apply to the footer element. */
    footerClassName?: string;
    /** The style object to apply to the footer element. */
    footerStyle?: React__default.CSSProperties;
}

declare const SignatureTextInput: React__default.ForwardRefExoticComponent<SignatureTextInputProps & React__default.RefAttributes<HTMLDivElement>>;

interface DevtoolsProps extends StylingProps {
    /** The children to render. */
    children: React__default.ReactNode;
    /**
     * The stating string for the class name.
     *
     * @default "BaselineUI"
     */
    classStartsWith?: string;
}

declare const Devtools: React__default.ForwardRefExoticComponent<DevtoolsProps & React__default.RefAttributes<HTMLDivElement>>;

interface StampProps extends StylingProps, AriaLabelingProps {
    /** The height of the stamp. */
    height?: number;
    /** The minimum width of the stamp. */
    minWidth?: number;
    /** The maximum width of the stamp. */
    maxWidth?: number;
    /** The width of the stamp's stroke. */
    strokeWidth?: number;
    /** The color of the stamp's stroke. */
    strokeColor?: string;
    /** The fill color of the stamp. */
    fillColor?: string;
    /** The color of the stamp's font. */
    fontColor?: string;
    /** The font family of the stamp. */
    fontFamily?: string;
    /** The description of the stamp. This is displayed below the `text`. */
    subText?: string;
    /**
     * The main text of the stamp. This is displayed at the center of the stamp if
     * no `subText` is provided. Otherwise, it is displayed above the `subText`.
     */
    text?: string;
    /**
     * The label of the stamp. This is used as the title of the SVG element. It is
     * not visible to the user.
     */
    label?: string | React__default.ReactNode;
    /**
     * The callback function that is called when the dimensions of the stamp
     * change.
     */
    onDimensionsChange?: (dimensions: {
        width: number;
        height: number;
    }) => void;
}

declare const Stamp: React__default.ForwardRefExoticComponent<StampProps & React__default.RefAttributes<SVGSVGElement>>;

interface CommentProps extends StylingProps {
    /**
     * The list of reactions to show in the comment. If this is not provided, the
     * reactions will not be shown.
     */
    reactions?: ReactionProps[];
    /** The name shown in the comment header. */
    name?: React__default.ReactNode;
    /**
     * The description shown in the comment header. This can be a string or a
     * React node.
     */
    description?: React__default.ReactNode;
    /** The content of the comment. This can be a string or a React node. */
    content: React__default.ReactNode;
    /**
     * The type of the content string. If this is set to "html", the content will
     * be rendered as HTML. If this is set to "plain", the content will be
     * rendered as plain text. This prop is only used when the `content` prop is a
     * string.
     *
     * @default "plain"
     */
    contentStringType?: "plain" | "html";
    /**
     * Indicates whether the element is displayed inline.
     *
     * @default true
     */
    isInline?: boolean;
    /**
     * The avatar properties to be passed to the `Avatar` component. If this is
     * not provided, the avatar will not be shown.
     */
    avatarProps?: Omit<AvatarProps, "size">;
    /**
     * The `aria-label` property assigned to the menu trigger element of the
     * component.
     */
    menuProps?: MenuProps;
    /**
     * The `children` property is an optional prop that accepts a JSX element or
     * an array of JSX elements. The children is displayed at the bottom of the
     * comment.
     */
    children?: React__default.ReactNode;
    /** Represents the unique identifier of a comment in the data. */
    "data-comment-id"?: string;
}

declare const Comment: React__default.ForwardRefExoticComponent<CommentProps & React__default.RefAttributes<HTMLDivElement>>;

interface TaggedPaginationProps extends StylingProps, Pick<PaginationProps, "isDisabled" | "decrementAriaLabel" | "incrementAriaLabel">, Omit<AriaTextFieldOptions<"input">, keyof AriaLabelingProps | "defaultValue"> {
    /** Whether the decrement button is disabled. */
    isDecrementButtonDisabled?: boolean;
    /** Whether the increment button is disabled. */
    isIncrementButtonDisabled?: boolean;
    /**
     * Determines the size of the component.
     *
     * @default "md"
     */
    size?: "xs" | "sm" | "md";
    /** The label for the component. */
    label?: React__default.ReactNode;
    /** The callback that is called when the decrement button is pressed. */
    onDecrementPress?: () => void;
    /** The callback that is called when the increment button is pressed. */
    onIncrementPress?: () => void;
    /** The aria-label for the description. */
    descriptionAriaLabel?: string;
    /** The ref handle that can be used to set the value of the component. */
    componentHandleRef?: React__default.RefObject<{
        setValue: (value: string) => void;
    }>;
}

declare const TaggedPagination: React__default.ForwardRefExoticComponent<TaggedPaginationProps & React__default.RefAttributes<HTMLDivElement>>;

declare const App_Comment: typeof Comment;
type App_CommentProps = CommentProps;
declare const App_Devtools: typeof Devtools;
type App_DevtoolsProps = DevtoolsProps;
declare const App_SignatureTextInput: typeof SignatureTextInput;
type App_SignatureTextInputProps = SignatureTextInputProps;
declare const App_Stamp: typeof Stamp;
type App_StampProps = StampProps;
declare const App_TaggedPagination: typeof TaggedPagination;
type App_TaggedPaginationProps = TaggedPaginationProps;
declare namespace App {
  export { App_Comment as Comment, type App_CommentProps as CommentProps, App_Devtools as Devtools, type App_DevtoolsProps as DevtoolsProps, App_SignatureTextInput as SignatureTextInput, type App_SignatureTextInputProps as SignatureTextInputProps, App_Stamp as Stamp, type App_StampProps as StampProps, App_TaggedPagination as TaggedPagination, type App_TaggedPaginationProps as TaggedPaginationProps };
}

interface StandardLonghandProperties<TLength = (string & {}) | 0, TTime = string & {}> {
  /**
   * The CSS **`align-content`** property sets the distribution of space between and around content items along a flexbox's cross-axis or a grid's block axis.
   *
   * **Syntax**: `normal | <baseline-position> | <content-distribution> | <overflow-position>? <content-position>`
   *
   * **Initial value**: `normal`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :------: | :-----: | :-------: | :----: | :----: |
   * |  **29**  | **28**  |   **9**   | **12** | **11** |
   * | 21 _-x-_ |         | 6.1 _-x-_ |        |        |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/align-content
   */
  alignContent?: Property.AlignContent;
  /**
   * The CSS **`align-items`** property sets the `align-self` value on all direct children as a group. In Flexbox, it controls the alignment of items on the Cross Axis. In Grid Layout, it controls the alignment of items on the Block Axis within their grid area.
   *
   * **Syntax**: `normal | stretch | <baseline-position> | [ <overflow-position>? <self-position> ]`
   *
   * **Initial value**: `normal`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * |  Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :------: | :-----: | :-----: | :----: | :----: |
   * |  **52**  | **20**  |  **9**  | **12** | **11** |
   * | 21 _-x-_ |         | 7 _-x-_ |        |        |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/align-items
   */
  alignItems?: Property.AlignItems;
  /**
   * The **`align-self`** CSS property overrides a grid or flex item's `align-items` value. In Grid, it aligns the item inside the grid area. In Flexbox, it aligns the item on the cross axis.
   *
   * **Syntax**: `auto | normal | stretch | <baseline-position> | <overflow-position>? <self-position>`
   *
   * **Initial value**: `auto`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :------: | :-----: | :-------: | :----: | :----: |
   * |  **36**  | **20**  |   **9**   | **12** | **11** |
   * | 21 _-x-_ |         | 6.1 _-x-_ |        |        |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox |  Safari  |  Edge  |      IE      |
   * | :----: | :-----: | :------: | :----: | :----------: |
   * | **57** | **52**  | **10.1** | **16** | **10** _-x-_ |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/align-self
   */
  alignSelf?: Property.AlignSelf;
  /**
   * The **`align-tracks`** CSS property sets the alignment in the masonry axis for grid containers that have masonry in their block axis.
   *
   * **Syntax**: `[ normal | <baseline-position> | <content-distribution> | <overflow-position>? <content-position> ]#`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   |   n/a   |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/align-tracks
   */
  alignTracks?: Property.AlignTracks;
  /**
   * The **`animation-delay`** CSS property specifies the amount of time to wait from applying the animation to an element before beginning to perform the animation. The animation can start later, immediately from its beginning, or immediately and partway through the animation.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation-delay
   */
  animationDelay?: Property.AnimationDelay<TTime>;
  /**
   * The **`animation-direction`** CSS property sets whether an animation should play forward, backward, or alternate back and forth between playing the sequence forward and backward.
   *
   * **Syntax**: `<single-animation-direction>#`
   *
   * **Initial value**: `normal`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation-direction
   */
  animationDirection?: Property.AnimationDirection;
  /**
   * The **`animation-duration`** CSS property sets the length of time that an animation takes to complete one cycle.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation-duration
   */
  animationDuration?: Property.AnimationDuration<TTime>;
  /**
   * The **`animation-fill-mode`** CSS property sets how a CSS animation applies styles to its target before and after its execution.
   *
   * **Syntax**: `<single-animation-fill-mode>#`
   *
   * **Initial value**: `none`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 5 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation-fill-mode
   */
  animationFillMode?: Property.AnimationFillMode;
  /**
   * The **`animation-iteration-count`** CSS property sets the number of times an animation sequence should be played before stopping.
   *
   * **Syntax**: `<single-animation-iteration-count>#`
   *
   * **Initial value**: `1`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation-iteration-count
   */
  animationIterationCount?: Property.AnimationIterationCount;
  /**
   * The **`animation-name`** CSS property specifies the names of one or more `@keyframes` at-rules describing the animation or animations to apply to the element.
   *
   * **Syntax**: `[ none | <keyframes-name> ]#`
   *
   * **Initial value**: `none`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation-name
   */
  animationName?: Property.AnimationName;
  /**
   * The **`animation-play-state`** CSS property sets whether an animation is running or paused.
   *
   * **Syntax**: `<single-animation-play-state>#`
   *
   * **Initial value**: `running`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation-play-state
   */
  animationPlayState?: Property.AnimationPlayState;
  /**
   * The **`animation-timing-function`** CSS property sets how an animation progresses through the duration of each cycle.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation-timing-function
   */
  animationTimingFunction?: Property.AnimationTimingFunction;
  /**
   * The `**appearance**` CSS property is used to display an element using platform-native styling, based on the operating system's theme. The **`-moz-appearance`** and **`-webkit-appearance`** properties are non-standard versions of this property, used (respectively) by Gecko (Firefox) and by WebKit-based (e.g., Safari) and Blink-based (e.g., Chrome, Opera) browsers to achieve the same thing. Note that Firefox and Edge also support **`-webkit-appearance`**, for compatibility reasons.
   *
   * **Syntax**: `none | auto | textfield | menulist-button | <compat-auto>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome  | Firefox |   Safari    |   Edge   | IE  |
   * | :-----: | :-----: | :---------: | :------: | :-: |
   * | **84**  | **80**  | **3** _-x-_ |  **84**  | No  |
   * | 1 _-x-_ | 1 _-x-_ |             | 12 _-x-_ |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/appearance
   */
  appearance?: Property.Appearance;
  /**
   * The **`aspect-ratio`**  CSS property sets a **preferred aspect ratio** for the box, which will be used in the calculation of auto sizes and some other layout functions.
   *
   * **Syntax**: `auto | <ratio>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **88** |   n/a   |   No   | **88** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/aspect-ratio
   */
  aspectRatio?: Property.AspectRatio;
  /**
   * The **`backdrop-filter`** CSS property lets you apply graphical effects such as blurring or color shifting to the area behind an element. Because it applies to everything _behind_ the element, to see the effect you must make the element or its background at least partially transparent.
   *
   * **Syntax**: `none | <filter-function-list>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox |   Safari    |  Edge  | IE  |
   * | :----: | :-----: | :---------: | :----: | :-: |
   * | **76** |   n/a   | **9** _-x-_ | **17** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/backdrop-filter
   */
  backdropFilter?: Property.BackdropFilter;
  /**
   * The **`backface-visibility`** CSS property sets whether the back face of an element is visible when turned towards the user.
   *
   * **Syntax**: `visible | hidden`
   *
   * **Initial value**: `visible`
   *
   * |  Chrome  | Firefox  |    Safari     |  Edge  |   IE   |
   * | :------: | :------: | :-----------: | :----: | :----: |
   * |  **36**  |  **16**  | **5.1** _-x-_ | **12** | **10** |
   * | 12 _-x-_ | 10 _-x-_ |               |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/backface-visibility
   */
  backfaceVisibility?: Property.BackfaceVisibility;
  /**
   * The **`background-attachment`** CSS property sets whether a background image's position is fixed within the viewport, or scrolls with its containing block.
   *
   * **Syntax**: `<attachment>#`
   *
   * **Initial value**: `scroll`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-attachment
   */
  backgroundAttachment?: Property.BackgroundAttachment;
  /**
   * The **`background-blend-mode`** CSS property sets how an element's background images should blend with each other and with the element's background color.
   *
   * **Syntax**: `<blend-mode>#`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **35** | **30**  | **8**  | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-blend-mode
   */
  backgroundBlendMode?: Property.BackgroundBlendMode;
  /**
   * The **`background-clip`** CSS property sets whether an element's background extends underneath its border box, padding box, or content box.
   *
   * **Syntax**: `<box>#`
   *
   * **Initial value**: `border-box`
   *
   * | Chrome | Firefox |   Safari    |  Edge  |  IE   |
   * | :----: | :-----: | :---------: | :----: | :---: |
   * | **1**  |  **4**  | **3** _-x-_ | **12** | **9** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-clip
   */
  backgroundClip?: Property.BackgroundClip;
  /**
   * The **`background-color`** CSS property sets the background color of an element.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `transparent`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-color
   */
  backgroundColor?: Property.BackgroundColor;
  /**
   * The **`background-image`** CSS property sets one or more background images on an element.
   *
   * **Syntax**: `<bg-image>#`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-image
   */
  backgroundImage?: Property.BackgroundImage;
  /**
   * The **`background-origin`** CSS property sets the background's origin: from the border start, inside the border, or inside the padding.
   *
   * **Syntax**: `<box>#`
   *
   * **Initial value**: `padding-box`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **4**  | **3**  | **12** | **9** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-origin
   */
  backgroundOrigin?: Property.BackgroundOrigin;
  /**
   * The **`background-position-x`** CSS property sets the initial horizontal position for each background image. The position is relative to the position layer set by `background-origin`.
   *
   * **Syntax**: `[ center | [ [ left | right | x-start | x-end ]? <length-percentage>? ]! ]#`
   *
   * **Initial value**: `left`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  | **49**  | **1**  | **12** | **6** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-position-x
   */
  backgroundPositionX?: Property.BackgroundPositionX<TLength>;
  /**
   * The **`background-position-y`** CSS property sets the initial vertical position for each background image. The position is relative to the position layer set by `background-origin`.
   *
   * **Syntax**: `[ center | [ [ top | bottom | y-start | y-end ]? <length-percentage>? ]! ]#`
   *
   * **Initial value**: `top`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  | **49**  | **1**  | **12** | **6** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-position-y
   */
  backgroundPositionY?: Property.BackgroundPositionY<TLength>;
  /**
   * The **`background-repeat`** CSS property sets how background images are repeated. A background image can be repeated along the horizontal and vertical axes, or not repeated at all.
   *
   * **Syntax**: `<repeat-style>#`
   *
   * **Initial value**: `repeat`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-repeat
   */
  backgroundRepeat?: Property.BackgroundRepeat;
  /**
   * The **`background-size`** CSS property sets the size of the element's background image. The image can be left to its natural size, stretched, or constrained to fit the available space.
   *
   * **Syntax**: `<bg-size>#`
   *
   * **Initial value**: `auto auto`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |  IE   |
   * | :-----: | :-----: | :-----: | :----: | :---: |
   * |  **3**  |  **4**  |  **5**  | **12** | **9** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-size
   */
  backgroundSize?: Property.BackgroundSize<TLength>;
  /**
   * **Syntax**: `clip | ellipsis | <string>`
   *
   * **Initial value**: `clip`
   */
  blockOverflow?: Property.BlockOverflow;
  /**
   * The **`block-size`** CSS property defines the horizontal or vertical size of an element's block, depending on its writing mode. It corresponds to either the `width` or the `height` property, depending on the value of `writing-mode`.
   *
   * **Syntax**: `<'width'>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/block-size
   */
  blockSize?: Property.BlockSize<TLength>;
  /**
   * The **`border-block-color`** CSS property defines the color of the logical block borders of an element, which maps to a physical border color depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-color` and `border-bottom-color`, or `border-right-color` and `border-left-color` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-color'>{1,2}`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-color
   */
  borderBlockColor?: Property.BorderBlockColor;
  /**
   * The **`border-block-end-color`** CSS property defines the color of the logical block-end border of an element, which maps to a physical border color depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-color`, `border-right-color`, `border-bottom-color`, or `border-left-color` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-color'>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-end-color
   */
  borderBlockEndColor?: Property.BorderBlockEndColor;
  /**
   * The **`border-block-end-style`** CSS property defines the style of the logical block-end border of an element, which maps to a physical border style depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-style`, `border-right-style`, `border-bottom-style`, or `border-left-style` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-style'>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-end-style
   */
  borderBlockEndStyle?: Property.BorderBlockEndStyle;
  /**
   * The **`border-block-end-width`** CSS property defines the width of the logical block-end border of an element, which maps to a physical border width depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-width`, `border-right-width`, `border-bottom-width`, or `border-left-width` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-width'>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-end-width
   */
  borderBlockEndWidth?: Property.BorderBlockEndWidth<TLength>;
  /**
   * The **`border-block-start-color`** CSS property defines the color of the logical block-start border of an element, which maps to a physical border color depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-color`, `border-right-color`, `border-bottom-color`, or `border-left-color` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-color'>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-start-color
   */
  borderBlockStartColor?: Property.BorderBlockStartColor;
  /**
   * The **`border-block-start-style`** CSS property defines the style of the logical block start border of an element, which maps to a physical border style depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-style`, `border-right-style`, `border-bottom-style`, or `border-left-style` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-style'>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-start-style
   */
  borderBlockStartStyle?: Property.BorderBlockStartStyle;
  /**
   * The **`border-block-start-width`** CSS property defines the width of the logical block-start border of an element, which maps to a physical border width depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-width`, `border-right-width`, `border-bottom-width`, or `border-left-width` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-width'>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-start-width
   */
  borderBlockStartWidth?: Property.BorderBlockStartWidth<TLength>;
  /**
   * The **`border-block-style`** CSS property defines the style of the logical block borders of an element, which maps to a physical border style depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-style` and `border-bottom-style`, or `border-left-style` and `border-right-style` properties depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-style'>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-style
   */
  borderBlockStyle?: Property.BorderBlockStyle;
  /**
   * The **`border-block-width`** CSS property defines the width of the logical block borders of an element, which maps to a physical border width depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-width` and `border-bottom-width`, or `border-left-width`, and `border-right-width` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-width'>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-width
   */
  borderBlockWidth?: Property.BorderBlockWidth<TLength>;
  /**
   * The **`border-bottom-color`** CSS property sets the color of an element's bottom border. It can also be set with the shorthand CSS properties `border-color` or `border-bottom`.
   *
   * **Syntax**: `<'border-top-color'>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-bottom-color
   */
  borderBottomColor?: Property.BorderBottomColor;
  /**
   * The **`border-bottom-left-radius`** CSS property rounds the bottom-left corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |  IE   |
   * | :-----: | :-----: | :-----: | :----: | :---: |
   * |  **4**  |  **4**  |  **5**  | **12** | **9** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-bottom-left-radius
   */
  borderBottomLeftRadius?: Property.BorderBottomLeftRadius<TLength>;
  /**
   * The **`border-bottom-right-radius`** CSS property rounds the bottom-right corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |  IE   |
   * | :-----: | :-----: | :-----: | :----: | :---: |
   * |  **4**  |  **4**  |  **5**  | **12** | **9** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-bottom-right-radius
   */
  borderBottomRightRadius?: Property.BorderBottomRightRadius<TLength>;
  /**
   * The **`border-bottom-style`** CSS property sets the line style of an element's bottom `border`.
   *
   * **Syntax**: `<line-style>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-bottom-style
   */
  borderBottomStyle?: Property.BorderBottomStyle;
  /**
   * The **`border-bottom-width`** CSS property sets the width of the bottom border of an element.
   *
   * **Syntax**: `<line-width>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-bottom-width
   */
  borderBottomWidth?: Property.BorderBottomWidth<TLength>;
  /**
   * The **`border-collapse`** CSS property sets whether cells inside a `<table>` have shared or separate borders.
   *
   * **Syntax**: `collapse | separate`
   *
   * **Initial value**: `separate`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  |  **1**  | **1.2** | **12** | **5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-collapse
   */
  borderCollapse?: Property.BorderCollapse;
  /**
   * The **`border-end-end-radius`** CSS property defines a logical border radius on an element, which maps to a physical border radius that depends on the element's `writing-mode`, `direction`, and `text-orientation`. This is useful when building styles to work regardless of the text orientation and writing mode.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **89** | **66**  |   No   | **89** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-end-end-radius
   */
  borderEndEndRadius?: Property.BorderEndEndRadius<TLength>;
  /**
   * The **`border-end-start-radius`** CSS property defines a logical border radius on an element, which maps to a physical border radius depending on the element's `writing-mode`, `direction`, and `text-orientation`. This is useful when building styles to work regardless of the text orientation and writing mode.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **89** | **66**  |   No   | **89** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-end-start-radius
   */
  borderEndStartRadius?: Property.BorderEndStartRadius<TLength>;
  /**
   * The **`border-image-outset`** CSS property sets the distance by which an element's border image is set out from its border box.
   *
   * **Syntax**: `[ <length> | <number> ]{1,4}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **15** | **15**  | **6**  | **12** | **11** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-image-outset
   */
  borderImageOutset?: Property.BorderImageOutset<TLength>;
  /**
   * The **`border-image-repeat`** CSS property defines how the edge regions of a source image are adjusted to fit the dimensions of an element's border image.
   *
   * **Syntax**: `[ stretch | repeat | round | space ]{1,2}`
   *
   * **Initial value**: `stretch`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **15** | **15**  | **6**  | **12** | **11** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-image-repeat
   */
  borderImageRepeat?: Property.BorderImageRepeat;
  /**
   * The **`border-image-slice`** CSS property divides the image specified by `border-image-source` into regions. These regions form the components of an element's border image.
   *
   * **Syntax**: `<number-percentage>{1,4} && fill?`
   *
   * **Initial value**: `100%`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **15** | **15**  | **6**  | **12** | **11** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-image-slice
   */
  borderImageSlice?: Property.BorderImageSlice;
  /**
   * The **`border-image-source`** CSS property sets the source image used to create an element's border image.
   *
   * **Syntax**: `none | <image>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **15** | **15**  | **6**  | **12** | **11** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-image-source
   */
  borderImageSource?: Property.BorderImageSource;
  /**
   * The **`border-image-width`** CSS property sets the width of an element's border image.
   *
   * **Syntax**: `[ <length-percentage> | <number> | auto ]{1,4}`
   *
   * **Initial value**: `1`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **15** | **13**  | **6**  | **12** | **11** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-image-width
   */
  borderImageWidth?: Property.BorderImageWidth<TLength>;
  /**
   * The **`border-inline-color`** CSS property defines the color of the logical inline borders of an element, which maps to a physical border color depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-color` and `border-bottom-color`, or `border-right-color` and `border-left-color` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-color'>{1,2}`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-color
   */
  borderInlineColor?: Property.BorderInlineColor;
  /**
   * The **`border-inline-end-color`** CSS property defines the color of the logical inline-end border of an element, which maps to a physical border color depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-color`, `border-right-color`, `border-bottom-color`, or `border-left-color` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-color'>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome |           Firefox           |  Safari  |  Edge  | IE  |
   * | :----: | :-------------------------: | :------: | :----: | :-: |
   * | **69** |           **41**            | **12.1** | **79** | No  |
   * |        | 3 _(-moz-border-end-color)_ |          |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-end-color
   */
  borderInlineEndColor?: Property.BorderInlineEndColor;
  /**
   * The **`border-inline-end-style`** CSS property defines the style of the logical inline end border of an element, which maps to a physical border style depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-style`, `border-right-style`, `border-bottom-style`, or `border-left-style` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-style'>`
   *
   * **Initial value**: `none`
   *
   * | Chrome |           Firefox           |  Safari  |  Edge  | IE  |
   * | :----: | :-------------------------: | :------: | :----: | :-: |
   * | **69** |           **41**            | **12.1** | **79** | No  |
   * |        | 3 _(-moz-border-end-style)_ |          |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-end-style
   */
  borderInlineEndStyle?: Property.BorderInlineEndStyle;
  /**
   * The **`border-inline-end-width`** CSS property defines the width of the logical inline-end border of an element, which maps to a physical border width depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-width`, `border-right-width`, `border-bottom-width`, or `border-left-width` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-width'>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome |           Firefox           |  Safari  |  Edge  | IE  |
   * | :----: | :-------------------------: | :------: | :----: | :-: |
   * | **69** |           **41**            | **12.1** | **79** | No  |
   * |        | 3 _(-moz-border-end-width)_ |          |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-end-width
   */
  borderInlineEndWidth?: Property.BorderInlineEndWidth<TLength>;
  /**
   * The **`border-inline-start-color`** CSS property defines the color of the logical inline start border of an element, which maps to a physical border color depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-color`, `border-right-color`, `border-bottom-color`, or `border-left-color` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-color'>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome |            Firefox            |  Safari  |  Edge  | IE  |
   * | :----: | :---------------------------: | :------: | :----: | :-: |
   * | **69** |            **41**             | **12.1** | **79** | No  |
   * |        | 3 _(-moz-border-start-color)_ |          |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-start-color
   */
  borderInlineStartColor?: Property.BorderInlineStartColor;
  /**
   * The **`border-inline-start-style`** CSS property defines the style of the logical inline start border of an element, which maps to a physical border style depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-style`, `border-right-style`, `border-bottom-style`, or `border-left-style` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-style'>`
   *
   * **Initial value**: `none`
   *
   * | Chrome |            Firefox            |  Safari  |  Edge  | IE  |
   * | :----: | :---------------------------: | :------: | :----: | :-: |
   * | **69** |            **41**             | **12.1** | **79** | No  |
   * |        | 3 _(-moz-border-start-style)_ |          |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-start-style
   */
  borderInlineStartStyle?: Property.BorderInlineStartStyle;
  /**
   * The **`border-inline-start-width`** CSS property defines the width of the logical inline-start border of an element, which maps to a physical border width depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-width`, `border-right-width`, `border-bottom-width`, or `border-left-width` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-width'>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-start-width
   */
  borderInlineStartWidth?: Property.BorderInlineStartWidth<TLength>;
  /**
   * The **`border-inline-style`** CSS property defines the style of the logical inline borders of an element, which maps to a physical border style depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-style` and `border-bottom-style`, or `border-left-style` and `border-right-style` properties depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-style'>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-style
   */
  borderInlineStyle?: Property.BorderInlineStyle;
  /**
   * The **`border-inline-width`** CSS property defines the width of the logical inline borders of an element, which maps to a physical border width depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-width` and `border-bottom-width`, or `border-left-width`, and `border-right-width` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-width'>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-width
   */
  borderInlineWidth?: Property.BorderInlineWidth<TLength>;
  /**
   * The **`border-left-color`** CSS property sets the color of an element's left border. It can also be set with the shorthand CSS properties `border-color` or `border-left`.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-left-color
   */
  borderLeftColor?: Property.BorderLeftColor;
  /**
   * The **`border-left-style`** CSS property sets the line style of an element's left `border`.
   *
   * **Syntax**: `<line-style>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-left-style
   */
  borderLeftStyle?: Property.BorderLeftStyle;
  /**
   * The **`border-left-width`** CSS property sets the width of the left border of an element.
   *
   * **Syntax**: `<line-width>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-left-width
   */
  borderLeftWidth?: Property.BorderLeftWidth<TLength>;
  /**
   * The **`border-right-color`** CSS property sets the color of an element's right border. It can also be set with the shorthand CSS properties `border-color` or `border-right`.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-right-color
   */
  borderRightColor?: Property.BorderRightColor;
  /**
   * The **`border-right-style`** CSS property sets the line style of an element's right `border`.
   *
   * **Syntax**: `<line-style>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-right-style
   */
  borderRightStyle?: Property.BorderRightStyle;
  /**
   * The **`border-right-width`** CSS property sets the width of the right border of an element.
   *
   * **Syntax**: `<line-width>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-right-width
   */
  borderRightWidth?: Property.BorderRightWidth<TLength>;
  /**
   * The **`border-spacing`** CSS property sets the distance between the borders of adjacent `<table>` cells. This property applies only when `border-collapse` is `separate`.
   *
   * **Syntax**: `<length> <length>?`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-spacing
   */
  borderSpacing?: Property.BorderSpacing<TLength>;
  /**
   * The **`border-start-end-radius`** CSS property defines a logical border radius on an element, which maps to a physical border radius depending on the element's `writing-mode`, `direction`, and `text-orientation`. This is useful when building styles to work regardless of the text orientation and writing mode.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **89** | **66**  |   No   | **89** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-start-end-radius
   */
  borderStartEndRadius?: Property.BorderStartEndRadius<TLength>;
  /**
   * The **`border-start-start-radius`** CSS property defines a logical border radius on an element, which maps to a physical border radius that depends on the element's `writing-mode`, `direction`, and `text-orientation`. This is useful when building styles to work regardless of the text orientation and writing mode.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **89** | **66**  |   No   | **89** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-start-start-radius
   */
  borderStartStartRadius?: Property.BorderStartStartRadius<TLength>;
  /**
   * The **`border-top-color`** CSS property sets the color of an element's top border. It can also be set with the shorthand CSS properties `border-color` or `border-top`.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-top-color
   */
  borderTopColor?: Property.BorderTopColor;
  /**
   * The **`border-top-left-radius`** CSS property rounds the top-left corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |  IE   |
   * | :-----: | :-----: | :-----: | :----: | :---: |
   * |  **4**  |  **4**  |  **5**  | **12** | **9** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-top-left-radius
   */
  borderTopLeftRadius?: Property.BorderTopLeftRadius<TLength>;
  /**
   * The **`border-top-right-radius`** CSS property rounds the top-right corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |  IE   |
   * | :-----: | :-----: | :-----: | :----: | :---: |
   * |  **4**  |  **4**  |  **5**  | **12** | **9** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-top-right-radius
   */
  borderTopRightRadius?: Property.BorderTopRightRadius<TLength>;
  /**
   * The **`border-top-style`** CSS property sets the line style of an element's top `border`.
   *
   * **Syntax**: `<line-style>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-top-style
   */
  borderTopStyle?: Property.BorderTopStyle;
  /**
   * The **`border-top-width`** CSS property sets the width of the top border of an element.
   *
   * **Syntax**: `<line-width>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-top-width
   */
  borderTopWidth?: Property.BorderTopWidth<TLength>;
  /**
   * The **`bottom`** CSS property participates in setting the vertical position of a positioned element. It has no effect on non-positioned elements.
   *
   * **Syntax**: `<length> | <percentage> | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/bottom
   */
  bottom?: Property.Bottom<TLength>;
  /**
   * The **`box-decoration-break`** CSS property specifies how an element's fragments should be rendered when broken across multiple lines, columns, or pages.
   *
   * **Syntax**: `slice | clone`
   *
   * **Initial value**: `slice`
   *
   * |    Chrome    | Firefox |    Safari     |     Edge     | IE  |
   * | :----------: | :-----: | :-----------: | :----------: | :-: |
   * | **22** _-x-_ | **32**  | **6.1** _-x-_ | **79** _-x-_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/box-decoration-break
   */
  boxDecorationBreak?: Property.BoxDecorationBreak;
  /**
   * The **`box-shadow`** CSS property adds shadow effects around an element's frame. You can set multiple effects separated by commas. A box shadow is described by X and Y offsets relative to the element, blur and spread radius, and color.
   *
   * **Syntax**: `none | <shadow>#`
   *
   * **Initial value**: `none`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |  IE   |
   * | :-----: | :-----: | :-----: | :----: | :---: |
   * | **10**  |  **4**  | **5.1** | **12** | **9** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/box-shadow
   */
  boxShadow?: Property.BoxShadow;
  /**
   * The **`box-sizing`** CSS property sets how the total width and height of an element is calculated.
   *
   * **Syntax**: `content-box | border-box`
   *
   * **Initial value**: `content-box`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |  IE   |
   * | :-----: | :-----: | :-----: | :----: | :---: |
   * | **10**  | **29**  | **5.1** | **12** | **8** |
   * | 1 _-x-_ | 1 _-x-_ | 3 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/box-sizing
   */
  boxSizing?: Property.BoxSizing;
  /**
   * The **`break-after`** CSS property sets how page, column, or region breaks should behave after a generated box. If there is no generated box, the property is ignored.
   *
   * **Syntax**: `auto | avoid | always | all | avoid-page | page | left | right | recto | verso | avoid-column | column | avoid-region | region`
   *
   * **Initial value**: `auto`
   *
   * ---
   *
   * _Supported in Multi-column Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **50** | **65**  |   No   | **12** | **10** |
   *
   * ---
   *
   * _Supported in Paged Media_
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **50** | **65**  | **10** | **12** | **10** |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/break-after
   */
  breakAfter?: Property.BreakAfter;
  /**
   * The **`break-before`** CSS property sets how page, column, or region breaks should behave before a generated box. If there is no generated box, the property is ignored.
   *
   * **Syntax**: `auto | avoid | always | all | avoid-page | page | left | right | recto | verso | avoid-column | column | avoid-region | region`
   *
   * **Initial value**: `auto`
   *
   * ---
   *
   * _Supported in Multi-column Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **50** | **65**  |   No   | **12** | **10** |
   *
   * ---
   *
   * _Supported in Paged Media_
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **50** | **65**  | **10** | **12** | **10** |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/break-before
   */
  breakBefore?: Property.BreakBefore;
  /**
   * The **`break-inside`** CSS property sets how page, column, or region breaks should behave inside a generated box. If there is no generated box, the property is ignored.
   *
   * **Syntax**: `auto | avoid | avoid-page | avoid-column | avoid-region`
   *
   * **Initial value**: `auto`
   *
   * ---
   *
   * _Supported in Multi-column Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **50** | **65**  | **10** | **12** | **10** |
   *
   * ---
   *
   * _Supported in Paged Media_
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **50** | **65**  | **10** | **12** | **10** |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/break-inside
   */
  breakInside?: Property.BreakInside;
  /**
   * The **`caption-side`** CSS property puts the content of a table's `<caption>` on the specified side. The values are relative to the `writing-mode` of the table.
   *
   * **Syntax**: `top | bottom | block-start | block-end | inline-start | inline-end`
   *
   * **Initial value**: `top`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/caption-side
   */
  captionSide?: Property.CaptionSide;
  /**
   * The **`caret-color`** CSS property sets the color of the **insertion caret**, the visible marker where the next character typed will be inserted. This is sometimes referred to as the **text input cursor**. The caret appears in elements such as `<input>` or those with the `contenteditable` attribute. The caret is typically a thin vertical line that flashes to help make it more noticeable. By default, it is black, but its color can be altered with this property.
   *
   * **Syntax**: `auto | <color>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **53**  | **11.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/caret-color
   */
  caretColor?: Property.CaretColor;
  /**
   * The **`clear`** CSS property sets whether an element must be moved below (cleared) floating elements that precede it. The `clear` property applies to floating and non-floating elements.
   *
   * **Syntax**: `none | left | right | both | inline-start | inline-end`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/clear
   */
  clear?: Property.Clear;
  /**
   * The `**clip-path**` CSS property creates a clipping region that sets what part of an element should be shown. Parts that are inside the region are shown, while those outside are hidden.
   *
   * **Syntax**: `<clip-source> | [ <basic-shape> || <geometry-box> ] | none`
   *
   * **Initial value**: `none`
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :------: | :-----: | :-------: | :----: | :----: |
   * |  **55**  | **3.5** |  **9.1**  | **12** | **10** |
   * | 23 _-x-_ |         | 6.1 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/clip-path
   */
  clipPath?: Property.ClipPath;
  /**
   * The **`color`** CSS property sets the foreground color value of an element's text and text decorations, and sets the `<currentcolor>` value. `currentcolor` may be used as an indirect value on _other_ properties and is the default for other color properties, such as `border-color`.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: Varies from one browser to another
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/color
   */
  color?: Property.Color;
  /**
   * The **`color-adjust`** CSS property sets what, if anything, the user agent may do to optimize the appearance of the element on the output device. By default, the browser is allowed to make any adjustments to the element's appearance it determines to be necessary and prudent given the type and capabilities of the output device.
   *
   * **Syntax**: `economy | exact`
   *
   * **Initial value**: `economy`
   *
   * |                Chrome                 | Firefox |                Safari                |                 Edge                  | IE  |
   * | :-----------------------------------: | :-----: | :----------------------------------: | :-----------------------------------: | :-: |
   * | **49** _(-webkit-print-color-adjust)_ | **48**  | **6** _(-webkit-print-color-adjust)_ | **79** _(-webkit-print-color-adjust)_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/color-adjust
   */
  colorAdjust?: Property.ColorAdjust;
  /**
   * The **`color-scheme`** CSS property allows an element to indicate which color schemes it can comfortably be rendered in.
   *
   * **Syntax**: `normal | [ light | dark | <custom-ident> ]+`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **81** |   No    | **13** | **81** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/color-scheme
   */
  colorScheme?: Property.ColorScheme;
  /**
   * The **`column-count`** CSS property breaks an element's content into the specified number of columns.
   *
   * **Syntax**: `<integer> | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **50**  | **52**  |  **9**  | **12** | **10** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-count
   */
  columnCount?: Property.ColumnCount;
  /**
   * The **`column-fill`** CSS property controls how an element's contents are balanced when broken into columns.
   *
   * **Syntax**: `auto | balance | balance-all`
   *
   * **Initial value**: `balance`
   *
   * | Chrome | Firefox | Safari  |  Edge  |   IE   |
   * | :----: | :-----: | :-----: | :----: | :----: |
   * | **50** | **52**  |  **9**  | **12** | **10** |
   * |        |         | 8 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-fill
   */
  columnFill?: Property.ColumnFill;
  /**
   * The **`column-gap`** CSS property sets the size of the gap (gutter) between an element's columns.
   *
   * **Syntax**: `normal | <length-percentage>`
   *
   * **Initial value**: `normal`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **84** | **63**  |   No   | **84** | No  |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * |         Chrome         |        Firefox         |          Safari          |  Edge  | IE  |
   * | :--------------------: | :--------------------: | :----------------------: | :----: | :-: |
   * |         **66**         |         **61**         |         **12.1**         | **16** | No  |
   * | 57 _(grid-column-gap)_ | 52 _(grid-column-gap)_ | 10.1 _(grid-column-gap)_ |        |     |
   *
   * ---
   *
   * _Supported in Multi-column Layout_
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **50**  | **52**  | **10**  | **12** | **10** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |        |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-gap
   */
  columnGap?: Property.ColumnGap<TLength>;
  /**
   * The **`column-rule-color`** CSS property sets the color of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **50**  | **52**  |  **9**  | **12** | **10** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-rule-color
   */
  columnRuleColor?: Property.ColumnRuleColor;
  /**
   * The **`column-rule-style`** CSS property sets the style of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'border-style'>`
   *
   * **Initial value**: `none`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **50**  | **52**  |  **9**  | **12** | **10** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-rule-style
   */
  columnRuleStyle?: Property.ColumnRuleStyle;
  /**
   * The **`column-rule-width`** CSS property sets the width of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'border-width'>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **50**  | **52**  |  **9**  | **12** | **10** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-rule-width
   */
  columnRuleWidth?: Property.ColumnRuleWidth<TLength>;
  /**
   * The **`column-span`** CSS property makes it possible for an element to span across all columns when its value is set to `all`.
   *
   * **Syntax**: `none | all`
   *
   * **Initial value**: `none`
   *
   * | Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :-----: | :-----: | :-------: | :----: | :----: |
   * | **50**  | **71**  |   **9**   | **12** | **10** |
   * | 6 _-x-_ |         | 5.1 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-span
   */
  columnSpan?: Property.ColumnSpan;
  /**
   * The **`column-width`** CSS property sets the ideal column width in a multi-column layout. The container will have as many columns as can fit without any of them having a width less than the `column-width` value. If the width of the container is narrower than the specified value, the single column's width will be smaller than the declared column width.
   *
   * **Syntax**: `<length> | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **50**  | **50**  |  **9**  | **12** | **10** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-width
   */
  columnWidth?: Property.ColumnWidth<TLength>;
  /**
   * The **`contain`** CSS property allows an author to indicate that an element and its contents are, as much as possible, _independent_ of the rest of the document tree. This allows the browser to recalculate layout, style, paint, size, or any combination of them for a limited area of the DOM and not the entire page, leading to obvious performance benefits.
   *
   * **Syntax**: `none | strict | content | [ size || layout || style || paint ]`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **52** | **69**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/contain
   */
  contain?: Property.Contain;
  /**
   * The **`content`** CSS property replaces an element with a generated value. Objects inserted using the `content` property are **anonymous replaced elements**_._
   *
   * **Syntax**: `normal | none | [ <content-replacement> | <content-list> ] [/ <string> ]?`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/content
   */
  content?: Property.Content;
  /**
   * The **`content-visibility`** CSS property controls whether or not an element renders its contents at all, along with forcing a strong set of containments, allowing user agents to potentially omit large swathes of layout and rendering work until it becomes needed. Basically it enables the user agent to skip an element's rendering work, including layout and painting, until it is needed, makes the initial page load much faster.
   *
   * **Syntax**: `visible | auto | hidden`
   *
   * **Initial value**: `visible`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **85** |   No    |   No   | **85** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/content-visibility
   */
  contentVisibility?: Property.ContentVisibility;
  /**
   * The **`counter-increment`** CSS property increases or decreases the value of a CSS counter by a given value.
   *
   * **Syntax**: `[ <custom-ident> <integer>? ]+ | none`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **2**  |  **1**  | **3**  | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/counter-increment
   */
  counterIncrement?: Property.CounterIncrement;
  /**
   * The **`counter-reset`** CSS property resets a CSS counter to a given value.
   *
   * **Syntax**: `[ <custom-ident> <integer>? ]+ | none`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **2**  |  **1**  | **3**  | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/counter-reset
   */
  counterReset?: Property.CounterReset;
  /**
   * The **`counter-set`** CSS property sets a CSS counter to a given value. It manipulates the value of existing counters, and will only create new counters if there isn't already a counter of the given name on the element.
   *
   * **Syntax**: `[ <custom-ident> <integer>? ]+ | none`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **85** | **68**  |   No   | **85** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/counter-set
   */
  counterSet?: Property.CounterSet;
  /**
   * The **`cursor`** CSS property sets the type of mouse cursor, if any, to show when the mouse pointer is over an element.
   *
   * **Syntax**: `[ [ <url> [ <x> <y> ]? , ]* [ auto | default | none | context-menu | help | pointer | progress | wait | cell | crosshair | text | vertical-text | alias | copy | move | no-drop | not-allowed | e-resize | n-resize | ne-resize | nw-resize | s-resize | se-resize | sw-resize | w-resize | ew-resize | ns-resize | nesw-resize | nwse-resize | col-resize | row-resize | all-scroll | zoom-in | zoom-out | grab | grabbing ] ]`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  |  **1**  | **1.2** | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/cursor
   */
  cursor?: Property.Cursor;
  /**
   * The **`direction`** CSS property sets the direction of text, table columns, and horizontal overflow. Use `rtl` for languages written from right to left (like Hebrew or Arabic), and `ltr` for those written from left to right (like English and most other languages).
   *
   * **Syntax**: `ltr | rtl`
   *
   * **Initial value**: `ltr`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **2**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/direction
   */
  direction?: Property.Direction;
  /**
   * The **`display`** CSS property sets whether an element is treated as a block or inline element and the layout used for its children, such as flow layout, grid or flex.
   *
   * **Syntax**: `[ <display-outside> || <display-inside> ] | <display-listitem> | <display-internal> | <display-box> | <display-legacy>`
   *
   * **Initial value**: `inline`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/display
   */
  display?: Property.Display;
  /**
   * The **`empty-cells`** CSS property sets whether borders and backgrounds appear around `<table>` cells that have no visible content.
   *
   * **Syntax**: `show | hide`
   *
   * **Initial value**: `show`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  |  **1**  | **1.2** | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/empty-cells
   */
  emptyCells?: Property.EmptyCells;
  /**
   * The **`filter`** CSS property applies graphical effects like blur or color shift to an element. Filters are commonly used to adjust the rendering of images, backgrounds, and borders.
   *
   * **Syntax**: `none | <filter-function-list>`
   *
   * **Initial value**: `none`
   *
   * |  Chrome  | Firefox | Safari  |  Edge  | IE  |
   * | :------: | :-----: | :-----: | :----: | :-: |
   * |  **53**  | **35**  | **9.1** | **12** | No  |
   * | 18 _-x-_ |         | 6 _-x-_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/filter
   */
  filter?: Property.Filter;
  /**
   * The **`flex-basis`** CSS property sets the initial main size of a flex item. It sets the size of the content box unless otherwise set with `box-sizing`.
   *
   * **Syntax**: `content | <'width'>`
   *
   * **Initial value**: `auto`
   *
   * |  Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :------: | :-----: | :-----: | :----: | :----: |
   * |  **29**  | **22**  |  **9**  | **12** | **11** |
   * | 22 _-x-_ |         | 7 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/flex-basis
   */
  flexBasis?: Property.FlexBasis<TLength>;
  /**
   * The **`flex-direction`** CSS property sets how flex items are placed in the flex container defining the main axis and the direction (normal or reversed).
   *
   * **Syntax**: `row | row-reverse | column | column-reverse`
   *
   * **Initial value**: `row`
   *
   * |  Chrome  | Firefox | Safari  |  Edge  |    IE    |
   * | :------: | :-----: | :-----: | :----: | :------: |
   * |  **29**  | **20**  |  **9**  | **12** |  **11**  |
   * | 21 _-x-_ |         | 7 _-x-_ |        | 10 _-x-_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/flex-direction
   */
  flexDirection?: Property.FlexDirection;
  /**
   * The **`flex-grow`** CSS property sets the flex grow factor of a flex item main size.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `0`
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  |            IE            |
   * | :------: | :-----: | :-------: | :----: | :----------------------: |
   * |  **29**  | **20**  |   **9**   | **12** |          **11**          |
   * | 22 _-x-_ |         | 6.1 _-x-_ |        | 10 _(-ms-flex-positive)_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/flex-grow
   */
  flexGrow?: Property.FlexGrow;
  /**
   * The **`flex-shrink`** CSS property sets the flex shrink factor of a flex item. If the size of all flex items is larger than the flex container, items shrink to fit according to `flex-shrink`.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `1`
   *
   * |  Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :------: | :-----: | :-----: | :----: | :----: |
   * |  **29**  | **20**  |  **9**  | **12** | **10** |
   * | 22 _-x-_ |         | 8 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/flex-shrink
   */
  flexShrink?: Property.FlexShrink;
  /**
   * The **`flex-wrap`** CSS property sets whether flex items are forced onto one line or can wrap onto multiple lines. If wrapping is allowed, it sets the direction that lines are stacked.
   *
   * **Syntax**: `nowrap | wrap | wrap-reverse`
   *
   * **Initial value**: `nowrap`
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :------: | :-----: | :-------: | :----: | :----: |
   * |  **29**  | **28**  |   **9**   | **12** | **11** |
   * | 21 _-x-_ |         | 6.1 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/flex-wrap
   */
  flexWrap?: Property.FlexWrap;
  /**
   * The **`float`** CSS property places an element on the left or right side of its container, allowing text and inline elements to wrap around it. The element is removed from the normal flow of the page, though still remaining a part of the flow (in contrast to absolute positioning).
   *
   * **Syntax**: `left | right | none | inline-start | inline-end`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/float
   */
  float?: Property.Float;
  /**
   * The **`font-family`** CSS property specifies a prioritized list of one or more font family names and/or generic family names for the selected element.
   *
   * **Syntax**: `[ <family-name> | <generic-family> ]#`
   *
   * **Initial value**: depends on user agent
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-family
   */
  fontFamily?: Property.FontFamily;
  /**
   * The **`font-feature-settings`** CSS property controls advanced typographic features in OpenType fonts.
   *
   * **Syntax**: `normal | <feature-tag-value>#`
   *
   * **Initial value**: `normal`
   *
   * |  Chrome  | Firefox  | Safari  |  Edge  |   IE   |
   * | :------: | :------: | :-----: | :----: | :----: |
   * |  **48**  |  **34**  | **9.1** | **15** | **10** |
   * | 16 _-x-_ | 15 _-x-_ |         |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-feature-settings
   */
  fontFeatureSettings?: Property.FontFeatureSettings;
  /**
   * The **`font-kerning`** CSS property sets the use of the kerning information stored in a font.
   *
   * **Syntax**: `auto | normal | none`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari  |  Edge  | IE  |
   * | :----: | :-----: | :-----: | :----: | :-: |
   * | **33** | **32**  |  **9**  | **79** | No  |
   * |        |         | 6 _-x-_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-kerning
   */
  fontKerning?: Property.FontKerning;
  /**
   * The **`font-language-override`** CSS property controls the use of language-specific glyphs in a typeface.
   *
   * **Syntax**: `normal | <string>`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **34**  |   No   |  No  | No  |
   * |        | 4 _-x-_ |        |      |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-language-override
   */
  fontLanguageOverride?: Property.FontLanguageOverride;
  /**
   * The **`font-optical-sizing`** CSS property sets whether text rendering is optimized for viewing at different sizes.
   *
   * **Syntax**: `auto | none`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **79** | **62**  | **11** | **17** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-optical-sizing
   */
  fontOpticalSizing?: Property.FontOpticalSizing;
  /**
   * The **`font-size`** CSS property sets the size of the font. Changing the font size also updates the sizes of the font size-relative `<length>` units, such as `em`, `ex`, and so forth.
   *
   * **Syntax**: `<absolute-size> | <relative-size> | <length-percentage>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-size
   */
  fontSize?: Property.FontSize<TLength>;
  /**
   * The **`font-size-adjust`** CSS property sets the size of lower-case letters relative to the current font size (which defines the size of upper-case letters).
   *
   * **Syntax**: `none | <number>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |  n/a   |  **1**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-size-adjust
   */
  fontSizeAdjust?: Property.FontSizeAdjust;
  /**
   * The **`font-smooth`** CSS property controls the application of anti-aliasing when fonts are rendered.
   *
   * **Syntax**: `auto | never | always | <absolute-size> | <length>`
   *
   * **Initial value**: `auto`
   *
   * |              Chrome              |              Firefox               |              Safari              |               Edge                | IE  |
   * | :------------------------------: | :--------------------------------: | :------------------------------: | :-------------------------------: | :-: |
   * | **5** _(-webkit-font-smoothing)_ | **25** _(-moz-osx-font-smoothing)_ | **4** _(-webkit-font-smoothing)_ | **79** _(-webkit-font-smoothing)_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-smooth
   */
  fontSmooth?: Property.FontSmooth<TLength>;
  /**
   * The **`font-stretch`** CSS property selects a normal, condensed, or expanded face from a font.
   *
   * **Syntax**: `<font-stretch-absolute>`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **60** |  **9**  | **11** | **12** | **9** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-stretch
   */
  fontStretch?: Property.FontStretch;
  /**
   * The **`font-style`** CSS property sets whether a font should be styled with a normal, italic, or oblique face from its `font-family`.
   *
   * **Syntax**: `normal | italic | oblique <angle>?`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-style
   */
  fontStyle?: Property.FontStyle;
  /**
   * The **`font-synthesis`** CSS property controls which missing typefaces, bold or italic, may be synthesized by the browser.
   *
   * **Syntax**: `none | [ weight || style ]`
   *
   * **Initial value**: `weight style`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **34**  | **9**  |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-synthesis
   */
  fontSynthesis?: Property.FontSynthesis;
  /**
   * The **`font-variant`** CSS shorthand property allows you to set all the font variants for a font.
   *
   * **Syntax**: `normal | none | [ <common-lig-values> || <discretionary-lig-values> || <historical-lig-values> || <contextual-alt-values> || stylistic( <feature-value-name> ) || historical-forms || styleset( <feature-value-name># ) || character-variant( <feature-value-name># ) || swash( <feature-value-name> ) || ornaments( <feature-value-name> ) || annotation( <feature-value-name> ) || [ small-caps | all-small-caps | petite-caps | all-petite-caps | unicase | titling-caps ] || <numeric-figure-values> || <numeric-spacing-values> || <numeric-fraction-values> || ordinal || slashed-zero || <east-asian-variant-values> || <east-asian-width-values> || ruby ]`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-variant
   */
  fontVariant?: Property.FontVariant;
  /**
   * The **`font-variant-caps`** CSS property controls the use of alternate glyphs for capital letters.
   *
   * **Syntax**: `normal | small-caps | all-small-caps | petite-caps | all-petite-caps | unicase | titling-caps`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari  |  Edge  | IE  |
   * | :----: | :-----: | :-----: | :----: | :-: |
   * | **52** | **34**  | **9.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-variant-caps
   */
  fontVariantCaps?: Property.FontVariantCaps;
  /**
   * The **`font-variant-east-asian`** CSS property controls the use of alternate glyphs for East Asian scripts, like Japanese and Chinese.
   *
   * **Syntax**: `normal | [ <east-asian-variant-values> || <east-asian-width-values> || ruby ]`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **63** | **34**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-variant-east-asian
   */
  fontVariantEastAsian?: Property.FontVariantEastAsian;
  /**
   * The **`font-variant-ligatures`** CSS property controls which ligatures and contextual forms are used in textual content of the elements it applies to. This leads to more harmonized forms in the resulting text.
   *
   * **Syntax**: `normal | none | [ <common-lig-values> || <discretionary-lig-values> || <historical-lig-values> || <contextual-alt-values> ]`
   *
   * **Initial value**: `normal`
   *
   * |  Chrome  | Firefox | Safari  |  Edge  | IE  |
   * | :------: | :-----: | :-----: | :----: | :-: |
   * |  **34**  | **34**  | **9.1** | **79** | No  |
   * | 31 _-x-_ |         | 7 _-x-_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-variant-ligatures
   */
  fontVariantLigatures?: Property.FontVariantLigatures;
  /**
   * The **`font-variant-numeric`** CSS property controls the usage of alternate glyphs for numbers, fractions, and ordinal markers.
   *
   * **Syntax**: `normal | [ <numeric-figure-values> || <numeric-spacing-values> || <numeric-fraction-values> || ordinal || slashed-zero ]`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari  |  Edge  | IE  |
   * | :----: | :-----: | :-----: | :----: | :-: |
   * | **52** | **34**  | **9.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-variant-numeric
   */
  fontVariantNumeric?: Property.FontVariantNumeric;
  /**
   * The **`font-variant-position`** CSS property controls the use of alternate, smaller glyphs that are positioned as superscript or subscript.
   *
   * **Syntax**: `normal | sub | super`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **34**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-variant-position
   */
  fontVariantPosition?: Property.FontVariantPosition;
  /**
   * The **`font-variation-settings`** CSS property provides low-level control over variable font characteristics, by specifying the four letter axis names of the characteristics you want to vary, along with their values.
   *
   * **Syntax**: `normal | [ <string> <number> ]#`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **62** | **62**  | **11** | **17** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-variation-settings
   */
  fontVariationSettings?: Property.FontVariationSettings;
  /**
   * The **`font-weight`** CSS property sets the weight (or boldness) of the font. The weights available depend on the `font-family` that is currently set.
   *
   * **Syntax**: `<font-weight-absolute> | bolder | lighter`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **2**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font-weight
   */
  fontWeight?: Property.FontWeight;
  /**
   * The **`forced-color-adjust`** CSS property allows authors to opt certain elements out of forced colors mode. This then restores the control of those values to CSS.
   *
   * **Syntax**: `auto | none`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |              Edge               |                 IE                  |
   * | :----: | :-----: | :----: | :-----------------------------: | :---------------------------------: |
   * | **89** |   No    |   No   |             **79**              | **10** _(-ms-high-contrast-adjust)_ |
   * |        |         |        | 12 _(-ms-high-contrast-adjust)_ |                                     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/forced-color-adjust
   */
  forcedColorAdjust?: Property.ForcedColorAdjust;
  /**
   * The **`grid-auto-columns`** CSS property specifies the size of an implicitly-created grid column track or pattern of tracks.
   *
   * **Syntax**: `<track-size>+`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  |             IE              |
   * | :----: | :-----: | :------: | :----: | :-------------------------: |
   * | **57** | **70**  | **10.1** | **16** | **10** _(-ms-grid-columns)_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-auto-columns
   */
  gridAutoColumns?: Property.GridAutoColumns<TLength>;
  /**
   * The **`grid-auto-flow`** CSS property controls how the auto-placement algorithm works, specifying exactly how auto-placed items get flowed into the grid.
   *
   * **Syntax**: `[ row | column ] || dense`
   *
   * **Initial value**: `row`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-auto-flow
   */
  gridAutoFlow?: Property.GridAutoFlow;
  /**
   * The **`grid-auto-rows`** CSS property specifies the size of an implicitly-created grid row track or pattern of tracks.
   *
   * **Syntax**: `<track-size>+`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  |            IE            |
   * | :----: | :-----: | :------: | :----: | :----------------------: |
   * | **57** | **70**  | **10.1** | **16** | **10** _(-ms-grid-rows)_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-auto-rows
   */
  gridAutoRows?: Property.GridAutoRows<TLength>;
  /**
   * The **`grid-column-end`** CSS property specifies a grid item’s end position within the grid column by contributing a line, a span, or nothing (automatic) to its grid placement, thereby specifying the block-end edge of its grid area.
   *
   * **Syntax**: `<grid-line>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-column-end
   */
  gridColumnEnd?: Property.GridColumnEnd;
  /**
   * The **`grid-column-start`** CSS property specifies a grid item’s start position within the grid column by contributing a line, a span, or nothing (automatic) to its grid placement. This start position defines the block-start edge of the grid area.
   *
   * **Syntax**: `<grid-line>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-column-start
   */
  gridColumnStart?: Property.GridColumnStart;
  /**
   * The **`grid-row-end`** CSS property specifies a grid item’s end position within the grid row by contributing a line, a span, or nothing (automatic) to its grid placement, thereby specifying the inline-end edge of its grid area.
   *
   * **Syntax**: `<grid-line>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-row-end
   */
  gridRowEnd?: Property.GridRowEnd;
  /**
   * The **`grid-row-start`** CSS property specifies a grid item’s start position within the grid row by contributing a line, a span, or nothing (automatic) to its grid placement, thereby specifying the inline-start edge of its grid area.
   *
   * **Syntax**: `<grid-line>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-row-start
   */
  gridRowStart?: Property.GridRowStart;
  /**
   * The **`grid-template-areas`** CSS property specifies named grid areas, establishing the cells in the grid and assigning them names.
   *
   * **Syntax**: `none | <string>+`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-template-areas
   */
  gridTemplateAreas?: Property.GridTemplateAreas;
  /**
   * The **`grid-template-columns`** CSS property defines the line names and track sizing functions of the grid columns.
   *
   * **Syntax**: `none | <track-list> | <auto-track-list> | subgrid <line-name-list>?`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox |  Safari  |  Edge  |             IE              |
   * | :----: | :-----: | :------: | :----: | :-------------------------: |
   * | **57** | **52**  | **10.1** | **16** | **10** _(-ms-grid-columns)_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-template-columns
   */
  gridTemplateColumns?: Property.GridTemplateColumns<TLength>;
  /**
   * The **`grid-template-rows`** CSS property defines the line names and track sizing functions of the grid rows.
   *
   * **Syntax**: `none | <track-list> | <auto-track-list> | subgrid <line-name-list>?`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox |  Safari  |  Edge  |            IE            |
   * | :----: | :-----: | :------: | :----: | :----------------------: |
   * | **57** | **52**  | **10.1** | **16** | **10** _(-ms-grid-rows)_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-template-rows
   */
  gridTemplateRows?: Property.GridTemplateRows<TLength>;
  /**
   * The **`hanging-punctuation`** CSS property specifies whether a punctuation mark should hang at the start or end of a line of text. Hanging punctuation may be placed outside the line box.
   *
   * **Syntax**: `none | [ first || [ force-end | allow-end ] || last ]`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   |   No    | **10** |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/hanging-punctuation
   */
  hangingPunctuation?: Property.HangingPunctuation;
  /**
   * The **`height`** CSS property specifies the height of an element. By default, the property defines the height of the content area. If `box-sizing` is set to `border-box`, however, it instead determines the height of the border area.
   *
   * **Syntax**: `auto | <length> | <percentage> | min-content | max-content | fit-content | fit-content(<length-percentage>)`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/height
   */
  height?: Property.Height<TLength>;
  /**
   * The **`hyphens`** CSS property specifies how words should be hyphenated when text wraps across multiple lines. It can prevent hyphenation entirely, hyphenate at manually-specified points within the text, or let the browser automatically insert hyphens where appropriate.
   *
   * **Syntax**: `none | manual | auto`
   *
   * **Initial value**: `manual`
   *
   * |  Chrome  | Firefox |    Safari     |     Edge     |      IE      |
   * | :------: | :-----: | :-----------: | :----------: | :----------: |
   * |  **55**  | **43**  | **5.1** _-x-_ | **12** _-x-_ | **10** _-x-_ |
   * | 13 _-x-_ | 6 _-x-_ |               |              |              |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/hyphens
   */
  hyphens?: Property.Hyphens;
  /**
   * The **`image-orientation`** CSS property specifies a layout-independent correction to the orientation of an image. It should _not_ be used for any other orientation adjustments; instead, the `transform` property should be used with the `rotate` `<transform-function>`.
   *
   * **Syntax**: `from-image | <angle> | [ <angle>? flip ]`
   *
   * **Initial value**: `from-image`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **81** | **26**  | **13.1** | **81** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/image-orientation
   */
  imageOrientation?: Property.ImageOrientation;
  /**
   * The **`image-rendering`** CSS property sets an image scaling algorithm. The property applies to an element itself, to any images set in its other properties, and to its descendants.
   *
   * **Syntax**: `auto | crisp-edges | pixelated`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **13** | **3.6** | **6**  | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/image-rendering
   */
  imageRendering?: Property.ImageRendering;
  /**
   * **Syntax**: `[ from-image || <resolution> ] && snap?`
   *
   * **Initial value**: `1dppx`
   */
  imageResolution?: Property.ImageResolution;
  /**
   * The `initial-letter` CSS property sets styling for dropped, raised, and sunken initial letters.
   *
   * **Syntax**: `normal | [ <number> <integer>? ]`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   |   No    | **9**  |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/initial-letter
   */
  initialLetter?: Property.InitialLetter;
  /**
   * The **`inline-size`** CSS property defines the horizontal or vertical size of an element's block, depending on its writing mode. It corresponds to either the `width` or the `height` property, depending on the value of `writing-mode`.
   *
   * **Syntax**: `<'width'>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/inline-size
   */
  inlineSize?: Property.InlineSize<TLength>;
  /**
   * The **`inset`** CSS property is a shorthand that corresponds to the `top`, `right`, `bottom`, and/or `left` properties. It has the same multi-value syntax of the `margin` shorthand.
   *
   * **Syntax**: `<'top'>{1,4}`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/inset
   */
  inset?: Property.Inset<TLength>;
  /**
   * The **`inset-inline`** CSS property defines the logical start and end offsets of an element in the inline direction, which maps to physical offsets depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top` and `bottom`, or `right` and `left` properties depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>{1,2}`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **63**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/inset-block
   */
  insetBlock?: Property.InsetBlock<TLength>;
  /**
   * The **`inset-block-end`** CSS property defines the logical block end offset of an element, which maps to a physical inset depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top`, `right`, `bottom`, or `left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **63**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/inset-block-end
   */
  insetBlockEnd?: Property.InsetBlockEnd<TLength>;
  /**
   * The **`inset-block-start`** CSS property defines the logical block start offset of an element, which maps to a physical inset depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top`, `right`, `bottom`, or `left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **63**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/inset-block-start
   */
  insetBlockStart?: Property.InsetBlockStart<TLength>;
  /**
   * The **`inset-inline`** CSS property defines the logical start and end offsets of an element in the inline direction, which maps to physical offsets depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top` and `bottom`, or `right` and `left` properties depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>{1,2}`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **63**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/inset-inline
   */
  insetInline?: Property.InsetInline<TLength>;
  /**
   * The **`inset-inline-end`** CSS property defines the logical inline end inset of an element, which maps to a physical offset depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top`, `right`, `bottom`, or `left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **63**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/inset-inline-end
   */
  insetInlineEnd?: Property.InsetInlineEnd<TLength>;
  /**
   * The **`inset-inline-start`** CSS property defines the logical inline start inset of an element, which maps to a physical offset depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top`, `right`, `bottom`, or `left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **63**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/inset-inline-start
   */
  insetInlineStart?: Property.InsetInlineStart<TLength>;
  /**
   * The **`isolation`** CSS property determines whether an element must create a new stacking context.
   *
   * **Syntax**: `auto | isolate`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **41** | **36**  | **8**  | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/isolation
   */
  isolation?: Property.Isolation;
  /**
   * The CSS **`justify-content`** property defines how the browser distributes space between and around content items along the main-axis of a flex container, and the inline axis of a grid container.
   *
   * **Syntax**: `normal | <content-distribution> | <overflow-position>? [ <content-position> | left | right ]`
   *
   * **Initial value**: `normal`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :------: | :-----: | :-------: | :----: | :----: |
   * |  **52**  | **20**  |   **9**   | **12** | **11** |
   * | 21 _-x-_ |         | 6.1 _-x-_ |        |        |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/justify-content
   */
  justifyContent?: Property.JustifyContent;
  /**
   * The CSS **`justify-items`** property defines the default `justify-self` for all items of the box, giving them all a default way of justifying each box along the appropriate axis.
   *
   * **Syntax**: `normal | stretch | <baseline-position> | <overflow-position>? [ <self-position> | left | right ] | legacy | legacy && [ left | right | center ]`
   *
   * **Initial value**: `legacy`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **52** | **20**  | **9**  | **12** | **11** |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **45**  | **10.1** | **16** | No  |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/justify-items
   */
  justifyItems?: Property.JustifyItems;
  /**
   * The CSS **`justify-self`** property sets the way a box is justified inside its alignment container along the appropriate axis.
   *
   * **Syntax**: `auto | normal | stretch | <baseline-position> | <overflow-position>? [ <self-position> | left | right ]`
   *
   * **Initial value**: `auto`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **45**  | **10.1** | **16** | No  |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox |  Safari  |  Edge  |      IE      |
   * | :----: | :-----: | :------: | :----: | :----------: |
   * | **57** | **45**  | **10.1** | **16** | **10** _-x-_ |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/justify-self
   */
  justifySelf?: Property.JustifySelf;
  /**
   * The **`justify-tracks`** CSS property sets the alignment in the masonry axis for grid containers that have masonry in their inline axis.
   *
   * **Syntax**: `[ normal | <content-distribution> | <overflow-position>? [ <content-position> | left | right ] ]#`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   |   n/a   |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/justify-tracks
   */
  justifyTracks?: Property.JustifyTracks;
  /**
   * The **`left`** CSS property participates in specifying the horizontal position of a positioned element. It has no effect on non-positioned elements.
   *
   * **Syntax**: `<length> | <percentage> | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/left
   */
  left?: Property.Left<TLength>;
  /**
   * The **`letter-spacing`** CSS property sets the horizontal spacing behavior between text characters. This value is added to the natural spacing between characters while rendering the text. Positive values of `letter-spacing` causes characters to spread farther apart, while negative values of `letter-spacing` bring characters closer together.
   *
   * **Syntax**: `normal | <length>`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/letter-spacing
   */
  letterSpacing?: Property.LetterSpacing<TLength>;
  /**
   * The **`line-break`** CSS property sets how to break lines of Chinese, Japanese, or Korean (CJK) text when working with punctuation and symbols.
   *
   * **Syntax**: `auto | loose | normal | strict | anywhere`
   *
   * **Initial value**: `auto`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE    |
   * | :-----: | :-----: | :-----: | :----: | :-----: |
   * | **58**  | **69**  | **11**  | **14** | **5.5** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |         |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/line-break
   */
  lineBreak?: Property.LineBreak;
  /**
   * The **`line-height`** CSS property sets the height of a line box. It's commonly used to set the distance between lines of text. On block-level elements, it specifies the minimum height of line boxes within the element. On non-replaced inline elements, it specifies the height that is used to calculate line box height.
   *
   * **Syntax**: `normal | <number> | <length> | <percentage>`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/line-height
   */
  lineHeight?: Property.LineHeight<TLength>;
  /**
   * The **`line-height-step`** CSS property sets the step unit for line box heights. When the property is set, line box heights are rounded up to the closest multiple of the unit.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |  n/a   |   No    |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/line-height-step
   */
  lineHeightStep?: Property.LineHeightStep<TLength>;
  /**
   * The **`list-style-image`** CSS property sets an image to be used as the list item marker.
   *
   * **Syntax**: `<image> | none`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/list-style-image
   */
  listStyleImage?: Property.ListStyleImage;
  /**
   * The **`list-style-position`** CSS property sets the position of the `::marker` relative to a list item.
   *
   * **Syntax**: `inside | outside`
   *
   * **Initial value**: `outside`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/list-style-position
   */
  listStylePosition?: Property.ListStylePosition;
  /**
   * The **`list-style-type`** CSS property sets the marker (such as a disc, character, or custom counter style) of a list item element.
   *
   * **Syntax**: `<counter-style> | <string> | none`
   *
   * **Initial value**: `disc`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/list-style-type
   */
  listStyleType?: Property.ListStyleType;
  /**
   * The **`margin-block`** CSS shorthand property defines the logical block start and end margins of an element, which maps to physical margins depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'margin-left'>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-block
   */
  marginBlock?: Property.MarginBlock<TLength>;
  /**
   * The **`margin-block-end`** CSS property defines the logical block end margin of an element, which maps to a physical margin depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'margin-left'>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-block-end
   */
  marginBlockEnd?: Property.MarginBlockEnd<TLength>;
  /**
   * The **`margin-block-start`** CSS property defines the logical block start margin of an element, which maps to a physical margin depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'margin-left'>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-block-start
   */
  marginBlockStart?: Property.MarginBlockStart<TLength>;
  /**
   * The **`margin-bottom`** CSS property sets the margin area on the bottom of an element. A positive value places it farther from its neighbors, while a negative value places it closer.
   *
   * **Syntax**: `<length> | <percentage> | auto`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-bottom
   */
  marginBottom?: Property.MarginBottom<TLength>;
  /**
   * The **`margin-inline`** CSS shorthand property is a shorthand property that defines both the logical inline start and end margins of an element, which maps to physical margins depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'margin-left'>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-inline
   */
  marginInline?: Property.MarginInline<TLength>;
  /**
   * The **`margin-inline-end`** CSS property defines the logical inline end margin of an element, which maps to a physical margin depending on the element's writing mode, directionality, and text orientation. In other words, it corresponds to the `margin-top`, `margin-right`, `margin-bottom` or `margin-left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'margin-left'>`
   *
   * **Initial value**: `0`
   *
   * |          Chrome          |        Firefox        |          Safari          |  Edge  | IE  |
   * | :----------------------: | :-------------------: | :----------------------: | :----: | :-: |
   * |          **69**          |        **41**         |         **12.1**         | **79** | No  |
   * | 2 _(-webkit-margin-end)_ | 3 _(-moz-margin-end)_ | 3 _(-webkit-margin-end)_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-inline-end
   */
  marginInlineEnd?: Property.MarginInlineEnd<TLength>;
  /**
   * The **`margin-inline-start`** CSS property defines the logical inline start margin of an element, which maps to a physical margin depending on the element's writing mode, directionality, and text orientation. It corresponds to the `margin-top`, `margin-right`, `margin-bottom`, or `margin-left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'margin-left'>`
   *
   * **Initial value**: `0`
   *
   * |           Chrome           |         Firefox         |           Safari           |  Edge  | IE  |
   * | :------------------------: | :---------------------: | :------------------------: | :----: | :-: |
   * |           **69**           |         **41**          |          **12.1**          | **79** | No  |
   * | 2 _(-webkit-margin-start)_ | 3 _(-moz-margin-start)_ | 3 _(-webkit-margin-start)_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-inline-start
   */
  marginInlineStart?: Property.MarginInlineStart<TLength>;
  /**
   * The **`margin-left`** CSS property sets the margin area on the left side of an element. A positive value places it farther from its neighbors, while a negative value places it closer.
   *
   * **Syntax**: `<length> | <percentage> | auto`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-left
   */
  marginLeft?: Property.MarginLeft<TLength>;
  /**
   * The **`margin-right`** CSS property sets the margin area on the right side of an element. A positive value places it farther from its neighbors, while a negative value places it closer.
   *
   * **Syntax**: `<length> | <percentage> | auto`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-right
   */
  marginRight?: Property.MarginRight<TLength>;
  /**
   * The **`margin-top`** CSS property sets the margin area on the top of an element. A positive value places it farther from its neighbors, while a negative value places it closer.
   *
   * **Syntax**: `<length> | <percentage> | auto`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin-top
   */
  marginTop?: Property.MarginTop<TLength>;
  /**
   * The **`mask-border-mode`** CSS property specifies the blending mode used in a mask border.
   *
   * **Syntax**: `luminance | alpha`
   *
   * **Initial value**: `alpha`
   */
  maskBorderMode?: Property.MaskBorderMode;
  /**
   * The **`mask-border-outset`** CSS property specifies the distance by which an element's mask border is set out from its border box.
   *
   * **Syntax**: `[ <length> | <number> ]{1,4}`
   *
   * **Initial value**: `0`
   *
   * |                 Chrome                  | Firefox |                  Safari                   |                   Edge                   | IE  |
   * | :-------------------------------------: | :-----: | :---------------------------------------: | :--------------------------------------: | :-: |
   * | **1** _(-webkit-mask-box-image-outset)_ |   No    | **3.1** _(-webkit-mask-box-image-outset)_ | **79** _(-webkit-mask-box-image-outset)_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-border-outset
   */
  maskBorderOutset?: Property.MaskBorderOutset<TLength>;
  /**
   * The **`mask-border-repeat`** CSS property sets how the edge regions of a source image are adjusted to fit the dimensions of an element's mask border.
   *
   * **Syntax**: `[ stretch | repeat | round | space ]{1,2}`
   *
   * **Initial value**: `stretch`
   *
   * |                 Chrome                  | Firefox |                  Safari                   |                   Edge                   | IE  |
   * | :-------------------------------------: | :-----: | :---------------------------------------: | :--------------------------------------: | :-: |
   * | **1** _(-webkit-mask-box-image-repeat)_ |   No    | **3.1** _(-webkit-mask-box-image-repeat)_ | **79** _(-webkit-mask-box-image-repeat)_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-border-repeat
   */
  maskBorderRepeat?: Property.MaskBorderRepeat;
  /**
   * The **`mask-border-slice`** CSS property divides the image set by `mask-border-source` into regions. These regions are used to form the components of an element's mask border.
   *
   * **Syntax**: `<number-percentage>{1,4} fill?`
   *
   * **Initial value**: `0`
   *
   * |                 Chrome                 | Firefox |                  Safari                  |                  Edge                   | IE  |
   * | :------------------------------------: | :-----: | :--------------------------------------: | :-------------------------------------: | :-: |
   * | **1** _(-webkit-mask-box-image-slice)_ |   No    | **3.1** _(-webkit-mask-box-image-slice)_ | **79** _(-webkit-mask-box-image-slice)_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-border-slice
   */
  maskBorderSlice?: Property.MaskBorderSlice;
  /**
   * The **`mask-border-source`** CSS property sets the source image used to create an element's mask border.
   *
   * **Syntax**: `none | <image>`
   *
   * **Initial value**: `none`
   *
   * |                 Chrome                  | Firefox |                  Safari                   |                   Edge                   | IE  |
   * | :-------------------------------------: | :-----: | :---------------------------------------: | :--------------------------------------: | :-: |
   * | **1** _(-webkit-mask-box-image-source)_ |   No    | **3.1** _(-webkit-mask-box-image-source)_ | **79** _(-webkit-mask-box-image-source)_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-border-source
   */
  maskBorderSource?: Property.MaskBorderSource;
  /**
   * The **`mask-border-width`** CSS property sets the width of an element's mask border.
   *
   * **Syntax**: `[ <length-percentage> | <number> | auto ]{1,4}`
   *
   * **Initial value**: `auto`
   *
   * |                 Chrome                 | Firefox |                  Safari                  |                  Edge                   | IE  |
   * | :------------------------------------: | :-----: | :--------------------------------------: | :-------------------------------------: | :-: |
   * | **1** _(-webkit-mask-box-image-width)_ |   No    | **3.1** _(-webkit-mask-box-image-width)_ | **79** _(-webkit-mask-box-image-width)_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-border-width
   */
  maskBorderWidth?: Property.MaskBorderWidth<TLength>;
  /**
   * The **`mask-clip`** CSS property determines the area which is affected by a mask. The painted content of an element must be restricted to this area.
   *
   * **Syntax**: `[ <geometry-box> | no-clip ]#`
   *
   * **Initial value**: `border-box`
   *
   * |   Chrome    | Firefox |   Safari    |     Edge     | IE  |
   * | :---------: | :-----: | :---------: | :----------: | :-: |
   * | **1** _-x-_ | **53**  | **4** _-x-_ | **79** _-x-_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-clip
   */
  maskClip?: Property.MaskClip;
  /**
   * The **`mask-composite`** CSS property represents a compositing operation used on the current mask layer with the mask layers below it.
   *
   * **Syntax**: `<compositing-operator>#`
   *
   * **Initial value**: `add`
   *
   * | Chrome | Firefox | Safari | Edge  | IE  |
   * | :----: | :-----: | :----: | :---: | :-: |
   * |   No   | **53**  |   No   | 18-79 | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-composite
   */
  maskComposite?: Property.MaskComposite;
  /**
   * The **`mask-image`** CSS property sets the image that is used as mask layer for an element.
   *
   * **Syntax**: `<mask-reference>#`
   *
   * **Initial value**: `none`
   *
   * |   Chrome    | Firefox |   Safari    |  Edge  | IE  |
   * | :---------: | :-----: | :---------: | :----: | :-: |
   * | **1** _-x-_ | **53**  | **4** _-x-_ | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-image
   */
  maskImage?: Property.MaskImage;
  /**
   * The **`mask-mode`** CSS property sets whether the mask reference defined by `mask-image` is treated as a luminance or alpha mask.
   *
   * **Syntax**: `<masking-mode>#`
   *
   * **Initial value**: `match-source`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **53**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-mode
   */
  maskMode?: Property.MaskMode;
  /**
   * The **`mask-origin`** CSS property sets the origin of a mask.
   *
   * **Syntax**: `<geometry-box>#`
   *
   * **Initial value**: `border-box`
   *
   * |   Chrome    | Firefox |   Safari    |     Edge     | IE  |
   * | :---------: | :-----: | :---------: | :----------: | :-: |
   * | **1** _-x-_ | **53**  | **4** _-x-_ | **79** _-x-_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-origin
   */
  maskOrigin?: Property.MaskOrigin;
  /**
   * The **`mask-position`** CSS property sets the initial position, relative to the mask position layer set by `mask-origin`, for each defined mask image.
   *
   * **Syntax**: `<position>#`
   *
   * **Initial value**: `center`
   *
   * |   Chrome    | Firefox |    Safari     |  Edge  | IE  |
   * | :---------: | :-----: | :-----------: | :----: | :-: |
   * | **1** _-x-_ | **53**  | **3.2** _-x-_ | **18** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-position
   */
  maskPosition?: Property.MaskPosition<TLength>;
  /**
   * The **`mask-repeat`** CSS property sets how mask images are repeated. A mask image can be repeated along the horizontal axis, the vertical axis, both axes, or not repeated at all.
   *
   * **Syntax**: `<repeat-style>#`
   *
   * **Initial value**: `no-repeat`
   *
   * |   Chrome    | Firefox |    Safari     |  Edge  | IE  |
   * | :---------: | :-----: | :-----------: | :----: | :-: |
   * | **1** _-x-_ | **53**  | **3.2** _-x-_ | **18** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-repeat
   */
  maskRepeat?: Property.MaskRepeat;
  /**
   * The **`mask-size`** CSS property specifies the sizes of the mask images. The size of the image can be fully or partially constrained in order to preserve its intrinsic ratio.
   *
   * **Syntax**: `<bg-size>#`
   *
   * **Initial value**: `auto`
   *
   * |   Chrome    | Firefox |   Safari    |  Edge  | IE  |
   * | :---------: | :-----: | :---------: | :----: | :-: |
   * | **4** _-x-_ | **53**  | **4** _-x-_ | **18** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-size
   */
  maskSize?: Property.MaskSize<TLength>;
  /**
   * The **`mask-type`** CSS property sets whether an SVG `<mask>` element is used as a _luminance_ or an _alpha_ mask. It applies to the `<mask>` element itself.
   *
   * **Syntax**: `luminance | alpha`
   *
   * **Initial value**: `luminance`
   *
   * | Chrome | Firefox | Safari  |  Edge  | IE  |
   * | :----: | :-----: | :-----: | :----: | :-: |
   * | **24** | **35**  | **6.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-type
   */
  maskType?: Property.MaskType;
  /**
   * The `math-style` property indicates whether MathML equations should render with normal or compact height.
   *
   * **Syntax**: `normal | compact`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |  n/a   |   n/a   |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/math-style
   */
  mathStyle?: Property.MathStyle;
  /**
   * The `**max-block-size**` CSS property specifies the maximum size of an element in the direction opposite that of the writing direction as specified by `writing-mode`. That is, if the writing direction is horizontal, then `max-block-size` is equivalent to `max-height`; if the writing direction is vertical, `max-block-size` is the same as `max-width`.
   *
   * **Syntax**: `<'max-width'>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/max-block-size
   */
  maxBlockSize?: Property.MaxBlockSize<TLength>;
  /**
   * The **`max-height`** CSS property sets the maximum height of an element. It prevents the used value of the `height` property from becoming larger than the value specified for `max-height`.
   *
   * **Syntax**: `none | <length-percentage> | min-content | max-content | fit-content | fit-content(<length-percentage>)`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **18** |  **1**  | **1.3** | **12** | **7** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/max-height
   */
  maxHeight?: Property.MaxHeight<TLength>;
  /**
   * The **`max-inline-size`** CSS property defines the horizontal or vertical maximum size of an element's block, depending on its writing mode. It corresponds to either the `max-width` or the `max-height` property, depending on the value of `writing-mode`.
   *
   * **Syntax**: `<'max-width'>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |   Safari   |  Edge  | IE  |
   * | :----: | :-----: | :--------: | :----: | :-: |
   * | **57** | **41**  |  **12.1**  | **79** | No  |
   * |        |         | 10.1 _-x-_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/max-inline-size
   */
  maxInlineSize?: Property.MaxInlineSize<TLength>;
  /**
   * **Syntax**: `none | <integer>`
   *
   * **Initial value**: `none`
   */
  maxLines?: Property.MaxLines;
  /**
   * The **`max-width`** CSS property sets the maximum width of an element. It prevents the used value of the `width` property from becoming larger than the value specified by `max-width`.
   *
   * **Syntax**: `none | <length-percentage> | min-content | max-content | fit-content | fit-content(<length-percentage>)`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **7** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/max-width
   */
  maxWidth?: Property.MaxWidth<TLength>;
  /**
   * The **`min-block-size`** CSS property defines the minimum horizontal or vertical size of an element's block, depending on its writing mode. It corresponds to either the `min-width` or the `min-height` property, depending on the value of `writing-mode`.
   *
   * **Syntax**: `<'min-width'>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/min-block-size
   */
  minBlockSize?: Property.MinBlockSize<TLength>;
  /**
   * The **`min-height`** CSS property sets the minimum height of an element. It prevents the used value of the `height` property from becoming smaller than the value specified for `min-height`.
   *
   * **Syntax**: `auto | <length> | <percentage> | min-content | max-content | fit-content | fit-content(<length-percentage>)`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  |  **3**  | **1.3** | **12** | **7** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/min-height
   */
  minHeight?: Property.MinHeight<TLength>;
  /**
   * The **`min-inline-size`** CSS property defines the horizontal or vertical minimal size of an element's block, depending on its writing mode. It corresponds to either the `min-width` or the `min-height` property, depending on the value of `writing-mode`.
   *
   * **Syntax**: `<'min-width'>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/min-inline-size
   */
  minInlineSize?: Property.MinInlineSize<TLength>;
  /**
   * The **`min-width`** CSS property sets the minimum width of an element. It prevents the used value of the `width` property from becoming smaller than the value specified for `min-width`.
   *
   * **Syntax**: `auto | <length> | <percentage> | min-content | max-content | fit-content | fit-content(<length-percentage>)`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **7** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/min-width
   */
  minWidth?: Property.MinWidth<TLength>;
  /**
   * The **`mix-blend-mode`** CSS property sets how an element's content should blend with the content of the element's parent and the element's background.
   *
   * **Syntax**: `<blend-mode>`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **41** | **32**  | **8**  | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mix-blend-mode
   */
  mixBlendMode?: Property.MixBlendMode;
  /**
   * The **`offset-distance`** CSS property specifies a position along an `offset-path` for an element to be placed.
   *
   * **Syntax**: `<length-percentage>`
   *
   * **Initial value**: `0`
   *
   * |         Chrome         | Firefox | Safari |  Edge  | IE  |
   * | :--------------------: | :-----: | :----: | :----: | :-: |
   * |         **55**         | **72**  |   No   | **79** | No  |
   * | 46 _(motion-distance)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset-distance
   */
  motionDistance?: Property.OffsetDistance<TLength>;
  /**
   * The **`offset-path`** CSS property specifies a motion path for an element to follow and defines the element's positioning within the parent container or SVG coordinate system.
   *
   * **Syntax**: `none | ray( [ <angle> && <size> && contain? ] ) | <path()> | <url> | [ <basic-shape> || <geometry-box> ]`
   *
   * **Initial value**: `none`
   *
   * |       Chrome       | Firefox | Safari |  Edge  | IE  |
   * | :----------------: | :-----: | :----: | :----: | :-: |
   * |       **55**       | **72**  |   No   | **79** | No  |
   * | 46 _(motion-path)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset-path
   */
  motionPath?: Property.OffsetPath;
  /**
   * The **`offset-rotate`** CSS property defines the orientation/direction of the element as it is positioned along the `offset-path`.
   *
   * **Syntax**: `[ auto | reverse ] || <angle>`
   *
   * **Initial value**: `auto`
   *
   * |         Chrome         | Firefox | Safari |  Edge  | IE  |
   * | :--------------------: | :-----: | :----: | :----: | :-: |
   * |         **56**         | **72**  |   No   | **79** | No  |
   * | 46 _(motion-rotation)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset-rotate
   */
  motionRotation?: Property.OffsetRotate;
  /**
   * The **`object-fit`** CSS property sets how the content of a replaced element, such as an `<img>` or `<video>`, should be resized to fit its container.
   *
   * **Syntax**: `fill | contain | cover | none | scale-down`
   *
   * **Initial value**: `fill`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **32** | **36**  | **10** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/object-fit
   */
  objectFit?: Property.ObjectFit;
  /**
   * The **`object-position`** CSS property specifies the alignment of the selected replaced element's contents within the element's box. Areas of the box which aren't covered by the replaced element's object will show the element's background.
   *
   * **Syntax**: `<position>`
   *
   * **Initial value**: `50% 50%`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **32** | **36**  | **10** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/object-position
   */
  objectPosition?: Property.ObjectPosition<TLength>;
  /**
   * **Syntax**: `auto | <position>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **79** | **72**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset-anchor
   */
  offsetAnchor?: Property.OffsetAnchor<TLength>;
  /**
   * The **`offset-distance`** CSS property specifies a position along an `offset-path` for an element to be placed.
   *
   * **Syntax**: `<length-percentage>`
   *
   * **Initial value**: `0`
   *
   * |         Chrome         | Firefox | Safari |  Edge  | IE  |
   * | :--------------------: | :-----: | :----: | :----: | :-: |
   * |         **55**         | **72**  |   No   | **79** | No  |
   * | 46 _(motion-distance)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset-distance
   */
  offsetDistance?: Property.OffsetDistance<TLength>;
  /**
   * The **`offset-path`** CSS property specifies a motion path for an element to follow and defines the element's positioning within the parent container or SVG coordinate system.
   *
   * **Syntax**: `none | ray( [ <angle> && <size> && contain? ] ) | <path()> | <url> | [ <basic-shape> || <geometry-box> ]`
   *
   * **Initial value**: `none`
   *
   * |       Chrome       | Firefox | Safari |  Edge  | IE  |
   * | :----------------: | :-----: | :----: | :----: | :-: |
   * |       **55**       | **72**  |   No   | **79** | No  |
   * | 46 _(motion-path)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset-path
   */
  offsetPath?: Property.OffsetPath;
  /**
   * The **`offset-rotate`** CSS property defines the orientation/direction of the element as it is positioned along the `offset-path`.
   *
   * **Syntax**: `[ auto | reverse ] || <angle>`
   *
   * **Initial value**: `auto`
   *
   * |         Chrome         | Firefox | Safari |  Edge  | IE  |
   * | :--------------------: | :-----: | :----: | :----: | :-: |
   * |         **56**         | **72**  |   No   | **79** | No  |
   * | 46 _(motion-rotation)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset-rotate
   */
  offsetRotate?: Property.OffsetRotate;
  /**
   * The **`offset-rotate`** CSS property defines the orientation/direction of the element as it is positioned along the `offset-path`.
   *
   * **Syntax**: `[ auto | reverse ] || <angle>`
   *
   * **Initial value**: `auto`
   *
   * |         Chrome         | Firefox | Safari |  Edge  | IE  |
   * | :--------------------: | :-----: | :----: | :----: | :-: |
   * |         **56**         | **72**  |   No   | **79** | No  |
   * | 46 _(motion-rotation)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset-rotate
   */
  offsetRotation?: Property.OffsetRotate;
  /**
   * The **`opacity`** CSS property sets the opacity of an element. Opacity is the degree to which content behind an element is hidden, and is the opposite of transparency.
   *
   * **Syntax**: `<alpha-value>`
   *
   * **Initial value**: `1.0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **2**  | **12** | **9** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/opacity
   */
  opacity?: Property.Opacity;
  /**
   * The **`order`** CSS property sets the order to lay out an item in a flex or grid container. Items in a container are sorted by ascending `order` value and then by their source code order.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `0`
   *
   * |  Chrome  | Firefox | Safari  |  Edge  |    IE    |
   * | :------: | :-----: | :-----: | :----: | :------: |
   * |  **29**  | **20**  |  **9**  | **12** |  **11**  |
   * | 21 _-x-_ |         | 7 _-x-_ |        | 10 _-x-_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/order
   */
  order?: Property.Order;
  /**
   * The **`orphans`** CSS property sets the minimum number of lines in a block container that must be shown at the _bottom_ of a page, region, or column.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `2`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **25** |   No    | **1.3** | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/orphans
   */
  orphans?: Property.Orphans;
  /**
   * The **`outline-color`** CSS property sets the color of an element's outline.
   *
   * **Syntax**: `<color> | invert`
   *
   * **Initial value**: `invert`, for browsers supporting it, `currentColor` for the other
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  | **1.5** | **1.2** | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/outline-color
   */
  outlineColor?: Property.OutlineColor;
  /**
   * The **`outline-offset`** CSS property sets the amount of space between an outline and the edge or border of an element.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari  |  Edge  | IE  |
   * | :----: | :-----: | :-----: | :----: | :-: |
   * | **1**  | **1.5** | **1.2** | **15** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/outline-offset
   */
  outlineOffset?: Property.OutlineOffset<TLength>;
  /**
   * The **`outline-style`** CSS property sets the style of an element's outline. An outline is a line that is drawn around an element, outside the `border`.
   *
   * **Syntax**: `auto | <'border-style'>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  | **1.5** | **1.2** | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/outline-style
   */
  outlineStyle?: Property.OutlineStyle;
  /**
   * The CSS **`outline-width`** property sets the thickness of an element's outline. An outline is a line that is drawn around an element, outside the `border`.
   *
   * **Syntax**: `<line-width>`
   *
   * **Initial value**: `medium`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  | **1.5** | **1.2** | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/outline-width
   */
  outlineWidth?: Property.OutlineWidth<TLength>;
  /**
   * **Syntax**: `auto | none`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **56** | **66**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overflow-anchor
   */
  overflowAnchor?: Property.OverflowAnchor;
  /**
   * **Syntax**: `visible | hidden | clip | scroll | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **69**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overflow-block
   */
  overflowBlock?: Property.OverflowBlock;
  /**
   * The **`overflow-clip-box`** CSS property specifies relative to which box the clipping happens when there is an overflow. It is short hand for the `overflow-clip-box-inline` and `overflow-clip-box-block` properties.
   *
   * **Syntax**: `padding-box | content-box`
   *
   * **Initial value**: `padding-box`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **29**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Mozilla/Gecko/Chrome/CSS/overflow-clip-box
   */
  overflowClipBox?: Property.OverflowClipBox;
  /**
   * **Syntax**: `<visual-box> || <length [0,∞]>`
   *
   * **Initial value**: `0px`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **90** |   No    |   No   | **90** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overflow-clip-margin
   */
  overflowClipMargin?: Property.OverflowClipMargin<TLength>;
  /**
   * **Syntax**: `visible | hidden | clip | scroll | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **69**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overflow-inline
   */
  overflowInline?: Property.OverflowInline;
  /**
   * The `**overflow-wrap**` CSS property applies to inline elements, setting whether the browser should insert line breaks within an otherwise unbreakable string to prevent text from overflowing its line box.
   *
   * **Syntax**: `normal | break-word | anywhere`
   *
   * **Initial value**: `normal`
   *
   * |     Chrome      |      Firefox      |     Safari      |       Edge       |          IE           |
   * | :-------------: | :---------------: | :-------------: | :--------------: | :-------------------: |
   * |     **23**      |      **49**       |     **6.1**     |      **18**      | **5.5** _(word-wrap)_ |
   * | 1 _(word-wrap)_ | 3.5 _(word-wrap)_ | 1 _(word-wrap)_ | 12 _(word-wrap)_ |                       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overflow-wrap
   */
  overflowWrap?: Property.OverflowWrap;
  /**
   * The **`overflow-x`** CSS property sets what shows when content overflows a block-level element's left and right edges. This may be nothing, a scroll bar, or the overflow content.
   *
   * **Syntax**: `visible | hidden | clip | scroll | auto`
   *
   * **Initial value**: `visible`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  | **3.5** | **3**  | **12** | **5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overflow-x
   */
  overflowX?: Property.OverflowX;
  /**
   * The **`overflow-y`** CSS property sets what shows when content overflows a block-level element's top and bottom edges. This may be nothing, a scroll bar, or the overflow content.
   *
   * **Syntax**: `visible | hidden | clip | scroll | auto`
   *
   * **Initial value**: `visible`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  | **3.5** | **3**  | **12** | **5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overflow-y
   */
  overflowY?: Property.OverflowY;
  /**
   * The **`overscroll-behavior-block`** CSS property sets the browser's behavior when the block direction boundary of a scrolling area is reached.
   *
   * **Syntax**: `contain | none | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **77** | **73**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overscroll-behavior-block
   */
  overscrollBehaviorBlock?: Property.OverscrollBehaviorBlock;
  /**
   * The **`overscroll-behavior-inline`** CSS property sets the browser's behavior when the inline direction boundary of a scrolling area is reached.
   *
   * **Syntax**: `contain | none | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **77** | **73**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overscroll-behavior-inline
   */
  overscrollBehaviorInline?: Property.OverscrollBehaviorInline;
  /**
   * The **`overscroll-behavior-x`** CSS property sets the browser's behavior when the horizontal boundary of a scrolling area is reached.
   *
   * **Syntax**: `contain | none | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **63** | **59**  |   No   | **18** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overscroll-behavior-x
   */
  overscrollBehaviorX?: Property.OverscrollBehaviorX;
  /**
   * The **`overscroll-behavior-y`** CSS property sets the browser's behavior when the vertical boundary of a scrolling area is reached.
   *
   * **Syntax**: `contain | none | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **63** | **59**  |   No   | **18** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overscroll-behavior-y
   */
  overscrollBehaviorY?: Property.OverscrollBehaviorY;
  /**
   * The **`padding-block`** CSS shorthand property defines the logical block start and end padding of an element, which maps to physical padding properties depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-block
   */
  paddingBlock?: Property.PaddingBlock<TLength>;
  /**
   * The **`padding-block-end`** CSS property defines the logical block end padding of an element, which maps to a physical padding depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-block-end
   */
  paddingBlockEnd?: Property.PaddingBlockEnd<TLength>;
  /**
   * The **`padding-block-start`** CSS property defines the logical block start padding of an element, which maps to a physical padding depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-block-start
   */
  paddingBlockStart?: Property.PaddingBlockStart<TLength>;
  /**
   * The **`padding-bottom`** CSS property sets the height of the padding area on the bottom of an element.
   *
   * **Syntax**: `<length> | <percentage>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-bottom
   */
  paddingBottom?: Property.PaddingBottom<TLength>;
  /**
   * The **`padding-inline`** CSS shorthand property defines the logical inline start and end padding of an element, which maps to physical padding properties depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-inline
   */
  paddingInline?: Property.PaddingInline<TLength>;
  /**
   * The **`padding-inline-end`** CSS property defines the logical inline end padding of an element, which maps to a physical padding depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>`
   *
   * **Initial value**: `0`
   *
   * |          Chrome           |        Firefox         |          Safari           |  Edge  | IE  |
   * | :-----------------------: | :--------------------: | :-----------------------: | :----: | :-: |
   * |          **69**           |         **41**         |         **12.1**          | **79** | No  |
   * | 2 _(-webkit-padding-end)_ | 3 _(-moz-padding-end)_ | 3 _(-webkit-padding-end)_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-inline-end
   */
  paddingInlineEnd?: Property.PaddingInlineEnd<TLength>;
  /**
   * The **`padding-inline-start`** CSS property defines the logical inline start padding of an element, which maps to a physical padding depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>`
   *
   * **Initial value**: `0`
   *
   * |           Chrome            |         Firefox          |           Safari            |  Edge  | IE  |
   * | :-------------------------: | :----------------------: | :-------------------------: | :----: | :-: |
   * |           **69**            |          **41**          |          **12.1**           | **79** | No  |
   * | 2 _(-webkit-padding-start)_ | 3 _(-moz-padding-start)_ | 3 _(-webkit-padding-start)_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-inline-start
   */
  paddingInlineStart?: Property.PaddingInlineStart<TLength>;
  /**
   * The **`padding-left`** CSS property sets the width of the padding area to the left of an element.
   *
   * **Syntax**: `<length> | <percentage>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-left
   */
  paddingLeft?: Property.PaddingLeft<TLength>;
  /**
   * The **`padding-right`** CSS property sets the width of the padding area on the right of an element.
   *
   * **Syntax**: `<length> | <percentage>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-right
   */
  paddingRight?: Property.PaddingRight<TLength>;
  /**
   * The **`padding-top`** CSS property sets the height of the padding area on the top of an element.
   *
   * **Syntax**: `<length> | <percentage>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding-top
   */
  paddingTop?: Property.PaddingTop<TLength>;
  /**
   * The **`page-break-after`** CSS property adjusts page breaks _after_ the current element.
   *
   * **Syntax**: `auto | always | avoid | left | right | recto | verso`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  |  **1**  | **1.2** | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/page-break-after
   */
  pageBreakAfter?: Property.PageBreakAfter;
  /**
   * The **`page-break-before`** CSS property adjusts page breaks _before_ the current element.
   *
   * **Syntax**: `auto | always | avoid | left | right | recto | verso`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  |  **1**  | **1.2** | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/page-break-before
   */
  pageBreakBefore?: Property.PageBreakBefore;
  /**
   * The **`page-break-inside`** CSS property adjusts page breaks _inside_ the current element.
   *
   * **Syntax**: `auto | avoid`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  | **19**  | **1.3** | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/page-break-inside
   */
  pageBreakInside?: Property.PageBreakInside;
  /**
   * The **`paint-order`** CSS property lets you control the order in which the fill and stroke (and painting markers) of text content and shapes are drawn.
   *
   * **Syntax**: `normal | [ fill || stroke || markers ]`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **35** | **60**  | **8**  | **17** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/paint-order
   */
  paintOrder?: Property.PaintOrder;
  /**
   * The **`perspective`** CSS property determines the distance between the z=0 plane and the user in order to give a 3D-positioned element some perspective.
   *
   * **Syntax**: `none | <length>`
   *
   * **Initial value**: `none`
   *
   * |  Chrome  | Firefox  | Safari  |  Edge  |   IE   |
   * | :------: | :------: | :-----: | :----: | :----: |
   * |  **36**  |  **16**  |  **9**  | **12** | **10** |
   * | 12 _-x-_ | 10 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/perspective
   */
  perspective?: Property.Perspective<TLength>;
  /**
   * The **`perspective-origin`** CSS property determines the position at which the viewer is looking. It is used as the _vanishing point_ by the `perspective` property.
   *
   * **Syntax**: `<position>`
   *
   * **Initial value**: `50% 50%`
   *
   * |  Chrome  | Firefox  | Safari  |  Edge  |   IE   |
   * | :------: | :------: | :-----: | :----: | :----: |
   * |  **36**  |  **16**  |  **9**  | **12** | **10** |
   * | 12 _-x-_ | 10 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/perspective-origin
   */
  perspectiveOrigin?: Property.PerspectiveOrigin<TLength>;
  /**
   * The `**place-content**` CSS shorthand property allows you to align content along both the block and inline directions at once (i.e. the `align-content` and `justify-content` properties) in a relevant layout system such as Grid or Flexbox.
   *
   * **Syntax**: `<'align-content'> <'justify-content'>?`
   *
   * **Initial value**: `normal`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **59** | **45**  | **9**  | **79** | No  |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **59** | **53**  | **11** | **79** | No  |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/place-content
   */
  placeContent?: Property.PlaceContent;
  /**
   * The **`pointer-events`** CSS property sets under what circumstances (if any) a particular graphic element can become the target of pointer events.
   *
   * **Syntax**: `auto | none | visiblePainted | visibleFill | visibleStroke | visible | painted | fill | stroke | all | inherit`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * | **1**  | **1.5** | **4**  | **12** | **11** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/pointer-events
   */
  pointerEvents?: Property.PointerEvents;
  /**
   * The **`position`** CSS property sets how an element is positioned in a document. The `top`, `right`, `bottom`, and `left` properties determine the final location of positioned elements.
   *
   * **Syntax**: `static | relative | absolute | sticky | fixed`
   *
   * **Initial value**: `static`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/position
   */
  position?: Property.Position;
  /**
   * The **`quotes`** CSS property sets how the browser should render quotation marks that are added using the `open-quotes` or `close-quotes` values of the CSS `content` property.
   *
   * **Syntax**: `none | auto | [ <string> <string> ]+`
   *
   * **Initial value**: depends on user agent
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **11** | **1.5** | **9**  | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/quotes
   */
  quotes?: Property.Quotes;
  /**
   * The **`resize`** CSS property sets whether an element is resizable, and if so, in which directions.
   *
   * **Syntax**: `none | both | horizontal | vertical | block | inline`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **1**  |  **4**  | **3**  | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/resize
   */
  resize?: Property.Resize;
  /**
   * The **`right`** CSS property participates in specifying the horizontal position of a positioned element. It has no effect on non-positioned elements.
   *
   * **Syntax**: `<length> | <percentage> | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/right
   */
  right?: Property.Right<TLength>;
  /**
   * The **`rotate`** CSS property allows you to specify rotation transforms individually and independently of the `transform` property. This maps better to typical user interface usage, and saves having to remember the exact order of transform functions to specify in the `transform` property.
   *
   * **Syntax**: `none | <angle> | [ x | y | z | <number>{3} ] && <angle>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **72**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/rotate
   */
  rotate?: Property.Rotate;
  /**
   * The **`row-gap`** CSS property sets the size of the gap (gutter) between an element's grid rows.
   *
   * **Syntax**: `normal | <length-percentage>`
   *
   * **Initial value**: `normal`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **84** | **63**  |   No   | **84** | No  |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * |       Chrome        |       Firefox       |        Safari         |  Edge  | IE  |
   * | :-----------------: | :-----------------: | :-------------------: | :----: | :-: |
   * |       **66**        |       **61**        |       **12.1**        | **16** | No  |
   * | 57 _(grid-row-gap)_ | 52 _(grid-row-gap)_ | 10.1 _(grid-row-gap)_ |        |     |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/row-gap
   */
  rowGap?: Property.RowGap<TLength>;
  /**
   * The `**ruby-align**` CSS property defines the distribution of the different ruby elements over the base.
   *
   * **Syntax**: `start | center | space-between | space-around`
   *
   * **Initial value**: `space-around`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **38**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/ruby-align
   */
  rubyAlign?: Property.RubyAlign;
  /**
   * **Syntax**: `separate | collapse | auto`
   *
   * **Initial value**: `separate`
   */
  rubyMerge?: Property.RubyMerge;
  /**
   * The `**ruby-position**` CSS property defines the position of a ruby element relatives to its base element. It can be position over the element (`over`), under it (`under`), or between the characters, on their right side (`inter-character`).
   *
   * **Syntax**: `[ alternate || [ over | under ] ] | inter-character`
   *
   * **Initial value**: `alternate`
   *
   * | Chrome  | Firefox |    Safari     | Edge  | IE  |
   * | :-----: | :-----: | :-----------: | :---: | :-: |
   * | **84**  | **38**  | **6.1** _-x-_ | 12-79 | No  |
   * | 1 _-x-_ |         |               |       |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/ruby-position
   */
  rubyPosition?: Property.RubyPosition;
  /**
   * The **`scale`** CSS property allows you to specify scale transforms individually and independently of the `transform` property. This maps better to typical user interface usage, and saves having to remember the exact order of transform functions to specify in the `transform` value.
   *
   * **Syntax**: `none | <number>{1,3}`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **72**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scale
   */
  scale?: Property.Scale;
  /**
   * The **`scroll-behavior`** CSS property sets the behavior for a scrolling box when scrolling is triggered by the navigation or CSSOM scrolling APIs.
   *
   * **Syntax**: `auto | smooth`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **61** | **36**  |  n/a   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-behavior
   */
  scrollBehavior?: Property.ScrollBehavior;
  /**
   * The **`scroll-margin`** shorthand property sets all of the scroll margins of an element at once, assigning values much like the `margin` property does for margins of an element.
   *
   * **Syntax**: `<length>{1,4}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |            Safari             |  Edge  | IE  |
   * | :----: | :-----: | :---------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin
   */
  scrollMargin?: Property.ScrollMargin<TLength>;
  /**
   * The `scroll-margin-block` shorthand property sets the scroll margins of an element in the block dimension.
   *
   * **Syntax**: `<length>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-block
   */
  scrollMarginBlock?: Property.ScrollMarginBlock<TLength>;
  /**
   * The `scroll-margin-block-end` property defines the margin of the scroll snap area at the end of the block dimension that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-block-end
   */
  scrollMarginBlockEnd?: Property.ScrollMarginBlockEnd<TLength>;
  /**
   * The `scroll-margin-block-start` property defines the margin of the scroll snap area at the start of the block dimension that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-block-start
   */
  scrollMarginBlockStart?: Property.ScrollMarginBlockStart<TLength>;
  /**
   * The `scroll-margin-bottom` property defines the bottom margin of the scroll snap area that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |                Safari                |  Edge  | IE  |
   * | :----: | :-----: | :----------------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin-bottom)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-bottom
   */
  scrollMarginBottom?: Property.ScrollMarginBottom<TLength>;
  /**
   * The `scroll-margin-inline` shorthand property sets the scroll margins of an element in the inline dimension.
   *
   * **Syntax**: `<length>{1,2}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **68**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-inline
   */
  scrollMarginInline?: Property.ScrollMarginInline<TLength>;
  /**
   * The `scroll-margin-inline-end` property defines the margin of the scroll snap area at the end of the inline dimension that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-inline-end
   */
  scrollMarginInlineEnd?: Property.ScrollMarginInlineEnd<TLength>;
  /**
   * The `scroll-margin-inline-start` property defines the margin of the scroll snap area at the start of the inline dimension that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-inline-start
   */
  scrollMarginInlineStart?: Property.ScrollMarginInlineStart<TLength>;
  /**
   * The `scroll-margin-left` property defines the left margin of the scroll snap area that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |               Safari               |  Edge  | IE  |
   * | :----: | :-----: | :--------------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin-left)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-left
   */
  scrollMarginLeft?: Property.ScrollMarginLeft<TLength>;
  /**
   * The `scroll-margin-right` property defines the right margin of the scroll snap area that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |               Safari                |  Edge  | IE  |
   * | :----: | :-----: | :---------------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin-right)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-right
   */
  scrollMarginRight?: Property.ScrollMarginRight<TLength>;
  /**
   * The `scroll-margin-top` property defines the top margin of the scroll snap area that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |              Safari               |  Edge  | IE  |
   * | :----: | :-----: | :-------------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin-top)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-top
   */
  scrollMarginTop?: Property.ScrollMarginTop<TLength>;
  /**
   * The **`scroll-padding`** shorthand property sets scroll padding on all sides of an element at once, much like the `padding` property does for padding on an element.
   *
   * **Syntax**: `[ auto | <length-percentage> ]{1,4}`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  | **11** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding
   */
  scrollPadding?: Property.ScrollPadding<TLength>;
  /**
   * The `scroll-padding-block` shorthand property sets the scroll padding of an element in the block dimension.
   *
   * **Syntax**: `[ auto | <length-percentage> ]{1,2}`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-block
   */
  scrollPaddingBlock?: Property.ScrollPaddingBlock<TLength>;
  /**
   * The `scroll-padding-block-end` property defines offsets for the end edge in the block dimension of the optimal viewing region of the scrollport: the region used as the target region for placing things in view of the user. This allows the author to exclude regions of the scrollport that are obscured by other content (such as fixed-positioned toolbars or sidebars) or simply to put more breathing room between a targetted element and the edges of the scrollport.
   *
   * **Syntax**: `auto | <length-percentage>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-block-end
   */
  scrollPaddingBlockEnd?: Property.ScrollPaddingBlockEnd<TLength>;
  /**
   * The `scroll-padding-block-start` property defines offsets for the start edge in the block dimension of the optimal viewing region of the scrollport: the region used as the target region for placing things in view of the user. This allows the author to exclude regions of the scrollport that are obscured by other content (such as fixed-positioned toolbars or sidebars) or simply to put more breathing room between a targetted element and the edges of the scrollport.
   *
   * **Syntax**: `auto | <length-percentage>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-block-start
   */
  scrollPaddingBlockStart?: Property.ScrollPaddingBlockStart<TLength>;
  /**
   * The `scroll-padding-bottom` property defines offsets for the bottom of the optimal viewing region of the scrollport: the region used as the target region for placing things in view of the user. This allows the author to exclude regions of the scrollport that are obscured by other content (such as fixed-positioned toolbars or sidebars) or simply to put more breathing room between a targetted element and the edges of the scrollport.
   *
   * **Syntax**: `auto | <length-percentage>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  | **11** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-bottom
   */
  scrollPaddingBottom?: Property.ScrollPaddingBottom<TLength>;
  /**
   * The `scroll-padding-inline` shorthand property sets the scroll padding of an element in the inline dimension.
   *
   * **Syntax**: `[ auto | <length-percentage> ]{1,2}`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-inline
   */
  scrollPaddingInline?: Property.ScrollPaddingInline<TLength>;
  /**
   * The `scroll-padding-inline-end` property defines offsets for the end edge in the inline dimension of the optimal viewing region of the scrollport: the region used as the target region for placing things in view of the user. This allows the author to exclude regions of the scrollport that are obscured by other content (such as fixed-positioned toolbars or sidebars) or simply to put more breathing room between a targetted element and the edges of the scrollport.
   *
   * **Syntax**: `auto | <length-percentage>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-inline-end
   */
  scrollPaddingInlineEnd?: Property.ScrollPaddingInlineEnd<TLength>;
  /**
   * The `scroll-padding-inline-start` property defines offsets for the start edge in the inline dimension of the optimal viewing region of the scrollport: the region used as the target region for placing things in view of the user. This allows the author to exclude regions of the scrollport that are obscured by other content (such as fixed-positioned toolbars or sidebars) or simply to put more breathing room between a targetted element and the edges of the scrollport.
   *
   * **Syntax**: `auto | <length-percentage>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-inline-start
   */
  scrollPaddingInlineStart?: Property.ScrollPaddingInlineStart<TLength>;
  /**
   * The `scroll-padding-left` property defines offsets for the left of the optimal viewing region of the scrollport: the region used as the target region for placing things in view of the user. This allows the author to exclude regions of the scrollport that are obscured by other content (such as fixed-positioned toolbars or sidebars) or simply to put more breathing room between a targetted element and the edges of the scrollport.
   *
   * **Syntax**: `auto | <length-percentage>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  | **11** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-left
   */
  scrollPaddingLeft?: Property.ScrollPaddingLeft<TLength>;
  /**
   * The `scroll-padding-right` property defines offsets for the right of the optimal viewing region of the scrollport: the region used as the target region for placing things in view of the user. This allows the author to exclude regions of the scrollport that are obscured by other content (such as fixed-positioned toolbars or sidebars) or simply to put more breathing room between a targetted element and the edges of the scrollport.
   *
   * **Syntax**: `auto | <length-percentage>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  | **11** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-right
   */
  scrollPaddingRight?: Property.ScrollPaddingRight<TLength>;
  /**
   * The **`scroll-padding-top`** property defines offsets for the top of the optimal viewing region of the scrollport: the region used as the target region for placing things in view of the user. This allows the author to exclude regions of the scrollport that are obscured by other content (such as fixed-positioned toolbars or sidebars) or simply to put more breathing room between a targetted element and the edges of the scrollport.
   *
   * **Syntax**: `auto | <length-percentage>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  | **11** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-padding-top
   */
  scrollPaddingTop?: Property.ScrollPaddingTop<TLength>;
  /**
   * The `scroll-snap-align` property specifies the box’s snap position as an alignment of its snap area (as the alignment subject) within its snap container’s snapport (as the alignment container). The two values specify the snapping alignment in the block axis and inline axis, respectively. If only one value is specified, the second value defaults to the same value.
   *
   * **Syntax**: `[ none | start | end | center ]{1,2}`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **69** | **68**  | **11** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-snap-align
   */
  scrollSnapAlign?: Property.ScrollSnapAlign;
  /**
   * The **`scroll-margin`** shorthand property sets all of the scroll margins of an element at once, assigning values much like the `margin` property does for margins of an element.
   *
   * **Syntax**: `<length>{1,4}`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |            Safari             |  Edge  | IE  |
   * | :----: | :-----: | :---------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin
   */
  scrollSnapMargin?: Property.ScrollMargin<TLength>;
  /**
   * The `scroll-margin-bottom` property defines the bottom margin of the scroll snap area that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |                Safari                |  Edge  | IE  |
   * | :----: | :-----: | :----------------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin-bottom)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-bottom
   */
  scrollSnapMarginBottom?: Property.ScrollMarginBottom<TLength>;
  /**
   * The `scroll-margin-left` property defines the left margin of the scroll snap area that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |               Safari               |  Edge  | IE  |
   * | :----: | :-----: | :--------------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin-left)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-left
   */
  scrollSnapMarginLeft?: Property.ScrollMarginLeft<TLength>;
  /**
   * The `scroll-margin-right` property defines the right margin of the scroll snap area that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |               Safari                |  Edge  | IE  |
   * | :----: | :-----: | :---------------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin-right)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-right
   */
  scrollSnapMarginRight?: Property.ScrollMarginRight<TLength>;
  /**
   * The `scroll-margin-top` property defines the top margin of the scroll snap area that is used for snapping this box to the snapport. The scroll snap area is determined by taking the transformed border box, finding its rectangular bounding box (axis-aligned in the scroll container’s coordinate space), then adding the specified outsets.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |              Safari               |  Edge  | IE  |
   * | :----: | :-----: | :-------------------------------: | :----: | :-: |
   * | **69** | **68**  | **11** _(scroll-snap-margin-top)_ | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-margin-top
   */
  scrollSnapMarginTop?: Property.ScrollMarginTop<TLength>;
  /**
   * The **`scroll-snap-stop`** CSS property defines whether the scroll container is allowed to "pass over" possible snap positions.
   *
   * **Syntax**: `normal | always`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **75** |   No    |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-snap-stop
   */
  scrollSnapStop?: Property.ScrollSnapStop;
  /**
   * The **`scroll-snap-type`** CSS property sets how strictly snap points are enforced on the scroll container in case there is one.
   *
   * **Syntax**: `none | [ x | y | block | inline | both ] [ mandatory | proximity ]?`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari  |     Edge     |      IE      |
   * | :----: | :-----: | :-----: | :----------: | :----------: |
   * | **69** |  39-68  | **11**  | **12** _-x-_ | **10** _-x-_ |
   * |        |         | 9 _-x-_ |              |              |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scroll-snap-type
   */
  scrollSnapType?: Property.ScrollSnapType;
  /**
   * The **`scrollbar-color`** CSS property sets the color of the scrollbar track and thumb.
   *
   * **Syntax**: `auto | dark | light | <color>{2}`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **64**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scrollbar-color
   */
  scrollbarColor?: Property.ScrollbarColor;
  /**
   * The **`scrollbar-gutter`** CSS property allows authors to reserve space for the scrollbar, preventing unwanted layout changes as the content grows while also avoiding unnecessary visuals when scrolling isn't needed.
   *
   * **Syntax**: `auto | [ stable | always ] && both? && force?`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |  n/a   |   No    |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scrollbar-gutter
   */
  scrollbarGutter?: Property.ScrollbarGutter;
  /**
   * The **`scrollbar-width`** property allows the author to set the maximum thickness of an element’s scrollbars when they are shown.
   *
   * **Syntax**: `auto | thin | none`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **64**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/scrollbar-width
   */
  scrollbarWidth?: Property.ScrollbarWidth;
  /**
   * The **`shape-image-threshold`** CSS property sets the alpha channel threshold used to extract the shape using an image as the value for `shape-outside`.
   *
   * **Syntax**: `<alpha-value>`
   *
   * **Initial value**: `0.0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **37** | **62**  | **10.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/shape-image-threshold
   */
  shapeImageThreshold?: Property.ShapeImageThreshold;
  /**
   * The **`shape-margin`** CSS property sets a margin for a CSS shape created using `shape-outside`.
   *
   * **Syntax**: `<length-percentage>`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **37** | **62**  | **10.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/shape-margin
   */
  shapeMargin?: Property.ShapeMargin<TLength>;
  /**
   * The **`shape-outside`** CSS property defines a shape—which may be non-rectangular—around which adjacent inline content should wrap. By default, inline content wraps around its margin box; `shape-outside` provides a way to customize this wrapping, making it possible to wrap text around complex objects rather than simple boxes.
   *
   * **Syntax**: `none | [ <shape-box> || <basic-shape> ] | <image>`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **37** | **62**  | **10.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/shape-outside
   */
  shapeOutside?: Property.ShapeOutside;
  /**
   * The **`tab-size`** CSS property is used to customize the width of tab characters (U+0009).
   *
   * **Syntax**: `<integer> | <length>`
   *
   * **Initial value**: `8`
   *
   * | Chrome |   Firefox   | Safari  |  Edge  | IE  |
   * | :----: | :---------: | :-----: | :----: | :-: |
   * | **21** | **4** _-x-_ | **6.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/tab-size
   */
  tabSize?: Property.TabSize<TLength>;
  /**
   * The **`table-layout`** CSS property sets the algorithm used to lay out `<table>` cells, rows, and columns.
   *
   * **Syntax**: `auto | fixed`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **14** |  **1**  | **1**  | **12** | **5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/table-layout
   */
  tableLayout?: Property.TableLayout;
  /**
   * The **`text-align`** CSS property sets the horizontal alignment of a block element or table-cell box. This means it works like `vertical-align` but in the horizontal direction.
   *
   * **Syntax**: `start | end | left | right | center | justify | match-parent`
   *
   * **Initial value**: `start`, or a nameless value that acts as `left` if _direction_ is `ltr`, `right` if _direction_ is `rtl` if `start` is not supported by the browser.
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-align
   */
  textAlign?: Property.TextAlign;
  /**
   * The **`text-align-last`** CSS property sets how the last line of a block or a line, right before a forced line break, is aligned.
   *
   * **Syntax**: `auto | start | end | left | right | center | justify`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **47** | **49**  |   No   | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-align-last
   */
  textAlignLast?: Property.TextAlignLast;
  /**
   * The **`text-combine-upright`** CSS property sets the combination of characters into the space of a single character. If the combined text is wider than 1em, the user agent must fit the contents within 1em. The resulting composition is treated as a single upright glyph for layout and decoration. This property only has an effect in vertical writing modes.
   *
   * **Syntax**: `none | all | [ digits <integer>? ]`
   *
   * **Initial value**: `none`
   *
   * |           Chrome           | Firefox |              Safari              | Edge  |                   IE                   |
   * | :------------------------: | :-----: | :------------------------------: | :---: | :------------------------------------: |
   * |           **48**           | **48**  | **5.1** _(-webkit-text-combine)_ | 15-79 | **11** _(-ms-text-combine-horizontal)_ |
   * | 9 _(-webkit-text-combine)_ |         |                                  |       |                                        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-combine-upright
   */
  textCombineUpright?: Property.TextCombineUpright;
  /**
   * The **`text-decoration-color`** CSS property sets the color of decorations added to text by `text-decoration-line`.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **36**  | **12.1** | **79** | No  |
   * |        |         | 8 _-x-_  |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-decoration-color
   */
  textDecorationColor?: Property.TextDecorationColor;
  /**
   * The **`text-decoration-line`** CSS property sets the kind of decoration that is used on text in an element, such as an underline or overline.
   *
   * **Syntax**: `none | [ underline || overline || line-through || blink ] | spelling-error | grammar-error`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **36**  | **12.1** | **79** | No  |
   * |        |         | 8 _-x-_  |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-decoration-line
   */
  textDecorationLine?: Property.TextDecorationLine;
  /**
   * The **`text-decoration-skip`** CSS property sets what parts of an element’s content any text decoration affecting the element must skip over. It controls all text decoration lines drawn by the element and also any text decoration lines drawn by its ancestors.
   *
   * **Syntax**: `none | [ objects || [ spaces | [ leading-spaces || trailing-spaces ] ] || edges || box-decoration ]`
   *
   * **Initial value**: `objects`
   *
   * | Chrome | Firefox |  Safari  | Edge | IE  |
   * | :----: | :-----: | :------: | :--: | :-: |
   * | 57-64  |   No    | **12.1** |  No  | No  |
   * |        |         | 8 _-x-_  |      |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-decoration-skip
   */
  textDecorationSkip?: Property.TextDecorationSkip;
  /**
   * The **`text-decoration-skip-ink`** CSS property specifies how overlines and underlines are drawn when they pass over glyph ascenders and descenders.
   *
   * **Syntax**: `auto | all | none`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **64** | **70**  |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-decoration-skip-ink
   */
  textDecorationSkipInk?: Property.TextDecorationSkipInk;
  /**
   * The **`text-decoration-style`** CSS property sets the style of the lines specified by `text-decoration-line`. The style applies to all lines that are set with `text-decoration-line`.
   *
   * **Syntax**: `solid | double | dotted | dashed | wavy`
   *
   * **Initial value**: `solid`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **36**  | **12.1** | **79** | No  |
   * |        |         | 8 _-x-_  |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-decoration-style
   */
  textDecorationStyle?: Property.TextDecorationStyle;
  /**
   * The **`text-decoration-thickness`** CSS property sets the stroke thickness of the decoration line that is used on text in an element, such as a line-through, underline, or overline.
   *
   * **Syntax**: `auto | from-font | <length> | <percentage> `
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **89** | **70**  | **12.1** | **89** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-decoration-thickness
   */
  textDecorationThickness?: Property.TextDecorationThickness<TLength>;
  /**
   * The **`text-decoration-thickness`** CSS property sets the stroke thickness of the decoration line that is used on text in an element, such as a line-through, underline, or overline.
   *
   * **Syntax**: `auto | from-font | <length> | <percentage> `
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  | Edge  | IE  |
   * | :----: | :-----: | :------: | :---: | :-: |
   * | 87-89  | **70**  | **12.1** | 87-89 | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-decoration-thickness
   */
  textDecorationWidth?: Property.TextDecorationThickness<TLength>;
  /**
   * The **`text-emphasis-color`** CSS property sets the color of emphasis marks. This value can also be set using the `text-emphasis` shorthand.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   *
   * |    Chrome    | Firefox | Safari  |     Edge     | IE  |
   * | :----------: | :-----: | :-----: | :----------: | :-: |
   * | **25** _-x-_ | **46**  | **6.1** | **79** _-x-_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-emphasis-color
   */
  textEmphasisColor?: Property.TextEmphasisColor;
  /**
   * The **`text-emphasis-position`** CSS property sets where emphasis marks are drawn. Like ruby text, if there isn't enough room for emphasis marks, the line height is increased.
   *
   * **Syntax**: `[ over | under ] && [ right | left ]`
   *
   * **Initial value**: `over right`
   *
   * |    Chrome    | Firefox | Safari  |     Edge     | IE  |
   * | :----------: | :-----: | :-----: | :----------: | :-: |
   * | **25** _-x-_ | **46**  | **6.1** | **79** _-x-_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-emphasis-position
   */
  textEmphasisPosition?: Property.TextEmphasisPosition;
  /**
   * The **`text-emphasis-style`** CSS property sets the appearance of emphasis marks. It can also be set, and reset, using the `text-emphasis` shorthand.
   *
   * **Syntax**: `none | [ [ filled | open ] || [ dot | circle | double-circle | triangle | sesame ] ] | <string>`
   *
   * **Initial value**: `none`
   *
   * |    Chrome    | Firefox | Safari  |     Edge     | IE  |
   * | :----------: | :-----: | :-----: | :----------: | :-: |
   * | **25** _-x-_ | **46**  | **6.1** | **79** _-x-_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-emphasis-style
   */
  textEmphasisStyle?: Property.TextEmphasisStyle;
  /**
   * The **`text-indent`** CSS property sets the length of empty space (indentation) that is put before lines of text in a block.
   *
   * **Syntax**: `<length-percentage> && hanging? && each-line?`
   *
   * **Initial value**: `0`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-indent
   */
  textIndent?: Property.TextIndent<TLength>;
  /**
   * The **`text-justify`** CSS property sets what type of justification should be applied to text when `text-align``: justify;` is set on an element.
   *
   * **Syntax**: `auto | inter-character | inter-word | none`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE   |
   * | :----: | :-----: | :----: | :----: | :----: |
   * |  n/a   | **55**  |   No   | **12** | **11** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-justify
   */
  textJustify?: Property.TextJustify;
  /**
   * The **`text-orientation`** CSS property sets the orientation of the text characters in a line. It only affects text in vertical mode (when `writing-mode` is not `horizontal-tb`). It is useful for controlling the display of languages that use vertical script, and also for making vertical table headers.
   *
   * **Syntax**: `mixed | upright | sideways`
   *
   * **Initial value**: `mixed`
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  | IE  |
   * | :------: | :-----: | :-------: | :----: | :-: |
   * |  **48**  | **41**  |  **14**   | **79** | No  |
   * | 11 _-x-_ |         | 5.1 _-x-_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-orientation
   */
  textOrientation?: Property.TextOrientation;
  /**
   * The **`text-overflow`** CSS property sets how hidden overflow content is signaled to users. It can be clipped, display an ellipsis ('`…`'), or display a custom string.
   *
   * **Syntax**: `[ clip | ellipsis | <string> ]{1,2}`
   *
   * **Initial value**: `clip`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  |  **7**  | **1.3** | **12** | **6** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-overflow
   */
  textOverflow?: Property.TextOverflow;
  /**
   * The **`text-rendering`** CSS property provides information to the rendering engine about what to optimize for when rendering text.
   *
   * **Syntax**: `auto | optimizeSpeed | optimizeLegibility | geometricPrecision`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **4**  |  **1**  | **5**  | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-rendering
   */
  textRendering?: Property.TextRendering;
  /**
   * The **`text-shadow`** CSS property adds shadows to text. It accepts a comma-separated list of shadows to be applied to the text and any of its `decorations`. Each shadow is described by some combination of X and Y offsets from the element, blur radius, and color.
   *
   * **Syntax**: `none | <shadow-t>#`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari  |  Edge  |   IE   |
   * | :----: | :-----: | :-----: | :----: | :----: |
   * | **2**  | **3.5** | **1.1** | **12** | **10** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-shadow
   */
  textShadow?: Property.TextShadow;
  /**
   * The **`text-size-adjust`** CSS property controls the text inflation algorithm used on some smartphones and tablets. Other browsers will ignore this property.
   *
   * **Syntax**: `none | auto | <percentage>`
   *
   * **Initial value**: `auto` for smartphone browsers supporting inflation, `none` in other cases (and then not modifiable).
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **54** |   No    |   No   | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-size-adjust
   */
  textSizeAdjust?: Property.TextSizeAdjust;
  /**
   * The **`text-transform`** CSS property specifies how to capitalize an element's text. It can be used to make text appear in all-uppercase or all-lowercase, or with each word capitalized. It also can help improve legibility for ruby.
   *
   * **Syntax**: `none | capitalize | uppercase | lowercase | full-width | full-size-kana`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-transform
   */
  textTransform?: Property.TextTransform;
  /**
   * The **`text-underline-offset`** CSS property sets the offset distance of an underline text decoration line (applied using `text-decoration`) from its original position.
   *
   * **Syntax**: `auto | <length> | <percentage> `
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **87** | **70**  | **12.1** | **87** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-underline-offset
   */
  textUnderlineOffset?: Property.TextUnderlineOffset<TLength>;
  /**
   * The **`text-underline-position`** CSS property specifies the position of the underline which is set using the `text-decoration` property's `underline` value.
   *
   * **Syntax**: `auto | from-font | [ under || [ left | right ] ]`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox |  Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :------: | :----: | :---: |
   * | **33** | **74**  | **12.1** | **12** | **6** |
   * |        |         | 9 _-x-_  |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-underline-position
   */
  textUnderlinePosition?: Property.TextUnderlinePosition;
  /**
   * The **`top`** CSS property participates in specifying the vertical position of a positioned element. It has no effect on non-positioned elements.
   *
   * **Syntax**: `<length> | <percentage> | auto`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/top
   */
  top?: Property.Top<TLength>;
  /**
   * The **`touch-action`** CSS property sets how an element's region can be manipulated by a touchscreen user (for example, by zooming features built into the browser).
   *
   * **Syntax**: `auto | none | [ [ pan-x | pan-left | pan-right ] || [ pan-y | pan-up | pan-down ] || pinch-zoom ] | manipulation`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |    IE    |
   * | :----: | :-----: | :----: | :----: | :------: |
   * | **36** | **52**  | **13** | **12** |  **11**  |
   * |        |         |        |        | 10 _-x-_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/touch-action
   */
  touchAction?: Property.TouchAction;
  /**
   * The **`transform`** CSS property lets you rotate, scale, skew, or translate an element. It modifies the coordinate space of the CSS visual formatting model.
   *
   * **Syntax**: `none | <transform-list>`
   *
   * **Initial value**: `none`
   *
   * | Chrome  | Firefox |  Safari   |  Edge  |   IE    |
   * | :-----: | :-----: | :-------: | :----: | :-----: |
   * | **36**  | **16**  |   **9**   | **12** | **10**  |
   * | 1 _-x-_ |         | 3.1 _-x-_ |        | 9 _-x-_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transform
   */
  transform?: Property.Transform;
  /**
   * The **`transform-box`** CSS property defines the layout box to which the `transform` and `transform-origin` properties relate.
   *
   * **Syntax**: `content-box | border-box | fill-box | stroke-box | view-box`
   *
   * **Initial value**: `view-box`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **64** | **55**  | **11** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transform-box
   */
  transformBox?: Property.TransformBox;
  /**
   * The **`transform-origin`** CSS property sets the origin for an element's transformations.
   *
   * **Syntax**: `[ <length-percentage> | left | center | right | top | bottom ] | [ [ <length-percentage> | left | center | right ] && [ <length-percentage> | top | center | bottom ] ] <length>?`
   *
   * **Initial value**: `50% 50% 0`
   *
   * | Chrome  |  Firefox  | Safari  |  Edge  |   IE    |
   * | :-----: | :-------: | :-----: | :----: | :-----: |
   * | **36**  |  **16**   |  **9**  | **12** | **10**  |
   * | 1 _-x-_ | 3.5 _-x-_ | 2 _-x-_ |        | 9 _-x-_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transform-origin
   */
  transformOrigin?: Property.TransformOrigin<TLength>;
  /**
   * The **`transform-style`** CSS property sets whether children of an element are positioned in the 3D space or are flattened in the plane of the element.
   *
   * **Syntax**: `flat | preserve-3d`
   *
   * **Initial value**: `flat`
   *
   * |  Chrome  | Firefox  | Safari  |  Edge  | IE  |
   * | :------: | :------: | :-----: | :----: | :-: |
   * |  **36**  |  **16**  |  **9**  | **12** | No  |
   * | 12 _-x-_ | 10 _-x-_ | 4 _-x-_ |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transform-style
   */
  transformStyle?: Property.TransformStyle;
  /**
   * The **`transition-delay`** CSS property specifies the duration to wait before starting a property's transition effect when its value changes.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **26**  | **16**  |  **9**  | **12** | **10** |
   * | 1 _-x-_ | 4 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transition-delay
   */
  transitionDelay?: Property.TransitionDelay<TTime>;
  /**
   * The **`transition-duration`** CSS property sets the length of time a transition animation should take to complete. By default, the value is `0s`, meaning that no animation will occur.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   *
   * | Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :-----: | :-----: | :-------: | :----: | :----: |
   * | **26**  | **16**  |   **9**   | **12** | **10** |
   * | 1 _-x-_ | 4 _-x-_ | 3.1 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transition-duration
   */
  transitionDuration?: Property.TransitionDuration<TTime>;
  /**
   * The **`transition-property`** CSS property sets the CSS properties to which a transition effect should be applied.
   *
   * **Syntax**: `none | <single-transition-property>#`
   *
   * **Initial value**: all
   *
   * | Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :-----: | :-----: | :-------: | :----: | :----: |
   * | **26**  | **16**  |   **9**   | **12** | **10** |
   * | 1 _-x-_ | 4 _-x-_ | 3.1 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transition-property
   */
  transitionProperty?: Property.TransitionProperty;
  /**
   * The **`transition-timing-function`** CSS property sets how intermediate values are calculated for CSS properties being affected by a transition effect.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   *
   * | Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :-----: | :-----: | :-------: | :----: | :----: |
   * | **26**  | **16**  |   **9**   | **12** | **10** |
   * | 1 _-x-_ | 4 _-x-_ | 3.1 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transition-timing-function
   */
  transitionTimingFunction?: Property.TransitionTimingFunction;
  /**
   * The **`translate`** CSS property allows you to specify translation transforms individually and independently of the `transform` property. This maps better to typical user interface usage, and saves having to remember the exact order of transform functions to specify in the `transform` value.
   *
   * **Syntax**: `none | <length-percentage> [ <length-percentage> <length>? ]?`
   *
   * **Initial value**: `none`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * |   No   | **72**  |   No   |  No  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/translate
   */
  translate?: Property.Translate<TLength>;
  /**
   * The **`unicode-bidi`** CSS property, together with the `direction` property, determines how bidirectional text in a document is handled. For example, if a block of content contains both left-to-right and right-to-left text, the user-agent uses a complex Unicode algorithm to decide how to display the text. The `unicode-bidi` property overrides this algorithm and allows the developer to control the text embedding.
   *
   * **Syntax**: `normal | embed | isolate | bidi-override | isolate-override | plaintext`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari  |  Edge  |   IE    |
   * | :----: | :-----: | :-----: | :----: | :-----: |
   * | **2**  |  **1**  | **1.3** | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/unicode-bidi
   */
  unicodeBidi?: Property.UnicodeBidi;
  /**
   * The `**user-select**` CSS property controls whether the user can select text. This doesn't have any effect on content loaded as chrome, except in textboxes.
   *
   * **Syntax**: `auto | text | none | contain | all`
   *
   * **Initial value**: `auto`
   *
   * | Chrome  | Firefox |   Safari    |     Edge     |      IE      |
   * | :-----: | :-----: | :---------: | :----------: | :----------: |
   * | **54**  | **69**  | **3** _-x-_ | **12** _-x-_ | **10** _-x-_ |
   * | 1 _-x-_ | 1 _-x-_ |             |              |              |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/user-select
   */
  userSelect?: Property.UserSelect;
  /**
   * The **`vertical-align`** CSS property sets vertical alignment of an inline, inline-block or table-cell box.
   *
   * **Syntax**: `baseline | sub | super | text-top | text-bottom | middle | top | bottom | <percentage> | <length>`
   *
   * **Initial value**: `baseline`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/vertical-align
   */
  verticalAlign?: Property.VerticalAlign<TLength>;
  /**
   * The **`visibility`** CSS property shows or hides an element without changing the layout of a document. The property can also hide rows or columns in a `<table>`.
   *
   * **Syntax**: `visible | hidden | collapse`
   *
   * **Initial value**: `visible`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/visibility
   */
  visibility?: Property.Visibility;
  /**
   * The **`white-space`** CSS property sets how white space inside an element is handled.
   *
   * **Syntax**: `normal | pre | nowrap | pre-wrap | pre-line | break-spaces`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/white-space
   */
  whiteSpace?: Property.WhiteSpace;
  /**
   * The **`widows`** CSS property sets the minimum number of lines in a block container that must be shown at the _top_ of a page, region, or column.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `2`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **25** |   No    | **1.3** | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/widows
   */
  widows?: Property.Widows;
  /**
   * The **`width`** CSS property sets an element's width. By default, it sets the width of the content area, but if `box-sizing` is set to `border-box`, it sets the width of the border area.
   *
   * **Syntax**: `auto | <length> | <percentage> | min-content | max-content | fit-content | fit-content(<length-percentage>)`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/width
   */
  width?: Property.Width<TLength>;
  /**
   * The **`will-change`** CSS property hints to browsers how an element is expected to change. Browsers may set up optimizations before an element is actually changed. These kinds of optimizations can increase the responsiveness of a page by doing potentially expensive work before they are actually required.
   *
   * **Syntax**: `auto | <animateable-feature>#`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari  |  Edge  | IE  |
   * | :----: | :-----: | :-----: | :----: | :-: |
   * | **36** | **36**  | **9.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/will-change
   */
  willChange?: Property.WillChange;
  /**
   * The **`word-break`** CSS property sets whether line breaks appear wherever the text would otherwise overflow its content box.
   *
   * **Syntax**: `normal | break-all | keep-all | break-word`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  | **15**  | **3**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/word-break
   */
  wordBreak?: Property.WordBreak;
  /**
   * The **`word-spacing`** CSS property sets the length of space between words and between tags.
   *
   * **Syntax**: `normal | <length-percentage>`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **6** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/word-spacing
   */
  wordSpacing?: Property.WordSpacing<TLength>;
  /**
   * The `**overflow-wrap**` CSS property applies to inline elements, setting whether the browser should insert line breaks within an otherwise unbreakable string to prevent text from overflowing its line box.
   *
   * **Syntax**: `normal | break-word`
   *
   * **Initial value**: `normal`
   */
  wordWrap?: Property.WordWrap;
  /**
   * The **`writing-mode`** CSS property sets whether lines of text are laid out horizontally or vertically, as well as the direction in which blocks progress. When set for an entire document, it should be set on the root element (`html` element for HTML documents).
   *
   * **Syntax**: `horizontal-tb | vertical-rl | vertical-lr | sideways-rl | sideways-lr`
   *
   * **Initial value**: `horizontal-tb`
   *
   * | Chrome  | Firefox |  Safari   |  Edge  |  IE   |
   * | :-----: | :-----: | :-------: | :----: | :---: |
   * | **48**  | **41**  | **10.1**  | **12** | **9** |
   * | 8 _-x-_ |         | 5.1 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/writing-mode
   */
  writingMode?: Property.WritingMode;
  /**
   * The **`z-index`** CSS property sets the z-order of a positioned element and its descendants or flex items. Overlapping elements with a larger z-index cover those with a smaller one.
   *
   * **Syntax**: `auto | <integer>`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/z-index
   */
  zIndex?: Property.ZIndex;
  /**
   * The non-standard **`zoom`** CSS property can be used to control the magnification level of an element. `transform: scale()` should be used instead of this property, if possible. However, unlike CSS Transforms, `zoom` affects the layout size of the element.
   *
   * **Syntax**: `normal | reset | <number> | <percentage>`
   *
   * **Initial value**: `normal`
   *
   * | Chrome | Firefox | Safari  |  Edge  |   IE    |
   * | :----: | :-----: | :-----: | :----: | :-----: |
   * | **1**  |   No    | **3.1** | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/zoom
   */
  zoom?: Property.Zoom;
}

interface StandardShorthandProperties<TLength = (string & {}) | 0, TTime = string & {}> {
  /**
   * The `**all**` shorthand CSS property resets all of an element's properties except `unicode-bidi`, `direction`, and CSS Custom Properties. It can set properties to their initial or inherited values, or to the values specified in another stylesheet origin.
   *
   * **Syntax**: `initial | inherit | unset | revert`
   *
   * **Initial value**: There is no practical initial value for it.
   *
   * | Chrome | Firefox | Safari  |  Edge  | IE  |
   * | :----: | :-----: | :-----: | :----: | :-: |
   * | **37** | **27**  | **9.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/all
   */
  all?: Property.All;
  /**
   * The **`animation`** shorthand CSS property applies an animation between styles. It is a shorthand for `animation-name`, `animation-duration`, `animation-timing-function`, `animation-delay`, `animation-iteration-count`, `animation-direction`, `animation-fill-mode`, and `animation-play-state`.
   *
   * **Syntax**: `<single-animation>#`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **43**  | **16**  |  **9**  | **12** | **10** |
   * | 3 _-x-_ | 5 _-x-_ | 4 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/animation
   */
  animation?: Property.Animation<TTime>;
  /**
   * The **`background`** shorthand CSS property sets all background style properties at once, such as color, image, origin and size, or repeat method.
   *
   * **Syntax**: `[ <bg-layer> , ]* <final-bg-layer>`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background
   */
  background?: Property.Background<TLength>;
  /**
   * The **`background-position`** CSS property sets the initial position for each background image. The position is relative to the position layer set by `background-origin`.
   *
   * **Syntax**: `<bg-position>#`
   *
   * **Initial value**: `0% 0%`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/background-position
   */
  backgroundPosition?: Property.BackgroundPosition<TLength>;
  /**
   * The **`border`** shorthand CSS property sets an element's border. It sets the values of `border-width`, `border-style`, and `border-color`.
   *
   * **Syntax**: `<line-width> || <line-style> || <color>`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border
   */
  border?: Property.Border<TLength>;
  /**
   * The **`border-block`** CSS property is a shorthand property for setting the individual logical block border property values in a single place in the style sheet.
   *
   * **Syntax**: `<'border-top-width'> || <'border-top-style'> || <color>`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block
   */
  borderBlock?: Property.BorderBlock<TLength>;
  /**
   * The **`border-block-end`** CSS property is a shorthand property for setting the individual logical block-end border property values in a single place in the style sheet.
   *
   * **Syntax**: `<'border-top-width'> || <'border-top-style'> || <color>`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-end
   */
  borderBlockEnd?: Property.BorderBlockEnd<TLength>;
  /**
   * The **`border-block-start`** CSS property is a shorthand property for setting the individual logical block-start border property values in a single place in the style sheet.
   *
   * **Syntax**: `<'border-top-width'> || <'border-top-style'> || <color>`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-block-start
   */
  borderBlockStart?: Property.BorderBlockStart<TLength>;
  /**
   * The **`border-bottom`** shorthand CSS property sets an element's bottom border. It sets the values of `border-bottom-width`, `border-bottom-style` and `border-bottom-color`.
   *
   * **Syntax**: `<line-width> || <line-style> || <color>`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-bottom
   */
  borderBottom?: Property.BorderBottom<TLength>;
  /**
   * The **`border-color`** shorthand CSS property sets the color of an element's border.
   *
   * **Syntax**: `<color>{1,4}`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-color
   */
  borderColor?: Property.BorderColor;
  /**
   * The **`border-image`** CSS property draws an image around a given element. It replaces the element's regular border.
   *
   * **Syntax**: `<'border-image-source'> || <'border-image-slice'> [ / <'border-image-width'> | / <'border-image-width'>? / <'border-image-outset'> ]? || <'border-image-repeat'>`
   *
   * | Chrome  |  Firefox  | Safari  |  Edge  |   IE   |
   * | :-----: | :-------: | :-----: | :----: | :----: |
   * | **16**  |  **15**   |  **6**  | **12** | **11** |
   * | 7 _-x-_ | 3.5 _-x-_ | 3 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-image
   */
  borderImage?: Property.BorderImage;
  /**
   * The **`border-inline`** CSS property is a shorthand property for setting the individual logical inline border property values in a single place in the style sheet.
   *
   * **Syntax**: `<'border-top-width'> || <'border-top-style'> || <color>`
   *
   * | Chrome | Firefox | Safari | Edge | IE  |
   * | :----: | :-----: | :----: | :--: | :-: |
   * | **87** | **66**  |   No   | n/a  | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline
   */
  borderInline?: Property.BorderInline<TLength>;
  /**
   * The **`border-inline-end`** CSS property is a shorthand property for setting the individual logical inline-end border property values in a single place in the style sheet.
   *
   * **Syntax**: `<'border-top-width'> || <'border-top-style'> || <color>`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-end
   */
  borderInlineEnd?: Property.BorderInlineEnd<TLength>;
  /**
   * The **`border-inline-start`** CSS property is a shorthand property for setting the individual logical inline-start border property values in a single place in the style sheet.
   *
   * **Syntax**: `<'border-top-width'> || <'border-top-style'> || <color>`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **69** | **41**  | **12.1** | **79** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-inline-start
   */
  borderInlineStart?: Property.BorderInlineStart<TLength>;
  /**
   * The **`border-left`** shorthand CSS property sets all the properties of an element's left border.
   *
   * **Syntax**: `<line-width> || <line-style> || <color>`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-left
   */
  borderLeft?: Property.BorderLeft<TLength>;
  /**
   * The **`border-radius`** CSS property rounds the corners of an element's outer border edge. You can set a single radius to make circular corners, or two radii to make elliptical corners.
   *
   * **Syntax**: `<length-percentage>{1,4} [ / <length-percentage>{1,4} ]?`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |  IE   |
   * | :-----: | :-----: | :-----: | :----: | :---: |
   * |  **4**  |  **4**  |  **5**  | **12** | **9** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |       |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-radius
   */
  borderRadius?: Property.BorderRadius<TLength>;
  /**
   * The **`border-right`** shorthand CSS property sets all the properties of an element's right border.
   *
   * **Syntax**: `<line-width> || <line-style> || <color>`
   *
   * | Chrome | Firefox | Safari |  Edge  |   IE    |
   * | :----: | :-----: | :----: | :----: | :-----: |
   * | **1**  |  **1**  | **1**  | **12** | **5.5** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-right
   */
  borderRight?: Property.BorderRight<TLength>;
  /**
   * The **`border-style`** shorthand CSS property sets the line style for all four sides of an element's border.
   *
   * **Syntax**: `<line-style>{1,4}`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-style
   */
  borderStyle?: Property.BorderStyle;
  /**
   * The **`border-top`** shorthand CSS property sets all the properties of an element's top border.
   *
   * **Syntax**: `<line-width> || <line-style> || <color>`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-top
   */
  borderTop?: Property.BorderTop<TLength>;
  /**
   * The **`border-width`** shorthand CSS property sets the width of an element's border.
   *
   * **Syntax**: `<line-width>{1,4}`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/border-width
   */
  borderWidth?: Property.BorderWidth<TLength>;
  /**
   * The **`column-rule`** shorthand CSS property sets the width, style, and color of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'column-rule-width'> || <'column-rule-style'> || <'column-rule-color'>`
   *
   * | Chrome  | Firefox | Safari  |  Edge  |   IE   |
   * | :-----: | :-----: | :-----: | :----: | :----: |
   * | **50**  | **52**  |  **9**  | **12** | **10** |
   * | 1 _-x-_ |         | 3 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/column-rule
   */
  columnRule?: Property.ColumnRule<TLength>;
  /**
   * The **`columns`** CSS shorthand property sets the number of columns to use when drawing an element's contents, as well as those columns' widths.
   *
   * **Syntax**: `<'column-width'> || <'column-count'>`
   *
   * | Chrome | Firefox | Safari  |  Edge  |   IE   |
   * | :----: | :-----: | :-----: | :----: | :----: |
   * | **50** | **52**  |  **9**  | **12** | **10** |
   * |        |         | 3 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/columns
   */
  columns?: Property.Columns<TLength>;
  /**
   * The **`flex`** CSS shorthand property sets how a flex _item_ will grow or shrink to fit the space available in its flex container.
   *
   * **Syntax**: `none | [ <'flex-grow'> <'flex-shrink'>? || <'flex-basis'> ]`
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  |    IE    |
   * | :------: | :-----: | :-------: | :----: | :------: |
   * |  **29**  | **20**  |   **9**   | **12** |  **11**  |
   * | 21 _-x-_ |         | 6.1 _-x-_ |        | 10 _-x-_ |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/flex
   */
  flex?: Property.Flex<TLength>;
  /**
   * The **`flex-flow`** CSS shorthand property specifies the direction of a flex container, as well as its wrapping behavior.
   *
   * **Syntax**: `<'flex-direction'> || <'flex-wrap'>`
   *
   * |  Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :------: | :-----: | :-------: | :----: | :----: |
   * |  **29**  | **28**  |   **9**   | **12** | **11** |
   * | 21 _-x-_ |         | 6.1 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/flex-flow
   */
  flexFlow?: Property.FlexFlow;
  /**
   * The **`font`** CSS shorthand property sets all the different properties of an element's font. Alternatively, it sets an element's font to a system font.
   *
   * **Syntax**: `[ [ <'font-style'> || <font-variant-css21> || <'font-weight'> || <'font-stretch'> ]? <'font-size'> [ / <'line-height'> ]? <'font-family'> ] | caption | icon | menu | message-box | small-caption | status-bar`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/font
   */
  font?: Property.Font;
  /**
   * The **`gap`** CSS property sets the gaps (gutters) between rows and columns. It is a shorthand for `row-gap` and `column-gap`.
   *
   * **Syntax**: `<'row-gap'> <'column-gap'>?`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **84** | **63**  |   No   | **84** | No  |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * |     Chrome      |     Firefox     |      Safari       |  Edge  | IE  |
   * | :-------------: | :-------------: | :---------------: | :----: | :-: |
   * |     **66**      |     **61**      |      **12**       | **16** | No  |
   * | 57 _(grid-gap)_ | 52 _(grid-gap)_ | 10.1 _(grid-gap)_ |        |     |
   *
   * ---
   *
   * _Supported in Multi-column Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **66** | **61**  |   No   | **16** | No  |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/gap
   */
  gap?: Property.Gap<TLength>;
  /**
   * The **`grid`** CSS property is a shorthand property that sets all of the explicit and implicit grid properties in a single declaration.
   *
   * **Syntax**: `<'grid-template'> | <'grid-template-rows'> / [ auto-flow && dense? ] <'grid-auto-columns'>? | [ auto-flow && dense? ] <'grid-auto-rows'>? / <'grid-template-columns'>`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid
   */
  grid?: Property.Grid;
  /**
   * The **`grid-area`** CSS shorthand property specifies a grid item’s size and location within a grid by contributing a line, a span, or nothing (automatic) to its grid placement, thereby specifying the edges of its grid area.
   *
   * **Syntax**: `<grid-line> [ / <grid-line> ]{0,3}`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-area
   */
  gridArea?: Property.GridArea;
  /**
   * The **`grid-column`** CSS shorthand property specifies a grid item's size and location within a grid column by contributing a line, a span, or nothing (automatic) to its grid placement, thereby specifying the inline-start and inline-end edge of its grid area.
   *
   * **Syntax**: `<grid-line> [ / <grid-line> ]?`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-column
   */
  gridColumn?: Property.GridColumn;
  /**
   * The **`grid-row`** CSS shorthand property specifies a grid item’s size and location within the grid row by contributing a line, a span, or nothing (automatic) to its grid placement, thereby specifying the inline-start and inline-end edge of its grid area.
   *
   * **Syntax**: `<grid-line> [ / <grid-line> ]?`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-row
   */
  gridRow?: Property.GridRow;
  /**
   * The **`grid-template`** CSS property is a shorthand property for defining grid columns, rows, and areas.
   *
   * **Syntax**: `none | [ <'grid-template-rows'> / <'grid-template-columns'> ] | [ <line-names>? <string> <track-size>? <line-names>? ]+ [ / <explicit-track-list> ]?`
   *
   * | Chrome | Firefox |  Safari  |  Edge  | IE  |
   * | :----: | :-----: | :------: | :----: | :-: |
   * | **57** | **52**  | **10.1** | **16** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/grid-template
   */
  gridTemplate?: Property.GridTemplate;
  /**
   * **Syntax**: `none | <integer>`
   *
   * **Initial value**: `none`
   */
  lineClamp?: Property.LineClamp;
  /**
   * The **`list-style`** CSS shorthand property allows you set all the list style properties at once.
   *
   * **Syntax**: `<'list-style-type'> || <'list-style-position'> || <'list-style-image'>`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/list-style
   */
  listStyle?: Property.ListStyle;
  /**
   * The **`margin`** CSS property sets the margin area on all four sides of an element. It is a shorthand for `margin-top`, `margin-right`, `margin-bottom`, and `margin-left`.
   *
   * **Syntax**: `[ <length> | <percentage> | auto ]{1,4}`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/margin
   */
  margin?: Property.Margin<TLength>;
  /**
   * The **`mask`** CSS shorthand property hides an element (partially or fully) by masking or clipping the image at specific points.
   *
   * **Syntax**: `<mask-layer>#`
   *
   * | Chrome | Firefox | Safari  |  Edge  | IE  |
   * | :----: | :-----: | :-----: | :----: | :-: |
   * | **1**  |  **2**  | **3.2** | **12** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask
   */
  mask?: Property.Mask<TLength>;
  /**
   * The **`mask-border`** CSS shorthand property lets you create a mask along the edge of an element's border.
   *
   * **Syntax**: `<'mask-border-source'> || <'mask-border-slice'> [ / <'mask-border-width'>? [ / <'mask-border-outset'> ]? ]? || <'mask-border-repeat'> || <'mask-border-mode'>`
   *
   * |              Chrome              | Firefox |               Safari               |               Edge                | IE  |
   * | :------------------------------: | :-----: | :--------------------------------: | :-------------------------------: | :-: |
   * | **1** _(-webkit-mask-box-image)_ |   No    | **3.1** _(-webkit-mask-box-image)_ | **79** _(-webkit-mask-box-image)_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/mask-border
   */
  maskBorder?: Property.MaskBorder;
  /**
   * The **`offset`** CSS shorthand property sets all the properties required for animating an element along a defined path.
   *
   * **Syntax**: `[ <'offset-position'>? [ <'offset-path'> [ <'offset-distance'> || <'offset-rotate'> ]? ]? ]! [ / <'offset-anchor'> ]?`
   *
   * |    Chrome     | Firefox | Safari |  Edge  | IE  |
   * | :-----------: | :-----: | :----: | :----: | :-: |
   * |    **55**     | **72**  |   No   | **79** | No  |
   * | 46 _(motion)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset
   */
  motion?: Property.Offset<TLength>;
  /**
   * The **`offset`** CSS shorthand property sets all the properties required for animating an element along a defined path.
   *
   * **Syntax**: `[ <'offset-position'>? [ <'offset-path'> [ <'offset-distance'> || <'offset-rotate'> ]? ]? ]! [ / <'offset-anchor'> ]?`
   *
   * |    Chrome     | Firefox | Safari |  Edge  | IE  |
   * | :-----------: | :-----: | :----: | :----: | :-: |
   * |    **55**     | **72**  |   No   | **79** | No  |
   * | 46 _(motion)_ |         |        |        |     |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/offset
   */
  offset?: Property.Offset<TLength>;
  /**
   * The **`outline`** CSS shorthand property set all the outline properties in a single declaration.
   *
   * **Syntax**: `[ <'outline-color'> || <'outline-style'> || <'outline-width'> ]`
   *
   * | Chrome | Firefox | Safari  |  Edge  |  IE   |
   * | :----: | :-----: | :-----: | :----: | :---: |
   * | **1**  | **1.5** | **1.2** | **12** | **8** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/outline
   */
  outline?: Property.Outline<TLength>;
  /**
   * The **`overflow`** CSS shorthand property sets the desired behavior for an element's overflow — i.e. when an element's content is too big to fit in its block formatting context — in both directions.
   *
   * **Syntax**: `[ visible | hidden | clip | scroll | auto ]{1,2}`
   *
   * **Initial value**: `visible`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overflow
   */
  overflow?: Property.Overflow;
  /**
   * The **`overscroll-behavior`** CSS property sets what a browser does when reaching the boundary of a scrolling area. It's a shorthand for `overscroll-behavior-x` and `overscroll-behavior-y`.
   *
   * **Syntax**: `[ contain | none | auto ]{1,2}`
   *
   * **Initial value**: `auto`
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **63** | **59**  |   No   | **18** | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/overscroll-behavior
   */
  overscrollBehavior?: Property.OverscrollBehavior;
  /**
   * The **`padding`** CSS shorthand property sets the padding area on all four sides of an element at once.
   *
   * **Syntax**: `[ <length> | <percentage> ]{1,4}`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **4** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/padding
   */
  padding?: Property.Padding<TLength>;
  /**
   * The CSS **`place-items`** shorthand property allows you to align items along both the block and inline directions at once (i.e. the `align-items` and `justify-items` properties) in a relevant layout system such as Grid or Flexbox. If the second value is not set, the first value is also used for it.
   *
   * **Syntax**: `<'align-items'> <'justify-items'>?`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **59** | **45**  | **11** | **79** | No  |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **59** | **45**  | **11** | **79** | No  |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/place-items
   */
  placeItems?: Property.PlaceItems;
  /**
   * The **`place-self`** CSS shorthand property allows you to align an individual item in both the block and inline directions at once (i.e. the `align-self` and `justify-self` properties) in a relevant layout system such as Grid or Flexbox. If the second value is not present, the first value is also used for it.
   *
   * **Syntax**: `<'align-self'> <'justify-self'>?`
   *
   * ---
   *
   * _Supported in Flex Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **59** | **45**  | **11** | **79** | No  |
   *
   * ---
   *
   * _Supported in Grid Layout_
   *
   * | Chrome | Firefox | Safari |  Edge  | IE  |
   * | :----: | :-----: | :----: | :----: | :-: |
   * | **59** | **45**  | **11** | **79** | No  |
   *
   * ---
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/place-self
   */
  placeSelf?: Property.PlaceSelf;
  /**
   * The **`text-decoration`** shorthand CSS property sets the appearance of decorative lines on text. It is a shorthand for `text-decoration-line`, `text-decoration-color`, `text-decoration-style`, and the newer `text-decoration-thickness` property.
   *
   * **Syntax**: `<'text-decoration-line'> || <'text-decoration-style'> || <'text-decoration-color'> || <'text-decoration-thickness'>`
   *
   * | Chrome | Firefox | Safari |  Edge  |  IE   |
   * | :----: | :-----: | :----: | :----: | :---: |
   * | **1**  |  **1**  | **1**  | **12** | **3** |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-decoration
   */
  textDecoration?: Property.TextDecoration<TLength>;
  /**
   * The **`text-emphasis`** CSS property applies emphasis marks to text (except spaces and control characters). It is a shorthand for `text-emphasis-style` and `text-emphasis-color`.
   *
   * **Syntax**: `<'text-emphasis-style'> || <'text-emphasis-color'>`
   *
   * |    Chrome    | Firefox | Safari  |     Edge     | IE  |
   * | :----------: | :-----: | :-----: | :----------: | :-: |
   * | **25** _-x-_ | **46**  | **6.1** | **79** _-x-_ | No  |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/text-emphasis
   */
  textEmphasis?: Property.TextEmphasis;
  /**
   * The **`transition`** CSS property is a shorthand property for `transition-property`, `transition-duration`, `transition-timing-function`, and `transition-delay`.
   *
   * **Syntax**: `<single-transition>#`
   *
   * | Chrome  | Firefox |  Safari   |  Edge  |   IE   |
   * | :-----: | :-----: | :-------: | :----: | :----: |
   * | **26**  | **16**  |   **9**   | **12** | **10** |
   * | 1 _-x-_ | 4 _-x-_ | 3.1 _-x-_ |        |        |
   *
   * @see https://developer.mozilla.org/docs/Web/CSS/transition
   */
  transition?: Property.Transition<TTime>;
}

interface StandardProperties<TLength = (string & {}) | 0, TTime = string & {}>
  extends StandardLonghandProperties<TLength, TTime>,
    StandardShorthandProperties<TLength, TTime> {}

interface VendorLonghandProperties<TLength = (string & {}) | 0, TTime = string & {}> {
  /**
   * The **`animation-delay`** CSS property specifies the amount of time to wait from applying the animation to an element before beginning to perform the animation. The animation can start later, immediately from its beginning, or immediately and partway through the animation.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  MozAnimationDelay?: Property.AnimationDelay<TTime>;
  /**
   * The **`animation-direction`** CSS property sets whether an animation should play forward, backward, or alternate back and forth between playing the sequence forward and backward.
   *
   * **Syntax**: `<single-animation-direction>#`
   *
   * **Initial value**: `normal`
   */
  MozAnimationDirection?: Property.AnimationDirection;
  /**
   * The **`animation-duration`** CSS property sets the length of time that an animation takes to complete one cycle.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  MozAnimationDuration?: Property.AnimationDuration<TTime>;
  /**
   * The **`animation-fill-mode`** CSS property sets how a CSS animation applies styles to its target before and after its execution.
   *
   * **Syntax**: `<single-animation-fill-mode>#`
   *
   * **Initial value**: `none`
   */
  MozAnimationFillMode?: Property.AnimationFillMode;
  /**
   * The **`animation-iteration-count`** CSS property sets the number of times an animation sequence should be played before stopping.
   *
   * **Syntax**: `<single-animation-iteration-count>#`
   *
   * **Initial value**: `1`
   */
  MozAnimationIterationCount?: Property.AnimationIterationCount;
  /**
   * The **`animation-name`** CSS property specifies the names of one or more `@keyframes` at-rules describing the animation or animations to apply to the element.
   *
   * **Syntax**: `[ none | <keyframes-name> ]#`
   *
   * **Initial value**: `none`
   */
  MozAnimationName?: Property.AnimationName;
  /**
   * The **`animation-play-state`** CSS property sets whether an animation is running or paused.
   *
   * **Syntax**: `<single-animation-play-state>#`
   *
   * **Initial value**: `running`
   */
  MozAnimationPlayState?: Property.AnimationPlayState;
  /**
   * The **`animation-timing-function`** CSS property sets how an animation progresses through the duration of each cycle.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   */
  MozAnimationTimingFunction?: Property.AnimationTimingFunction;
  /**
   * The `**appearance**` CSS property is used to display an element using platform-native styling, based on the operating system's theme. The **`-moz-appearance`** and **`-webkit-appearance`** properties are non-standard versions of this property, used (respectively) by Gecko (Firefox) and by WebKit-based (e.g., Safari) and Blink-based (e.g., Chrome, Opera) browsers to achieve the same thing. Note that Firefox and Edge also support **`-webkit-appearance`**, for compatibility reasons.
   *
   * **Syntax**: `none | button | button-arrow-down | button-arrow-next | button-arrow-previous | button-arrow-up | button-bevel | button-focus | caret | checkbox | checkbox-container | checkbox-label | checkmenuitem | dualbutton | groupbox | listbox | listitem | menuarrow | menubar | menucheckbox | menuimage | menuitem | menuitemtext | menulist | menulist-button | menulist-text | menulist-textfield | menupopup | menuradio | menuseparator | meterbar | meterchunk | progressbar | progressbar-vertical | progresschunk | progresschunk-vertical | radio | radio-container | radio-label | radiomenuitem | range | range-thumb | resizer | resizerpanel | scale-horizontal | scalethumbend | scalethumb-horizontal | scalethumbstart | scalethumbtick | scalethumb-vertical | scale-vertical | scrollbarbutton-down | scrollbarbutton-left | scrollbarbutton-right | scrollbarbutton-up | scrollbarthumb-horizontal | scrollbarthumb-vertical | scrollbartrack-horizontal | scrollbartrack-vertical | searchfield | separator | sheet | spinner | spinner-downbutton | spinner-textfield | spinner-upbutton | splitter | statusbar | statusbarpanel | tab | tabpanel | tabpanels | tab-scroll-arrow-back | tab-scroll-arrow-forward | textfield | textfield-multiline | toolbar | toolbarbutton | toolbarbutton-dropdown | toolbargripper | toolbox | tooltip | treeheader | treeheadercell | treeheadersortarrow | treeitem | treeline | treetwisty | treetwistyopen | treeview | -moz-mac-unified-toolbar | -moz-win-borderless-glass | -moz-win-browsertabbar-toolbox | -moz-win-communicationstext | -moz-win-communications-toolbox | -moz-win-exclude-glass | -moz-win-glass | -moz-win-mediatext | -moz-win-media-toolbox | -moz-window-button-box | -moz-window-button-box-maximized | -moz-window-button-close | -moz-window-button-maximize | -moz-window-button-minimize | -moz-window-button-restore | -moz-window-frame-bottom | -moz-window-frame-left | -moz-window-frame-right | -moz-window-titlebar | -moz-window-titlebar-maximized`
   *
   * **Initial value**: `none` (but this value is overridden in the user agent CSS)
   */
  MozAppearance?: Property.MozAppearance;
  /**
   * The **`backface-visibility`** CSS property sets whether the back face of an element is visible when turned towards the user.
   *
   * **Syntax**: `visible | hidden`
   *
   * **Initial value**: `visible`
   */
  MozBackfaceVisibility?: Property.BackfaceVisibility;
  /**
   * In Mozilla applications like Firefox, the **`-moz-border-bottom-colors`** CSS property sets a list of colors for the bottom border.
   *
   * **Syntax**: `<color>+ | none`
   *
   * **Initial value**: `none`
   */
  MozBorderBottomColors?: Property.MozBorderBottomColors;
  /**
   * The **`border-inline-end-color`** CSS property defines the color of the logical inline-end border of an element, which maps to a physical border color depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-color`, `border-right-color`, `border-bottom-color`, or `border-left-color` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-color'>`
   *
   * **Initial value**: `currentcolor`
   */
  MozBorderEndColor?: Property.BorderInlineEndColor;
  /**
   * The **`border-inline-end-style`** CSS property defines the style of the logical inline end border of an element, which maps to a physical border style depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-style`, `border-right-style`, `border-bottom-style`, or `border-left-style` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-style'>`
   *
   * **Initial value**: `none`
   */
  MozBorderEndStyle?: Property.BorderInlineEndStyle;
  /**
   * The **`border-inline-end-width`** CSS property defines the width of the logical inline-end border of an element, which maps to a physical border width depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-width`, `border-right-width`, `border-bottom-width`, or `border-left-width` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-width'>`
   *
   * **Initial value**: `medium`
   */
  MozBorderEndWidth?: Property.BorderInlineEndWidth<TLength>;
  /**
   * In Mozilla applications like Firefox, the **`-moz-border-left-colors`** CSS property sets a list of colors for the left border.
   *
   * **Syntax**: `<color>+ | none`
   *
   * **Initial value**: `none`
   */
  MozBorderLeftColors?: Property.MozBorderLeftColors;
  /**
   * In Mozilla applications like Firefox, the **`-moz-border-right-colors`** CSS property sets a list of colors for the right border.
   *
   * **Syntax**: `<color>+ | none`
   *
   * **Initial value**: `none`
   */
  MozBorderRightColors?: Property.MozBorderRightColors;
  /**
   * The **`border-inline-start-color`** CSS property defines the color of the logical inline start border of an element, which maps to a physical border color depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-color`, `border-right-color`, `border-bottom-color`, or `border-left-color` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-color'>`
   *
   * **Initial value**: `currentcolor`
   */
  MozBorderStartColor?: Property.BorderInlineStartColor;
  /**
   * The **`border-inline-start-style`** CSS property defines the style of the logical inline start border of an element, which maps to a physical border style depending on the element's writing mode, directionality, and text orientation. It corresponds to the `border-top-style`, `border-right-style`, `border-bottom-style`, or `border-left-style` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'border-top-style'>`
   *
   * **Initial value**: `none`
   */
  MozBorderStartStyle?: Property.BorderInlineStartStyle;
  /**
   * In Mozilla applications like Firefox, the **`-moz-border-top-colors`** CSS property sets a list of colors for the top border.
   *
   * **Syntax**: `<color>+ | none`
   *
   * **Initial value**: `none`
   */
  MozBorderTopColors?: Property.MozBorderTopColors;
  /**
   * The **`box-sizing`** CSS property sets how the total width and height of an element is calculated.
   *
   * **Syntax**: `content-box | border-box`
   *
   * **Initial value**: `content-box`
   */
  MozBoxSizing?: Property.BoxSizing;
  /**
   * The **`column-count`** CSS property breaks an element's content into the specified number of columns.
   *
   * **Syntax**: `<integer> | auto`
   *
   * **Initial value**: `auto`
   */
  MozColumnCount?: Property.ColumnCount;
  /**
   * The **`column-fill`** CSS property controls how an element's contents are balanced when broken into columns.
   *
   * **Syntax**: `auto | balance | balance-all`
   *
   * **Initial value**: `balance`
   */
  MozColumnFill?: Property.ColumnFill;
  /**
   * The **`column-gap`** CSS property sets the size of the gap (gutter) between an element's columns.
   *
   * **Syntax**: `normal | <length-percentage>`
   *
   * **Initial value**: `normal`
   */
  MozColumnGap?: Property.ColumnGap<TLength>;
  /**
   * The **`column-rule-color`** CSS property sets the color of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   */
  MozColumnRuleColor?: Property.ColumnRuleColor;
  /**
   * The **`column-rule-style`** CSS property sets the style of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'border-style'>`
   *
   * **Initial value**: `none`
   */
  MozColumnRuleStyle?: Property.ColumnRuleStyle;
  /**
   * The **`column-rule-width`** CSS property sets the width of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'border-width'>`
   *
   * **Initial value**: `medium`
   */
  MozColumnRuleWidth?: Property.ColumnRuleWidth<TLength>;
  /**
   * The **`column-width`** CSS property sets the ideal column width in a multi-column layout. The container will have as many columns as can fit without any of them having a width less than the `column-width` value. If the width of the container is narrower than the specified value, the single column's width will be smaller than the declared column width.
   *
   * **Syntax**: `<length> | auto`
   *
   * **Initial value**: `auto`
   */
  MozColumnWidth?: Property.ColumnWidth<TLength>;
  /**
   * The `**-moz-context-properties**` property can be used within privileged contexts in Firefox to share the values of specified properties of the element with a child SVG image.
   *
   * **Syntax**: `none | [ fill | fill-opacity | stroke | stroke-opacity ]#`
   *
   * **Initial value**: `none`
   */
  MozContextProperties?: Property.MozContextProperties;
  /**
   * The **`font-feature-settings`** CSS property controls advanced typographic features in OpenType fonts.
   *
   * **Syntax**: `normal | <feature-tag-value>#`
   *
   * **Initial value**: `normal`
   */
  MozFontFeatureSettings?: Property.FontFeatureSettings;
  /**
   * The **`font-language-override`** CSS property controls the use of language-specific glyphs in a typeface.
   *
   * **Syntax**: `normal | <string>`
   *
   * **Initial value**: `normal`
   */
  MozFontLanguageOverride?: Property.FontLanguageOverride;
  /**
   * The **`hyphens`** CSS property specifies how words should be hyphenated when text wraps across multiple lines. It can prevent hyphenation entirely, hyphenate at manually-specified points within the text, or let the browser automatically insert hyphens where appropriate.
   *
   * **Syntax**: `none | manual | auto`
   *
   * **Initial value**: `manual`
   */
  MozHyphens?: Property.Hyphens;
  /**
   * For certain XUL elements and pseudo-elements that use an image from the `list-style-image` property, this property specifies a region of the image that is used in place of the whole image. This allows elements to use different pieces of the same image to improve performance.
   *
   * **Syntax**: `<shape> | auto`
   *
   * **Initial value**: `auto`
   */
  MozImageRegion?: Property.MozImageRegion;
  /**
   * The **`margin-inline-end`** CSS property defines the logical inline end margin of an element, which maps to a physical margin depending on the element's writing mode, directionality, and text orientation. In other words, it corresponds to the `margin-top`, `margin-right`, `margin-bottom` or `margin-left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'margin-left'>`
   *
   * **Initial value**: `0`
   */
  MozMarginEnd?: Property.MarginInlineEnd<TLength>;
  /**
   * The **`margin-inline-start`** CSS property defines the logical inline start margin of an element, which maps to a physical margin depending on the element's writing mode, directionality, and text orientation. It corresponds to the `margin-top`, `margin-right`, `margin-bottom`, or `margin-left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'margin-left'>`
   *
   * **Initial value**: `0`
   */
  MozMarginStart?: Property.MarginInlineStart<TLength>;
  /**
   * The **`-moz-orient`** CSS property specifies the orientation of the element to which it's applied.
   *
   * **Syntax**: `inline | block | horizontal | vertical`
   *
   * **Initial value**: `inline`
   */
  MozOrient?: Property.MozOrient;
  /**
   * The **`font-smooth`** CSS property controls the application of anti-aliasing when fonts are rendered.
   *
   * **Syntax**: `auto | never | always | <absolute-size> | <length>`
   *
   * **Initial value**: `auto`
   */
  MozOsxFontSmoothing?: Property.FontSmooth<TLength>;
  /**
   * The **`padding-inline-end`** CSS property defines the logical inline end padding of an element, which maps to a physical padding depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>`
   *
   * **Initial value**: `0`
   */
  MozPaddingEnd?: Property.PaddingInlineEnd<TLength>;
  /**
   * The **`padding-inline-start`** CSS property defines the logical inline start padding of an element, which maps to a physical padding depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>`
   *
   * **Initial value**: `0`
   */
  MozPaddingStart?: Property.PaddingInlineStart<TLength>;
  /**
   * The **`perspective`** CSS property determines the distance between the z=0 plane and the user in order to give a 3D-positioned element some perspective.
   *
   * **Syntax**: `none | <length>`
   *
   * **Initial value**: `none`
   */
  MozPerspective?: Property.Perspective<TLength>;
  /**
   * The **`perspective-origin`** CSS property determines the position at which the viewer is looking. It is used as the _vanishing point_ by the `perspective` property.
   *
   * **Syntax**: `<position>`
   *
   * **Initial value**: `50% 50%`
   */
  MozPerspectiveOrigin?: Property.PerspectiveOrigin<TLength>;
  /**
   * **`-moz-stack-sizing`** is an extended CSS property. Normally, a `<xul:stack>` will change its size so that all of its child elements are completely visible. For example, moving a child of the stack far to the right will widen the stack so the child remains visible.
   *
   * **Syntax**: `ignore | stretch-to-fit`
   *
   * **Initial value**: `stretch-to-fit`
   */
  MozStackSizing?: Property.MozStackSizing;
  /**
   * The **`tab-size`** CSS property is used to customize the width of tab characters (U+0009).
   *
   * **Syntax**: `<integer> | <length>`
   *
   * **Initial value**: `8`
   */
  MozTabSize?: Property.TabSize<TLength>;
  /**
   * The **`-moz-text-blink`** non-standard Mozilla CSS extension specifies the blink mode.
   *
   * **Syntax**: `none | blink`
   *
   * **Initial value**: `none`
   */
  MozTextBlink?: Property.MozTextBlink;
  /**
   * The **`text-size-adjust`** CSS property controls the text inflation algorithm used on some smartphones and tablets. Other browsers will ignore this property.
   *
   * **Syntax**: `none | auto | <percentage>`
   *
   * **Initial value**: `auto` for smartphone browsers supporting inflation, `none` in other cases (and then not modifiable).
   */
  MozTextSizeAdjust?: Property.TextSizeAdjust;
  /**
   * The **`transform-origin`** CSS property sets the origin for an element's transformations.
   *
   * **Syntax**: `[ <length-percentage> | left | center | right | top | bottom ] | [ [ <length-percentage> | left | center | right ] && [ <length-percentage> | top | center | bottom ] ] <length>?`
   *
   * **Initial value**: `50% 50% 0`
   */
  MozTransformOrigin?: Property.TransformOrigin<TLength>;
  /**
   * The **`transform-style`** CSS property sets whether children of an element are positioned in the 3D space or are flattened in the plane of the element.
   *
   * **Syntax**: `flat | preserve-3d`
   *
   * **Initial value**: `flat`
   */
  MozTransformStyle?: Property.TransformStyle;
  /**
   * The **`transition-delay`** CSS property specifies the duration to wait before starting a property's transition effect when its value changes.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  MozTransitionDelay?: Property.TransitionDelay<TTime>;
  /**
   * The **`transition-duration`** CSS property sets the length of time a transition animation should take to complete. By default, the value is `0s`, meaning that no animation will occur.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  MozTransitionDuration?: Property.TransitionDuration<TTime>;
  /**
   * The **`transition-property`** CSS property sets the CSS properties to which a transition effect should be applied.
   *
   * **Syntax**: `none | <single-transition-property>#`
   *
   * **Initial value**: all
   */
  MozTransitionProperty?: Property.TransitionProperty;
  /**
   * The **`transition-timing-function`** CSS property sets how intermediate values are calculated for CSS properties being affected by a transition effect.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   */
  MozTransitionTimingFunction?: Property.TransitionTimingFunction;
  /**
   * The **`-moz-user-focus`** CSS property is used to indicate whether an element can have the focus.
   *
   * **Syntax**: `ignore | normal | select-after | select-before | select-menu | select-same | select-all | none`
   *
   * **Initial value**: `none`
   */
  MozUserFocus?: Property.MozUserFocus;
  /**
   * The **`user-modify`** property has no effect in Firefox. It was originally planned to determine whether or not the content of an element can be edited by a user.
   *
   * **Syntax**: `read-only | read-write | write-only`
   *
   * **Initial value**: `read-only`
   */
  MozUserModify?: Property.MozUserModify;
  /**
   * The `**user-select**` CSS property controls whether the user can select text. This doesn't have any effect on content loaded as chrome, except in textboxes.
   *
   * **Syntax**: `auto | text | none | contain | all`
   *
   * **Initial value**: `auto`
   */
  MozUserSelect?: Property.UserSelect;
  /**
   * The **`-moz-window-dragging`** CSS property specifies whether a window is draggable or not. It only works in Chrome code, and only on Mac OS X.
   *
   * **Syntax**: `drag | no-drag`
   *
   * **Initial value**: `drag`
   */
  MozWindowDragging?: Property.MozWindowDragging;
  /**
   * The **`-moz-window-shadow`** CSS property specifies whether a window will have a shadow. It only works on Mac OS X.
   *
   * **Syntax**: `default | menu | tooltip | sheet | none`
   *
   * **Initial value**: `default`
   */
  MozWindowShadow?: Property.MozWindowShadow;
  /**
   * The **`-ms-accelerator`** CSS property is a Microsoft extension that sets or retrieves a string indicating whether the object represents a keyboard shortcut.
   *
   * **Syntax**: `false | true`
   *
   * **Initial value**: `false`
   */
  msAccelerator?: Property.MsAccelerator;
  /**
   * The **`align-self`** CSS property overrides a grid or flex item's `align-items` value. In Grid, it aligns the item inside the grid area. In Flexbox, it aligns the item on the cross axis.
   *
   * **Syntax**: `auto | normal | stretch | <baseline-position> | <overflow-position>? <self-position>`
   *
   * **Initial value**: `auto`
   */
  msAlignSelf?: Property.AlignSelf;
  /**
   * The **`-ms-block-progression`** CSS property is a Microsoft extension that specifies the block progression and layout orientation.
   *
   * **Syntax**: `tb | rl | bt | lr`
   *
   * **Initial value**: `tb`
   */
  msBlockProgression?: Property.MsBlockProgression;
  /**
   * The **`-ms-content-zoom-chaining`** CSS property is a Microsoft extension specifying the zoom behavior that occurs when a user hits the zoom limit during page manipulation.
   *
   * **Syntax**: `none | chained`
   *
   * **Initial value**: `none`
   */
  msContentZoomChaining?: Property.MsContentZoomChaining;
  /**
   * The **`-ms-content-zoom-limit-max`** CSS property is a Microsoft extension that specifies the selected elements' maximum zoom factor.
   *
   * **Syntax**: `<percentage>`
   *
   * **Initial value**: `400%`
   */
  msContentZoomLimitMax?: Property.MsContentZoomLimitMax;
  /**
   * The **`-ms-content-zoom-limit-min`** CSS property is a Microsoft extension that specifies the minimum zoom factor.
   *
   * **Syntax**: `<percentage>`
   *
   * **Initial value**: `100%`
   */
  msContentZoomLimitMin?: Property.MsContentZoomLimitMin;
  /**
   * The **`-ms-content-zoom-snap-points`** CSS property is a Microsoft extension that specifies where zoom snap-points are located.
   *
   * **Syntax**: `snapInterval( <percentage>, <percentage> ) | snapList( <percentage># )`
   *
   * **Initial value**: `snapInterval(0%, 100%)`
   */
  msContentZoomSnapPoints?: Property.MsContentZoomSnapPoints;
  /**
   * The **`-ms-content-zoom-snap-type`** CSS property is a Microsoft extension that specifies how zooming is affected by defined snap-points.
   *
   * **Syntax**: `none | proximity | mandatory`
   *
   * **Initial value**: `none`
   */
  msContentZoomSnapType?: Property.MsContentZoomSnapType;
  /**
   * The **`-ms-content-zooming`** CSS property is a Microsoft extension that specifies whether zooming is enabled.
   *
   * **Syntax**: `none | zoom`
   *
   * **Initial value**: zoom for the top level element, none for all other elements
   */
  msContentZooming?: Property.MsContentZooming;
  /**
   * The `-ms-filter` CSS property is a Microsoft extension that sets or retrieves the filter or collection of filters applied to an object.
   *
   * **Syntax**: `<string>`
   *
   * **Initial value**: "" (the empty string)
   */
  msFilter?: Property.MsFilter;
  /**
   * The **`flex-direction`** CSS property sets how flex items are placed in the flex container defining the main axis and the direction (normal or reversed).
   *
   * **Syntax**: `row | row-reverse | column | column-reverse`
   *
   * **Initial value**: `row`
   */
  msFlexDirection?: Property.FlexDirection;
  /**
   * The **`flex-grow`** CSS property sets the flex grow factor of a flex item main size.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `0`
   */
  msFlexPositive?: Property.FlexGrow;
  /**
   * The **`-ms-flow-from`** CSS property is a Microsoft extension that gets or sets a value identifying a region container in the document that accepts the content flow from the data source.
   *
   * **Syntax**: `[ none | <custom-ident> ]#`
   *
   * **Initial value**: `none`
   */
  msFlowFrom?: Property.MsFlowFrom;
  /**
   * The **`-ms-flow-into`** CSS property is a Microsoft extension that gets or sets a value identifying an iframe container in the document that serves as the region's data source.
   *
   * **Syntax**: `[ none | <custom-ident> ]#`
   *
   * **Initial value**: `none`
   */
  msFlowInto?: Property.MsFlowInto;
  /**
   * The **`grid-template-columns`** CSS property defines the line names and track sizing functions of the grid columns.
   *
   * **Syntax**: `none | <track-list> | <auto-track-list>`
   *
   * **Initial value**: `none`
   */
  msGridColumns?: Property.MsGridColumns<TLength>;
  /**
   * The **`grid-template-rows`** CSS property defines the line names and track sizing functions of the grid rows.
   *
   * **Syntax**: `none | <track-list> | <auto-track-list>`
   *
   * **Initial value**: `none`
   */
  msGridRows?: Property.MsGridRows<TLength>;
  /**
   * The **`-ms-high-contrast-adjust`** CSS property is a Microsoft extension that gets or sets a value indicating whether to override any CSS properties that would have been set in high contrast mode.
   *
   * **Syntax**: `auto | none`
   *
   * **Initial value**: `auto`
   */
  msHighContrastAdjust?: Property.MsHighContrastAdjust;
  /**
   * The **`-ms-hyphenate-limit-chars`** CSS property is a Microsoft extension that specifies one to three values indicating the minimum number of characters in a hyphenated word. If the word does not meet the required minimum number of characters in the word, before the hyphen, or after the hyphen, then the word is not hyphenated.
   *
   * **Syntax**: `auto | <integer>{1,3}`
   *
   * **Initial value**: `auto`
   */
  msHyphenateLimitChars?: Property.MsHyphenateLimitChars;
  /**
   * The **`-ms-hyphenate-limit-lines`** CSS property is a Microsoft extension specifying the maximum number of consecutive lines in an element that may be ended with a hyphenated word.
   *
   * **Syntax**: `no-limit | <integer>`
   *
   * **Initial value**: `no-limit`
   */
  msHyphenateLimitLines?: Property.MsHyphenateLimitLines;
  /**
   * The `**-ms-hyphenate-limit-zone**` CSS property is a Microsoft extension specifying the width of the hyphenation zone.
   *
   * **Syntax**: `<percentage> | <length>`
   *
   * **Initial value**: `0`
   */
  msHyphenateLimitZone?: Property.MsHyphenateLimitZone<TLength>;
  /**
   * The **`hyphens`** CSS property specifies how words should be hyphenated when text wraps across multiple lines. It can prevent hyphenation entirely, hyphenate at manually-specified points within the text, or let the browser automatically insert hyphens where appropriate.
   *
   * **Syntax**: `none | manual | auto`
   *
   * **Initial value**: `manual`
   */
  msHyphens?: Property.Hyphens;
  /**
   * The **`-ms-ime-align`** CSS property is a Microsoft extension aligning the Input Method Editor (IME) candidate window box relative to the element on which the IME composition is active. The extension is implemented in Microsoft Edge and Internet Explorer 11.
   *
   * **Syntax**: `auto | after`
   *
   * **Initial value**: `auto`
   */
  msImeAlign?: Property.MsImeAlign;
  /**
   * The CSS **`justify-self`** property sets the way a box is justified inside its alignment container along the appropriate axis.
   *
   * **Syntax**: `auto | normal | stretch | <baseline-position> | <overflow-position>? [ <self-position> | left | right ]`
   *
   * **Initial value**: `auto`
   */
  msJustifySelf?: Property.JustifySelf;
  /**
   * The **`line-break`** CSS property sets how to break lines of Chinese, Japanese, or Korean (CJK) text when working with punctuation and symbols.
   *
   * **Syntax**: `auto | loose | normal | strict | anywhere`
   *
   * **Initial value**: `auto`
   */
  msLineBreak?: Property.LineBreak;
  /**
   * The **`order`** CSS property sets the order to lay out an item in a flex or grid container. Items in a container are sorted by ascending `order` value and then by their source code order.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `0`
   */
  msOrder?: Property.Order;
  /**
   * The **`-ms-overflow-style`** CSS property is a Microsoft extension controlling the behavior of scrollbars when the content of an element overflows.
   *
   * **Syntax**: `auto | none | scrollbar | -ms-autohiding-scrollbar`
   *
   * **Initial value**: `auto`
   */
  msOverflowStyle?: Property.MsOverflowStyle;
  /**
   * The **`overflow-x`** CSS property sets what shows when content overflows a block-level element's left and right edges. This may be nothing, a scroll bar, or the overflow content.
   *
   * **Syntax**: `visible | hidden | clip | scroll | auto`
   *
   * **Initial value**: `visible`
   */
  msOverflowX?: Property.OverflowX;
  /**
   * The **`overflow-y`** CSS property sets what shows when content overflows a block-level element's top and bottom edges. This may be nothing, a scroll bar, or the overflow content.
   *
   * **Syntax**: `visible | hidden | clip | scroll | auto`
   *
   * **Initial value**: `visible`
   */
  msOverflowY?: Property.OverflowY;
  /**
   * The `**-ms-scroll-chaining**` CSS property is a Microsoft extension that specifies the scrolling behavior that occurs when a user hits the scroll limit during a manipulation.
   *
   * **Syntax**: `chained | none`
   *
   * **Initial value**: `chained`
   */
  msScrollChaining?: Property.MsScrollChaining;
  /**
   * The `**-ms-scroll-limit-x-max**` CSS property is a Microsoft extension that specifies the maximum value for the `Element.scrollLeft` property.
   *
   * **Syntax**: `auto | <length>`
   *
   * **Initial value**: `auto`
   */
  msScrollLimitXMax?: Property.MsScrollLimitXMax<TLength>;
  /**
   * The **`-ms-scroll-limit-x-min`** CSS property is a Microsoft extension that specifies the minimum value for the `Element.scrollLeft` property.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   */
  msScrollLimitXMin?: Property.MsScrollLimitXMin<TLength>;
  /**
   * The **`-ms-scroll-limit-y-max`** CSS property is a Microsoft extension that specifies the maximum value for the `Element.scrollTop` property.
   *
   * **Syntax**: `auto | <length>`
   *
   * **Initial value**: `auto`
   */
  msScrollLimitYMax?: Property.MsScrollLimitYMax<TLength>;
  /**
   * The **`-ms-scroll-limit-y-min`** CSS property is a Microsoft extension that specifies the minimum value for the `Element.scrollTop` property.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   */
  msScrollLimitYMin?: Property.MsScrollLimitYMin<TLength>;
  /**
   * The **`-ms-scroll-rails`** CSS property is a Microsoft extension that specifies whether scrolling locks to the primary axis of motion.
   *
   * **Syntax**: `none | railed`
   *
   * **Initial value**: `railed`
   */
  msScrollRails?: Property.MsScrollRails;
  /**
   * The **`-ms-scroll-snap-points-x`** CSS property is a Microsoft extension that specifies where snap-points will be located along the x-axis.
   *
   * **Syntax**: `snapInterval( <length-percentage>, <length-percentage> ) | snapList( <length-percentage># )`
   *
   * **Initial value**: `snapInterval(0px, 100%)`
   */
  msScrollSnapPointsX?: Property.MsScrollSnapPointsX;
  /**
   * The **`-ms-scroll-snap-points-y`** CSS property is a Microsoft extension that specifies where snap-points will be located along the y-axis.
   *
   * **Syntax**: `snapInterval( <length-percentage>, <length-percentage> ) | snapList( <length-percentage># )`
   *
   * **Initial value**: `snapInterval(0px, 100%)`
   */
  msScrollSnapPointsY?: Property.MsScrollSnapPointsY;
  /**
   * The **`scroll-snap-type`** CSS property sets how strictly snap points are enforced on the scroll container in case there is one.
   *
   * **Syntax**: `none | proximity | mandatory`
   *
   * **Initial value**: `none`
   */
  msScrollSnapType?: Property.MsScrollSnapType;
  /**
   * The **`-ms-scroll-translation`** CSS property is a Microsoft extension that specifies whether vertical-to-horizontal scroll wheel translation occurs on the specified element.
   *
   * **Syntax**: `none | vertical-to-horizontal`
   *
   * **Initial value**: `none`
   */
  msScrollTranslation?: Property.MsScrollTranslation;
  /**
   * The **`-ms-scrollbar-3dlight-color`** CSS property is a Microsoft extension specifying the color of the top and left edges of the scroll box and scroll arrows of a scroll bar.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: depends on user agent
   */
  msScrollbar3dlightColor?: Property.MsScrollbar3dlightColor;
  /**
   * The **`-ms-scrollbar-arrow-color`** CSS property is a Microsoft extension that specifies the color of the arrow elements of a scroll arrow.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `ButtonText`
   */
  msScrollbarArrowColor?: Property.MsScrollbarArrowColor;
  /**
   * The `**-ms-scrollbar-base-color**` CSS property is a Microsoft extension that specifies the base color of the main elements of a scroll bar.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: depends on user agent
   */
  msScrollbarBaseColor?: Property.MsScrollbarBaseColor;
  /**
   * The **`-ms-scrollbar-darkshadow-color`** CSS property is a Microsoft extension that specifies the color of a scroll bar's gutter.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `ThreeDDarkShadow`
   */
  msScrollbarDarkshadowColor?: Property.MsScrollbarDarkshadowColor;
  /**
   * The `**-ms-scrollbar-face-color**` CSS property is a Microsoft extension that specifies the color of the scroll box and scroll arrows of a scroll bar.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `ThreeDFace`
   */
  msScrollbarFaceColor?: Property.MsScrollbarFaceColor;
  /**
   * The `**-ms-scrollbar-highlight-color**` CSS property is a Microsoft extension that specifies the color of the slider tray, the top and left edges of the scroll box, and the scroll arrows of a scroll bar.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `ThreeDHighlight`
   */
  msScrollbarHighlightColor?: Property.MsScrollbarHighlightColor;
  /**
   * The **`-ms-scrollbar-shadow-color`** CSS property is a Microsoft extension that specifies the color of the bottom and right edges of the scroll box and scroll arrows of a scroll bar.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `ThreeDDarkShadow`
   */
  msScrollbarShadowColor?: Property.MsScrollbarShadowColor;
  /**
   * The **`-ms-text-autospace`** CSS property is a Microsoft extension that specifies the autospacing and narrow space width adjustment of text.
   *
   * **Syntax**: `none | ideograph-alpha | ideograph-numeric | ideograph-parenthesis | ideograph-space`
   *
   * **Initial value**: `none`
   */
  msTextAutospace?: Property.MsTextAutospace;
  /**
   * The **`text-combine-upright`** CSS property sets the combination of characters into the space of a single character. If the combined text is wider than 1em, the user agent must fit the contents within 1em. The resulting composition is treated as a single upright glyph for layout and decoration. This property only has an effect in vertical writing modes.
   *
   * **Syntax**: `none | all | [ digits <integer>? ]`
   *
   * **Initial value**: `none`
   */
  msTextCombineHorizontal?: Property.TextCombineUpright;
  /**
   * The **`text-overflow`** CSS property sets how hidden overflow content is signaled to users. It can be clipped, display an ellipsis ('`…`'), or display a custom string.
   *
   * **Syntax**: `[ clip | ellipsis | <string> ]{1,2}`
   *
   * **Initial value**: `clip`
   */
  msTextOverflow?: Property.TextOverflow;
  /**
   * The **`touch-action`** CSS property sets how an element's region can be manipulated by a touchscreen user (for example, by zooming features built into the browser).
   *
   * **Syntax**: `auto | none | [ [ pan-x | pan-left | pan-right ] || [ pan-y | pan-up | pan-down ] || pinch-zoom ] | manipulation`
   *
   * **Initial value**: `auto`
   */
  msTouchAction?: Property.TouchAction;
  /**
   * The **`-ms-touch-select`** CSS property is a Microsoft extension that toggles the gripper visual elements that enable touch text selection.
   *
   * **Syntax**: `grippers | none`
   *
   * **Initial value**: `grippers`
   */
  msTouchSelect?: Property.MsTouchSelect;
  /**
   * The **`transform`** CSS property lets you rotate, scale, skew, or translate an element. It modifies the coordinate space of the CSS visual formatting model.
   *
   * **Syntax**: `none | <transform-list>`
   *
   * **Initial value**: `none`
   */
  msTransform?: Property.Transform;
  /**
   * The **`transform-origin`** CSS property sets the origin for an element's transformations.
   *
   * **Syntax**: `[ <length-percentage> | left | center | right | top | bottom ] | [ [ <length-percentage> | left | center | right ] && [ <length-percentage> | top | center | bottom ] ] <length>?`
   *
   * **Initial value**: `50% 50% 0`
   */
  msTransformOrigin?: Property.TransformOrigin<TLength>;
  /**
   * The **`transition-delay`** CSS property specifies the duration to wait before starting a property's transition effect when its value changes.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  msTransitionDelay?: Property.TransitionDelay<TTime>;
  /**
   * The **`transition-duration`** CSS property sets the length of time a transition animation should take to complete. By default, the value is `0s`, meaning that no animation will occur.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  msTransitionDuration?: Property.TransitionDuration<TTime>;
  /**
   * The **`transition-property`** CSS property sets the CSS properties to which a transition effect should be applied.
   *
   * **Syntax**: `none | <single-transition-property>#`
   *
   * **Initial value**: all
   */
  msTransitionProperty?: Property.TransitionProperty;
  /**
   * The **`transition-timing-function`** CSS property sets how intermediate values are calculated for CSS properties being affected by a transition effect.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   */
  msTransitionTimingFunction?: Property.TransitionTimingFunction;
  /**
   * The `**user-select**` CSS property controls whether the user can select text. This doesn't have any effect on content loaded as chrome, except in textboxes.
   *
   * **Syntax**: `none | element | text`
   *
   * **Initial value**: `text`
   */
  msUserSelect?: Property.MsUserSelect;
  /**
   * The **`word-break`** CSS property sets whether line breaks appear wherever the text would otherwise overflow its content box.
   *
   * **Syntax**: `normal | break-all | keep-all | break-word`
   *
   * **Initial value**: `normal`
   */
  msWordBreak?: Property.WordBreak;
  /**
   * The **`-ms-wrap-flow`** CSS property is a Microsoft extension that specifies how exclusions impact inline content within block-level elements.
   *
   * **Syntax**: `auto | both | start | end | maximum | clear`
   *
   * **Initial value**: `auto`
   */
  msWrapFlow?: Property.MsWrapFlow;
  /**
   * The **`-ms-wrap-margin`** CSS property is a Microsoft extension that specifies a margin that offsets the inner wrap shape from other shapes.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   */
  msWrapMargin?: Property.MsWrapMargin<TLength>;
  /**
   * The **`-ms-wrap-through`** CSS property is a Microsoft extension that specifies how content should wrap around an exclusion element.
   *
   * **Syntax**: `wrap | none`
   *
   * **Initial value**: `wrap`
   */
  msWrapThrough?: Property.MsWrapThrough;
  /**
   * The **`writing-mode`** CSS property sets whether lines of text are laid out horizontally or vertically, as well as the direction in which blocks progress. When set for an entire document, it should be set on the root element (`html` element for HTML documents).
   *
   * **Syntax**: `horizontal-tb | vertical-rl | vertical-lr | sideways-rl | sideways-lr`
   *
   * **Initial value**: `horizontal-tb`
   */
  msWritingMode?: Property.WritingMode;
  /**
   * The CSS **`align-content`** property sets the distribution of space between and around content items along a flexbox's cross-axis or a grid's block axis.
   *
   * **Syntax**: `normal | <baseline-position> | <content-distribution> | <overflow-position>? <content-position>`
   *
   * **Initial value**: `normal`
   */
  WebkitAlignContent?: Property.AlignContent;
  /**
   * The CSS **`align-items`** property sets the `align-self` value on all direct children as a group. In Flexbox, it controls the alignment of items on the Cross Axis. In Grid Layout, it controls the alignment of items on the Block Axis within their grid area.
   *
   * **Syntax**: `normal | stretch | <baseline-position> | [ <overflow-position>? <self-position> ]`
   *
   * **Initial value**: `normal`
   */
  WebkitAlignItems?: Property.AlignItems;
  /**
   * The **`align-self`** CSS property overrides a grid or flex item's `align-items` value. In Grid, it aligns the item inside the grid area. In Flexbox, it aligns the item on the cross axis.
   *
   * **Syntax**: `auto | normal | stretch | <baseline-position> | <overflow-position>? <self-position>`
   *
   * **Initial value**: `auto`
   */
  WebkitAlignSelf?: Property.AlignSelf;
  /**
   * The **`animation-delay`** CSS property specifies the amount of time to wait from applying the animation to an element before beginning to perform the animation. The animation can start later, immediately from its beginning, or immediately and partway through the animation.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  WebkitAnimationDelay?: Property.AnimationDelay<TTime>;
  /**
   * The **`animation-direction`** CSS property sets whether an animation should play forward, backward, or alternate back and forth between playing the sequence forward and backward.
   *
   * **Syntax**: `<single-animation-direction>#`
   *
   * **Initial value**: `normal`
   */
  WebkitAnimationDirection?: Property.AnimationDirection;
  /**
   * The **`animation-duration`** CSS property sets the length of time that an animation takes to complete one cycle.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  WebkitAnimationDuration?: Property.AnimationDuration<TTime>;
  /**
   * The **`animation-fill-mode`** CSS property sets how a CSS animation applies styles to its target before and after its execution.
   *
   * **Syntax**: `<single-animation-fill-mode>#`
   *
   * **Initial value**: `none`
   */
  WebkitAnimationFillMode?: Property.AnimationFillMode;
  /**
   * The **`animation-iteration-count`** CSS property sets the number of times an animation sequence should be played before stopping.
   *
   * **Syntax**: `<single-animation-iteration-count>#`
   *
   * **Initial value**: `1`
   */
  WebkitAnimationIterationCount?: Property.AnimationIterationCount;
  /**
   * The **`animation-name`** CSS property specifies the names of one or more `@keyframes` at-rules describing the animation or animations to apply to the element.
   *
   * **Syntax**: `[ none | <keyframes-name> ]#`
   *
   * **Initial value**: `none`
   */
  WebkitAnimationName?: Property.AnimationName;
  /**
   * The **`animation-play-state`** CSS property sets whether an animation is running or paused.
   *
   * **Syntax**: `<single-animation-play-state>#`
   *
   * **Initial value**: `running`
   */
  WebkitAnimationPlayState?: Property.AnimationPlayState;
  /**
   * The **`animation-timing-function`** CSS property sets how an animation progresses through the duration of each cycle.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   */
  WebkitAnimationTimingFunction?: Property.AnimationTimingFunction;
  /**
   * The `**appearance**` CSS property is used to display an element using platform-native styling, based on the operating system's theme. The **`-moz-appearance`** and **`-webkit-appearance`** properties are non-standard versions of this property, used (respectively) by Gecko (Firefox) and by WebKit-based (e.g., Safari) and Blink-based (e.g., Chrome, Opera) browsers to achieve the same thing. Note that Firefox and Edge also support **`-webkit-appearance`**, for compatibility reasons.
   *
   * **Syntax**: `none | button | button-bevel | caret | checkbox | default-button | inner-spin-button | listbox | listitem | media-controls-background | media-controls-fullscreen-background | media-current-time-display | media-enter-fullscreen-button | media-exit-fullscreen-button | media-fullscreen-button | media-mute-button | media-overlay-play-button | media-play-button | media-seek-back-button | media-seek-forward-button | media-slider | media-sliderthumb | media-time-remaining-display | media-toggle-closed-captions-button | media-volume-slider | media-volume-slider-container | media-volume-sliderthumb | menulist | menulist-button | menulist-text | menulist-textfield | meter | progress-bar | progress-bar-value | push-button | radio | searchfield | searchfield-cancel-button | searchfield-decoration | searchfield-results-button | searchfield-results-decoration | slider-horizontal | slider-vertical | sliderthumb-horizontal | sliderthumb-vertical | square-button | textarea | textfield | -apple-pay-button`
   *
   * **Initial value**: `none` (but this value is overridden in the user agent CSS)
   */
  WebkitAppearance?: Property.WebkitAppearance;
  /**
   * The **`backdrop-filter`** CSS property lets you apply graphical effects such as blurring or color shifting to the area behind an element. Because it applies to everything _behind_ the element, to see the effect you must make the element or its background at least partially transparent.
   *
   * **Syntax**: `none | <filter-function-list>`
   *
   * **Initial value**: `none`
   */
  WebkitBackdropFilter?: Property.BackdropFilter;
  /**
   * The **`backface-visibility`** CSS property sets whether the back face of an element is visible when turned towards the user.
   *
   * **Syntax**: `visible | hidden`
   *
   * **Initial value**: `visible`
   */
  WebkitBackfaceVisibility?: Property.BackfaceVisibility;
  /**
   * The **`background-clip`** CSS property sets whether an element's background extends underneath its border box, padding box, or content box.
   *
   * **Syntax**: `<box>#`
   *
   * **Initial value**: `border-box`
   */
  WebkitBackgroundClip?: Property.BackgroundClip;
  /**
   * The **`background-origin`** CSS property sets the background's origin: from the border start, inside the border, or inside the padding.
   *
   * **Syntax**: `<box>#`
   *
   * **Initial value**: `padding-box`
   */
  WebkitBackgroundOrigin?: Property.BackgroundOrigin;
  /**
   * The **`background-size`** CSS property sets the size of the element's background image. The image can be left to its natural size, stretched, or constrained to fit the available space.
   *
   * **Syntax**: `<bg-size>#`
   *
   * **Initial value**: `auto auto`
   */
  WebkitBackgroundSize?: Property.BackgroundSize<TLength>;
  /**
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   */
  WebkitBorderBeforeColor?: Property.WebkitBorderBeforeColor;
  /**
   * **Syntax**: `<'border-style'>`
   *
   * **Initial value**: `none`
   */
  WebkitBorderBeforeStyle?: Property.WebkitBorderBeforeStyle;
  /**
   * **Syntax**: `<'border-width'>`
   *
   * **Initial value**: `medium`
   */
  WebkitBorderBeforeWidth?: Property.WebkitBorderBeforeWidth<TLength>;
  /**
   * The **`border-bottom-left-radius`** CSS property rounds the bottom-left corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   */
  WebkitBorderBottomLeftRadius?: Property.BorderBottomLeftRadius<TLength>;
  /**
   * The **`border-bottom-right-radius`** CSS property rounds the bottom-right corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   */
  WebkitBorderBottomRightRadius?: Property.BorderBottomRightRadius<TLength>;
  /**
   * The **`border-image-slice`** CSS property divides the image specified by `border-image-source` into regions. These regions form the components of an element's border image.
   *
   * **Syntax**: `<number-percentage>{1,4} && fill?`
   *
   * **Initial value**: `100%`
   */
  WebkitBorderImageSlice?: Property.BorderImageSlice;
  /**
   * The **`border-top-left-radius`** CSS property rounds the top-left corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   */
  WebkitBorderTopLeftRadius?: Property.BorderTopLeftRadius<TLength>;
  /**
   * The **`border-top-right-radius`** CSS property rounds the top-right corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   */
  WebkitBorderTopRightRadius?: Property.BorderTopRightRadius<TLength>;
  /**
   * The **`box-decoration-break`** CSS property specifies how an element's fragments should be rendered when broken across multiple lines, columns, or pages.
   *
   * **Syntax**: `slice | clone`
   *
   * **Initial value**: `slice`
   */
  WebkitBoxDecorationBreak?: Property.BoxDecorationBreak;
  /**
   * The **`-webkit-box-reflect`** CSS property lets you reflect the content of an element in one specific direction.
   *
   * **Syntax**: `[ above | below | right | left ]? <length>? <image>?`
   *
   * **Initial value**: `none`
   */
  WebkitBoxReflect?: Property.WebkitBoxReflect<TLength>;
  /**
   * The **`box-shadow`** CSS property adds shadow effects around an element's frame. You can set multiple effects separated by commas. A box shadow is described by X and Y offsets relative to the element, blur and spread radius, and color.
   *
   * **Syntax**: `none | <shadow>#`
   *
   * **Initial value**: `none`
   */
  WebkitBoxShadow?: Property.BoxShadow;
  /**
   * The **`box-sizing`** CSS property sets how the total width and height of an element is calculated.
   *
   * **Syntax**: `content-box | border-box`
   *
   * **Initial value**: `content-box`
   */
  WebkitBoxSizing?: Property.BoxSizing;
  /**
   * The `**clip-path**` CSS property creates a clipping region that sets what part of an element should be shown. Parts that are inside the region are shown, while those outside are hidden.
   *
   * **Syntax**: `<clip-source> | [ <basic-shape> || <geometry-box> ] | none`
   *
   * **Initial value**: `none`
   */
  WebkitClipPath?: Property.ClipPath;
  /**
   * The **`column-count`** CSS property breaks an element's content into the specified number of columns.
   *
   * **Syntax**: `<integer> | auto`
   *
   * **Initial value**: `auto`
   */
  WebkitColumnCount?: Property.ColumnCount;
  /**
   * The **`column-fill`** CSS property controls how an element's contents are balanced when broken into columns.
   *
   * **Syntax**: `auto | balance | balance-all`
   *
   * **Initial value**: `balance`
   */
  WebkitColumnFill?: Property.ColumnFill;
  /**
   * The **`column-gap`** CSS property sets the size of the gap (gutter) between an element's columns.
   *
   * **Syntax**: `normal | <length-percentage>`
   *
   * **Initial value**: `normal`
   */
  WebkitColumnGap?: Property.ColumnGap<TLength>;
  /**
   * The **`column-rule-color`** CSS property sets the color of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   */
  WebkitColumnRuleColor?: Property.ColumnRuleColor;
  /**
   * The **`column-rule-style`** CSS property sets the style of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'border-style'>`
   *
   * **Initial value**: `none`
   */
  WebkitColumnRuleStyle?: Property.ColumnRuleStyle;
  /**
   * The **`column-rule-width`** CSS property sets the width of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'border-width'>`
   *
   * **Initial value**: `medium`
   */
  WebkitColumnRuleWidth?: Property.ColumnRuleWidth<TLength>;
  /**
   * The **`column-span`** CSS property makes it possible for an element to span across all columns when its value is set to `all`.
   *
   * **Syntax**: `none | all`
   *
   * **Initial value**: `none`
   */
  WebkitColumnSpan?: Property.ColumnSpan;
  /**
   * The **`column-width`** CSS property sets the ideal column width in a multi-column layout. The container will have as many columns as can fit without any of them having a width less than the `column-width` value. If the width of the container is narrower than the specified value, the single column's width will be smaller than the declared column width.
   *
   * **Syntax**: `<length> | auto`
   *
   * **Initial value**: `auto`
   */
  WebkitColumnWidth?: Property.ColumnWidth<TLength>;
  /**
   * The **`filter`** CSS property applies graphical effects like blur or color shift to an element. Filters are commonly used to adjust the rendering of images, backgrounds, and borders.
   *
   * **Syntax**: `none | <filter-function-list>`
   *
   * **Initial value**: `none`
   */
  WebkitFilter?: Property.Filter;
  /**
   * The **`flex-basis`** CSS property sets the initial main size of a flex item. It sets the size of the content box unless otherwise set with `box-sizing`.
   *
   * **Syntax**: `content | <'width'>`
   *
   * **Initial value**: `auto`
   */
  WebkitFlexBasis?: Property.FlexBasis<TLength>;
  /**
   * The **`flex-direction`** CSS property sets how flex items are placed in the flex container defining the main axis and the direction (normal or reversed).
   *
   * **Syntax**: `row | row-reverse | column | column-reverse`
   *
   * **Initial value**: `row`
   */
  WebkitFlexDirection?: Property.FlexDirection;
  /**
   * The **`flex-grow`** CSS property sets the flex grow factor of a flex item main size.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `0`
   */
  WebkitFlexGrow?: Property.FlexGrow;
  /**
   * The **`flex-shrink`** CSS property sets the flex shrink factor of a flex item. If the size of all flex items is larger than the flex container, items shrink to fit according to `flex-shrink`.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `1`
   */
  WebkitFlexShrink?: Property.FlexShrink;
  /**
   * The **`flex-wrap`** CSS property sets whether flex items are forced onto one line or can wrap onto multiple lines. If wrapping is allowed, it sets the direction that lines are stacked.
   *
   * **Syntax**: `nowrap | wrap | wrap-reverse`
   *
   * **Initial value**: `nowrap`
   */
  WebkitFlexWrap?: Property.FlexWrap;
  /**
   * The **`font-feature-settings`** CSS property controls advanced typographic features in OpenType fonts.
   *
   * **Syntax**: `normal | <feature-tag-value>#`
   *
   * **Initial value**: `normal`
   */
  WebkitFontFeatureSettings?: Property.FontFeatureSettings;
  /**
   * The **`font-kerning`** CSS property sets the use of the kerning information stored in a font.
   *
   * **Syntax**: `auto | normal | none`
   *
   * **Initial value**: `auto`
   */
  WebkitFontKerning?: Property.FontKerning;
  /**
   * The **`font-smooth`** CSS property controls the application of anti-aliasing when fonts are rendered.
   *
   * **Syntax**: `auto | never | always | <absolute-size> | <length>`
   *
   * **Initial value**: `auto`
   */
  WebkitFontSmoothing?: Property.FontSmooth<TLength>;
  /**
   * The **`font-variant-ligatures`** CSS property controls which ligatures and contextual forms are used in textual content of the elements it applies to. This leads to more harmonized forms in the resulting text.
   *
   * **Syntax**: `normal | none | [ <common-lig-values> || <discretionary-lig-values> || <historical-lig-values> || <contextual-alt-values> ]`
   *
   * **Initial value**: `normal`
   */
  WebkitFontVariantLigatures?: Property.FontVariantLigatures;
  /**
   * The **`hyphens`** CSS property specifies how words should be hyphenated when text wraps across multiple lines. It can prevent hyphenation entirely, hyphenate at manually-specified points within the text, or let the browser automatically insert hyphens where appropriate.
   *
   * **Syntax**: `none | manual | auto`
   *
   * **Initial value**: `manual`
   */
  WebkitHyphens?: Property.Hyphens;
  /**
   * The CSS **`justify-content`** property defines how the browser distributes space between and around content items along the main-axis of a flex container, and the inline axis of a grid container.
   *
   * **Syntax**: `normal | <content-distribution> | <overflow-position>? [ <content-position> | left | right ]`
   *
   * **Initial value**: `normal`
   */
  WebkitJustifyContent?: Property.JustifyContent;
  /**
   * The **`line-break`** CSS property sets how to break lines of Chinese, Japanese, or Korean (CJK) text when working with punctuation and symbols.
   *
   * **Syntax**: `auto | loose | normal | strict | anywhere`
   *
   * **Initial value**: `auto`
   */
  WebkitLineBreak?: Property.LineBreak;
  /**
   * The **`-webkit-line-clamp`** CSS property allows limiting of the contents of a block container to the specified number of lines.
   *
   * **Syntax**: `none | <integer>`
   *
   * **Initial value**: `none`
   */
  WebkitLineClamp?: Property.WebkitLineClamp;
  /**
   * The **`margin-inline-end`** CSS property defines the logical inline end margin of an element, which maps to a physical margin depending on the element's writing mode, directionality, and text orientation. In other words, it corresponds to the `margin-top`, `margin-right`, `margin-bottom` or `margin-left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'margin-left'>`
   *
   * **Initial value**: `0`
   */
  WebkitMarginEnd?: Property.MarginInlineEnd<TLength>;
  /**
   * The **`margin-inline-start`** CSS property defines the logical inline start margin of an element, which maps to a physical margin depending on the element's writing mode, directionality, and text orientation. It corresponds to the `margin-top`, `margin-right`, `margin-bottom`, or `margin-left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'margin-left'>`
   *
   * **Initial value**: `0`
   */
  WebkitMarginStart?: Property.MarginInlineStart<TLength>;
  /**
   * If a `-webkit-mask-image` is specified, `-webkit-mask-attachment` determines whether the mask image's position is fixed within the viewport, or scrolls along with its containing block.
   *
   * **Syntax**: `<attachment>#`
   *
   * **Initial value**: `scroll`
   */
  WebkitMaskAttachment?: Property.WebkitMaskAttachment;
  /**
   * The **`mask-border-outset`** CSS property specifies the distance by which an element's mask border is set out from its border box.
   *
   * **Syntax**: `[ <length> | <number> ]{1,4}`
   *
   * **Initial value**: `0`
   */
  WebkitMaskBoxImageOutset?: Property.MaskBorderOutset<TLength>;
  /**
   * The **`mask-border-repeat`** CSS property sets how the edge regions of a source image are adjusted to fit the dimensions of an element's mask border.
   *
   * **Syntax**: `[ stretch | repeat | round | space ]{1,2}`
   *
   * **Initial value**: `stretch`
   */
  WebkitMaskBoxImageRepeat?: Property.MaskBorderRepeat;
  /**
   * The **`mask-border-slice`** CSS property divides the image set by `mask-border-source` into regions. These regions are used to form the components of an element's mask border.
   *
   * **Syntax**: `<number-percentage>{1,4} fill?`
   *
   * **Initial value**: `0`
   */
  WebkitMaskBoxImageSlice?: Property.MaskBorderSlice;
  /**
   * The **`mask-border-source`** CSS property sets the source image used to create an element's mask border.
   *
   * **Syntax**: `none | <image>`
   *
   * **Initial value**: `none`
   */
  WebkitMaskBoxImageSource?: Property.MaskBorderSource;
  /**
   * The **`mask-border-width`** CSS property sets the width of an element's mask border.
   *
   * **Syntax**: `[ <length-percentage> | <number> | auto ]{1,4}`
   *
   * **Initial value**: `auto`
   */
  WebkitMaskBoxImageWidth?: Property.MaskBorderWidth<TLength>;
  /**
   * The **`mask-clip`** CSS property determines the area which is affected by a mask. The painted content of an element must be restricted to this area.
   *
   * **Syntax**: `[ <box> | border | padding | content | text ]#`
   *
   * **Initial value**: `border`
   */
  WebkitMaskClip?: Property.WebkitMaskClip;
  /**
   * The **`-webkit-mask-composite`** property specifies the manner in which multiple mask images applied to the same element are composited with one another. Mask images are composited in the opposite order that they are declared with the `-webkit-mask-image` property.
   *
   * **Syntax**: `<composite-style>#`
   *
   * **Initial value**: `source-over`
   */
  WebkitMaskComposite?: Property.WebkitMaskComposite;
  /**
   * The **`mask-image`** CSS property sets the image that is used as mask layer for an element.
   *
   * **Syntax**: `<mask-reference>#`
   *
   * **Initial value**: `none`
   */
  WebkitMaskImage?: Property.WebkitMaskImage;
  /**
   * The **`mask-origin`** CSS property sets the origin of a mask.
   *
   * **Syntax**: `[ <box> | border | padding | content ]#`
   *
   * **Initial value**: `padding`
   */
  WebkitMaskOrigin?: Property.WebkitMaskOrigin;
  /**
   * The **`mask-position`** CSS property sets the initial position, relative to the mask position layer set by `mask-origin`, for each defined mask image.
   *
   * **Syntax**: `<position>#`
   *
   * **Initial value**: `0% 0%`
   */
  WebkitMaskPosition?: Property.WebkitMaskPosition<TLength>;
  /**
   * The `-webkit-mask-position-x` CSS property sets the initial horizontal position of a mask image.
   *
   * **Syntax**: `[ <length-percentage> | left | center | right ]#`
   *
   * **Initial value**: `0%`
   */
  WebkitMaskPositionX?: Property.WebkitMaskPositionX<TLength>;
  /**
   * The `-webkit-mask-position-y` CSS property sets the initial vertical position of a mask image.
   *
   * **Syntax**: `[ <length-percentage> | top | center | bottom ]#`
   *
   * **Initial value**: `0%`
   */
  WebkitMaskPositionY?: Property.WebkitMaskPositionY<TLength>;
  /**
   * The **`mask-repeat`** CSS property sets how mask images are repeated. A mask image can be repeated along the horizontal axis, the vertical axis, both axes, or not repeated at all.
   *
   * **Syntax**: `<repeat-style>#`
   *
   * **Initial value**: `repeat`
   */
  WebkitMaskRepeat?: Property.WebkitMaskRepeat;
  /**
   * The `-webkit-mask-repeat-x` property specifies whether and how a mask image is repeated (tiled) horizontally.
   *
   * **Syntax**: `repeat | no-repeat | space | round`
   *
   * **Initial value**: `repeat`
   */
  WebkitMaskRepeatX?: Property.WebkitMaskRepeatX;
  /**
   * The `-webkit-mask-repeat-y` property sets whether and how a mask image is repeated (tiled) vertically.
   *
   * **Syntax**: `repeat | no-repeat | space | round`
   *
   * **Initial value**: `repeat`
   */
  WebkitMaskRepeatY?: Property.WebkitMaskRepeatY;
  /**
   * The **`mask-size`** CSS property specifies the sizes of the mask images. The size of the image can be fully or partially constrained in order to preserve its intrinsic ratio.
   *
   * **Syntax**: `<bg-size>#`
   *
   * **Initial value**: `auto auto`
   */
  WebkitMaskSize?: Property.WebkitMaskSize<TLength>;
  /**
   * The **`max-inline-size`** CSS property defines the horizontal or vertical maximum size of an element's block, depending on its writing mode. It corresponds to either the `max-width` or the `max-height` property, depending on the value of `writing-mode`.
   *
   * **Syntax**: `<'max-width'>`
   *
   * **Initial value**: `0`
   */
  WebkitMaxInlineSize?: Property.MaxInlineSize<TLength>;
  /**
   * The **`order`** CSS property sets the order to lay out an item in a flex or grid container. Items in a container are sorted by ascending `order` value and then by their source code order.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `0`
   */
  WebkitOrder?: Property.Order;
  /**
   * The `-webkit-overflow-scrolling` CSS property controls whether or not touch devices use momentum-based scrolling for a given element.
   *
   * **Syntax**: `auto | touch`
   *
   * **Initial value**: `auto`
   */
  WebkitOverflowScrolling?: Property.WebkitOverflowScrolling;
  /**
   * The **`padding-inline-end`** CSS property defines the logical inline end padding of an element, which maps to a physical padding depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>`
   *
   * **Initial value**: `0`
   */
  WebkitPaddingEnd?: Property.PaddingInlineEnd<TLength>;
  /**
   * The **`padding-inline-start`** CSS property defines the logical inline start padding of an element, which maps to a physical padding depending on the element's writing mode, directionality, and text orientation.
   *
   * **Syntax**: `<'padding-left'>`
   *
   * **Initial value**: `0`
   */
  WebkitPaddingStart?: Property.PaddingInlineStart<TLength>;
  /**
   * The **`perspective`** CSS property determines the distance between the z=0 plane and the user in order to give a 3D-positioned element some perspective.
   *
   * **Syntax**: `none | <length>`
   *
   * **Initial value**: `none`
   */
  WebkitPerspective?: Property.Perspective<TLength>;
  /**
   * The **`perspective-origin`** CSS property determines the position at which the viewer is looking. It is used as the _vanishing point_ by the `perspective` property.
   *
   * **Syntax**: `<position>`
   *
   * **Initial value**: `50% 50%`
   */
  WebkitPerspectiveOrigin?: Property.PerspectiveOrigin<TLength>;
  /**
   * The **`color-adjust`** CSS property sets what, if anything, the user agent may do to optimize the appearance of the element on the output device. By default, the browser is allowed to make any adjustments to the element's appearance it determines to be necessary and prudent given the type and capabilities of the output device.
   *
   * **Syntax**: `economy | exact`
   *
   * **Initial value**: `economy`
   */
  WebkitPrintColorAdjust?: Property.ColorAdjust;
  /**
   * The `**ruby-position**` CSS property defines the position of a ruby element relatives to its base element. It can be position over the element (`over`), under it (`under`), or between the characters, on their right side (`inter-character`).
   *
   * **Syntax**: `[ alternate || [ over | under ] ] | inter-character`
   *
   * **Initial value**: `alternate`
   */
  WebkitRubyPosition?: Property.RubyPosition;
  /**
   * The **`scroll-snap-type`** CSS property sets how strictly snap points are enforced on the scroll container in case there is one.
   *
   * **Syntax**: `none | [ x | y | block | inline | both ] [ mandatory | proximity ]?`
   *
   * **Initial value**: `none`
   */
  WebkitScrollSnapType?: Property.ScrollSnapType;
  /**
   * The **`shape-margin`** CSS property sets a margin for a CSS shape created using `shape-outside`.
   *
   * **Syntax**: `<length-percentage>`
   *
   * **Initial value**: `0`
   */
  WebkitShapeMargin?: Property.ShapeMargin<TLength>;
  /**
   * **`-webkit-tap-highlight-color`** is a non-standard CSS property that sets the color of the highlight that appears over a link while it's being tapped. The highlighting indicates to the user that their tap is being successfully recognized, and indicates which element they're tapping on.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `black`
   */
  WebkitTapHighlightColor?: Property.WebkitTapHighlightColor;
  /**
   * The **`text-combine-upright`** CSS property sets the combination of characters into the space of a single character. If the combined text is wider than 1em, the user agent must fit the contents within 1em. The resulting composition is treated as a single upright glyph for layout and decoration. This property only has an effect in vertical writing modes.
   *
   * **Syntax**: `none | all | [ digits <integer>? ]`
   *
   * **Initial value**: `none`
   */
  WebkitTextCombine?: Property.TextCombineUpright;
  /**
   * The **`text-decoration-color`** CSS property sets the color of decorations added to text by `text-decoration-line`.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   */
  WebkitTextDecorationColor?: Property.TextDecorationColor;
  /**
   * The **`text-decoration-line`** CSS property sets the kind of decoration that is used on text in an element, such as an underline or overline.
   *
   * **Syntax**: `none | [ underline || overline || line-through || blink ] | spelling-error | grammar-error`
   *
   * **Initial value**: `none`
   */
  WebkitTextDecorationLine?: Property.TextDecorationLine;
  /**
   * The **`text-decoration-skip`** CSS property sets what parts of an element’s content any text decoration affecting the element must skip over. It controls all text decoration lines drawn by the element and also any text decoration lines drawn by its ancestors.
   *
   * **Syntax**: `none | [ objects || [ spaces | [ leading-spaces || trailing-spaces ] ] || edges || box-decoration ]`
   *
   * **Initial value**: `objects`
   */
  WebkitTextDecorationSkip?: Property.TextDecorationSkip;
  /**
   * The **`text-decoration-style`** CSS property sets the style of the lines specified by `text-decoration-line`. The style applies to all lines that are set with `text-decoration-line`.
   *
   * **Syntax**: `solid | double | dotted | dashed | wavy`
   *
   * **Initial value**: `solid`
   */
  WebkitTextDecorationStyle?: Property.TextDecorationStyle;
  /**
   * The **`text-emphasis-color`** CSS property sets the color of emphasis marks. This value can also be set using the `text-emphasis` shorthand.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   */
  WebkitTextEmphasisColor?: Property.TextEmphasisColor;
  /**
   * The **`text-emphasis-position`** CSS property sets where emphasis marks are drawn. Like ruby text, if there isn't enough room for emphasis marks, the line height is increased.
   *
   * **Syntax**: `[ over | under ] && [ right | left ]`
   *
   * **Initial value**: `over right`
   */
  WebkitTextEmphasisPosition?: Property.TextEmphasisPosition;
  /**
   * The **`text-emphasis-style`** CSS property sets the appearance of emphasis marks. It can also be set, and reset, using the `text-emphasis` shorthand.
   *
   * **Syntax**: `none | [ [ filled | open ] || [ dot | circle | double-circle | triangle | sesame ] ] | <string>`
   *
   * **Initial value**: `none`
   */
  WebkitTextEmphasisStyle?: Property.TextEmphasisStyle;
  /**
   * The **`-webkit-text-fill-color`** CSS property specifies the fill color of characters of text. If this property is not set, the value of the `color` property is used.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   */
  WebkitTextFillColor?: Property.WebkitTextFillColor;
  /**
   * The **`text-orientation`** CSS property sets the orientation of the text characters in a line. It only affects text in vertical mode (when `writing-mode` is not `horizontal-tb`). It is useful for controlling the display of languages that use vertical script, and also for making vertical table headers.
   *
   * **Syntax**: `mixed | upright | sideways`
   *
   * **Initial value**: `mixed`
   */
  WebkitTextOrientation?: Property.TextOrientation;
  /**
   * The **`text-size-adjust`** CSS property controls the text inflation algorithm used on some smartphones and tablets. Other browsers will ignore this property.
   *
   * **Syntax**: `none | auto | <percentage>`
   *
   * **Initial value**: `auto` for smartphone browsers supporting inflation, `none` in other cases (and then not modifiable).
   */
  WebkitTextSizeAdjust?: Property.TextSizeAdjust;
  /**
   * The **`-webkit-text-stroke-color`** CSS property specifies the stroke color of characters of text. If this property is not set, the value of the `color` property is used.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   */
  WebkitTextStrokeColor?: Property.WebkitTextStrokeColor;
  /**
   * The **`-webkit-text-stroke-width`** CSS property specifies the width of the stroke for text.
   *
   * **Syntax**: `<length>`
   *
   * **Initial value**: `0`
   */
  WebkitTextStrokeWidth?: Property.WebkitTextStrokeWidth<TLength>;
  /**
   * The **`text-underline-position`** CSS property specifies the position of the underline which is set using the `text-decoration` property's `underline` value.
   *
   * **Syntax**: `auto | from-font | [ under || [ left | right ] ]`
   *
   * **Initial value**: `auto`
   */
  WebkitTextUnderlinePosition?: Property.TextUnderlinePosition;
  /**
   * The `-webkit-touch-callout` CSS property controls the display of the default callout shown when you touch and hold a touch target.
   *
   * **Syntax**: `default | none`
   *
   * **Initial value**: `default`
   */
  WebkitTouchCallout?: Property.WebkitTouchCallout;
  /**
   * The **`transform`** CSS property lets you rotate, scale, skew, or translate an element. It modifies the coordinate space of the CSS visual formatting model.
   *
   * **Syntax**: `none | <transform-list>`
   *
   * **Initial value**: `none`
   */
  WebkitTransform?: Property.Transform;
  /**
   * The **`transform-origin`** CSS property sets the origin for an element's transformations.
   *
   * **Syntax**: `[ <length-percentage> | left | center | right | top | bottom ] | [ [ <length-percentage> | left | center | right ] && [ <length-percentage> | top | center | bottom ] ] <length>?`
   *
   * **Initial value**: `50% 50% 0`
   */
  WebkitTransformOrigin?: Property.TransformOrigin<TLength>;
  /**
   * The **`transform-style`** CSS property sets whether children of an element are positioned in the 3D space or are flattened in the plane of the element.
   *
   * **Syntax**: `flat | preserve-3d`
   *
   * **Initial value**: `flat`
   */
  WebkitTransformStyle?: Property.TransformStyle;
  /**
   * The **`transition-delay`** CSS property specifies the duration to wait before starting a property's transition effect when its value changes.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  WebkitTransitionDelay?: Property.TransitionDelay<TTime>;
  /**
   * The **`transition-duration`** CSS property sets the length of time a transition animation should take to complete. By default, the value is `0s`, meaning that no animation will occur.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   */
  WebkitTransitionDuration?: Property.TransitionDuration<TTime>;
  /**
   * The **`transition-property`** CSS property sets the CSS properties to which a transition effect should be applied.
   *
   * **Syntax**: `none | <single-transition-property>#`
   *
   * **Initial value**: all
   */
  WebkitTransitionProperty?: Property.TransitionProperty;
  /**
   * The **`transition-timing-function`** CSS property sets how intermediate values are calculated for CSS properties being affected by a transition effect.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   */
  WebkitTransitionTimingFunction?: Property.TransitionTimingFunction;
  /**
   * **Syntax**: `read-only | read-write | read-write-plaintext-only`
   *
   * **Initial value**: `read-only`
   */
  WebkitUserModify?: Property.WebkitUserModify;
  /**
   * The `**user-select**` CSS property controls whether the user can select text. This doesn't have any effect on content loaded as chrome, except in textboxes.
   *
   * **Syntax**: `auto | text | none | contain | all`
   *
   * **Initial value**: `auto`
   */
  WebkitUserSelect?: Property.UserSelect;
  /**
   * The **`writing-mode`** CSS property sets whether lines of text are laid out horizontally or vertically, as well as the direction in which blocks progress. When set for an entire document, it should be set on the root element (`html` element for HTML documents).
   *
   * **Syntax**: `horizontal-tb | vertical-rl | vertical-lr | sideways-rl | sideways-lr`
   *
   * **Initial value**: `horizontal-tb`
   */
  WebkitWritingMode?: Property.WritingMode;
}

interface VendorShorthandProperties<TLength = (string & {}) | 0, TTime = string & {}> {
  /**
   * The **`animation`** shorthand CSS property applies an animation between styles. It is a shorthand for `animation-name`, `animation-duration`, `animation-timing-function`, `animation-delay`, `animation-iteration-count`, `animation-direction`, `animation-fill-mode`, and `animation-play-state`.
   *
   * **Syntax**: `<single-animation>#`
   */
  MozAnimation?: Property.Animation<TTime>;
  /**
   * The **`border-image`** CSS property draws an image around a given element. It replaces the element's regular border.
   *
   * **Syntax**: `<'border-image-source'> || <'border-image-slice'> [ / <'border-image-width'> | / <'border-image-width'>? / <'border-image-outset'> ]? || <'border-image-repeat'>`
   */
  MozBorderImage?: Property.BorderImage;
  /**
   * The **`column-rule`** shorthand CSS property sets the width, style, and color of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'column-rule-width'> || <'column-rule-style'> || <'column-rule-color'>`
   */
  MozColumnRule?: Property.ColumnRule<TLength>;
  /**
   * The **`columns`** CSS shorthand property sets the number of columns to use when drawing an element's contents, as well as those columns' widths.
   *
   * **Syntax**: `<'column-width'> || <'column-count'>`
   */
  MozColumns?: Property.Columns<TLength>;
  /**
   * The **`transition`** CSS property is a shorthand property for `transition-property`, `transition-duration`, `transition-timing-function`, and `transition-delay`.
   *
   * **Syntax**: `<single-transition>#`
   */
  MozTransition?: Property.Transition<TTime>;
  /**
   * The **`-ms-content-zoom-limit`** CSS shorthand property is a Microsoft extension that specifies values for the `-ms-content-zoom-limit-min` and `-ms-content-zoom-limit-max` properties.
   *
   * **Syntax**: `<'-ms-content-zoom-limit-min'> <'-ms-content-zoom-limit-max'>`
   */
  msContentZoomLimit?: Property.MsContentZoomLimit;
  /**
   * The **`-ms-content-zoom-snap`** CSS shorthand property is a Microsoft extension that specifies values for the `-ms-content-zoom-snap-type` and `-ms-content-zoom-snap-points` properties.
   *
   * **Syntax**: `<'-ms-content-zoom-snap-type'> || <'-ms-content-zoom-snap-points'>`
   */
  msContentZoomSnap?: Property.MsContentZoomSnap;
  /**
   * The **`flex`** CSS shorthand property sets how a flex _item_ will grow or shrink to fit the space available in its flex container.
   *
   * **Syntax**: `none | [ <'flex-grow'> <'flex-shrink'>? || <'flex-basis'> ]`
   */
  msFlex?: Property.Flex<TLength>;
  /**
   * The **\-ms-scroll-limit** CSS property is a Microsoft extension that specifies values for the `-ms-scroll-limit-x-min`, `-ms-scroll-limit-y-min`, `-ms-scroll-limit-x-max`, and `-ms-scroll-limit-y-max` properties.
   *
   * **Syntax**: `<'-ms-scroll-limit-x-min'> <'-ms-scroll-limit-y-min'> <'-ms-scroll-limit-x-max'> <'-ms-scroll-limit-y-max'>`
   */
  msScrollLimit?: Property.MsScrollLimit;
  /**
   * The **`-ms-scroll-snap-x`** CSS shorthand property is a Microsoft extension that specifies values for the `-ms-scroll-snap-type` and `-ms-scroll-snap-points-x` properties.
   *
   * **Syntax**: `<'-ms-scroll-snap-type'> <'-ms-scroll-snap-points-x'>`
   */
  msScrollSnapX?: Property.MsScrollSnapX;
  /**
   * The **`-ms-scroll-snap-x`** CSS shorthand property is a Microsoft extension that specifies values for the `-ms-scroll-snap-type` and `-ms-scroll-snap-points-y` properties.
   *
   * **Syntax**: `<'-ms-scroll-snap-type'> <'-ms-scroll-snap-points-y'>`
   */
  msScrollSnapY?: Property.MsScrollSnapY;
  /**
   * The **`transition`** CSS property is a shorthand property for `transition-property`, `transition-duration`, `transition-timing-function`, and `transition-delay`.
   *
   * **Syntax**: `<single-transition>#`
   */
  msTransition?: Property.Transition<TTime>;
  /**
   * The **`animation`** shorthand CSS property applies an animation between styles. It is a shorthand for `animation-name`, `animation-duration`, `animation-timing-function`, `animation-delay`, `animation-iteration-count`, `animation-direction`, `animation-fill-mode`, and `animation-play-state`.
   *
   * **Syntax**: `<single-animation>#`
   */
  WebkitAnimation?: Property.Animation<TTime>;
  /**
   * The **`-webkit-border-before`** CSS property is a shorthand property for setting the individual logical block start border property values in a single place in the style sheet.
   *
   * **Syntax**: `<'border-width'> || <'border-style'> || <color>`
   */
  WebkitBorderBefore?: Property.WebkitBorderBefore<TLength>;
  /**
   * The **`border-image`** CSS property draws an image around a given element. It replaces the element's regular border.
   *
   * **Syntax**: `<'border-image-source'> || <'border-image-slice'> [ / <'border-image-width'> | / <'border-image-width'>? / <'border-image-outset'> ]? || <'border-image-repeat'>`
   */
  WebkitBorderImage?: Property.BorderImage;
  /**
   * The **`border-radius`** CSS property rounds the corners of an element's outer border edge. You can set a single radius to make circular corners, or two radii to make elliptical corners.
   *
   * **Syntax**: `<length-percentage>{1,4} [ / <length-percentage>{1,4} ]?`
   */
  WebkitBorderRadius?: Property.BorderRadius<TLength>;
  /**
   * The **`column-rule`** shorthand CSS property sets the width, style, and color of the line drawn between columns in a multi-column layout.
   *
   * **Syntax**: `<'column-rule-width'> || <'column-rule-style'> || <'column-rule-color'>`
   */
  WebkitColumnRule?: Property.ColumnRule<TLength>;
  /**
   * The **`columns`** CSS shorthand property sets the number of columns to use when drawing an element's contents, as well as those columns' widths.
   *
   * **Syntax**: `<'column-width'> || <'column-count'>`
   */
  WebkitColumns?: Property.Columns<TLength>;
  /**
   * The **`flex`** CSS shorthand property sets how a flex _item_ will grow or shrink to fit the space available in its flex container.
   *
   * **Syntax**: `none | [ <'flex-grow'> <'flex-shrink'>? || <'flex-basis'> ]`
   */
  WebkitFlex?: Property.Flex<TLength>;
  /**
   * The **`flex-flow`** CSS shorthand property specifies the direction of a flex container, as well as its wrapping behavior.
   *
   * **Syntax**: `<'flex-direction'> || <'flex-wrap'>`
   */
  WebkitFlexFlow?: Property.FlexFlow;
  /**
   * The **`mask`** CSS shorthand property hides an element (partially or fully) by masking or clipping the image at specific points.
   *
   * **Syntax**: `[ <mask-reference> || <position> [ / <bg-size> ]? || <repeat-style> || [ <box> | border | padding | content | text ] || [ <box> | border | padding | content ] ]#`
   */
  WebkitMask?: Property.WebkitMask<TLength>;
  /**
   * The **`mask-border`** CSS shorthand property lets you create a mask along the edge of an element's border.
   *
   * **Syntax**: `<'mask-border-source'> || <'mask-border-slice'> [ / <'mask-border-width'>? [ / <'mask-border-outset'> ]? ]? || <'mask-border-repeat'> || <'mask-border-mode'>`
   */
  WebkitMaskBoxImage?: Property.MaskBorder;
  /**
   * The **`text-emphasis`** CSS property applies emphasis marks to text (except spaces and control characters). It is a shorthand for `text-emphasis-style` and `text-emphasis-color`.
   *
   * **Syntax**: `<'text-emphasis-style'> || <'text-emphasis-color'>`
   */
  WebkitTextEmphasis?: Property.TextEmphasis;
  /**
   * The **`-webkit-text-stroke`** CSS property specifies the width and color of strokes for text characters. This is a shorthand property for the longhand properties `-webkit-text-stroke-width` and `-webkit-text-stroke-color`.
   *
   * **Syntax**: `<length> || <color>`
   */
  WebkitTextStroke?: Property.WebkitTextStroke<TLength>;
  /**
   * The **`transition`** CSS property is a shorthand property for `transition-property`, `transition-duration`, `transition-timing-function`, and `transition-delay`.
   *
   * **Syntax**: `<single-transition>#`
   */
  WebkitTransition?: Property.Transition<TTime>;
}

interface VendorProperties<TLength = (string & {}) | 0, TTime = string & {}> extends VendorLonghandProperties<TLength, TTime>, VendorShorthandProperties<TLength, TTime> {}

interface ObsoleteProperties<TLength = (string & {}) | 0, TTime = string & {}> {
  /**
   * In combination with `elevation`, the **`azimuth`** CSS property enables different audio sources to be positioned spatially for aural presentation. This is important in that it provides a natural way to tell several voices apart, as each can be positioned to originate at a different location on the sound stage. Stereo output produce a lateral sound stage, while binaural headphones and multi-speaker setups allow for a fully three-dimensional stage.
   *
   * **Syntax**: `<angle> | [ [ left-side | far-left | left | center-left | center | center-right | right | far-right | right-side ] || behind ] | leftwards | rightwards`
   *
   * **Initial value**: `center`
   *
   * @deprecated
   */
  azimuth?: Property.Azimuth;
  /**
   * The **`box-align`** CSS property specifies how an element aligns its contents across its layout in a perpendicular direction. The effect of the property is only visible if there is extra space in the box.
   *
   * **Syntax**: `start | center | end | baseline | stretch`
   *
   * **Initial value**: `stretch`
   *
   * @deprecated
   */
  boxAlign?: Property.BoxAlign;
  /**
   * The **`box-direction`** CSS property specifies whether a box lays out its contents normally (from the top or left edge), or in reverse (from the bottom or right edge).
   *
   * **Syntax**: `normal | reverse | inherit`
   *
   * **Initial value**: `normal`
   *
   * @deprecated
   */
  boxDirection?: Property.BoxDirection;
  /**
   * The **`-moz-box-flex`** and **`-webkit-box-flex`** CSS properties specify how a `-moz-box` or `-webkit-box` grows to fill the box that contains it, in the direction of the containing box's layout.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  boxFlex?: Property.BoxFlex;
  /**
   * The **`box-flex-group`** CSS property assigns the flexbox's child elements to a flex group.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `1`
   *
   * @deprecated
   */
  boxFlexGroup?: Property.BoxFlexGroup;
  /**
   * The **`box-lines`** CSS property determines whether the box may have a single or multiple lines (rows for horizontally oriented boxes, columns for vertically oriented boxes).
   *
   * **Syntax**: `single | multiple`
   *
   * **Initial value**: `single`
   *
   * @deprecated
   */
  boxLines?: Property.BoxLines;
  /**
   * The **`box-ordinal-group`** CSS property assigns the flexbox's child elements to an ordinal group.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `1`
   *
   * @deprecated
   */
  boxOrdinalGroup?: Property.BoxOrdinalGroup;
  /**
   * This is a property of the original CSS Flexible Box Layout Module draft, and has been replaced by a newer standard. See flexbox for information about the current standard.
   *
   * **Syntax**: `horizontal | vertical | inline-axis | block-axis | inherit`
   *
   * **Initial value**: `inline-axis` (`horizontal` in XUL)
   *
   * @deprecated
   */
  boxOrient?: Property.BoxOrient;
  /**
   * The **`-moz-box-pack`** and **`-webkit-box-pack`** CSS properties specify how a `-moz-box` or `-webkit-box` packs its contents in the direction of its layout. The effect of this is only visible if there is extra space in the box.
   *
   * **Syntax**: `start | center | end | justify`
   *
   * **Initial value**: `start`
   *
   * @deprecated
   */
  boxPack?: Property.BoxPack;
  /**
   * The **`clip`** CSS property defines a visible portion of an element. The `clip` property applies only to absolutely positioned elements — that is, elements with `position:absolute` or `position:fixed`.
   *
   * **Syntax**: `<shape> | auto`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  clip?: Property.Clip;
  /**
   * The **`font-variant-alternates`** CSS property controls the usage of alternate glyphs. These alternate glyphs may be referenced by alternative names defined in `@font-feature-values`.
   *
   * **Syntax**: `normal | [ stylistic( <feature-value-name> ) || historical-forms || styleset( <feature-value-name># ) || character-variant( <feature-value-name># ) || swash( <feature-value-name> ) || ornaments( <feature-value-name> ) || annotation( <feature-value-name> ) ]`
   *
   * **Initial value**: `normal`
   *
   * @deprecated
   */
  fontVariantAlternates?: Property.FontVariantAlternates;
  /**
   * The **`column-gap`** CSS property sets the size of the gap (gutter) between an element's columns.
   *
   * **Syntax**: `<length-percentage>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  gridColumnGap?: Property.GridColumnGap<TLength>;
  /**
   * The **`gap`** CSS property sets the gaps (gutters) between rows and columns. It is a shorthand for `row-gap` and `column-gap`.
   *
   * **Syntax**: `<'grid-row-gap'> <'grid-column-gap'>?`
   *
   * @deprecated
   */
  gridGap?: Property.GridGap<TLength>;
  /**
   * The **`row-gap`** CSS property sets the size of the gap (gutter) between an element's grid rows.
   *
   * **Syntax**: `<length-percentage>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  gridRowGap?: Property.GridRowGap<TLength>;
  /**
   * The **`ime-mode`** CSS property controls the state of the input method editor (IME) for text fields. This property is obsolete.
   *
   * **Syntax**: `auto | normal | active | inactive | disabled`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  imeMode?: Property.ImeMode;
  /**
   * The **`inset-inline`** CSS property defines the logical start and end offsets of an element in the inline direction, which maps to physical offsets depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top` and `bottom`, or `right` and `left` properties depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>{1,2}`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  offsetBlock?: Property.InsetBlock<TLength>;
  /**
   * The **`inset-block-end`** CSS property defines the logical block end offset of an element, which maps to a physical inset depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top`, `right`, `bottom`, or `left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  offsetBlockEnd?: Property.InsetBlockEnd<TLength>;
  /**
   * The **`inset-block-start`** CSS property defines the logical block start offset of an element, which maps to a physical inset depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top`, `right`, `bottom`, or `left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  offsetBlockStart?: Property.InsetBlockStart<TLength>;
  /**
   * The **`inset-inline`** CSS property defines the logical start and end offsets of an element in the inline direction, which maps to physical offsets depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top` and `bottom`, or `right` and `left` properties depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>{1,2}`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  offsetInline?: Property.InsetInline<TLength>;
  /**
   * The **`inset-inline-end`** CSS property defines the logical inline end inset of an element, which maps to a physical offset depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top`, `right`, `bottom`, or `left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  offsetInlineEnd?: Property.InsetInlineEnd<TLength>;
  /**
   * The **`inset-inline-start`** CSS property defines the logical inline start inset of an element, which maps to a physical offset depending on the element's writing mode, directionality, and text orientation. It corresponds to the `top`, `right`, `bottom`, or `left` property depending on the values defined for `writing-mode`, `direction`, and `text-orientation`.
   *
   * **Syntax**: `<'top'>`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  offsetInlineStart?: Property.InsetInlineStart<TLength>;
  /**
   * The **`scroll-snap-coordinate`** CSS property defines the x and y coordinate positions within an element that will align with its nearest ancestor scroll container's `scroll-snap-destination` for each respective axis.
   *
   * **Syntax**: `none | <position>#`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  scrollSnapCoordinate?: Property.ScrollSnapCoordinate<TLength>;
  /**
   * The **`scroll-snap-destination`** CSS property defines the position in x and y coordinates within the scroll container's visual viewport which element snap points align with.
   *
   * **Syntax**: `<position>`
   *
   * **Initial value**: `0px 0px`
   *
   * @deprecated
   */
  scrollSnapDestination?: Property.ScrollSnapDestination<TLength>;
  /**
   * The **`scroll-snap-points-x`** CSS property defines the horizontal positioning of snap points within the content of the scroll container they are applied to.
   *
   * **Syntax**: `none | repeat( <length-percentage> )`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  scrollSnapPointsX?: Property.ScrollSnapPointsX;
  /**
   * The **`scroll-snap-points-y`** CSS property defines the vertical positioning of snap points within the content of the scroll container they are applied to.
   *
   * **Syntax**: `none | repeat( <length-percentage> )`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  scrollSnapPointsY?: Property.ScrollSnapPointsY;
  /**
   * The **`scroll-snap-type-x`** CSS property defines how strictly snap points are enforced on the horizontal axis of the scroll container in case there is one.
   *
   * **Syntax**: `none | mandatory | proximity`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  scrollSnapTypeX?: Property.ScrollSnapTypeX;
  /**
   * The **`scroll-snap-type-y`** CSS property defines how strictly snap points are enforced on the vertical axis of the scroll container in case there is one.
   *
   * **Syntax**: `none | mandatory | proximity`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  scrollSnapTypeY?: Property.ScrollSnapTypeY;
  /**
   * The **`-ms-scrollbar-track-color`** CSS property is a Microsoft extension that specifies the color of the track element of a scrollbar.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `Scrollbar`
   *
   * @deprecated
   */
  scrollbarTrackColor?: Property.MsScrollbarTrackColor;
  /**
   * The **`box-align`** CSS property specifies how an element aligns its contents across its layout in a perpendicular direction. The effect of the property is only visible if there is extra space in the box.
   *
   * **Syntax**: `start | center | end | baseline | stretch`
   *
   * **Initial value**: `stretch`
   *
   * @deprecated
   */
  KhtmlBoxAlign?: Property.BoxAlign;
  /**
   * The **`box-direction`** CSS property specifies whether a box lays out its contents normally (from the top or left edge), or in reverse (from the bottom or right edge).
   *
   * **Syntax**: `normal | reverse | inherit`
   *
   * **Initial value**: `normal`
   *
   * @deprecated
   */
  KhtmlBoxDirection?: Property.BoxDirection;
  /**
   * The **`-moz-box-flex`** and **`-webkit-box-flex`** CSS properties specify how a `-moz-box` or `-webkit-box` grows to fill the box that contains it, in the direction of the containing box's layout.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  KhtmlBoxFlex?: Property.BoxFlex;
  /**
   * The **`box-flex-group`** CSS property assigns the flexbox's child elements to a flex group.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `1`
   *
   * @deprecated
   */
  KhtmlBoxFlexGroup?: Property.BoxFlexGroup;
  /**
   * The **`box-lines`** CSS property determines whether the box may have a single or multiple lines (rows for horizontally oriented boxes, columns for vertically oriented boxes).
   *
   * **Syntax**: `single | multiple`
   *
   * **Initial value**: `single`
   *
   * @deprecated
   */
  KhtmlBoxLines?: Property.BoxLines;
  /**
   * The **`box-ordinal-group`** CSS property assigns the flexbox's child elements to an ordinal group.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `1`
   *
   * @deprecated
   */
  KhtmlBoxOrdinalGroup?: Property.BoxOrdinalGroup;
  /**
   * This is a property of the original CSS Flexible Box Layout Module draft, and has been replaced by a newer standard. See flexbox for information about the current standard.
   *
   * **Syntax**: `horizontal | vertical | inline-axis | block-axis | inherit`
   *
   * **Initial value**: `inline-axis` (`horizontal` in XUL)
   *
   * @deprecated
   */
  KhtmlBoxOrient?: Property.BoxOrient;
  /**
   * The **`-moz-box-pack`** and **`-webkit-box-pack`** CSS properties specify how a `-moz-box` or `-webkit-box` packs its contents in the direction of its layout. The effect of this is only visible if there is extra space in the box.
   *
   * **Syntax**: `start | center | end | justify`
   *
   * **Initial value**: `start`
   *
   * @deprecated
   */
  KhtmlBoxPack?: Property.BoxPack;
  /**
   * The **`line-break`** CSS property sets how to break lines of Chinese, Japanese, or Korean (CJK) text when working with punctuation and symbols.
   *
   * **Syntax**: `auto | loose | normal | strict | anywhere`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  KhtmlLineBreak?: Property.LineBreak;
  /**
   * The **`opacity`** CSS property sets the opacity of an element. Opacity is the degree to which content behind an element is hidden, and is the opposite of transparency.
   *
   * **Syntax**: `<alpha-value>`
   *
   * **Initial value**: `1.0`
   *
   * @deprecated
   */
  KhtmlOpacity?: Property.Opacity;
  /**
   * The `**user-select**` CSS property controls whether the user can select text. This doesn't have any effect on content loaded as chrome, except in textboxes.
   *
   * **Syntax**: `auto | text | none | contain | all`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  KhtmlUserSelect?: Property.UserSelect;
  /**
   * The **`background-clip`** CSS property sets whether an element's background extends underneath its border box, padding box, or content box.
   *
   * **Syntax**: `<box>#`
   *
   * **Initial value**: `border-box`
   *
   * @deprecated
   */
  MozBackgroundClip?: Property.BackgroundClip;
  /**
   * The **`box-decoration-break`** CSS property specifies how an element's fragments should be rendered when broken across multiple lines, columns, or pages.
   *
   * **Syntax**: `slice | clone`
   *
   * **Initial value**: `slice`
   *
   * @deprecated
   */
  MozBackgroundInlinePolicy?: Property.BoxDecorationBreak;
  /**
   * The **`background-origin`** CSS property sets the background's origin: from the border start, inside the border, or inside the padding.
   *
   * **Syntax**: `<box>#`
   *
   * **Initial value**: `padding-box`
   *
   * @deprecated
   */
  MozBackgroundOrigin?: Property.BackgroundOrigin;
  /**
   * The **`background-size`** CSS property sets the size of the element's background image. The image can be left to its natural size, stretched, or constrained to fit the available space.
   *
   * **Syntax**: `<bg-size>#`
   *
   * **Initial value**: `auto auto`
   *
   * @deprecated
   */
  MozBackgroundSize?: Property.BackgroundSize<TLength>;
  /**
   * The **`-moz-binding`** CSS property is used by Mozilla-based applications to attach an XBL binding to a DOM element.
   *
   * **Syntax**: `<url> | none`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  MozBinding?: Property.MozBinding;
  /**
   * The **`border-radius`** CSS property rounds the corners of an element's outer border edge. You can set a single radius to make circular corners, or two radii to make elliptical corners.
   *
   * **Syntax**: `<length-percentage>{1,4} [ / <length-percentage>{1,4} ]?`
   *
   * @deprecated
   */
  MozBorderRadius?: Property.BorderRadius<TLength>;
  /**
   * The **`border-bottom-left-radius`** CSS property rounds the bottom-left corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozBorderRadiusBottomleft?: Property.BorderBottomLeftRadius<TLength>;
  /**
   * The **`border-bottom-right-radius`** CSS property rounds the bottom-right corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozBorderRadiusBottomright?: Property.BorderBottomRightRadius<TLength>;
  /**
   * The **`border-top-left-radius`** CSS property rounds the top-left corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozBorderRadiusTopleft?: Property.BorderTopLeftRadius<TLength>;
  /**
   * The **`border-top-right-radius`** CSS property rounds the top-right corner of an element by specifying the radius (or the radius of the semi-major and semi-minor axes) of the ellipse defining the curvature of the corner.
   *
   * **Syntax**: `<length-percentage>{1,2}`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozBorderRadiusTopright?: Property.BorderTopRightRadius<TLength>;
  /**
   * The **`box-align`** CSS property specifies how an element aligns its contents across its layout in a perpendicular direction. The effect of the property is only visible if there is extra space in the box.
   *
   * **Syntax**: `start | center | end | baseline | stretch`
   *
   * **Initial value**: `stretch`
   *
   * @deprecated
   */
  MozBoxAlign?: Property.BoxAlign;
  /**
   * The **`box-direction`** CSS property specifies whether a box lays out its contents normally (from the top or left edge), or in reverse (from the bottom or right edge).
   *
   * **Syntax**: `normal | reverse | inherit`
   *
   * **Initial value**: `normal`
   *
   * @deprecated
   */
  MozBoxDirection?: Property.BoxDirection;
  /**
   * The **`-moz-box-flex`** and **`-webkit-box-flex`** CSS properties specify how a `-moz-box` or `-webkit-box` grows to fill the box that contains it, in the direction of the containing box's layout.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozBoxFlex?: Property.BoxFlex;
  /**
   * The **`box-ordinal-group`** CSS property assigns the flexbox's child elements to an ordinal group.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `1`
   *
   * @deprecated
   */
  MozBoxOrdinalGroup?: Property.BoxOrdinalGroup;
  /**
   * This is a property of the original CSS Flexible Box Layout Module draft, and has been replaced by a newer standard. See flexbox for information about the current standard.
   *
   * **Syntax**: `horizontal | vertical | inline-axis | block-axis | inherit`
   *
   * **Initial value**: `inline-axis` (`horizontal` in XUL)
   *
   * @deprecated
   */
  MozBoxOrient?: Property.BoxOrient;
  /**
   * The **`-moz-box-pack`** and **`-webkit-box-pack`** CSS properties specify how a `-moz-box` or `-webkit-box` packs its contents in the direction of its layout. The effect of this is only visible if there is extra space in the box.
   *
   * **Syntax**: `start | center | end | justify`
   *
   * **Initial value**: `start`
   *
   * @deprecated
   */
  MozBoxPack?: Property.BoxPack;
  /**
   * The **`box-shadow`** CSS property adds shadow effects around an element's frame. You can set multiple effects separated by commas. A box shadow is described by X and Y offsets relative to the element, blur and spread radius, and color.
   *
   * **Syntax**: `none | <shadow>#`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  MozBoxShadow?: Property.BoxShadow;
  /**
   * The non-standard **`-moz-float-edge`** CSS property specifies whether the height and width properties of the element include the margin, border, or padding thickness.
   *
   * **Syntax**: `border-box | content-box | margin-box | padding-box`
   *
   * **Initial value**: `content-box`
   *
   * @deprecated
   */
  MozFloatEdge?: Property.MozFloatEdge;
  /**
   * The **`-moz-force-broken-image-icon`** extended CSS property can be used to force the broken image icon to be shown even when a broken image has an `alt` attribute.
   *
   * **Syntax**: `<integer [0,1]>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozForceBrokenImageIcon?: Property.MozForceBrokenImageIcon;
  /**
   * The **`opacity`** CSS property sets the opacity of an element. Opacity is the degree to which content behind an element is hidden, and is the opposite of transparency.
   *
   * **Syntax**: `<alpha-value>`
   *
   * **Initial value**: `1.0`
   *
   * @deprecated
   */
  MozOpacity?: Property.Opacity;
  /**
   * The **`outline`** CSS shorthand property set all the outline properties in a single declaration.
   *
   * **Syntax**: `[ <'outline-color'> || <'outline-style'> || <'outline-width'> ]`
   *
   * @deprecated
   */
  MozOutline?: Property.Outline<TLength>;
  /**
   * The **`outline-color`** CSS property sets the color of an element's outline.
   *
   * **Syntax**: `<color> | invert`
   *
   * **Initial value**: `invert`, for browsers supporting it, `currentColor` for the other
   *
   * @deprecated
   */
  MozOutlineColor?: Property.OutlineColor;
  /**
   * In Mozilla applications like Firefox, the **`-moz-outline-radius`** CSS shorthand property can be used to give an element's `outline` rounded corners.
   *
   * **Syntax**: `<outline-radius>{1,4} [ / <outline-radius>{1,4} ]?`
   *
   * @deprecated
   */
  MozOutlineRadius?: Property.MozOutlineRadius<TLength>;
  /**
   * In Mozilla applications, the **`-moz-outline-radius-bottomleft`** CSS property can be used to round the bottom-left corner of an element's `outline`.
   *
   * **Syntax**: `<outline-radius>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozOutlineRadiusBottomleft?: Property.MozOutlineRadiusBottomleft<TLength>;
  /**
   * In Mozilla applications, the **`-moz-outline-radius-bottomright`** CSS property can be used to round the bottom-right corner of an element's `outline`.
   *
   * **Syntax**: `<outline-radius>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozOutlineRadiusBottomright?: Property.MozOutlineRadiusBottomright<TLength>;
  /**
   * In Mozilla applications, the **`-moz-outline-radius-topleft`** CSS property can be used to round the top-left corner of an element's `outline`.
   *
   * **Syntax**: `<outline-radius>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozOutlineRadiusTopleft?: Property.MozOutlineRadiusTopleft<TLength>;
  /**
   * In Mozilla applications, the **`-moz-outline-radius-topright`** CSS property can be used to round the top-right corner of an element's `outline`.
   *
   * **Syntax**: `<outline-radius>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  MozOutlineRadiusTopright?: Property.MozOutlineRadiusTopright<TLength>;
  /**
   * The **`outline-style`** CSS property sets the style of an element's outline. An outline is a line that is drawn around an element, outside the `border`.
   *
   * **Syntax**: `auto | <'border-style'>`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  MozOutlineStyle?: Property.OutlineStyle;
  /**
   * The CSS **`outline-width`** property sets the thickness of an element's outline. An outline is a line that is drawn around an element, outside the `border`.
   *
   * **Syntax**: `<line-width>`
   *
   * **Initial value**: `medium`
   *
   * @deprecated
   */
  MozOutlineWidth?: Property.OutlineWidth<TLength>;
  /**
   * The **`text-align-last`** CSS property sets how the last line of a block or a line, right before a forced line break, is aligned.
   *
   * **Syntax**: `auto | start | end | left | right | center | justify`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  MozTextAlignLast?: Property.TextAlignLast;
  /**
   * The **`text-decoration-color`** CSS property sets the color of decorations added to text by `text-decoration-line`.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `currentcolor`
   *
   * @deprecated
   */
  MozTextDecorationColor?: Property.TextDecorationColor;
  /**
   * The **`text-decoration-line`** CSS property sets the kind of decoration that is used on text in an element, such as an underline or overline.
   *
   * **Syntax**: `none | [ underline || overline || line-through || blink ] | spelling-error | grammar-error`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  MozTextDecorationLine?: Property.TextDecorationLine;
  /**
   * The **`text-decoration-style`** CSS property sets the style of the lines specified by `text-decoration-line`. The style applies to all lines that are set with `text-decoration-line`.
   *
   * **Syntax**: `solid | double | dotted | dashed | wavy`
   *
   * **Initial value**: `solid`
   *
   * @deprecated
   */
  MozTextDecorationStyle?: Property.TextDecorationStyle;
  /**
   * In Mozilla applications, **`-moz-user-input`** determines if an element will accept user input.
   *
   * **Syntax**: `auto | none | enabled | disabled`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  MozUserInput?: Property.MozUserInput;
  /**
   * The **`ime-mode`** CSS property controls the state of the input method editor (IME) for text fields. This property is obsolete.
   *
   * **Syntax**: `auto | normal | active | inactive | disabled`
   *
   * **Initial value**: `auto`
   *
   * @deprecated
   */
  msImeMode?: Property.ImeMode;
  /**
   * The **`-ms-scrollbar-track-color`** CSS property is a Microsoft extension that specifies the color of the track element of a scrollbar.
   *
   * **Syntax**: `<color>`
   *
   * **Initial value**: `Scrollbar`
   *
   * @deprecated
   */
  msScrollbarTrackColor?: Property.MsScrollbarTrackColor;
  /**
   * The **`animation`** shorthand CSS property applies an animation between styles. It is a shorthand for `animation-name`, `animation-duration`, `animation-timing-function`, `animation-delay`, `animation-iteration-count`, `animation-direction`, `animation-fill-mode`, and `animation-play-state`.
   *
   * **Syntax**: `<single-animation>#`
   *
   * @deprecated
   */
  OAnimation?: Property.Animation<TTime>;
  /**
   * The **`animation-delay`** CSS property specifies the amount of time to wait from applying the animation to an element before beginning to perform the animation. The animation can start later, immediately from its beginning, or immediately and partway through the animation.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   *
   * @deprecated
   */
  OAnimationDelay?: Property.AnimationDelay<TTime>;
  /**
   * The **`animation-direction`** CSS property sets whether an animation should play forward, backward, or alternate back and forth between playing the sequence forward and backward.
   *
   * **Syntax**: `<single-animation-direction>#`
   *
   * **Initial value**: `normal`
   *
   * @deprecated
   */
  OAnimationDirection?: Property.AnimationDirection;
  /**
   * The **`animation-duration`** CSS property sets the length of time that an animation takes to complete one cycle.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   *
   * @deprecated
   */
  OAnimationDuration?: Property.AnimationDuration<TTime>;
  /**
   * The **`animation-fill-mode`** CSS property sets how a CSS animation applies styles to its target before and after its execution.
   *
   * **Syntax**: `<single-animation-fill-mode>#`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  OAnimationFillMode?: Property.AnimationFillMode;
  /**
   * The **`animation-iteration-count`** CSS property sets the number of times an animation sequence should be played before stopping.
   *
   * **Syntax**: `<single-animation-iteration-count>#`
   *
   * **Initial value**: `1`
   *
   * @deprecated
   */
  OAnimationIterationCount?: Property.AnimationIterationCount;
  /**
   * The **`animation-name`** CSS property specifies the names of one or more `@keyframes` at-rules describing the animation or animations to apply to the element.
   *
   * **Syntax**: `[ none | <keyframes-name> ]#`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  OAnimationName?: Property.AnimationName;
  /**
   * The **`animation-play-state`** CSS property sets whether an animation is running or paused.
   *
   * **Syntax**: `<single-animation-play-state>#`
   *
   * **Initial value**: `running`
   *
   * @deprecated
   */
  OAnimationPlayState?: Property.AnimationPlayState;
  /**
   * The **`animation-timing-function`** CSS property sets how an animation progresses through the duration of each cycle.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   *
   * @deprecated
   */
  OAnimationTimingFunction?: Property.AnimationTimingFunction;
  /**
   * The **`background-size`** CSS property sets the size of the element's background image. The image can be left to its natural size, stretched, or constrained to fit the available space.
   *
   * **Syntax**: `<bg-size>#`
   *
   * **Initial value**: `auto auto`
   *
   * @deprecated
   */
  OBackgroundSize?: Property.BackgroundSize<TLength>;
  /**
   * The **`border-image`** CSS property draws an image around a given element. It replaces the element's regular border.
   *
   * **Syntax**: `<'border-image-source'> || <'border-image-slice'> [ / <'border-image-width'> | / <'border-image-width'>? / <'border-image-outset'> ]? || <'border-image-repeat'>`
   *
   * @deprecated
   */
  OBorderImage?: Property.BorderImage;
  /**
   * The **`object-fit`** CSS property sets how the content of a replaced element, such as an `<img>` or `<video>`, should be resized to fit its container.
   *
   * **Syntax**: `fill | contain | cover | none | scale-down`
   *
   * **Initial value**: `fill`
   *
   * @deprecated
   */
  OObjectFit?: Property.ObjectFit;
  /**
   * The **`object-position`** CSS property specifies the alignment of the selected replaced element's contents within the element's box. Areas of the box which aren't covered by the replaced element's object will show the element's background.
   *
   * **Syntax**: `<position>`
   *
   * **Initial value**: `50% 50%`
   *
   * @deprecated
   */
  OObjectPosition?: Property.ObjectPosition<TLength>;
  /**
   * The **`tab-size`** CSS property is used to customize the width of tab characters (U+0009).
   *
   * **Syntax**: `<integer> | <length>`
   *
   * **Initial value**: `8`
   *
   * @deprecated
   */
  OTabSize?: Property.TabSize<TLength>;
  /**
   * The **`text-overflow`** CSS property sets how hidden overflow content is signaled to users. It can be clipped, display an ellipsis ('`…`'), or display a custom string.
   *
   * **Syntax**: `[ clip | ellipsis | <string> ]{1,2}`
   *
   * **Initial value**: `clip`
   *
   * @deprecated
   */
  OTextOverflow?: Property.TextOverflow;
  /**
   * The **`transform`** CSS property lets you rotate, scale, skew, or translate an element. It modifies the coordinate space of the CSS visual formatting model.
   *
   * **Syntax**: `none | <transform-list>`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  OTransform?: Property.Transform;
  /**
   * The **`transform-origin`** CSS property sets the origin for an element's transformations.
   *
   * **Syntax**: `[ <length-percentage> | left | center | right | top | bottom ] | [ [ <length-percentage> | left | center | right ] && [ <length-percentage> | top | center | bottom ] ] <length>?`
   *
   * **Initial value**: `50% 50% 0`
   *
   * @deprecated
   */
  OTransformOrigin?: Property.TransformOrigin<TLength>;
  /**
   * The **`transition`** CSS property is a shorthand property for `transition-property`, `transition-duration`, `transition-timing-function`, and `transition-delay`.
   *
   * **Syntax**: `<single-transition>#`
   *
   * @deprecated
   */
  OTransition?: Property.Transition<TTime>;
  /**
   * The **`transition-delay`** CSS property specifies the duration to wait before starting a property's transition effect when its value changes.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   *
   * @deprecated
   */
  OTransitionDelay?: Property.TransitionDelay<TTime>;
  /**
   * The **`transition-duration`** CSS property sets the length of time a transition animation should take to complete. By default, the value is `0s`, meaning that no animation will occur.
   *
   * **Syntax**: `<time>#`
   *
   * **Initial value**: `0s`
   *
   * @deprecated
   */
  OTransitionDuration?: Property.TransitionDuration<TTime>;
  /**
   * The **`transition-property`** CSS property sets the CSS properties to which a transition effect should be applied.
   *
   * **Syntax**: `none | <single-transition-property>#`
   *
   * **Initial value**: all
   *
   * @deprecated
   */
  OTransitionProperty?: Property.TransitionProperty;
  /**
   * The **`transition-timing-function`** CSS property sets how intermediate values are calculated for CSS properties being affected by a transition effect.
   *
   * **Syntax**: `<easing-function>#`
   *
   * **Initial value**: `ease`
   *
   * @deprecated
   */
  OTransitionTimingFunction?: Property.TransitionTimingFunction;
  /**
   * The **`box-align`** CSS property specifies how an element aligns its contents across its layout in a perpendicular direction. The effect of the property is only visible if there is extra space in the box.
   *
   * **Syntax**: `start | center | end | baseline | stretch`
   *
   * **Initial value**: `stretch`
   *
   * @deprecated
   */
  WebkitBoxAlign?: Property.BoxAlign;
  /**
   * The **`box-direction`** CSS property specifies whether a box lays out its contents normally (from the top or left edge), or in reverse (from the bottom or right edge).
   *
   * **Syntax**: `normal | reverse | inherit`
   *
   * **Initial value**: `normal`
   *
   * @deprecated
   */
  WebkitBoxDirection?: Property.BoxDirection;
  /**
   * The **`-moz-box-flex`** and **`-webkit-box-flex`** CSS properties specify how a `-moz-box` or `-webkit-box` grows to fill the box that contains it, in the direction of the containing box's layout.
   *
   * **Syntax**: `<number>`
   *
   * **Initial value**: `0`
   *
   * @deprecated
   */
  WebkitBoxFlex?: Property.BoxFlex;
  /**
   * The **`box-flex-group`** CSS property assigns the flexbox's child elements to a flex group.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `1`
   *
   * @deprecated
   */
  WebkitBoxFlexGroup?: Property.BoxFlexGroup;
  /**
   * The **`box-lines`** CSS property determines whether the box may have a single or multiple lines (rows for horizontally oriented boxes, columns for vertically oriented boxes).
   *
   * **Syntax**: `single | multiple`
   *
   * **Initial value**: `single`
   *
   * @deprecated
   */
  WebkitBoxLines?: Property.BoxLines;
  /**
   * The **`box-ordinal-group`** CSS property assigns the flexbox's child elements to an ordinal group.
   *
   * **Syntax**: `<integer>`
   *
   * **Initial value**: `1`
   *
   * @deprecated
   */
  WebkitBoxOrdinalGroup?: Property.BoxOrdinalGroup;
  /**
   * This is a property of the original CSS Flexible Box Layout Module draft, and has been replaced by a newer standard. See flexbox for information about the current standard.
   *
   * **Syntax**: `horizontal | vertical | inline-axis | block-axis | inherit`
   *
   * **Initial value**: `inline-axis` (`horizontal` in XUL)
   *
   * @deprecated
   */
  WebkitBoxOrient?: Property.BoxOrient;
  /**
   * The **`-moz-box-pack`** and **`-webkit-box-pack`** CSS properties specify how a `-moz-box` or `-webkit-box` packs its contents in the direction of its layout. The effect of this is only visible if there is extra space in the box.
   *
   * **Syntax**: `start | center | end | justify`
   *
   * **Initial value**: `start`
   *
   * @deprecated
   */
  WebkitBoxPack?: Property.BoxPack;
  /**
   * The **`scroll-snap-points-x`** CSS property defines the horizontal positioning of snap points within the content of the scroll container they are applied to.
   *
   * **Syntax**: `none | repeat( <length-percentage> )`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  WebkitScrollSnapPointsX?: Property.ScrollSnapPointsX;
  /**
   * The **`scroll-snap-points-y`** CSS property defines the vertical positioning of snap points within the content of the scroll container they are applied to.
   *
   * **Syntax**: `none | repeat( <length-percentage> )`
   *
   * **Initial value**: `none`
   *
   * @deprecated
   */
  WebkitScrollSnapPointsY?: Property.ScrollSnapPointsY;
}

interface SvgProperties<TLength = (string & {}) | 0, TTime = string & {}> {
  alignmentBaseline?: Property.AlignmentBaseline;
  baselineShift?: Property.BaselineShift<TLength>;
  clip?: Property.Clip;
  clipPath?: Property.ClipPath;
  clipRule?: Property.ClipRule;
  color?: Property.Color;
  colorInterpolation?: Property.ColorInterpolation;
  colorRendering?: Property.ColorRendering;
  cursor?: Property.Cursor;
  direction?: Property.Direction;
  display?: Property.Display;
  dominantBaseline?: Property.DominantBaseline;
  fill?: Property.Fill;
  fillOpacity?: Property.FillOpacity;
  fillRule?: Property.FillRule;
  filter?: Property.Filter;
  floodColor?: Property.FloodColor;
  floodOpacity?: Property.FloodOpacity;
  font?: Property.Font;
  fontFamily?: Property.FontFamily;
  fontSize?: Property.FontSize<TLength>;
  fontSizeAdjust?: Property.FontSizeAdjust;
  fontStretch?: Property.FontStretch;
  fontStyle?: Property.FontStyle;
  fontVariant?: Property.FontVariant;
  fontWeight?: Property.FontWeight;
  glyphOrientationVertical?: Property.GlyphOrientationVertical;
  imageRendering?: Property.ImageRendering;
  letterSpacing?: Property.LetterSpacing<TLength>;
  lightingColor?: Property.LightingColor;
  lineHeight?: Property.LineHeight<TLength>;
  marker?: Property.Marker;
  markerEnd?: Property.MarkerEnd;
  markerMid?: Property.MarkerMid;
  markerStart?: Property.MarkerStart;
  mask?: Property.Mask<TLength>;
  opacity?: Property.Opacity;
  overflow?: Property.Overflow;
  paintOrder?: Property.PaintOrder;
  pointerEvents?: Property.PointerEvents;
  shapeRendering?: Property.ShapeRendering;
  stopColor?: Property.StopColor;
  stopOpacity?: Property.StopOpacity;
  stroke?: Property.Stroke;
  strokeDasharray?: Property.StrokeDasharray<TLength>;
  strokeDashoffset?: Property.StrokeDashoffset<TLength>;
  strokeLinecap?: Property.StrokeLinecap;
  strokeLinejoin?: Property.StrokeLinejoin;
  strokeMiterlimit?: Property.StrokeMiterlimit;
  strokeOpacity?: Property.StrokeOpacity;
  strokeWidth?: Property.StrokeWidth<TLength>;
  textAnchor?: Property.TextAnchor;
  textDecoration?: Property.TextDecoration<TLength>;
  textRendering?: Property.TextRendering;
  unicodeBidi?: Property.UnicodeBidi;
  vectorEffect?: Property.VectorEffect;
  visibility?: Property.Visibility;
  whiteSpace?: Property.WhiteSpace;
  wordSpacing?: Property.WordSpacing<TLength>;
  writingMode?: Property.WritingMode;
}

interface Properties<TLength = (string & {}) | 0, TTime = string & {}>
  extends StandardProperties<TLength, TTime>,
    VendorProperties<TLength, TTime>,
    ObsoleteProperties<TLength, TTime>,
    SvgProperties<TLength, TTime> {}

type Globals = "-moz-initial" | "inherit" | "initial" | "revert" | "unset";

declare namespace Property {
  export type AlignContent = Globals | DataType.ContentDistribution | DataType.ContentPosition | "baseline" | "normal" | (string & {});

  export type AlignItems = Globals | DataType.SelfPosition | "baseline" | "normal" | "stretch" | (string & {});

  export type AlignSelf = Globals | DataType.SelfPosition | "auto" | "baseline" | "normal" | "stretch" | (string & {});

  export type AlignTracks = Globals | DataType.ContentDistribution | DataType.ContentPosition | "baseline" | "normal" | (string & {});

  export type All = Globals;

  export type Animation<TTime = string & {}> = Globals | DataType.SingleAnimation<TTime> | (string & {});

  export type AnimationDelay<TTime = string & {}> = Globals | TTime | (string & {});

  export type AnimationDirection = Globals | DataType.SingleAnimationDirection | (string & {});

  export type AnimationDuration<TTime = string & {}> = Globals | TTime | (string & {});

  export type AnimationFillMode = Globals | DataType.SingleAnimationFillMode | (string & {});

  export type AnimationIterationCount = Globals | "infinite" | (string & {}) | (number & {});

  export type AnimationName = Globals | "none" | (string & {});

  export type AnimationPlayState = Globals | "paused" | "running" | (string & {});

  export type AnimationTimingFunction = Globals | DataType.EasingFunction | (string & {});

  export type Appearance = Globals | DataType.CompatAuto | "auto" | "menulist-button" | "none" | "textfield";

  export type AspectRatio = Globals | "auto" | (string & {});

  export type Azimuth =
    | Globals
    | "behind"
    | "center"
    | "center-left"
    | "center-right"
    | "far-left"
    | "far-right"
    | "left"
    | "left-side"
    | "leftwards"
    | "right"
    | "right-side"
    | "rightwards"
    | (string & {});

  export type BackdropFilter = Globals | "none" | (string & {});

  export type BackfaceVisibility = Globals | "hidden" | "visible";

  export type Background<TLength = (string & {}) | 0> = Globals | DataType.FinalBgLayer<TLength> | (string & {});

  export type BackgroundAttachment = Globals | DataType.Attachment | (string & {});

  export type BackgroundBlendMode = Globals | DataType.BlendMode | (string & {});

  export type BackgroundClip = Globals | DataType.Box | (string & {});

  export type BackgroundColor = Globals | DataType.Color;

  export type BackgroundImage = Globals | "none" | (string & {});

  export type BackgroundOrigin = Globals | DataType.Box | (string & {});

  export type BackgroundPosition<TLength = (string & {}) | 0> = Globals | DataType.BgPosition<TLength> | (string & {});

  export type BackgroundPositionX<TLength = (string & {}) | 0> = Globals | TLength | "center" | "left" | "right" | "x-end" | "x-start" | (string & {});

  export type BackgroundPositionY<TLength = (string & {}) | 0> = Globals | TLength | "bottom" | "center" | "top" | "y-end" | "y-start" | (string & {});

  export type BackgroundRepeat = Globals | DataType.RepeatStyle | (string & {});

  export type BackgroundSize<TLength = (string & {}) | 0> = Globals | DataType.BgSize<TLength> | (string & {});

  export type BlockOverflow = Globals | "clip" | "ellipsis" | (string & {});

  export type BlockSize<TLength = (string & {}) | 0> =
    | Globals
    | TLength
    | "-moz-fit-content"
    | "-moz-max-content"
    | "-moz-min-content"
    | "-webkit-fill-available"
    | "auto"
    | "fit-content"
    | "max-content"
    | "min-content"
    | (string & {});

  export type Border<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength> | DataType.LineStyle | DataType.Color | (string & {});

  export type BorderBlock<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength> | DataType.LineStyle | DataType.Color | (string & {});

  export type BorderBlockColor = Globals | DataType.Color | (string & {});

  export type BorderBlockEnd<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength> | DataType.LineStyle | DataType.Color | (string & {});

  export type BorderBlockEndColor = Globals | DataType.Color;

  export type BorderBlockEndStyle = Globals | DataType.LineStyle;

  export type BorderBlockEndWidth<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength>;

  export type BorderBlockStart<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength> | DataType.LineStyle | DataType.Color | (string & {});

  export type BorderBlockStartColor = Globals | DataType.Color;

  export type BorderBlockStartStyle = Globals | DataType.LineStyle;

  export type BorderBlockStartWidth<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength>;

  export type BorderBlockStyle = Globals | DataType.LineStyle;

  export type BorderBlockWidth<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength>;

  export type BorderBottom<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength> | DataType.LineStyle | DataType.Color | (string & {});

  export type BorderBottomColor = Globals | DataType.Color;

  export type BorderBottomLeftRadius<TLength = (string & {}) | 0> = Globals | TLength | (string & {});

  export type BorderBottomRightRadius<TLength = (string & {}) | 0> = Globals | TLength | (string & {});

  export type BorderBottomStyle = Globals | DataType.LineStyle;

  export type BorderBottomWidth<TLength = (string & {}) | 0> = Globals | DataType.LineWidth<TLength>;

  export type BorderCollapse 